/*
 * Copyright 2021 Palantir Technologies, Inc. All rights reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// @ts-check

import { writeFileSync } from "node:fs";
import { join, resolve } from "node:path";
import { fileURLToPath } from "node:url";

export const scriptsDir = fileURLToPath(new URL(".", import.meta.url));
export const iconResourcesDir = resolve(scriptsDir, "../../../resources/icons");
export const generatedSrcDir = resolve(scriptsDir, "../src/generated-icons");
export const NS = "bp4";

/**
 * @typedef {Object} IconMetadata
 * @property {string} displayName - "Icon name" for display
 * @property {string} iconName - `icon-name` for IconName and CSS class
 * @property {string} tags - comma separated list of tags describing this icon
 * @property {string} group - group to which this icon belongs
 * @property {number} codepoint - icon font codepoint
 */

const { default: iconsMetadataJson } = await import("../icons.json", { assert: { type: "json" }});

/** @type {IconMetadata[]} */
export const iconsMetadata = iconsMetadataJson.sort(
    (a, b) => a.iconName.localeCompare(b.iconName),
);

/**
 * Writes lines to given filename in GENERATED_SRC_DIR.
 *
 * @param {string} filename
 * @param {Array<string>} lines
 */
export function writeLinesToFile(filename, ...lines) {
    const outputPath = join(generatedSrcDir, filename);
    const contents = [...lines, ""].join("\n");
    writeFileSync(outputPath, contents);
}
