/***********************************************/
/**
* @file tidesOceanPole.h
*
* @brief Ocean pole tide.
* Pole tides is generated by the centrifugal effect of polar motion on the oceans.
* @see Tides
*
* @author Torsten Mayer-Guerr
* @date 2014-05-23
*
*/
/***********************************************/

#ifndef __GROOPS_TIDESOCEANPOLE__
#define __GROOPS_TIDESOCEANPOLE__

// Latex documentation
#ifdef DOCSTRING_Tides
static const char *docstringTidesOceanPole = R"(
\subsection{OceanPoleTide}\label{tidesType:oceanPoleTide}
The ocean pole tide is generated by the centrifugal effect of polar motion on the oceans.
The potential coefficients of this effect is given by
IERS2003 conventions are given by
\begin{equation}
\begin{Bmatrix}
\Delta c_{nm}  \\
\Delta s_{nm}
\end{Bmatrix}=
\begin{Bmatrix}
c_{nm}^R  \\
s_{nm}^R
\end{Bmatrix}
(m_1\gamma^R+m_2\gamma^I)+
\begin{Bmatrix}
c_{nm}^I  \\
s_{nm}^I
\end{Bmatrix}
(m_2\gamma^R-m_1\gamma^I)
\end{equation}
where the coefficients are read from file \configFile{inputfileOceanPole}{oceanPoleTide},
$\gamma=\gamma^R+i\gamma^I$ is given by \config{gammaReal} and
\config{gammaImaginary} and $(m_1,m_2)$ are the wobble variables in radians.
They are related to the polar motion variables $(x_p,y_p)$ according to
\begin{equation}
\begin{split}
m_1 &=  (x_p - \bar{x}_p), \\
m_2 &= -(y_p - \bar{y}_p),
\end{split}
\end{equation}
The mean pole $(\bar{x}_p, \bar{y}_p)$ is approximated by a polynomial
read from \configFile{inputfileMeanPole}{meanPolarMotion}.

The computed result is multiplied with \config{factor}.
)";
#endif

/***********************************************/

#include "files/fileMeanPolarMotion.h"
#include "classes/earthRotation/earthRotation.h"
#include "classes/tides/tides.h"

/***** CLASS ***********************************/

/** @brief Ocean pole tide.
* @ingroup tidesGroup
* Pole tides is generated by the centrifugal effect of polar motion on the oceans.
* @see Tides */
class TidesOceanPole : public TidesBase
{
  Double              gammaR,    gammaI;
  SphericalHarmonics  harmReal, harmImag;
  MeanPolarMotion     meanPole;

  void pole(const Time &time, EarthRotationPtr rotation, Double &m1, Double &m2) const;

public:
  TidesOceanPole(Config &config);

  SphericalHarmonics sphericalHarmonics(const Time &time, const Rotary3d &rotEarth, EarthRotationPtr rotation, EphemeridesPtr ephemerides,
                                        UInt maxDegree, UInt minDegree, Double GM, Double R) const override;
  void deformation(const std::vector<Time> &time, const std::vector<Vector3d> &point, const std::vector<Rotary3d> &rotEarth,
                   EarthRotationPtr rotation, EphemeridesPtr ephemerides, const std::vector<Double> &gravity, const Vector &hn, const Vector &ln,
                   std::vector<std::vector<Vector3d>> &disp) const override;
};

/***********************************************/

#endif /* __GROOPS_TIDES__ */
