#include <qmessagebox.h>
#include <qtimer.h>
#include <kapp.h>
#include <kconfig.h>
#include <kdebug.h>
#include "khglobals.h"
#include "khfeatureprofile.h"
#include "khchip.h"

KHChip::KHChip( sensors_chip_name chipname, KHFeatureProfile *profile ) {

	//debug( "entering KHChip" );
	this->chipname = chipname;
	this->profile = profile;

	// enable system beep for sensor alarms
	writeData( GlobalBeep, 1 );
	// initially turn off all beeps
	writeData( BeepControl, 0 );

	config = kapp->getConfig();	

	// set all interrupts to 0 -> no critical state monitored yet
	for ( int i = 0; i < SensorIdSize; i++ )interruptStatus[i] = 0;
	
  // initially parse configuration and configure the chip
	
	parseFaultsSettings();
	parseBeepSettings();
	parseAlarmSettings();
	//debug("leaving KHC constr");
}

KHChip::~KHChip() {
	KDEBUG( KDEBUG_FATAL, 0, "Deleting KHChip" );
	// stop timer and reset chip to factory defaults
	timer->stop();
	delete timer;
	delete profile;
}


/** sets the low limit of the specified sensor to value */
void KHChip::setAlarm( int value, SensorId sid ){
	int ret = 0;

	switch ( sid ) {
		
		case Fan1:
		case Fan2:
		case Fan3:
			ret = sensors_set_feature( chipname, profile->minID( sid ), value );	
		break;
		case Temp1:
		case Temp2:
		case Temp3:
			ret = sensors_set_feature( chipname, profile->minID( sid ), value );	
			ret = sensors_set_feature( chipname, profile->maxID( sid ), value );	
		break;
		default:
			// the other Sensors have hi
			// and lo limits -> according
			// functions apply here
		break;
	}
	if ( ret ) debug( "KHChip: setting alarm for sensor %i failed", sid );
}

void KHChip::setSensorType( int type, SensorId sid ){

	switch ( sid ){
		case Temp1:
			sid = Sensor1;
		break;
		case Temp2:
			sid = Sensor2;
		break;
		case Temp3:
			sid = Sensor3;
		break;
		default:
			return;
		break;
	}

	switch ( type ){
		case 0:
			sensors_set_feature( chipname, profile->featureID( sid ), 3435 );
		break;
		case 1:
			sensors_set_feature( chipname, profile->featureID( sid ), 2 );
		break;
		case 2:
			sensors_set_feature( chipname, profile->featureID( sid ), 1 );
		break;
	}
}

void KHChip::switchAlarm( bool enabled, SensorId sid ) {
	
	int thisbeep = profile->alarmID( sid );
  double beeps;

	//debug( "alarm ID: %i", thisbeep );
	
	if ( enabled ) {
		beeps = readData( BeepControl );
		beeps = thisbeep | static_cast<int>(beeps);
		writeData( BeepControl, beeps );
	} else {
		beeps = readData( BeepControl );
		beeps = ~thisbeep & static_cast<int>(beeps);
		writeData( BeepControl, beeps );
	}
}


/** reads interrupt status registers and sets the software interrupts
	* if faults have been detected
	*	@return true if any interrupt is set
	*/
bool KHChip::reportInterruptStatus(){
	
	bool interrupt_set = false;
	double cur;
	double min;
	double max;

  if ( profile->checkFeature( Temp1 ) &&
      !sensors_get_feature( chipname, profile->featureID( Temp1 ), &cur) &&
      !sensors_get_feature( chipname, profile->maxID( Temp1 ), &max)) {
				if      ( !interruptStatus[Temp1] 	&&  ( cur > max 	) ) interruptStatus[Temp1] = 1;
				else if ( interruptStatus[Temp1] 	&& 	( cur > max 	) ) interruptStatus[Temp1]++;
				else if ( interruptStatus[Temp1] 	&&  ( cur <= max 	) ) interruptStatus[Temp1] = 0;
	};
	//debug( "%f\t%f\t%f", cur, min, max );

	if ( profile->checkFeature( Temp2 ) &&
      !sensors_get_feature( chipname, profile->featureID( Temp2 ), &cur) &&
      !sensors_get_feature( chipname, profile->maxID( Temp2 ), &max)) {
				if      ( !interruptStatus[Temp2] 	&&  ( cur > max 	) ) interruptStatus[Temp2] = 1;
				else if ( interruptStatus[Temp2] 	&& 	( cur > max		) ) interruptStatus[Temp2]++;
				else if ( interruptStatus[Temp2] 	&&  ( cur <= max 	) ) interruptStatus[Temp2] = 0;
	};
	//debug( "%f\t%f\t%f", cur, min, max );

	if ( profile->checkFeature( Temp3 ) &&
      !sensors_get_feature( chipname, profile->featureID( Temp3 ), &cur) &&
      !sensors_get_feature( chipname, profile->maxID( Temp3 ), &max)) {
				if      ( !interruptStatus[Temp3] 	&&  ( cur > max 	) ) interruptStatus[Temp3] = 1;
				else if ( interruptStatus[Temp3] 	&& 	( cur > max 	) ) interruptStatus[Temp3]++;
				else if ( interruptStatus[Temp3] 	&&  ( cur <= max 	) ) interruptStatus[Temp3] = 0;
	};
	//debug( "%f\t%f\t%f", cur, min, max );

	if ( profile->checkFeature( Fan1 ) &&
      !sensors_get_feature( chipname, profile->featureID( Fan1 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( Fan1 ), &min)) {
				if      ( !interruptStatus[Fan1] 	&&  ( cur < min 	) ) interruptStatus[Fan1] = 1;
				else if ( interruptStatus[Fan1] 	&& 	( cur < min 	) ) interruptStatus[Fan1]++;
				else if ( interruptStatus[Fan1] 	&&  ( cur >= min 	) ) interruptStatus[Fan1] = 0;
	};
	//debug( "%f\t%f\t%f", cur, min, max );

	if ( profile->checkFeature( Fan2 ) &&
      !sensors_get_feature( chipname, profile->featureID( Fan2 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( Fan2 ), &min)) {
				if      ( !interruptStatus[Fan2] 	&&  ( cur < min 	) ) interruptStatus[Fan2] = 1;
				else if ( interruptStatus[Fan2] 	&& 	( cur < min 	) ) interruptStatus[Fan2]++;
				else if ( interruptStatus[Fan2] 	&&  ( cur >= min 	) ) interruptStatus[Fan2] = 0;
	};
	//debug( "%f\t%f\t%f", cur, min, max );

	if ( profile->checkFeature( Fan3 ) &&
      !sensors_get_feature( chipname, profile->featureID( Fan3 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( Fan3 ), &min)) {
				if      ( !interruptStatus[Fan3] 	&&  ( cur < min 	) ) interruptStatus[Fan3] = 1;
				else if ( interruptStatus[Fan3] 	&& 	( cur < min 	) ) interruptStatus[Fan3]++;
				else if ( interruptStatus[Fan3] 	&&  ( cur >= min 	) ) interruptStatus[Fan3] = 0;
	};
	//debug( "%f\t%f\t%f", cur, min, max );

	if ( profile->checkFeature( VCore ) &&
      !sensors_get_feature( chipname, profile->featureID( VCore ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( VCore ), &min) 		&&
      !sensors_get_feature( chipname, profile->maxID( VCore ), &max)) {
				if      ( !interruptStatus[VCore] 	&&  ( ( cur < min ) || ( cur > max ) ) ) interruptStatus[VCore] = 1;
				else if ( interruptStatus[VCore] 	&& 	( ( cur < min	) || ( cur > max ) ) ) interruptStatus[VCore]++;
				else if ( interruptStatus[VCore] 	&&  ( ( cur >= min ) && ( cur <= max ) ) ) interruptStatus[VCore] = 0;
	};

	if ( profile->checkFeature( VCore2 ) &&
      !sensors_get_feature( chipname, profile->featureID( VCore2 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( VCore2 ), &min) 		&&
      !sensors_get_feature( chipname, profile->maxID( VCore2 ), &max)) {
				if      ( !interruptStatus[VCore2] 	&&  ( ( cur < min ) || ( cur > max ) ) ) interruptStatus[VCore2] = 1;
				else if ( interruptStatus[VCore2] 	&& 	( ( cur < min	) || ( cur > max ) ) ) interruptStatus[VCore2]++;
				else if ( interruptStatus[VCore2] 	&&  ( ( cur >= min ) && ( cur <= max ) ) ) interruptStatus[VCore2] = 0;
	};

	if ( profile->checkFeature( V33 ) &&
      !sensors_get_feature( chipname, profile->featureID( V33 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( V33 ), &min) 		&&
      !sensors_get_feature( chipname, profile->maxID( V33 ), &max)) {
				if      ( !interruptStatus[V33] 	&&  ( ( cur < min ) || ( cur > max ) ) ) interruptStatus[V33] = 1;
				else if ( interruptStatus[V33] 	&& 	( ( cur < min	) || ( cur > max ) ) ) interruptStatus[V33]++;
				else if ( interruptStatus[V33] 	&&  ( ( cur >= min ) && ( cur <= max ) ) ) interruptStatus[V33] = 0;
	};

	if ( profile->checkFeature( V5 ) &&
      !sensors_get_feature( chipname, profile->featureID( V5 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( V5 ), &min) 		&&
      !sensors_get_feature( chipname, profile->maxID( V5 ), &max)) {
				if      ( !interruptStatus[V5] 	&&  ( ( cur < min ) || ( cur > max ) ) ) interruptStatus[V5] = 1;
				else if ( interruptStatus[V5] 	&& 	( ( cur < min	) || ( cur > max ) ) ) interruptStatus[V5]++;
				else if ( interruptStatus[V5] 	&&  ( ( cur >= min ) && ( cur <= max ) ) ) interruptStatus[V5] = 0;
	};

	if ( profile->checkFeature( Vm5 ) &&
      !sensors_get_feature( chipname, profile->featureID( Vm5 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( Vm5 ), &min) 		&&
      !sensors_get_feature( chipname, profile->maxID( Vm5 ), &max)) {
				if      ( !interruptStatus[Vm5] 	&&  ( ( cur < min ) || ( cur > max ) ) ) interruptStatus[Vm5] = 1;
				else if ( interruptStatus[Vm5] 	&& 	( ( cur < min	) || ( cur > max ) ) ) interruptStatus[Vm5]++;
				else if ( interruptStatus[Vm5] 	&&  ( ( cur >= min ) && ( cur <= max ) ) ) interruptStatus[Vm5] = 0;
	};

	if ( profile->checkFeature( V12 ) &&
      !sensors_get_feature( chipname, profile->featureID( V12 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( V12 ), &min) 		&&
      !sensors_get_feature( chipname, profile->maxID( V12 ), &max)) {
				if      ( !interruptStatus[V12] 	&&  ( ( cur < min ) || ( cur > max ) ) ) interruptStatus[V12] = 1;
				else if ( interruptStatus[V12] 	&& 	( ( cur < min	) || ( cur > max ) ) ) interruptStatus[V12]++;
				else if ( interruptStatus[V12] 	&&  ( ( cur >= min ) && ( cur <= max ) ) ) interruptStatus[V12] = 0;
	};

	if ( profile->checkFeature( Vm12 ) &&
      !sensors_get_feature( chipname, profile->featureID( Vm12 ), &cur) &&
      !sensors_get_feature( chipname, profile->minID( Vm12 ), &min) 		&&
      !sensors_get_feature( chipname, profile->maxID( Vm12 ), &max)) {
				if      ( !interruptStatus[Vm12] 	&&  ( ( cur < min ) || ( cur > max ) ) ) interruptStatus[Vm12] = 1;
				else if ( interruptStatus[Vm12] 	&& 	( ( cur < min	) || ( cur > max ) ) ) interruptStatus[Vm12]++;
				else if ( interruptStatus[Vm12] 	&&  ( ( cur >= min ) && ( cur <= max ) ) ) interruptStatus[Vm12] = 0;
	};

	for ( SensorId sid = Fan1; sid < SensorIdSize; sid++ ) {
		if ( ( interruptStatus[sid] == Faults[sid] ) 	|| ( interruptStatus[sid] 	% ( NotificationInterval / 2 ) == Faults[sid] ) ) emit reportCond( sid, SensorRed );
		else
	  if ( !interruptStatus[sid] ) emit reportCond( sid, SensorGreen );
		if ( interruptStatus[sid] )	interrupt_set = true;
	}

	return interrupt_set;
}


/** switches system beep to the assigned value */
void KHChip::toggleUserBeep( bool enabled ) const {
	
}

void KHChip::setFaultsCount( int value, SensorId sid ){
	Faults[sid] = value;
}

int KHChip::setLoLim( int value, SensorId sid ){
	int ret;

  ret = sensors_set_feature( chipname, profile->minID( sid ), value / 10.0 );
	if ( ret ) debug( "%i: setting lo lim failed with message: %s", sid, sensors_strerror( ret ) );
	return ret;
}

int KHChip::setHiLim( int value, SensorId sid ){
  int ret = 0;

	ret = sensors_set_feature( chipname, profile->maxID( sid ), value / 10.0 );
	if ( ret ) debug( "%i: setting hi lim failed with message: %s", sid, sensors_strerror( ret ) );
	return ret;

}

void KHChip::start(){

	threadProc(); // initial display of values
	timer = new QTimer( this );
	connect( timer, SIGNAL( timeout() ),
						this, SLOT	( threadProc() ) );
	timer->start( 2000, FALSE );

}

void KHChip::threadProc( void ){

	static int i = 0;
	i++;

	//debug( "doing turn %i", i );

	//debug( "Register 0x56 set to 0x%X\n", readRegister( 0x56 ) );

	parseGeneralSettings();
	parseSensorSettings();

	emit newDataAcquired( readData( VCore ), VCore );
	emit newDataAcquired( readData( VCore2 ), VCore2 );
	emit newDataAcquired( readData( V33 ), V33 );
	emit newDataAcquired( readData( V5 ), V5  );
	emit newDataAcquired( readData( V12 ), V12 );
	emit newDataAcquired( readData( Vm12 ), Vm12 );
	emit newDataAcquired( readData( Vm5 ), Vm5 );
  emit newDataAcquired( readData( VID ), VID );
	emit newDataAcquired( static_cast<int>( readData( Temp1 ) ), Temp1 );
	emit newDataAcquired( static_cast<int>( readData( Fan1 ) ), Fan1 );
	emit newDataAcquired( static_cast<int>( readData( Fan2 ) ), Fan2 );
	emit newDataAcquired( static_cast<int>( readData( Fan3 ) ), Fan3 );
	emit newDataAcquired( static_cast<int>( readData( Temp2 ) ), Temp2 );
	emit newDataAcquired( static_cast<int>( readData( Temp3 ) ), Temp3 );
	reportInterruptStatus();
  emit threadCompleted();	// signal for redisplaying the widget
}

void KHChip::parseBeepSettings(){

	config->setGroup( "Beeping" );
	switchAlarm( config->readBoolEntry( BeepKeys[Fan1] ), Fan1 );
	switchAlarm( config->readBoolEntry( BeepKeys[Fan2] ), Fan2 );
	switchAlarm( config->readBoolEntry( BeepKeys[Fan3] ),	Fan3 );
	switchAlarm( config->readBoolEntry( BeepKeys[Temp1] ), Temp1 );
	switchAlarm( config->readBoolEntry( BeepKeys[Temp2] ), Temp2 );
	switchAlarm( config->readBoolEntry( BeepKeys[Temp3] ), Temp3 );
	switchAlarm( config->readBoolEntry( BeepKeys[VCore] ), VCore );
	switchAlarm( config->readBoolEntry( BeepKeys[VCore2] ), VCore2 );
	switchAlarm( config->readBoolEntry( BeepKeys[V33] ), V33 );
	switchAlarm( config->readBoolEntry( BeepKeys[V5] ), V5 );
	switchAlarm( config->readBoolEntry( BeepKeys[Vm5] ), Vm5 );
	switchAlarm( config->readBoolEntry( BeepKeys[V12] ), V12 );
	switchAlarm( config->readBoolEntry( BeepKeys[Vm12] ), Vm12 );
}

void KHChip::parseAlarmSettings(){
	
	config->setGroup( "Alarms" );
	setAlarm( config->readNumEntry( AlarmKeysLo[Fan1] ), Fan1 );
	setAlarm( config->readNumEntry( AlarmKeysLo[Fan2] ), Fan2 );
	setAlarm( config->readNumEntry( AlarmKeysLo[Fan3] ), Fan3 );
	setAlarm( config->readNumEntry( AlarmKeysHi[Temp1] ), Temp1 );
	setAlarm( config->readNumEntry( AlarmKeysHi[Temp2] ), Temp2 );
	setAlarm( config->readNumEntry( AlarmKeysHi[Temp3] ), Temp3 );
	setLoLim( config->readNumEntry( AlarmKeysLo[VCore] ), VCore );
	setHiLim( config->readNumEntry( AlarmKeysHi[VCore] ), VCore );
	setLoLim( config->readNumEntry( AlarmKeysLo[VCore2] ), VCore2 );
	setHiLim( config->readNumEntry( AlarmKeysHi[VCore2] ), VCore2 );
	setLoLim( config->readNumEntry( AlarmKeysLo[V33] ), V33 );
	setHiLim( config->readNumEntry( AlarmKeysHi[V33] ), V33 );
	setLoLim( config->readNumEntry( AlarmKeysLo[V5] ), V5 );
	setHiLim( config->readNumEntry( AlarmKeysHi[V5] ), V5 );
	setLoLim( config->readNumEntry( AlarmKeysLo[Vm5] ), Vm5 );
	setHiLim( config->readNumEntry( AlarmKeysHi[Vm5] ), Vm5 );
	setLoLim( config->readNumEntry( AlarmKeysLo[V12] ), V12 );
	setHiLim( config->readNumEntry( AlarmKeysHi[V12] ), V12 );
	setLoLim( config->readNumEntry( AlarmKeysLo[Vm12] ), Vm12 );
	setHiLim( config->readNumEntry( AlarmKeysHi[Vm12] ), Vm12 );
}


void KHChip::parseGeneralSettings(){
	config->setGroup( "General" );
	NotificationInterval = config->readNumEntry( GeneralKeys[0] );
}

void KHChip::parseFaultsSettings(){

	config->setGroup( "NumFaults" );
	Faults[Fan1] = config->readNumEntry	( FaultsKeys[Fan1] );
	Faults[Fan2] = config->readNumEntry	( FaultsKeys[Fan2] );
	Faults[Fan3] = config->readNumEntry	( FaultsKeys[Fan3] );
	Faults[Temp1] = config->readNumEntry( FaultsKeys[Temp1] );
	Faults[Temp2] = config->readNumEntry( FaultsKeys[Temp2] );
	Faults[Temp3] = config->readNumEntry( FaultsKeys[Temp3] );
	Faults[VCore] = config->readNumEntry( FaultsKeys[VCore] );
	Faults[VCore2] = config->readNumEntry( FaultsKeys[VCore2] );
	Faults[V33] = config->readNumEntry	( FaultsKeys[V33] );
	Faults[V5] = config->readNumEntry		( FaultsKeys[V5] );
	Faults[Vm5] = config->readNumEntry	( FaultsKeys[Vm5] );
	Faults[V12] = config->readNumEntry	( FaultsKeys[V12] );
	Faults[Vm12] = config->readNumEntry	( FaultsKeys[Vm12] );
}

void KHChip::parseSensorSettings(){

	config->setGroup( "SensorTypes" );
	setSensorType( config->readNumEntry( SensorKeys[0] ), Temp1 );
	setSensorType( config->readNumEntry( SensorKeys[1] ), Temp2 );
	setSensorType( config->readNumEntry( SensorKeys[2] ), Temp3 );
}

float KHChip::readData( SensorId sid ){
	double value;

	if ( profile->checkFeature( sid ) ) {
		if ( !sensors_get_feature( chipname, profile->featureID( sid ), &value ) )
			return static_cast<float>( value );
	  else {
			// if feature seems to exist, but cannot be read, return 0
			// and report error
			debug("KHChip: Internal error: Feature %i not existing in this chip", sid);
			return 0;
		}
	}
	return 0; // if feature doesn't exist, return 0
}

void KHChip::writeData( SensorId sid, double data ){
	if ( profile->checkFeature( sid ) )
		sensors_set_feature( chipname, profile->featureID( sid ), data );
  else debug("Internal error: Feature %i not existing in this chip", sid);
}

























