/***************************************************************************
                          winbond.h  -  description                              
                             -------------------                                         
    begin                : Sun Jul 18 1999                                           
    copyright            : (C) 1999 by Ruediger.Anlauf                         
    email                : Ruediger.Anlauf@gmx.net                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef KHCHIP_H
#define KHCHIP_H

#include <math.h>
extern "C" {
#include <sys/io.h>
}
#include <sensorid.h>
#include <qobject.h>
#include <sensors.h>

class QMessageBox;
class QTimer;
class KConfig;
class KHFeatureProfile;

typedef unsigned char byte;

/** Winbond is where all the communication with the Winbond chip is done.
	* A thread runs periodically and checks for new data the healthcare chip has
	* recorded. newDataAcquired () is emitted, containing the new data on each thread run.
	* In addition, threadCompleted () is emitted whenever a run of the thread completes.
	*	You may use the class's slots to set the values at which the chip will ring the alarm bell
	* for each monitored item.
	* @short does the communication (I/O) with the Winbond chip.
  * @author Ruediger Anlauf
  */
class KHChip : public QObject {

Q_OBJECT

protected: // Protected attributes

  /** current interrupt status for each sensor
		*
    *
		* counts, for how many runs of the thread the sensor has had a critical state
		* set by reportInterruptStatus
		* is reset to zero if no critical condition can be detected anymore
		* @see #reportInterruptStatus
  	*/
	unsigned int interruptStatus[SensorIdSize]; // SensorIdSize is global

  /** timer which controlls the I/O thread;
		* the I/O thread is called on each timeout() signal of timer,
		* which is every 500 ms
		*/
  QTimer * timer;

	/** pointer to the configuration object of this application */
	KConfig *config;

	/** interval in seconds in which condRed() for each monitored item is emitted */
	unsigned int NotificationInterval;	

	/** number of consecutive faults to be detected before the alarm signal is emitted */
	unsigned int Faults[SensorIdSize];

	/** contains which sensors to monitor */
	bool Monitors[SensorIdSize];

	sensors_chip_name chipname;

	KHFeatureProfile *profile;

public:
	/** constructor */
	KHChip( sensors_chip_name chipname, KHFeatureProfile *profile );

	/** destructor
		*
		*
		* stops and deletes the thread control timer, resets the Winbond chip to
		* power on defaults and releases I/O access to the chip's ports
		* @see #timer ,
		* @see #requestIOPermission
	  */
	~KHChip();

	/** starts the I/O thread via the controlling timer
		* @see #timer
		*/
  void start();

protected:
	/** sets the type of the thermal sensor
		* @param type 0 for thermistor, 1 for transistor, 2 for PII diode
		*/
	void setSensorType( int type, SensorId sid );

  /** switches system beep to the assigned value */
  void toggleUserBeep( bool enabled ) const ;

  /** reads interrupt status registers
			true, if interrupt has been set */
  bool reportInterruptStatus();

	/** parses the config file and sets the beep control flags accordingly
	  */
	void parseBeepSettings();

	/** parses the config file and sets the registers for high and low limits accordingly
	  */
	void parseAlarmSettings();

	/** parses the config file for general settings, as far as this class is
		* concerned, and sets the variables to the correct values
		*/
	void parseGeneralSettings();
	void parseFaultsSettings();
	void parseSensorSettings();

public slots:
	/** sets the low limit of the specified sensor to value */
	void setAlarm( int value, SensorId sid );

	/** sets the low limit of the specified sensor to value */
	int setLoLim( int value, SensorId sid );

  /** sets the high limit of the specified sensor to value */
	int setHiLim( int value, SensorId sid );

	/** switches the beeping of the specified sensor to enabled */
	void switchAlarm( bool enabled, SensorId sid );

	/** set the number of faults to count before alarm triggers */
	void setFaultsCount( int value, SensorId sid );

	float readData( SensorId );

	void writeData( SensorId, double data );

protected slots:

	/** the workhorse of the class
		*
		*
		* ThreadProc parses the user settings in the main window and config dialog
		* each time it runs and sets the chip's configuration registers to the user's choices.
		* Additionally, it reads the readout of those registers monitoring the system.
		* For each readout, the appropriate signal is emitted, containing
		* the actual value of the sensor.
		*
		*	This function is protected, because it is not meant for direct use. Call @ref #start ()
		*	instead to start the thread. It terminates on destruction of the object.
		* @see #start()
		*/
	void threadProc( void );

signals:
	/** emitted whenever a thread run has finished */
	void threadCompleted();

	/** emits the actual value and sensor ID for a specific sensor which has just been read.
		* this int method is meant for the fans and temperatures
		*/
	void newDataAcquired( int, SensorId );

	/** overloaded method for conveniance; meant for the voltages
		* @see #newDataAcquired ()
		*/
	void newDataAcquired( float, SensorId );

	/** emitted when a sensor is reporting a critical condition according to the user's settings */
	void reportCond( SensorId, SensorStatus );
};

#endif






































