/* -*- C -*-
 * This source file is part of the libinetd package.
 * 
 * Copyright (C) 1988 Jrgen Sigvardsson / jorgen@linux.nu
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA
 *
 * $Id: inetd.c,v 1.11 1998/05/28 18:39:46 jorgen Exp $
 * $Log: inetd.c,v $
 * Revision 1.11  1998/05/28 18:39:46  jorgen
 * inetd_AddService semantics changed. inetd_Find, inetd_RemoveService semantics
 * also changed. Added support for comment-meta information in inetd.conf.
 *
 * Revision 1.10  1998/05/24 01:29:43  jorgen
 * Added inetd_CreateEmptyConf()
 *
 * Revision 1.9  1998/05/23 18:20:34  jorgen
 * Moved the inetd_XXXName() macros to inetd.h
 *
 * Revision 1.8  1998/05/23 18:18:58  jorgen
 * Renamed structs xxx to xxx_ in order to keep g++ happy. (G++ did not like
 * the forward references)
 *
 * Revision 1.7  1998/05/23 13:50:24  jorgen
 * More bugfixes - inetd_CurrentService() fixed.
 *
 * Revision 1.6  1998/05/23 13:26:48  jorgen
 * Bugfixes in inetd_AddService().
 *
 * Revision 1.5  1998/05/23 13:11:27  jorgen
 * Added inetd_RemoveService() and changed the semantics for inetd_AddService().
 *
 * Revision 1.4  1998/05/23 10:55:51  jorgen
 * Beta-ready
 *
 * Revision 1.3  1998/05/22 19:06:52  jorgen
 * inetd_Find() and inetd_Free() now works.
 *
 * Revision 1.2  1998/05/22 18:53:55  jorgen
 * inetd_ReadConf() now reads in /etc/inetd.conf and inetd_Flush() now flushes
 * the inetd information correctly.
 *
 * Revision 1.1.1.1  1998/05/22 17:46:22  jorgen
 * Initial version
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "inetd.h"
#include "inetd_private.h"
#include "misc.h"

static const char rcsid[] = "$Id: inetd.c,v 1.11 1998/05/28 18:39:46 jorgen Exp $";

/* Constants and macros */
#define BUFSIZE      1024
#define DELIM        "\n\t "
#define ISNULL(x)    ((x) == NULL)

/* private functions */

static int inetd_DetermineSockType(const char* string) {
  if(!strcasecmp(string, "stream"))
    return SOCKTYPE_STREAM;
  else if(!strcasecmp(string, "dgram"))
    return SOCKTYPE_DGRAM;
  else
    return SOCKTYPE_UNKNOWN;
}

static int inetd_DetermineProto(const char* string) {
  if(!strcasecmp(string, "udp"))
    return PROTO_UDP;
  else if(!strcasecmp(string, "tcp"))
    return PROTO_TCP;
  else if(!strcasecmp(string, "rpc/udp"))
    return PROTO_RPCUDP;
  else if(!strcasecmp(string, "rpc/tcp"))
    return PROTO_RPCTCP;
  else
    return PROTO_UNKNOWN;
}

static int inetd_DetermineFlag(const char* string) {
  if(!strcasecmp(string, "wait"))
    return FLAG_WAIT;
  else if(!strcasecmp(string, "nowait"))
    return FLAG_NOWAIT;
  else
    return FLAG_UNKNOWN;
}

static inetd_Service inetd_ParseLine(char* line) {
  char* service_name;
  char* sock_type;
  char* proto;
  char* flags;
  char* user;
  char* server_path;
  char* arg;
  int on;

  inetd_Service aService;

  /* Assume the service is turned on */
  on = 1;
  if(!strncmp(line, OFFMARKER, strlen(OFFMARKER))) {
    line += strlen(OFFMARKER); /* skip marker */
    on = 0; /* turn off service */
  }


  service_name = strtok(line, DELIM);
  sock_type = strtok(NULL, DELIM);
  proto = strtok(NULL, DELIM);
  flags = strtok(NULL, DELIM);
  user = strtok(NULL, DELIM);
  server_path = strtok(NULL, DELIM);

  if(ISNULL(service_name) || ISNULL(sock_type) ||
     ISNULL(proto)        || ISNULL(flags) ||
     ISNULL(user)         || ISNULL(server_path)) {
    return NULL;
  }
  
  aService = (inetd_Service)calloc(1, sizeof(struct inetd_Service_));
  
  aService->service_name = strdup(service_name);
  aService->sock_type = inetd_DetermineSockType(sock_type);
  aService->proto = inetd_DetermineProto(proto);
  aService->flags = inetd_DetermineFlag(flags);
  aService->user = strdup(user);
  aService->server_path = strdup(server_path);
  aService->off = on;
  
  /* parse arguments */
  aService->num_args = 0; /* we assume 0 arguments to begin with */

  while((arg = strtok(NULL, DELIM))) {
    aService->args[aService->num_args++] = strdup(arg);
  }

  return aService;
}

static inetd_Service inetd_ReadService(FILE* fp) {
  char buf[BUFSIZE + 1];
  inetd_Service service;

  while(1) {
    if(!fgets(buf, BUFSIZE, fp)) { /* EOF reached */
      return NULL;
    } 
    
    /* If line was a comment, skip to next line */
    if(buf[0] == '#' && strncmp(buf, OFFMARKER, strlen(OFFMARKER)) != 0)
      continue;
    
    /* if line could not be parsed - bad syntax - skip to next line */
    /* Maybe a warning should be issued here? */
    if(!(service = inetd_ParseLine(buf)))
      continue;
    
    /* If we made it this far, we do have a service. */
    return service;
  }
}

static inetd_ListEl* inetd_ListElCreate(inetd_Service service) {
  inetd_ListEl* el;

  el = (inetd_ListEl*)calloc(1, sizeof(inetd_ListEl));
  el->service = service;
  return el;
}

static void inetd_AppendService(inetd_h handle, inetd_Service service) {
  inetd_List* list;

  list = handle->list;

  if(list->num_elements == 0) {
    list->head = list->tail = inetd_ListElCreate(service);
  } else {
    list->tail->next = inetd_ListElCreate(service);
    list->tail->next->prev = list->tail;
    list->tail = list->tail->next;
  }
  list->num_elements++;
}

static void inetd_FlushService(FILE* fp, inetd_Service serv) {
  char buf[BUFSIZE];
  int i;

  sprintf(buf, "%s\t%s\t%s\t%s\t%s\t%s\t",
	  serv->service_name, inetd_SockTypeName(serv->sock_type),
	  inetd_ProtoName(serv->proto), inetd_FlagName(serv->flags),
	  serv->user, serv->server_path);

  for(i = 0; i < serv->num_args; i++) {
    strcat(buf, serv->args[i]);
    strcat(buf, " ");
  }
  
  fprintf(fp, buf);
}

static void inetd_FreeService(inetd_Service serv) {
  int i;

  if(serv == NULL) return;

  free(serv->service_name);
  free(serv->user);
  free(serv->server_path);

  for(i = 0; i < serv->num_args; i++)
    free(serv->args[i]);

  free(serv);
}


/* Interface functions */

inetd_h inetd_ReadConf() {
  /* Pre = None */
  FILE* fp;
  inetd_h handle;
  inetd_Service service;

  if(!(fp = fopen("/etc/inetd.conf", "r")))
    return NULL;

  handle = (inetd_h)calloc(1, sizeof(struct inetd_h_));
  handle->list = (inetd_List*)calloc(1, sizeof(inetd_List));

  while((service = inetd_ReadService(fp)))
    inetd_AppendService(handle, service);

  fclose(fp);
  return handle;
}

inetd_h inetd_CreateEmptyConf() {
  inetd_h handle;

  handle = (inetd_h)calloc(1, sizeof(struct inetd_h_));
  handle->list = (inetd_List*)calloc(1, sizeof(inetd_List));

  return handle;
}

int inetd_Flush(inetd_h handle, const char* filename) {
  /* Pre = handle != NULL && filename != NULL */
  FILE* fp;
  inetd_ListEl* i;

  if(!(fp = fopen(filename, "w")))
    return -1;

  fprintf(fp, CONF_HEADER);

  for(i = handle->list->head; i != NULL; i = i->next) {
    inetd_FlushService(fp, i->service);
    fprintf(fp, "\n");
  }

  fclose(fp);
  
  return 0;
}

void inetd_Free(inetd_h handle) {
  /* Pre = handle != NULL */
  inetd_ListEl* i, *tmp;
  
  for(i = handle->list->head; i != NULL;) {
    tmp = i->next;
    inetd_FreeService(i->service);
    free(i);
    i = tmp;
  }
  
  free(handle->list);
  free(handle);
}

void inetd_RemoveService(inetd_h handle, inetd_Service service) {
  /* Pre = handle != NULL */
  inetd_ListEl* i;

  for(i = handle->list->head; i != NULL; i = i->next) {
    if(i->service == service) { /* service found - delink element */
      
      if(i == handle->list->tail) {
	if(handle->list->tail->prev)
	  handle->list->tail->prev->next = NULL;
	handle->list->tail = i->prev;
      } else if(i == handle->list->head) {
	if(handle->list->head->next)
	  handle->list->head->next->prev = NULL;
	handle->list->head = i->next;
      } else {
	if(i->next)
	  i->next->prev = i->prev;
	if(i->prev)
	  i->prev->next = i->next;
      }
      handle->list->num_elements--;
      inetd_FreeService(i->service);
      free(i);
      return;
    }
  } 
}

inetd_Service inetd_Find(inetd_h handle, const char* serv_name, int sock_type) {
  /* Pre = handle != NULL */
  inetd_ListEl* i;

  for(i = handle->list->head; i != NULL; i = i->next) {
    if(!strcasecmp(i->service->service_name, serv_name) && 
       i->service->sock_type == sock_type)
      return i->service;
  }

  return NULL;
}

void inetd_AddService(inetd_h handle, inetd_Service s) {
  /* Pre = handle != NULL and s != NULL */
  
  /* First check if service already exists.
   * If so, remove the old service and add the new.
   * If not, just add service.
   */

  inetd_AppendService(handle, s);
}

void inetd_BeginIter(inetd_h handle) {
  /* Pre = handle != NULL */
  handle->iter = handle->list->head;
}

inetd_Service inetd_CurrentService(inetd_h handle) {
  /* Pre handle != NULL && inetd_BeginIter() has been called earlier */
  if(handle->iter)
    return handle->iter->service;
  return NULL;
}

void inetd_Iterate(inetd_h handle) {
  /* Pre handle != NULL */

  handle->iter = handle->iter->next;
}
