/*
    toplevel.cpp - the main view of the KDE VideoTune

    Requires the Qt widget libraries, available at no cost at
    http://www.troll.no

    Copyright (C) 1997 Frithjof Brestrich
                       brestrich@kde.org


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "kvidtune.h"

#include <qpixmap.h>
#include <qpushbt.h>
#include <qtimer.h>
#include <qmsgbox.h>
#include <kiconloader.h>
#include <kmsgbox.h>
#include <kpopmenu.h>
#include <ktabctl.h>

#include "dlginfomonitor.h"
#include "dlginfoxserver.h"
#include "dlginfovideo.h"
#include "dlginfomodeline.h"
#include "dlgadvancedxstartup.h"
#include "dlgoptionsconfig.h"
#include "toplevel.h"
#include "config.h"

/***************************************************************************************/

TopLevel::TopLevel(KVidData *data):KTopLevelWidget()
{
    LifeAdjust  = TRUE;
    vidData     = data;
    Cfg         = data->vidCfg;
    VidMode     = data->xf86vid;

    VML_Akt     = new XF86VidModeLine;
    VidMode->GetModeLine(*VML_Akt);
    VML_Restore = new XF86VidModeLine(*VML_Akt);
    VML_Undo    = new XF86VidModeLine(*VML_Akt);

    setCaption(klocale->translate("Video Configuration for KDE"));

    createMenuBar();
    createToolBar();
    createMainWidget();
    createStatusBar();

    update_modeinfo();
    
    show();
}

TopLevel::~TopLevel()
{
}

void TopLevel::createMenuBar()
{
    menu_file     = new QPopupMenu();
    menu_settings = new QPopupMenu();
    menu_help     = new QPopupMenu();
    
    menu_item_save = menu_file->insertItem(klocale->translate("&Save"), this, SLOT(filesave()));
    menu_file->insertSeparator(-1);
    menu_file->insertItem(klocale->translate("&Exit"), 
                          KApplication::getKApplication(), SLOT(quit()));

    menu_settings->insertItem(klocale->translate("&Configure Program"), 
                              this, SLOT(settingsconfigure()));

    menu_help->insertItem(klocale->translate("&Using KVidTune"), this, SLOT(help()));
    menu_help->insertSeparator(-1);
    menu_help->insertItem(klocale->translate("&About KVidTune..."), this, SLOT(about()));
    menu_help->insertItem(klocale->translate("&About KDE..."), 
                          KApplication::getKApplication(), SLOT(aboutKDE()));
    menu_help->insertItem(klocale->translate("&About QT..."), this, SLOT(aboutQt()));
    
    menubar = new KMenuBar(this);
    menubar->insertItem(klocale->translate("&File"), menu_file);
    menubar->insertItem(klocale->translate("&Settings"), menu_settings);
    menubar->insertSeparator(-1);
    menubar->insertItem(klocale->translate("&Help"), menu_help);
    setMenu(menubar);
}
 

void TopLevel::createToolBar()
{
    KIconLoader IconLoader;
    
    toolbar = new KToolBar(this);
    
    toolbar->insertButton(IconLoader.loadIcon("kvidtune-save.xpm"), ID_TOOLBAR_SAVE, 
                          SIGNAL(clicked()), this, SLOT(filesave()), FALSE /*EnableItem*/, 
			  klocale->translate("Save configuration"));
    toolbar->insertSeparator(-1);
    toolbar->insertButton(IconLoader.loadIcon("help.xpm"), 0, SIGNAL(clicked()), 
                          this, SLOT(help()), TRUE, klocale->translate("Help"));
    
    toolbar->setBarPos(KToolBar::Top);

    addToolBar(toolbar);
}

void TopLevel::createStatusBar()
{
    statusbar = new KStatusBar(this);
    statusbar->insertItem(klocale->translate("No status available"), ID_STATUS_GENERAL);
    statusbar->setInsertOrder(KStatusBar::LeftToRight);
    setStatusBar(statusbar);
}

void TopLevel::createMainWidget()
{
    QWidget *card;

    tabctl = new KTabCtl(this,"MainTab");

    card = createTabVideoMode(tabctl);
    tabctl->addTab(card,klocale->translate("Video mode"));

//    card = createTabCompose(tabctl);
//    tabctl->addTab(card,klocale->translate("Compose new video mode"));

    card = createTabXStartup(tabctl);
    tabctl->addTab(card,klocale->translate("X Startup"));

    card = createTabMonitor(tabctl);
    tabctl->addTab(card,klocale->translate("Monitor"));

    card = createTabX11(tabctl);
    tabctl->addTab(card,klocale->translate("X11"));

//    card = createTabOptions(tabctl);
//    tabctl->addTab(card,klocale->translate("Program options"));

    tabctl->adjustSize();
    tabctl->setFixedSize(tabctl->size());

    setView(tabctl); 
}

QWidget *TopLevel::createTabVideoMode(QWidget *parent)
{
    QWidget *view = new QWidget(tabctl);

    vidtune = new KVidTune(view, NULL);

    vidtune->SetLifeAdjust(LifeAdjust);

    connect(vidtune,SIGNAL(VidVideoMode(int)),    this, SLOT(videomode(int)));
    connect(vidtune,SIGNAL(VidLifeAdjust(bool)),  this, SLOT(vidcmd_lifeadj(bool)));
    connect(vidtune,SIGNAL(VidMoveLeft(int)),     this, SLOT(vidmov_left(int)));
    connect(vidtune,SIGNAL(VidMoveRight(int)),    this, SLOT(vidmov_right(int)));
    connect(vidtune,SIGNAL(VidMoveUp(int)),       this, SLOT(vidmov_up(int)));
    connect(vidtune,SIGNAL(VidMoveDown(int)),     this, SLOT(vidmov_down(int)));
    connect(vidtune,SIGNAL(VidSizeWider(int)),    this, SLOT(vidsiz_hmore(int)));
    connect(vidtune,SIGNAL(VidSizeNarrower(int)), this, SLOT(vidsiz_hless(int)));
    connect(vidtune,SIGNAL(VidSizeTaller(int)),   this, SLOT(vidsiz_vmore(int)));
    connect(vidtune,SIGNAL(VidSizeShorter(int)),  this, SLOT(vidsiz_vless(int)));
    connect(vidtune,SIGNAL(VidActionDetails()),   this, SLOT(vidcmd_details()));
    connect(vidtune,SIGNAL(VidActionModeline()),  this, SLOT(vidcmd_modeline()));
    connect(vidtune,SIGNAL(VidActionUndo()),      this, SLOT(vidcmd_undo()));
    connect(vidtune,SIGNAL(VidActionTest()),      this, SLOT(vidcmd_test()));
    connect(vidtune,SIGNAL(VidActionRestore()),   this, SLOT(vidcmd_restore()));
    connect(vidtune,SIGNAL(VidActionSave()),      this, SLOT(filesave()));
    connect(vidtune,SIGNAL(VidInterlace(bool)),   this, SLOT(vidflg_interlace(bool)));
    connect(vidtune,SIGNAL(VidDoubleScan(bool)),  this, SLOT(vidflg_doublescan(bool)));
    connect(vidtune,SIGNAL(VidComposite(bool)),   this, SLOT(vidflg_composite(bool)));
    connect(vidtune,SIGNAL(VidPHSync(bool)),      this, SLOT(vidflg_phsync(bool)));
    connect(vidtune,SIGNAL(VidNHSync(bool)),      this, SLOT(vidflg_nhsync(bool)));
    connect(vidtune,SIGNAL(VidPVSync(bool)),      this, SLOT(vidflg_pvsync(bool)));
    connect(vidtune,SIGNAL(VidNVSync(bool)),      this, SLOT(vidflg_nvsync(bool)));
    connect(vidtune,SIGNAL(VidPCSync(bool)),      this, SLOT(vidflg_pcsync(bool)));
    connect(vidtune,SIGNAL(VidNCSync(bool)),      this, SLOT(vidflg_ncsync(bool)));

    vidtune->move(10,10);
    view->adjustSize();
    view->setFixedSize(view->size());

    return view;
}

QWidget *TopLevel::createTabCompose(QWidget *parent)
{
    return new QWidget(parent);
}

QWidget *TopLevel::createTabXStartup(QWidget *parent)
{
    tabXStartup = new DlgAdvancedXStartup(parent);
    return tabXStartup;
}

QWidget *TopLevel::createTabMonitor(QWidget *parent)
{
    tabMonitor = new DlgInfoMonitor(parent);
    return tabMonitor;
}

QWidget *TopLevel::createTabX11(QWidget *parent)
{
    tabXServer = new DlgInfoXServer(parent);
    return tabXServer;
}

QWidget *TopLevel::createTabOptions(QWidget *parent)
{
    return new QWidget(parent);
}

void TopLevel::setupTabVideoMode()
{
/*
    char **list;
    unsigned cnt;

    vidtune->SetLifeAdjust(LifeAdjust);

    if (!VidMode->GetAllModeLineList(list,cnt) ) {
        KMsgBox::message(this, klocale->translate("Error"), 
                         klocale->translate("Cannot get list of video modes"), 
                         KMsgBox::STOP, 
                         klocale->translate("Close"));
    } else {
        vidtune->SetVideoModeList(list);
        vidtune->SetVideoModeSelection(VidMode->GetMode());
        free (list);
    }
*/
/*
    char **list;
    XF86VidModeLine *modelines;
    unsigned i, cnt;
    QString  name;

    vidtune->SetLifeAdjust(LifeAdjust);

    if (!VidMode->GetModeLineList(modelines,cnt) ) {
        KMsgBox::message(this, klocale->translate("Error"), 
                         klocale->translate("Cannot get list of video modes"), 
                         KMsgBox::STOP, 
                         klocale->translate("Close"));
    } else {
        list = (char**) malloc( (cnt+1) * sizeof(char*) );
	memset(list,0,(cnt+1) * sizeof(char*) );
	for ( i = 0; i < cnt ; i++ ) {
           //name    = vidData->resolveVidModeName(modelines[i]);
	   name = vidData->doc->monitor.modeLines.at(i)->GetName();
	   if ( name.isNull() ) name =" ";
	   list[i] = strdup(name);
	}
        vidtune->SetVideoModeList(list);
        vidtune->SetVideoModeSelection(VidMode->GetMode());
	for ( i = 0; i < cnt ; i++ ) 
	   free(list[i]);
        free (list);
    }
*/
    QStrList list;

    list = vidData->getVideoModeList();
    vidtune->SetVideoModeList(&list);
    vidtune->SetVideoModeSelection(VidMode->GetMode());
}

void TopLevel::setupTabCompose()
{
}

void TopLevel::setupTabXStartup()
{
/*
    unsigned                 colordepth;
    XF86ConfigResolutionList reslist;

    if ( !Cfg->xf86cfg ) return;

    if ( !Cfg->xf86cfg->GetResolutionList(Cfg->screen.driver,reslist) ) {
        KMsgBox::message(this, klocale->translate("Error"), 
	                 klocale->translate("Cannot get list of video modes from configuration file"), 
			 KMsgBox::STOP, klocale->translate("Close"));
    } else {
        colordepth = VidMode->GetColorDepth();
	tabXStartup->setData(colordepth,reslist);
    }
*/
    tabXStartup->setDataObj(vidData);
}

void TopLevel::setupTabMonitor()
{
    char *vendor    = VidMode->MonitorVendor();
    char *model     = VidMode->MonitorModel();
    char *bandwidth = VidMode->MonitorBandwidth();
    char *hsync     = VidMode->MonitorSyncRatesH();
    char *vsync     = VidMode->MonitorSyncRatesV();
  		     
    tabMonitor->FillLabels(vendor,model,bandwidth,hsync,vsync);
  
    free(vendor);
    free(model);
    free(bandwidth);
    free(hsync),
    free(vsync);
}

void TopLevel::setupTabX11()
{
    char *protocol   = VidMode->X11ProtocolVersion();
    char *vendor     = VidMode->X11Vendor();
    char *release    = VidMode->X11ReleaseVersion();
    char *extversion = VidMode->XF86VidModeVersion();
    
    tabXServer->FillLabels(protocol,vendor,release,extversion);
  
    free(protocol);
    free(vendor);
    free(release);
    free(extversion);
}

void TopLevel::setupTabOptions()
{
}

void TopLevel::setupContent()
{
    menu_file->setItemEnabled(menu_item_save,vidData->xf86cfg!=NULL);
    toolbar->setItemEnabled(ID_TOOLBAR_SAVE,vidData->xf86cfg!=NULL);

    setupTabVideoMode();
    setupTabCompose();
    setupTabXStartup();
    setupTabMonitor();
    setupTabX11();
    setupTabOptions();
}

void TopLevel::updateContent()
{
}


void TopLevel::ProcessStatus(const char *topic, const char *msg, bool msgbox = TRUE)
{
    if ( msg ) {
        statusbar->changeItem(msg, ID_STATUS_GENERAL);
        if ( msgbox ) 
            KMsgBox::message(this, topic, msg, KMsgBox::STOP, klocale->translate("Close"));
    } else {
        statusbar->changeItem(klocale->translate("Success"), ID_STATUS_GENERAL);
    }
}

void TopLevel::ProcessStatus(const char *topic, bool msgbox = TRUE)
{
    char *et;

    et = VidMode->LastErrorText();

    ProcessStatus(topic,et,msgbox);

    if ( et ) free (et);
}

void TopLevel::update_modeinfo()
{
    char res[64],vsc[64],hsc[64];
  
    snprintf(res,sizeof(res),"%u x %u",VML_Akt->GetHorizontalResolution(),
                                       VML_Akt->GetVerticalResolution());
    snprintf(vsc,sizeof(vsc),"%.1f Hz",VML_Akt->GetVerticalSyncClock());
    snprintf(hsc,sizeof(hsc),"%.1f kHz",VML_Akt->GetHorizontalSyncClock()/1000.0);
  
    vidtune->SetValueBox(res,hsc,vsc);
    vidtune->SetFlagCheckBoxes(VML_Akt->IsInterlace(),VML_Akt->IsDoublescan(),
                               VML_Akt->IsComposite(),VML_Akt->IsPHSync(),VML_Akt->IsNHSync(),
                               VML_Akt->IsPVSync(),VML_Akt->IsNVSync(),VML_Akt->IsPCSync(),
                               VML_Akt->IsNCSync());
}

void TopLevel::videomode(int mode)
{
    bool b;

    b = VidMode->SetMode(mode);
    ProcessStatus(klocale->translate("Mode switch failed"),!b);
  
    if ( b ) {
        VidMode->GetModeLine(*VML_Akt);
        *VML_Undo = *VML_Restore = *VML_Akt;
    }
  
    int m = VidMode->GetMode();
    vidtune->SetVideoModeSelection(m);

    update_modeinfo();
}

bool TopLevel::vidcmd_apply( bool doit )
{
  bool b = TRUE;
  
  if ( LifeAdjust || doit ) {
    XF86VidModeLine ml;
    if (VidMode->GetModeLine(ml)) {
      // It seems to be possible that "set" works and "get" fails on bad settings
      if (VidMode->SetModeLine(*VML_Akt) && VidMode->GetModeLine(*VML_Akt)) {
        *VML_Undo = ml;
      } else {
        *VML_Akt = ml; // undo faild request
        b = FALSE; 
      }
    } else {
      VML_Akt = VML_Undo; // forced UNDO to last saved and working settings
    }
    ProcessStatus(klocale->translate("Failed to set new settings"),TRUE);
  } else {
  }
  
  update_modeinfo();
  
  return b;
}

void TopLevel::filesave()
{
   if ( vidData->xf86cfg == NULL ) return;

   if ( !vidData->storeXf86Config(vidData->vidCfg->cfgConfigFile) ) {
       ProcessStatus(klocale->translate("Save configuration"),vidData->xf86cfg->LastErrorText());
       ProcessStatus(NULL,klocale->translate("Error - configuration not saved"),FALSE);
   } else {
       ProcessStatus(NULL,klocale->translate("Configuration saved"),FALSE);
   }

/*    
    DlgFileSaveWizzardTopW *dlg;

    dlg = new DlgFileSaveWizzardTopW(this,NULL,VidMode,
                                     Cfg->cfgConfigFile,Cfg->cfgWorkingDirectory);
    dlg->exec();
  
    delete dlg;
*/
}

void TopLevel::prevvideomode()
{
    videomode(VidMode->GetMode()-1);
}

void TopLevel::nextvideomode()
{
    videomode(VidMode->GetMode()+1);
}

void TopLevel::vidmov_left(int steps)
{
    if (VML_Akt->Move(-steps,0)) {
        vidcmd_apply();
    } else {
        ProcessStatus(klocale->translate("Failed to move screen"),
                      klocale->translate("Reached left boarder"),TRUE);
    }
}

void TopLevel::vidmov_right(int steps)
{
  if (VML_Akt->Move(steps,0)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to move screen"),
                  klocale->translate("Reached right boarder"),TRUE);
}

void TopLevel::vidmov_up(int steps)
{
  if (VML_Akt->Move(0,-steps)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to move screen"),
                  klocale->translate("Reached upper boarder"),TRUE);
}

void TopLevel::vidmov_down(int steps)
{
  if (VML_Akt->Move(0,steps)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to move screen"),
                  klocale->translate("Reached lower boarder"),TRUE);
}

void TopLevel::vidsiz_hmore(int steps)
{
  if (VML_Akt->Resize(steps,0)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to resize screen"), 
                  klocale->translate("Requested horizontal values are out of range"),TRUE);
}

void TopLevel::vidsiz_hless(int steps)
{
  if (VML_Akt->Resize(-steps,0)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to resize screen"), 
                  klocale->translate("Requested horizontal values are out of range"),TRUE);
}

void TopLevel::vidsiz_vmore(int steps)
{
  if (VML_Akt->Resize(0,steps)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to resize screen"), 
                  klocale->translate("Requested vertical values are out of range"),TRUE);
}

void TopLevel::vidsiz_vless(int steps)
{
  if (VML_Akt->Resize(0,-steps)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to resize screen"), 
                  klocale->translate("Requested vertical values are out of range"),TRUE);
}

void TopLevel::vidcmd_lifeadj(bool state)
{
   LifeAdjust = state;
   vidcmd_apply();
}

void TopLevel::vidcmd_test()
{
    if (LifeAdjust) return;

    XF86VidModeLine normal;

    if (!VidMode->GetModeLine(normal)) {
      ProcessStatus(klocale->translate("Preparation for test failed."),TRUE);
      return;
    }

    if (VidMode->SetModeLine(*VML_Akt) && VidMode->GetModeLine(*VML_Akt)) {
      QTimer timer;
      KMsgBox msgbox(this, klocale->translate("Test new video mode"),
                     klocale->translate("If you can read this press Apply or Close\n\n"
		     "Old Video mode will be restored in 10 seconds!"),
		     KMsgBox::EXCLAMATION | KMsgBox::DB_THIRD, 
		     klocale->translate("&Apply"),
		     klocale->translate("&Close"),klocale->translate("A&bort"));
      msgbox.setFixedSize(msgbox.size());
      
      connect( &timer, SIGNAL(timeout()), &msgbox, SLOT(b4Pressed()));
      timer.start(10*1000,TRUE);

      switch (msgbox.exec()) {
        /* APPLY=1 */
        case 1:  *VML_Undo = normal;
	         break;
	/* CLOSE=2 */
	case 2:  VidMode->SetModeLine(normal);
	         *VML_Undo = normal;
	         break;
	/* ABORT=3, TIMEOUT=4 */
	default: VidMode->SetModeLine(normal);
	         break;
      }
    } else 
      ProcessStatus(klocale->translate("Failed to set new settings for test"),TRUE);

}

void TopLevel::vidcmd_undo()
{
    if (LifeAdjust) {
      *VML_Akt = *VML_Undo;
      vidcmd_apply();
    } else {
      XF86VidModeLine tmp;
      tmp = *VML_Akt;
      *VML_Akt = *VML_Undo; 
      *VML_Undo = tmp;
      update_modeinfo();
    }
}

void TopLevel::vidcmd_restore()
{
    *VML_Undo = *VML_Akt;
    *VML_Akt = *VML_Restore;
    vidcmd_apply();
}

void TopLevel::vidcmd_modeline()
{
  DlgInfoModeLine *dlg = new DlgInfoModeLine;

  char *ml = VML_Akt->GetModeline();
  dlg->FillLabels( QString("Modeline ") + ml );
  free (ml);

  dlg->exec();
  delete dlg;
}

void TopLevel::vidcmd_details()
{
  DlgInfoVideo *dlg = new DlgInfoVideo;

  dlg->FillLabels(VML_Akt->GetHorizontalResolution(),
                  VML_Akt->GetVerticalResolution(),
		  VML_Akt->GetHorizontalSyncStart(),
		  VML_Akt->GetHorizontalSyncEnd(),
		  VML_Akt->GetHorizontalSyncTotal(),
		  VML_Akt->GetVerticalSyncStart(),
		  VML_Akt->GetVerticalSyncEnd(),
		  VML_Akt->GetVerticalSyncTotal(),
		  VML_Akt->GetHorizontalSyncClock()/1000.0,
		  VML_Akt->GetVerticalSyncClock(),
		  VML_Akt->GetPixelClock(),
		  VML_Akt->GetFlags());

  dlg->exec();
  delete dlg;
}

void TopLevel::vidflg_interlace(bool state)
{
  if (VML_Akt->Interlace(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"Interlace\""),TRUE);
}

void TopLevel::vidflg_doublescan(bool state)
{
  if (VML_Akt->Doublescan(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"Doublescan\""),TRUE);
}

void TopLevel::vidflg_composite(bool state)
{
  if (VML_Akt->Composite(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"Composite\""),TRUE);
}

void TopLevel::vidflg_phsync(bool state)
{
  if (VML_Akt->PHSync(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"+HSync\""),TRUE);
}

void TopLevel::vidflg_nhsync(bool state)
{
  if (VML_Akt->NHSync(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"-HSync\""),TRUE);
}

void TopLevel::vidflg_pvsync(bool state)
{
  if (VML_Akt->PVSync(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"+VSync\""),TRUE);
}

void TopLevel::vidflg_nvsync(bool state)
{
  if (VML_Akt->NVSync(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"-VSync\""),TRUE);
}

void TopLevel::vidflg_pcsync(bool state)
{
  if (VML_Akt->PCSync(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"+CSync\""),TRUE);
}

void TopLevel::vidflg_ncsync(bool state)
{
  if (VML_Akt->NCSync(state)) {
    vidcmd_apply();
  } else 
    ProcessStatus(klocale->translate("Failed to set flag"),
                  klocale->translate("Cannot change Option \"-CSync\""),TRUE);
}

void TopLevel::settingsconfigure()
{
  DlgOptionsConfig *dlg = new DlgOptionsConfig;

  dlg->FillLabels(Cfg->cfgDisplayWarning,Cfg->cfgConfigFile);
       
  if (dlg->exec() == 1) {
    dlg->ReadLabels(Cfg->cfgDisplayWarning,Cfg->cfgConfigFile);
  }

  delete dlg;
}

void TopLevel::about()
{
    QMessageBox::about(this, klocale->translate("About KVidTune"),
                       klocale->translate("KDE X-Server Video Configuration and Tuning Utility\n\n"
		  			  "Version "VERSION" (alpha)\n\n"
					  "(c) 1997-1999 by Arnim Frithjof Brestrich\n"
					  "(brestrich@kde.org)\n"));
}

void TopLevel::aboutQt()
{
    QMessageBox::aboutQt(this, klocale->translate("About Qt"));
}

void TopLevel::help()
{
    kapp->invokeHTMLHelp( "", "" );    
}
