/*

    Copyright (C) 1997 Frithjof Brestrich
                       brestrich@kde.org


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/ 

#include <config.h>

#include <stdlib.h>
#include <stdio.h>

#include <kapp.h>

#include <stdlib.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include <utime.h>
#include <math.h> 


#include "xf86configfile.h"

XF86ConfigFile::XF86ConfigFile()
{
  ErrorText = "";
  Data      = new tXF86CfgDirectory(NULL);
  File      = NULL;
  Stream    = NULL;
}

XF86ConfigFile::~XF86ConfigFile()
{
  CloseFile();
  delete Data;
}

void XF86ConfigFile::SetErrorText(const QString &txt)
{
  ErrorText = txt;
}

const QString &XF86ConfigFile::LastErrorText()
{
  return ErrorText;
}

bool XF86ConfigFile::OpenFile(const QString &fname, int flags)
{
  File = new QFile(fname);
  if ( !File ) { SetErrorText(klocale->translate("cannot create file object")); return FALSE; }

  if (File->open( flags )) { 
    File->at(0);
    Stream = new QTextStream(File);
  } else {
     SetErrorText(klocale->translate("cannot connect file object to stream object"));
     return FALSE;
  }

  return TRUE;
}

bool XF86ConfigFile::CloseFile()
{
  if (!File || !Stream ) return FALSE;

  File->close();

  delete Stream;
  delete File;
  Stream = NULL;
  File = NULL;

  return TRUE;
}

bool XF86ConfigFile::Load(const QString &fname)
{
  bool b = TRUE;

  if (!OpenFile(fname, IO_ReadOnly )) {
    SetErrorText(klocale->translate("Cannot open file for reading.\nCheck permissons and existance!"));
    return FALSE;
  }

  while ( File->isOpen() && !Stream->eof() ) {

    if (!ReadSection(Data)) {
      b = FALSE;
      break;
    }
  }

  CloseFile(); // ignore error code

  return b;
}

bool XF86ConfigFile::Store(const QString &fname, bool backup)
{
  bool b = TRUE;

  if ( backup ) {
    if (!BackupFile(fname,fname+backupPostfixCharacter))
        return FALSE;
  }

  if (!OpenFile(fname, IO_Truncate | IO_WriteOnly )) {
    SetErrorText(klocale->translate("Cannot open file for writing. Check permissons!"));
    return FALSE;
  }

  b = WriteSection(Data);
  
  CloseFile(); // ignore error code

  return b;
}

bool XF86ConfigFile::BackupFile(const QString &masterfile, const QString &backupfile)
{
    QFile f(masterfile);

    // backup copy, care for hard links!
    if ( f.exists() ) {
        QFile bf(backupfile);
        /*
         * Readonly should be enough, but this way we get an error message ( writeonly by root)
         * before we make a backup - deleting old backup + confusing user
         */
        if ( !f.open(IO_ReadWrite) ) {
            SetErrorText(klocale->translate("Configuration file cannot be accessed for reading and writing. Check file permissons and existance."));
            return FALSE;
        }
        if ( !bf.open( IO_Truncate | IO_WriteOnly) ) {
            SetErrorText(klocale->translate("Backup file cannot be created. Check rights and files space."));
            return FALSE;
        }
        /*
         * stupid copy
         */
        int i;
        char buf[256];
        while ( ( i = f.readBlock( buf, sizeof(buf) ) ) > 0 ) {
            if ( bf.writeBlock( buf, i ) != i ) {
                SetErrorText(klocale->translate("Backup file cannot be written. Check disk space."
));
                 return FALSE;
            }
        }
        bf.close();
        f.close();
        /*
         * copy file permissions/time/owner etc...
         */
        struct stat st;
        struct utimbuf tb;
        if ( stat(masterfile, &st) == 0 ) {
            chmod(backupfile,st.st_mode);
            chown(backupfile,st.st_uid,st.st_gid);
            tb.actime = st.st_atime;
            tb.modtime = st.st_mtime;
            utime(backupfile,&tb);
        }
    }

    return TRUE;
}


tXF86CfgEntry* XF86ConfigFile::NewCfgEntry(const QString &key, const QString &value, bool isstring)
{
    tXF86CfgEntry *e;

    if ( (e = new tXF86CfgEntry) == NULL )  { 
        SetErrorText(klocale->translate("out of memory"));
	e = NULL; // I'm paranoid (slow down a low memory system)
    } else {
        e->Key.setStr(key);
	e->Value.setStr(value);
	e->IsString = isstring;
    }
    return e;
}

bool XF86ConfigFile::SetCfgEntry(tXF86CfgDirectory* root, const char* key, const char* value, bool isstring)
{
  tXF86CfgEntry  *e;

  e = FindEntry(&(root->Data),key);

  if ( e == NULL ) {
      e = NewCfgEntry(key,value,isstring);
      if ( e == NULL ) return FALSE;
      root->Data.append(e);
  } else {
      e->IsString = isstring;
      e->Value.setStr(value);
  }

  return TRUE;
}

bool XF86ConfigFile::ReadMonitorModeSection(tXF86CfgDirectory *Dir, const QString &name)
{
  QString s,key,val;
  tXF86CfgEntry *e;
  unsigned sep,space,tab;
  bool    b = TRUE;
  QString DotClock = "0";
  QString HTimings = "0 0 0 0";
  QString VTimings = "0 0 0 0";
  QString Flags    = "";
  QString HSkew    = "";
  QString buf;

  if (!Stream) return FALSE;
  
  while ( File->isOpen() && !Stream->eof() ) {
  
    if ( (s = Stream->readLine() ) == NULL ) { 
      SetErrorText(klocale->translate("error reading a line"));
      b = FALSE; 
      break; 
    }

    s = s.stripWhiteSpace();
    if ( s[0] == '#'  || s.length() == 0 ) {
    } else {
      space = s.find(' ',0,FALSE);
      tab   = s.find('\t',0,FALSE);
      sep   = space < tab ? space : tab;
      sep   = sep < s.length() ? sep : s.length();

      key.setStr(s).truncate(sep);
      val = s.mid(sep,s.size()).stripWhiteSpace();

      if ( ! stricmp(key,"DotClock") ) {
        DotClock.setStr(val);
      } else if ( ! stricmp(key,"HTimings") ) { 
        HTimings.setStr(val);
      } else if ( ! stricmp(key,"VTimings") ) { 
        VTimings.setStr(val);
      } else if ( ! stricmp(key,"Flags") ) { 
        Flags.setStr(val);
      } else if ( ! stricmp(key,"HSkew") ) { 
        HSkew.setStr(val);
      } else if ( ! stricmp(key,"EndMode") ) {
        break;
      }

    }

  }

  buf.sprintf("%s %s %s %s %s %s\n",(const char*) name,(const char*)DotClock,
                         (const char*)HTimings,(const char*)VTimings,
                         (const char*)Flags,(const char*)HSkew);

  if ( ! (e = NewCfgEntry("Modeline",buf,FALSE))  )  { 
    b = FALSE; 
  } else {
    Dir->Data.append(e);
  }

  return b;
}

bool XF86ConfigFile::ReadSection(tXF86CfgDirectory *Dir)
{
  QString s,key,val;
  tXF86CfgEntry *e;
  unsigned sep,space,tab;
  bool   b = TRUE;
  bool   isMonitorSection;

  if (!Stream) return FALSE;

  isMonitorSection = ( Dir->Name == "Monitor");

  while ( File->isOpen() && !Stream->eof() ) {

    if ( (s = Stream->readLine() ) == NULL ) { 
      SetErrorText(klocale->translate("error reading a line"));
      b = FALSE; 
      break; 
    }

    s = s.stripWhiteSpace();
    if ( s[0] == '#'  || s.length() == 0 ) {
    
      e = Dir->Data.getLast();

      // remove blank lines which occur more than once
      if ( e==NULL || e->Key!=NULL || e->Value!=NULL || s.length() ) {

        if (s.length()!=0) { // empty strings become NULL
	  if ( ! (e = NewCfgEntry(NULL,s,FALSE))  )  { b = FALSE; break; }
	} else {
	  if ( ! (e = NewCfgEntry(NULL,NULL,FALSE))  ) { b = FALSE; break; }
	}

        Dir->Data.append(e);

      }
    
    } else {
    
      space = s.find(' ',0,FALSE);
      tab   = s.find('\t',0,FALSE);
      sep   = space < tab ? space : tab;
      sep   = sep < s.length() ? sep : s.length();

      key.setStr(s).truncate(sep);
      val = s.mid(sep,s.size()).stripWhiteSpace();

      if ( !stricmp(key,"SubSection") || !stricmp(key,"Section") ) {
        tXF86CfgDirectory *d = NULL;

	if ( (val.length()<2) || (val[0]!='"') || (val[val.length()-1]!='"') ) {
	   SetErrorText(klocale->translate("invalid section statement"));
	   b = FALSE; 
	   break; 
	}

        if ( (d = new tXF86CfgDirectory(val.mid(1,val.length()-2))) == NULL )  { 
	   SetErrorText(klocale->translate("out of memory"));
	   b = FALSE; 
	   break; 
	}

        if (!ReadSection(d))  { 
	  // error set by ReadSection()
	  b = FALSE; 
	  break; 
	}

	Dir->Dir.append(d);

      } else if ( !stricmp(key,"EndSection") || !stricmp(key,"EndSubSection") ) {
      
        break;
      
      } else if ( isMonitorSection && !stricmp(key,"Mode") ) {

        if (!ReadMonitorModeSection(Dir,val)) {
	  b = FALSE; 
	  break; 
	};
      
      } else {
      
	if ( val.contains('"') == 2 && val[0] == '"' && val[val.length()-1] == '"' ) {
	  if ( ! (e = NewCfgEntry(key,val.mid(1,val.length()-2),TRUE))  )  { b = FALSE; break; }
	} else {
	  if ( ! (e = NewCfgEntry(key,val,FALSE))  )  { b = FALSE; break; }
	}

        Dir->Data.append(e);
      
      }
    }
  }

  return b;
}

bool XF86ConfigFile::WriteSection(tXF86CfgDirectory *Dir, int Deep)
{
  tXF86CfgDirectory *d;
  tXF86CfgEntry     *e;
  QString           s,buf;
  int               i;

  for (i=1;i<Deep;i++) s+="\t";

  if ( Deep == 1 ) *Stream << "\nSection \"" << Dir->Name << "\"\n";
  else if ( Deep > 1) *Stream << s << "SubSection \"" << Dir->Name << "\"\n";

  s+="\t";
  
  for ( e = Dir->Data.first(); e != NULL ; e = Dir->Data.next() ) {
    if ( e->Key == NULL ) {
      if ( e->Value != NULL ) *Stream << e->Value;
      *Stream << '\n';
    } else {
      if (e->IsString) {
        buf.sprintf("%s%-15s \"%s\"\n",(const char*)s,(const char*)e->Key,(const char*)e->Value);
      } else {
        if ( e->Value == NULL )
          buf.sprintf("%s%s\n",(const char*)s,(const char*)e->Key);
	else 

          buf.sprintf("%s%-15s %s\n",(const char*)s,(const char*)e->Key,(const char*)e->Value);
      }
      *Stream << buf;
    }
  }

  for ( d = Dir->Dir.first(); d != NULL ; d = Dir->Dir.next() ) 
    if (!WriteSection(d,Deep+1)) 
      return FALSE;

  if ( Deep == 1 ) *Stream << "EndSection\n";
  else if ( Deep > 1) *Stream << s.left(s.length()-1) << "EndSubSection\n";

  return TRUE;
}


tXF86CfgDirectory* XF86ConfigFile::FindSection(tXF86CfgDirectory* root,const char* name, unsigned skip)
{
  tXF86CfgDirectory *d;

  if ( root == NULL ) 
    root = Data;

  for ( d = root->Dir.last(); d != NULL ; d = root->Dir.prev() ) {

    if ( d->Name == name ) {
      if ( skip )
        --skip;
      else
        return d;
    }

  }

  SetErrorText(klocale->translate("requested section not found")); 
  return NULL;
}

tXF86CfgEntry* XF86ConfigFile::FindEntry(tXF86CfgEntryList* root, const char* key,
                                         const char* value)
{
  tXF86CfgEntry *e;
  
  if ( root == NULL )
    root = &(Data->Data);

  for ( e = root->last(); e != NULL ; e = root->prev() ) 
    if ( e->Key != NULL  &&  !stricmp(e->Key,key) ) {
      if (value == NULL ) {
        return e;
      } else if ( e->Value != NULL && !stricmp(e->Value,value) ) {
        return e;
      }
    }

  SetErrorText(klocale->translate("requested entry not found")); 
  return NULL;
}

bool XF86ConfigFile::ExistEntry(tXF86CfgDirectory* root, const char* key, 
                                const char* value)
{
  return (FindEntry(&(root->Data),key,value) != NULL);
}

bool XF86ConfigFile::RemoveEntry(tXF86CfgDirectory* root, const char* key, const char* value)
{
    tXF86CfgEntry  *e;

    for ( e = root->Data.first(); e != NULL ; e = root->Data.next() ) 
        if ( (!e->Key.isNull()) && !strcmp(e->Key,key)) {
	    if ( (value == NULL ) || ( ! strcmp(e->Value,value) ) )
            if (root->Data.remove()) {
                root->Data.prev();
                //delete e;
	    }
	}

    return TRUE;
}

const char *XF86ConfigFile::GetValueStr(tXF86CfgDirectory* root, const char* key)
{
  tXF86CfgEntry  *e;

  e = FindEntry(&(root->Data),key);
  if ( e == NULL ) return NULL;

  return e->Value;
}

unsigned XF86ConfigFile::GetValueInt(tXF86CfgDirectory* root, const char* key)
{
  tXF86CfgEntry     *e;
  unsigned           i;

  i = 0;

  e = FindEntry(&(root->Data),key);
  if ( e == NULL ) return 0;

  sscanf(e->Value,"%u",&i);

  return i;
}

QList<XF86VidSyncRange> XF86ConfigFile::GetValueRangeList(tXF86CfgDirectory* root, const char* key, unsigned scale)
{
  tXF86CfgEntry           *e;
  XF86VidSyncRange        range;
  QList<XF86VidSyncRange> list;
  QString                 s, t, low, high;
  int                     a, b, c;
  bool                    ok;

  list.clear();

  e = FindEntry(&(root->Data),key);
  if ( e == NULL ) return list;

  // e->Value looks like: 30-70.5, 71-75MHz
  s = e->Value.upper().stripWhiteSpace();

  if ( s.right(3) == "MHZ" ) {
     s.truncate(s.length()-3); 
     scale = 1000000;
  } else if ( s.right(3) == "KHZ" ) {
     s.truncate(s.length()-3); 
     scale = 1000;
  } else if ( s.right(2) == "HZ" ) {
     s.truncate(s.length()-2); 
     scale = 1;
  }

  a = b = c = 0;
  while ( (unsigned) a < s.length() ) {
      b = s.find(',',a);
      if ( b == -1 ) {
          t = s.mid(a,s.length()).stripWhiteSpace();
      } else {
          t = s.mid(a,b-a).stripWhiteSpace();
      }
      if ( t.isEmpty() ) break;
      
      // now we have one element which should look like: >35Khz< or >28.2-78.0<
      c    = t.find('-',0);
      if ( c == -1 ) {
          high = low = t.stripWhiteSpace();
      } else {
          low  = t.mid(0,c).stripWhiteSpace();
          high = t.mid(c+1,t.length()).stripWhiteSpace();
      }
      
      ok = FALSE;
      range.start = (unsigned) rint( low.toDouble(&ok) * scale);
      if ( ok ) {
        ok = FALSE;
        range.end   = (unsigned) rint( high.toDouble(&ok) * scale);
        if ( ok ) 
          list.append(new XF86VidSyncRange(range) );
      }

      if ( b == -1 ) break;

      a = b+1;
  }

  return list;
}

QList<XF86VidModeLine> XF86ConfigFile::GetValueModelines(tXF86CfgDirectory* root, const char* key)
{
    QList<XF86VidModeLine> list;
    XF86VidModeLine        *modeline;
    XF86VidModeModeInfo    minfo;
    tXF86CfgEntry          *e;
    QString                modename;
    double                 dotclock;
    QString                options;
    char                   buf[1024];
    QString                s;
    int                    a;

    list.clear();

    if ( root == NULL )
        root = Data;

    for ( e = root->Data.first(); e != NULL ; e = root->Data.next() ) {
        
	memset(&minfo,0,sizeof(minfo));
        memset(buf,0,sizeof(buf));
        dotclock = 0.0;
	options  = "";

        if ( e->Value == NULL ) continue;
        if ( e->Key != NULL && !stricmp(e->Key,key) ) {
	    s = e->Value.stripWhiteSpace();

	    if ( s[0] != '"' ) continue;

	    a = s.find('"',1);
	    if ( a == -1 ) continue;
	    modename = s.mid(1,a-1);
	    s = s.mid(a+1,s.length()).simplifyWhiteSpace();

	    // example for s: "1024x768" 80.00 1024 1044 1164 1344 768 772 775 803
	    if ( sscanf(s,"%lf %hu %hu %hu %hu %hu %hu %hu %hu %1023c",
	           &dotclock,
	           &minfo.hdisplay, &minfo.hsyncstart, &minfo.hsyncend, &minfo.htotal,
		   &minfo.vdisplay, &minfo.vsyncstart, &minfo.vsyncend, &minfo.vtotal,
		   buf)
	       < 9 )
	    {
	        continue;
	    }
	    minfo.dotclock = (unsigned) (dotclock * 1000);
	    options = buf;

            while ( ! options.isEmpty() ) {
               QString param;
               
	       a = options.find(' ');
	       if ( a == -1 ) a = options.length();
	       param = options.mid(0,a).lower();
	       options.remove(0,a+1);

	       if ( param == "+hsync" )     minfo.flags |= VM_FLAG_PHSYNC;
	       if ( param == "-hsync" )     minfo.flags |= VM_FLAG_NHSYNC;
	       if ( param == "+vsync" )     minfo.flags |= VM_FLAG_PVSYNC;
	       if ( param == "-vsync" )     minfo.flags |= VM_FLAG_NVSYNC;
	       if ( param == "interlace" )  minfo.flags |= VM_FLAG_INTERLACE;
	       if ( param == "composite" )  minfo.flags |= VM_FLAG_CSYNC;
	       if ( param == "+csync" )     minfo.flags |= VM_FLAG_PCSYNC;
	       if ( param == "-csync" )     minfo.flags |= VM_FLAG_NCSYNC;
	       if ( param == "doublescan" ) minfo.flags |= VM_FLAG_DBLSCAN;
	    }

	    modeline = new XF86VidModeLine;
	    *modeline = minfo;
	    modeline->SetName(modename);
	    list.append(modeline);
        } 
    }

    return list;
}

bool XF86ConfigFile::SetValueStr(tXF86CfgDirectory* root, const char* key, const char* value)
{
    return SetCfgEntry(root,key,value,TRUE);
}

bool XF86ConfigFile::SetValueInt(tXF86CfgDirectory* root, const char* key, unsigned value)
{
    QString        s;

    s.sprintf("%u",value);

    return SetCfgEntry(root,key,s,FALSE);
}

bool XF86ConfigFile::SetValueRangeList(tXF86CfgDirectory* root, const char* key,
                                       QList<struct XF86VidSyncRange> &list, unsigned scale)
{
  QString          s, value;
  XF86VidSyncRange *range;

  value = "";
  s     = "";

  for ( range = list.first(); range != NULL ; range = list.next() ) {
      if ( range->start == range->end ) {
          s.sprintf("%0.15g", (double)range->start/scale );
      } else {
          s.sprintf("%0.15g-%0.15g", (double)range->start/scale, (double)range->end/scale );
      }

      if ( ! value.isEmpty() ) value += ", ";

      value += s;
  }
  switch ( scale ) {
      case 1:       value += " Hz";  break;
      case 1000:    value += " kHz"; break;
      case 1000000: value += " MHz"; break;
  }
  
  return SetCfgEntry(root,key,value,FALSE);
}

bool XF86ConfigFile::SetValueModelines(tXF86CfgDirectory* root, const char* key, 
                                       QList<class XF86VidModeLine> &list)
{
    tXF86CfgEntry    *e;
    XF86VidModeLine  *ml;
    char             *s;

    RemoveEntry(root,key);

    for ( ml = list.first(); ml != NULL ; ml = list.next() ) {
        s = ml->GetModeline();
        e = NewCfgEntry(key,s,FALSE);
	free(s);
	if ( e != NULL ) {
	    root->Data.append(e);
	} else {
	    return FALSE;
	}
    }

    return TRUE;
}


tXF86CfgDirectory* XF86ConfigFile::GetSection(tXF86CfgDirectory* root,
                                              const char* section, 
                                              const char* key, const char* value)
{
  tXF86CfgDirectory *d;
  tXF86CfgEntry     *e;
  unsigned          nr;

  nr = 0;
  while ( (d = FindSection(root,section,nr)) != NULL ) {
    ++nr;
    e  = FindEntry(&(d->Data),key);
    if ( e->Key != NULL  &&  e->Value == value ) {
      return d;
    }
  }
  
  SetErrorText(klocale->translate("requested section not found")); 
  return NULL;
}

tXF86CfgDirectory* XF86ConfigFile::GetSection(tXF86CfgDirectory* root, const char* section, 
                                              const char* key, unsigned value)
{
    QString s;

    s.sprintf("%u",value);

    return GetSection(root, section, key, s);
}
    
bool XF86ConfigFile::GetScreenSections(tXF86CfgDirectoryList &list,
                                       const char* vendor, const char* model)
{
  tXF86CfgDirectory *d, *mon;
  tXF86CfgEntry     *e, *driver, *device, *monitor;
  unsigned          nr;

  list.clear();

  nr = 0;
  while ( (d = FindSection("Screen",nr)) != NULL ) {
    ++nr;
    driver  = FindEntry(&(d->Data),"Driver");
    device  = FindEntry(&(d->Data),"Device");
    monitor = FindEntry(&(d->Data),"Monitor");
    if ( driver && device && monitor ) {
      mon = GetMonitorSection(monitor->Value);
      if ( mon && GetDeviceSection(device->Value) ) {
        if ( vendor != NULL ) {
          e = FindEntry(&(mon->Data),"VendorName");
          if ( e->Value != vendor ) continue;
	}
	if ( model != NULL ) {
          e = FindEntry(&(mon->Data),"ModelName");
	  if ( e->Value != model ) continue;
	}
        list.append(d);
      }
    }
  }

  if (list.isEmpty()) {
    SetErrorText(klocale->translate("no valid driver section found")); 
    return FALSE;
  } else {
    return TRUE;
  }
}

/*
bool XF86ConfigFile::SetModeLines(char *vendor, char *model, XF86VidModeLine* &modelines,unsigned count)
{
  tXF86CfgDirectory *d;
  tXF86CfgEntry     *e;
  char              *s;
  unsigned           n;
  unsigned           monitornr;

  monitornr = 0;
  while ( (d = FindSection("Monitor",monitornr)) != NULL ) {
    ++monitornr;
    if (FindEntry(&(d->Data),"VendorName",vendor) == NULL ) continue;
    if (FindEntry(&(d->Data),"ModelName",model) == NULL ) continue;
    break;
  }

  if ( d == NULL ) { 
    SetErrorText("no entry for specified monitor found"); 
    return FALSE; 
  }

  // delete all existing modelines
  for ( e = d->Data.first(); e != NULL ; e = d->Data.next() ) 
    if ( (e->Key!=NULL) && !stricmp(e->Key,"Modeline"))  
      if (d->Data.remove())     
        d->Data.prev();

  // insert all new/current modlines 
  for ( n=0 ; n<count ; n++ ) {
    if ( (e = new tXF86CfgEntry) == NULL ) {
      SetErrorText("out of memory"); 
      return FALSE; 
    }
    e->Key.setStr("Modeline");
    e->IsString = FALSE;
    s = modelines[n].GetModeline();
    e->Value.setStr(s);
    free(s);
    d->Data.append(e);
  }
      
  return TRUE;
}
*/
