/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [iemsi.cpp]            FSC-0056.001 IEMSI Client                           */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* Copyright (c) 1997 by Lars Doelle                                          */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is available under the GNU Public Software Licence.      */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

#include <strings.h>
#include <stdio.h>
#include <time.h>

#include "iemsi.moc"

/* FIXME:
   - handle quotation of `{}[]' and non-ascii character hex denotation
   - improve scaning of msg (see above), length, chksum.
   - timeout.
*/

/* --| crc interface |------------------------------------------------------- */

#include "crc.h"
#include "iemsi.h"

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

/* Some notes on emsi encoding and check sums

   **EMSI_xyz0005{\00}CCCC
     +---------------+ checksum region (i.e. check sum starts in 'E' of EMSI)

   Both check sum and length refere to the encoded (i.e. quoted) text.

   Characters are quoted the following way:
     }              }}   if appears outside a subfield (Fixme: correct?)
     ]              ]]   if appears inside  a subfield (Fixme: correct?)
     0x00 - 0x1f    \xx
     0x80 - 0xff    \xx

   When sending emsi frames, we don't care for doubling '}]',
   but simply encode them hex.

   We thus have to deal with with '}}' only in data received.
 
*/

/* --| incoming message types |---------------------------------------------- */

#define EMSI_ACK 1
#define EMSI_NAK 2
#define EMSI_IRQ 3
#define EMSI_ISI 6
#define EMSI_ICI 7

static struct
{
  int   typ;
  char* ide;
  int   dta; /* bool */
  int   c32; /* 4/8  */
} msg_typ[] =
{
  { EMSI_ACK, "ACK", 0, 4 },
  { EMSI_NAK, "NAK", 0, 4 },
  { EMSI_IRQ, "IRQ", 0, 4 },
  { EMSI_ISI, "ISI", 1, 8 },
  { EMSI_ICI, "ICI", 1, 8 },
  { 0       , ""   , 0, 4 }
};

/* --| outgoing messages |--------------------------------------------------- */

char *ici_hdr = "EMSI_ICI";

void Iemsi::send(int typ, char* dta)
/* msg composition */
/* FIXME: handle characters > 127, {[]} */
{ char msg[EMSI_MSG_LEN]; int m,i,len = 0; int dtalen = strlen(dta);
  for (m=0; msg_typ[m].typ != 0 && msg_typ[m].typ != typ; m++);
  sprintf(msg,"**EMSI_%s",msg_typ[m].ide);
  len = 10;
  if (msg_typ[m].dta)
  {
    sprintf(msg+len,"%04X",dtalen);
    len += 4;
    for (i = 0; i < dtalen; i++) msg[len+i] = dta[i];
    len += dtalen;
  }
  if (msg_typ[m].c32 == 8)
    sprintf(msg+len,"%08lX",crc32(msg+2,len-2));
  else
    sprintf(msg+len,"%04X",crc16(msg+2,len-2));
  len += msg_typ[m].c32;
  msg[len] = '\r'; len += 1;
  msg[len] = 0;
  mo->send_bytes(msg,len);
  // if (log) fprintf(log,"[send '%s' %d]",msg,len);
}

/* --| protocol logic |------------------------------------------------------ */

void Iemsi::protocol(int msg_typ, int ok, char* msg, int len)
{
  // if (log) fprintf(log,"[decide %d %d '%s' %d]",msg_typ,ok,msg,len);
  switch( msg_typ )
  {
    case EMSI_NAK :
    case EMSI_IRQ :
      // connection older than 1 minute --> ignore
      // check_sum error                --> ignore
      // not configured                 --> ignore
      // replied more than 3 times      --> ignore
      //                                --> reply ICI (for IRQ)
      //FIXME: should be retransmitted every 20 sec if no ISI
      if (ok) send(EMSI_ICI,ici_bdy);
    break;
    case EMSI_ISI :
      //FIXME: improve scanning, this may be really more complex
{ char sys[100], box[100], loc[100], sop[100];
  time_t tim; char not[100], wai[100], cab[100];
  cab[0] = 0;
  sscanf(msg,"{%[^}]}{%[^}]}{%[^}]}{%[^}]}{%x}{%[^}]}{%[^}]}{%[^}]}\r",
              sys,box,loc,sop,tim,not,wai,cab);
  IemsiServerInfo *isi = new IemsiServerInfo;
  isi->system       = sys;
  isi->box          = box;
  isi->location     = loc;
  isi->sysop        = sop;
  isi->localtime    = tim;
  isi->notice       = not;
  isi->wait         = wai;
  isi->capabilities = cab;
  emit rcv_isi(isi);
}
      // connection older than 1 minute --> ignore
      // not configured                 --> ignore
      // replied more than 3 times      --> ignore
      // check_sum error                --> reply NAK (for ISI)
      //                                --> reply ACK (for ISI)
      send(ok?EMSI_ACK:EMSI_NAK,"");
      if (ok) delete this;
    break;
    default:
      delete this;
      break;
  }
}

/* --| incoming events |----------------------------------------------------- */

/*FIXME:
  - This should eventually be replaced by a more flexible device to detect
    arbitrary incoming strings.
  - If the string is recognized, this device should signal, that it likes
    to take full control over the chanel, ie. deactivating all other
    control sequence detections.
  - we could eventuall use this technic to handle console character sequences
    also.
*/

void Iemsi::rcv_byte(int cc)
/* notify event */
{
  switch (emsi_scn_state)
  {
    case EMSI_SCAN : 
    { int i;
      for (i = 0; i < EMSI_HDR_LEN; i++) 
        emsi_msg[i] = emsi_msg[i+1];
      emsi_msg[EMSI_HDR_LEN-1] = cc;
      if (strncmp(emsi_msg,"**EMSI_",7) == 0)
      /* emsi header seen */
      {
        emsi_scn_state = EMSI_READ;
        /* now we check how to handle it */
        for (i=0;strncmp(emsi_msg+7,msg_typ[i].ide,strlen(msg_typ[i].ide));i++);
        if (msg_typ[i].dta)
        { 
          sscanf(emsi_msg+10,"%04x",&emsi_msg_len);   /*FIXME: improve */
          if (emsi_msg_len <= 2048) 
          {
            emsi_msg_len  += msg_typ[i].c32;
            emsi_msg_len  += EMSI_HDR_LEN+1;     /* means cr included */
            emsi_scn_pos   = EMSI_HDR_LEN;   
            emsi_scn_state = EMSI_READ;
          }
          else
          {
            emsi_scn_state = EMSI_SCAN;
            delete this; // error,leave //FIXME: proper error handling,NAK
          }
        }
        else
        {
          emsi_msg_len = 0;
          emsi_msg_len  += EMSI_HDR_LEN+1; /* means cr included */
          emsi_scn_pos   = EMSI_HDR_LEN;   
          emsi_scn_state = EMSI_READ;
        }
      }
    }
    break;
    case EMSI_READ : 
    {
      emsi_msg[emsi_scn_pos] = cc; 
      emsi_scn_pos += 1;
      if (emsi_scn_pos >= emsi_msg_len)
      { int i; int ok;
        emsi_msg[emsi_scn_pos] = 0;
        for (i=0;strncmp(emsi_msg+7,msg_typ[i].ide,strlen(msg_typ[i].ide));i++);
        if (msg_typ[i].c32 == 4)
        { unsigned int chk16;
          sscanf(emsi_msg+emsi_msg_len-4-1,"%04x",&chk16);
          ok = (chk16 == crc16(emsi_msg+2,emsi_msg_len-2-4-1));
        }
        else
        { unsigned long chk32;
          sscanf(emsi_msg+emsi_msg_len-8-1,"%08lx",&chk32);
          ok = (chk32 == crc32(emsi_msg+2,emsi_msg_len-2-8-1));
        }
        if (msg_typ[i].dta)
        { int len; sscanf(emsi_msg+10,"%04x",&len);
          protocol(msg_typ[i].typ,ok,emsi_msg+14,len);
        }
        else
          protocol(msg_typ[i].typ,ok,emsi_msg+10,0);
        emsi_scn_state = EMSI_SCAN;
      }
    }
    break;
  }
  return;
}

void Iemsi::stopProtocol()
// signal: terminate yourself
{
  delete this;
}

Iemsi::Iemsi(Modem* mo, IemsiClientInfo* ici)
{
  this->mo = mo;
  strcpy(emsi_msg,"0123456789xxxx"); /* init header */
  emsi_scn_state = EMSI_SCAN;
  /*FIXME: below translation is more complex */
  sprintf(ici_bdy,
          "{%s}"   /* Name         */
          "{%s}"   /* Alias        */
          "{%s}"   /* Location     */
          "{%s}"   /* Data#        */
          "{%s}"   /* Voice#       */
          "{%s}"   /* Password     */
          "{%x}"   /* Birthdate    */
          "{%s}"   /* CrtDef       */
          "{%s}"   /* Protocols    */
          "{%s}"   /* Capabilities */
          "{%s}"   /* Requests     */
          "{%s}"   /* Software     */
          "{%s}",  /* XlatTbl      */
          ici->user.data(),
          ici->alias.data(),
          ici->location.data(),
          ici->data_pno.data(),
          ici->voice_pno.data(),
          ici->password.data(),
          ici->birthdate,
          ici->crtdef.data(),
          ici->protocols.data(),
          ici->capabilities.data(),
          ici->requests.data(),
          ici->software.data(),
          ici->xlattbl.data()
         );
}

void Iemsi::start(char* initial)
// start receiving from modem after 'initial'
{
  while(*initial) rcv_byte(*initial++);
  connect( mo, SIGNAL(data_in(int)), this, SLOT(rcv_byte(int)) );
}

Iemsi::~Iemsi()
{
  doneProtocol(); /* leave */
}
/*
char* test_isi =
"{Feld 1}"
"{Feld 2}"
"{Feld 3}"
"{Feld 4}"
"{0}"
"{Feld 6}"
"{Feld 7}"
"{Feld 8}";
send(EMSI_ISI,test_isi);
*/
