/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [modem.cpp]                   Modem Control                                */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* Copyright (c) 1997 by Lars Doelle                                          */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is available under the GNU Public Software Licence.      */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

//FIXME: add some non-configured dummy modem material

/* some posix terminal handling taken from minicom. */

#include <stdio.h>
#include <stdlib.h>

#include <termios.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include <assert.h>
#include <time.h>

#include "modem.h"

#include "modem.moc"

#define HERE fprintf(stdout,"%s(%d): here\n",__FILE__,__LINE__)

/* ---| Modem Parameters |--------------------------------------------------- */

int Modem::getdcd()
/* Get the data carrier detect.      */
/* i.e. tells if we are connected.   */
/* this is platform depended (sorry) */
/* Is it right, by the way?          */
{ unsigned int value;
  if (cmdev) return 1;
  ioctl(fd, TIOCMGET, &value);
  return (value & TIOCM_CAR) ? 1 : 0;
}

/* should all be POSIX */

void Modem::dtrtoggle() 
/* Drop DTR line and raise it again. */
/* Used to initialize and to hangup. */
{ struct termios tty, old;
  if (cmdev) return;
  /* Posix - set baudrate to 0 and back */
  tcgetattr(fd, &tty);
  tcgetattr(fd, &old);
  cfsetospeed(&tty, B0);
  cfsetispeed(&tty, B0);
  tcsetattr(fd, TCSANOW, &tty);
  //FIXME: split into two routines to avoid application blocking for 1 sec.
  sleep(2);
  tcsetattr(fd, TCSANOW, &old);
}

void Modem::setparms(long baudr, char par, int bits, bool hwf, bool swf)
/* Set baudrate, parity and number of bits. */
{
  int spd = -1;

  struct termios tty;

  tcgetattr(fd, &tty);

  switch(baudr)
  {
    case 0:       spd = B0;      break;
    case 300:     spd = B300;    break;
    case 600:     spd = B600;    break;
    case 1200:    spd = B1200;   break;
    case 2400:    spd = B2400;   break;
    case 4800:    spd = B4800;   break;
    case 9600:    spd = B9600;   break;
    case 19200:   spd = B19200;  break;
    case 38400:   spd = B38400;  break;
    case 57600:   spd = B57600;  break;
    case 115200:  spd = B115200; break;
  }

  if (spd != -1)
  {
    cfsetospeed(&tty, (speed_t)spd);
    cfsetispeed(&tty, (speed_t)spd);
  }

  switch (bits) {
   case 5:
    tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS5;
    break;
   case 6:
    tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS6;
    break;
   case 7:
    tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS7;
    break;
   case 8:
   default:
    tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS8;
    break;
  }  
  /* Set into raw, no echo mode */
  tty.c_iflag = IGNBRK;
  tty.c_lflag = 0;
  tty.c_oflag = 0;
  tty.c_cflag |= CLOCAL | CREAD;
  tty.c_cc[VMIN] = 1;
  tty.c_cc[VTIME] = 5;

  if (swf)
    tty.c_iflag |= IXON | IXOFF;
  else
    tty.c_iflag &= ~(IXON|IXOFF|IXANY);

  tty.c_cflag &= ~(PARENB | PARODD);
  if (par == 'E') tty.c_cflag |= PARENB;
  if (par == 'O') tty.c_cflag |= PARODD;

  if (hwf)
	tty.c_cflag |= CRTSCTS;
  else
	tty.c_cflag &= ~CRTSCTS;

  tcsetattr(fd, TCSANOW, &tty);
}

Modem::Modem
       ( const char* device = "/dev/modem",
         long baudr   = 38400,
         char par     = 'N',
         int  bits    = 8,
         bool hwf     = TRUE, 
         bool swf     = FALSE
       )
/* setup modem */
{
fprintf(stderr,"device = %s\n",device);
  cmdev = 0; // !strcmp("none",device);
  fd = open(device, O_RDWR);
  assert( fd > 0 ); //FIXME: catch user's bug
  assert( 0 == tcgetattr( fd, &tp ) );
  setparms(baudr, par, bits, hwf, swf);
  status = connected();
  dtrtoggle();
  mn = new QSocketNotifier(fd, QSocketNotifier::Read);
  mw = new QSocketNotifier(fd, QSocketNotifier::Write);
  connect( mn, SIGNAL(activated(int)), this, SLOT(DataReceived(int)) );
  connect( mw, SIGNAL(activated(int)), this, SLOT(DataWritten(int)) );
  tick = new QTimer();
  connect( tick, SIGNAL(timeout()), this, SLOT(chk_connect()));
  tick->start(1000); // once a second
}

Modem::~Modem()
/* restore modem */
{
  delete mn;
  delete tick;
  if (cmdev) return;
  assert( 0 == tcsetattr( fd, TCSANOW, &tp ) );
}

void Modem::send_byte(char c)
{ 
  write(fd,&c,1); mw->setEnabled(TRUE);
}

void Modem::send_string(const char* s)
{
  write(fd,s,strlen(s)); mw->setEnabled(TRUE);
}

void Modem::send_bytes(char* s, int len)
{
  write(fd,s,len); mw->setEnabled(TRUE);
}

bool Modem::connected()
{
  return getdcd() != 0;
}

void Modem::hangup()
{
  dtrtoggle();
}

void Modem::chk_connect()
{
  if (status != getdcd())
  {
    status = !status;
    emit chgConnect(status);
  }
}

void Modem::DataReceived(int fd)
{ char buf[128]; char *s;
  int n = read(fd, buf, 128);
  for (int i = 0; i < n; i++)
  {
    data_in(buf[i]);
//  if (log) fputc(buf[i],log);
  }
}

void Modem::DataWritten(int fd)
{
  mw->setEnabled(FALSE); written();
}
