/***************************************************************************
                          bibletime.cpp  -  description                              
                             -------------------                                         
    begin                : Tue Jun 15 15:34:26 CEST 1999
                                           
    copyright            : (C) 1999 by Torsten Uhlmann                         
    email                : TUhlmann@gmx.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "bibletime.h"
#include "bibletimeview.h"
#include "bibletimedoc.h"
#include "ktipofday.h"
#include "optionsdialog/coptionsdialog.h"
#include "cbugreportdlg.h"

#include <qmsgbox.h>

BibleTimeApp::BibleTimeApp()
{
	config=kapp->getConfig();
	fontList = 0;
	key_accel=0;
		
  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initMenuBar();
  initToolBar();
  initStatusBar();
  initDocument();
  initView();
  initPrinter();
	readOptions();
  view->readOptions();

  initKeyAccel();
  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar items at startup
	disableCommand(ID_FILE_PRINT);
	disableCommand(ID_FILE_EMPTY_PRINT_QUEUE);
	
	disableCommand(ID_EDIT_COPY);
	disableCommand(ID_EDIT_CUT);
	disableCommand(ID_EDIT_PASTE);	

	browserHasSelection=false;
	notesEditorHasSelection=false;
}

BibleTimeApp::~BibleTimeApp()
{
}

void BibleTimeApp::initKeyAccel()
{
	key_accel = new CKNewAccel( this );

//file menu
	key_accel->connectItem( KAccel::Close, 	this, SLOT( slotFileClose() ),true, ID_FILE_CLOSE );
	key_accel->connectItem( KAccel::Print, 	this, SLOT( slotFilePrint() ),true, ID_FILE_PRINT );
	key_accel->connectItem( KAccel::Find, 	view, SLOT( slotSearchText()),true, ID_FILE_SEARCH );
	key_accel->connectItem( KAccel::Quit, 	this, SLOT( slotFileQuit() ),	true, ID_FILE_QUIT );

//edit menu	
	key_accel->connectItem( KAccel::Copy, 	view, SLOT( slotCopy() ),	true, ID_EDIT_COPY );
	key_accel->connectItem( KAccel::Cut, 		view, SLOT( slotCut() ),	true, ID_EDIT_CUT );
	key_accel->connectItem( KAccel::Paste, 	this, SLOT( slotEditPaste() ),true, ID_EDIT_PASTE ); 	

//view menu
	key_accel->insertItem(i18n("Show / hide noteseditor"), "notes_editor", IDK_VIEW_NOTES_EDITOR);
	key_accel->connectItem("notes_editor", this, SLOT(slotViewTNotesEditor()), true, ID_VIEW_NOTES_EDITOR);
	
	key_accel->insertItem(i18n("Show browser maximized /normal"), "browser", IDK_VIEW_ONLY_BROWSER);
	key_accel->connectItem("browser", this, SLOT(slotViewBigBrowser()), true, ID_VIEW_BIG_BROWSER);

	key_accel->insertItem(i18n("Toggle text toolbar"), "toggle_text_toolbar", IDK_VIEW_TEXT_TOOLBAR);
	key_accel->connectItem("toggle_text_toolbar", this, SLOT(slotViewTextToolBar()), true, ID_VIEW_TEXT_TOOLBAR);
	
	key_accel->insertItem(i18n("Show strongs"), "show_strongs", IDK_VIEW_DISPLAY_STRONGS);
	key_accel->connectItem("show_strongs", this, SLOT(slotViewDisplayStrongs()), true, ID_VIEW_DISPLAY_STRONGS);

	key_accel->insertItem(i18n("Show footnotes"), "show_footnotes", IDK_VIEW_DISPLAY_FOOTNOTES);
	key_accel->connectItem("show_footnotes", this, SLOT(slotViewDisplayFootnotes()), true, ID_VIEW_DISPLAY_FOOTNOTES);

//help menu	
 	key_accel->connectItem( KAccel::Help,	this, SLOT( slotHelpContents() ),true, ID_HELP_CONTENTS );

	
//accels for the toolbar
	key_accel->insertItem(i18n("Move to previous verse"), "previous_verse", IDK_MOVE_TO_PREV_VERSE);
	key_accel->connectItem("previous_verse", doc, SLOT(slotMoveToPreviousVerse()), true, ID_TOOLBAR_PREV_VERSE);
	key_accel->insertItem(i18n("Move to next verse"), "next_verse", IDK_MOVE_TO_NEXT_VERSE);
	key_accel->connectItem("next_verse", doc, SLOT(slotMoveToNextVerse()), true, ID_TOOLBAR_NEXT_VERSE);

	key_accel->insertItem(i18n("Change to previous chapter"), "previous_chapter", IDK_MOVE_TO_PREV_CHAPTER);
	key_accel->connectItem("previous_chapter",doc, SLOT(slotMoveToPreviousChapter()),true, ID_TOOLBAR_PREV_CHAPTER);
	key_accel->insertItem(i18n("Change to next chapter"), "next_chapter", IDK_MOVE_TO_NEXT_CHAPTER);
	key_accel->connectItem("next_chapter",doc, SLOT(slotMoveToNextChapter()),true, ID_TOOLBAR_NEXT_CHAPTER);

	key_accel->insertItem(i18n("Move to previous book"), "previous_book", IDK_MOVE_TO_PREV_BOOK);
	key_accel->connectItem("previous_book", doc, SLOT(slotMoveToPreviousBook()), true, ID_TOOLBAR_PREV_BOOK);
	key_accel->insertItem(i18n("Move to next book"), "next_book", IDK_MOVE_TO_NEXT_BOOK);
	key_accel->connectItem("next_book", doc, SLOT(slotMoveToNextBook()), true, ID_TOOLBAR_NEXT_BOOK);
	

	key_accel->insertItem(i18n("Toggle instant lookup"), "instant_lookup", IDK_TOOLBAR_INSTANT_LOOKUP);
	key_accel->connectItem("instant_lookup", this, SLOT(slotToggleLookupCheckBox()), true);

//update the entries of the menus
 	key_accel->readSettings(config, true);
	this->setKeyAccels();
}

void BibleTimeApp::initMenuBar()
{
  ///////////////////////////////////////////////////////////////////
  // menuBar entry file_menu
  file_menu = new QPopupMenu();
  file_menu->insertItem(Icon("mini/mini-bibletime.xpm"), i18n("New &Window"), ID_FILE_NEW_WINDOW );
  file_menu->insertSeparator();
  file_menu->insertItem(i18n("&Close"), ID_FILE_CLOSE );
  file_menu->insertSeparator();
  file_menu->insertItem(Icon("mini-print.xpm"), i18n("&Print..."), ID_FILE_PRINT );
  file_menu->insertItem(Icon("mini-bomb.xpm"), i18n("Empty Print &Queue..."), ID_FILE_EMPTY_PRINT_QUEUE );
  file_menu->insertSeparator();
  file_menu->insertItem(Icon("mini-lookup.xpm"), i18n("&Search the viewed text..."), ID_FILE_SEARCH );
  file_menu->insertSeparator();
  file_menu->insertItem(i18n("E&xit"), ID_FILE_QUIT );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry edit_menu
  edit_menu=new QPopupMenu();
  edit_menu->insertItem(ICON("mini-copy.xpm"),i18n("Copy"), ID_EDIT_COPY);
  edit_menu->insertItem(ICON("mini-cut.xpm"),i18n("Cut"), ID_EDIT_CUT);
  edit_menu->insertItem(ICON("mini-paste.xpm"),i18n("Paste"), ID_EDIT_PASTE);
//  this->slotClipboardChanged();
  ///////////////////////////////////////////////////////////////////
  // menuBar entry view_menu
  view_menu = new QPopupMenu();
  view_menu->setCheckable(true);
  view_menu->insertItem(i18n("&Generic Toolbar")	, ID_VIEW_TOOLBAR);
  view_menu->insertItem(i18n("&Text Toolbar")			,	ID_VIEW_TEXT_TOOLBAR);
  view_menu->insertItem(i18n("&Statusbar")				, ID_VIEW_STATUSBAR );
  view_menu->insertSeparator();
  view_menu->insertItem(i18n("Notes &Editor")			, ID_VIEW_NOTES_EDITOR );
  view_menu->insertItem(i18n("Big &browser")			,	ID_VIEW_BIG_BROWSER );
  view_menu->insertSeparator();
  view_menu->insertItem(i18n("Display Strong &Numbers")	, ID_VIEW_DISPLAY_STRONGS );
  view_menu->insertItem(i18n("Display &Footnotes")			, ID_VIEW_DISPLAY_FOOTNOTES );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry settings_menu
  settings_menu = new QPopupMenu();
  settings_menu->insertItem(i18n("Notes Editor..."), ID_SETTINGS_EDITOR	);
  settings_menu->insertItem(i18n("Editor Colors..."), ID_SETTINGS_EDITOR_COLORS );
  settings_menu->insertItem(i18n("Editor Defaults..."), ID_SETTINGS_EDITOR_SYHI_DEF );
  settings_menu->insertItem(i18n("Syntax Highlighting..."), ID_SETTINGS_EDITOR_SYHI );
  settings_menu->insertSeparator();
  settings_menu->insertItem(i18n("Options..."), ID_SETTINGS_OPTIONS );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry help_menu
	help_menu = new QPopupMenu();
	help_menu ->insertItem(Icon("help.xpm"),i18n("Help..."), ID_HELP_CONTENTS);
	help_menu ->insertItem(Icon("mini-bug1.xpm"),i18n("Bug report..."), ID_HELP_BUG_REPORT);
	help_menu ->insertItem(i18n("Tip of the day"), ID_HELP_TIP_OF_DAY);
	help_menu ->insertSeparator();
	help_menu ->insertItem(Icon("mini/mini-bibletime.xpm"),i18n("About BibleTime..."), ID_HELP_ABOUT);
	help_menu ->insertItem(i18n("About KDE..."), kapp, SLOT(aboutKDE()));

  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // insert your popup menus with the according menubar entries in the order
  // they will appear later from left to right
  menuBar()->insertItem(i18n("&File"), file_menu);
  menuBar()->insertItem(i18n("Edit"), edit_menu);
  menuBar()->insertItem(i18n("&View"), view_menu);
  menuBar()->insertItem(i18n("&Settings"), settings_menu);
  menuBar()->insertItem(i18n("&Help"), help_menu);

  ///////////////////////////////////////////////////////////////////
  // CONNECT THE MENU SLOTS WITH SIGNALS
  // for execution slots and statusbar messages
  connect(file_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(file_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(edit_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(edit_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(view_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
  connect(view_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(settings_menu,SIGNAL(activated(int))	,SLOT(commandCallback(int)));
  connect(settings_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

  connect(help_menu,SIGNAL(activated(int))	,SLOT(commandCallback(int)));
  connect(help_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));

}

void BibleTimeApp::initToolBar()
{

  ///////////////////////////////////////////////////////////////////
  // TOOLBAR
	toolBar()->insertButton(Icon("mini-print.xpm"), ID_FILE_PRINT, true, i18n("Print") );
	toolBar()->insertButton(Icon("mini-bomb.xpm"), ID_FILE_EMPTY_PRINT_QUEUE, true, i18n("Remove all entries from the print queue") );
  toolBar()->insertButton(Icon("mini-lookup.xpm"), ID_FILE_SEARCH, true, i18n("Search in viewed text") );
  toolBar()->insertSeparator();

	history_list = new HistoryList;
	history_list->setAutoDelete(true);
	
	history_prev = new QPopupMenu();
	toolBar()->insertButton(Icon("back.xpm"), ID_HISTORY_PREV,false, i18n("move backwards"));	
	toolBar()->setDelayedPopup(ID_HISTORY_PREV, history_prev);
	connect(history_prev, SIGNAL(activated(int)), this, SLOT(slotHistoryBack(int)));

	history_next = new QPopupMenu();	
	toolBar()->insertButton(Icon("forward.xpm"), ID_HISTORY_NEXT,false, i18n("move forwards"));	
	toolBar()->setDelayedPopup(ID_HISTORY_NEXT, history_next);
	connect(history_next, SIGNAL(activated(int)), this, SLOT(slotHistoryNext(int)));
	toolBar()->insertSeparator();

  toolBar()->insertButton(Icon("previous_book.xpm"), ID_TOOLBAR_PREV_BOOK, true, i18n("moves to the previous book"));
  toolBar()->insertButton(Icon("previous_chapter.xpm"), ID_TOOLBAR_PREV_CHAPTER, true, i18n("moves to the previous chapter"));
  toolBar()->insertButton(Icon("previous_verse.xpm"), ID_TOOLBAR_PREV_VERSE, true, i18n("moves to the previous verse"));
	toolBar()->insertButton(Icon("next_verse.xpm"),  ID_TOOLBAR_NEXT_VERSE, true, i18n("moves to the next verse"));
	toolBar()->insertButton(Icon("next_chapter.xpm"),  ID_TOOLBAR_NEXT_CHAPTER, true, i18n("moves to the next chapter"));
  toolBar()->insertButton(Icon("next_book.xpm"),  ID_TOOLBAR_NEXT_BOOK, true, i18n("moves to the next book"));
	toolBar()->insertSeparator();

  toolBar()->insertButton(Icon("mini-help.xpm"), ID_HELP_CONTENTS, true, i18n("Help"));


  ///////////////////////////////////////////////////////////////////
  // INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE WITH toolBar(n)
	lookupLine = new KLined(toolBar(ID_TEXT_TOOLBAR));
	lookupLine->setFocusPolicy( QWidget::StrongFocus );
	lookupLine->resize(175,70);
	toolBar(ID_TEXT_TOOLBAR)->insertWidget(ID_TOOLBAR_LOOKUP_LINE,175, lookupLine, -1);
	KQuickHelp::add(lookupLine, i18n("Type in the <b>versename</b> or the <b>keyword</b>\nyou want to see in the browser widget."));
	
	toolBar(ID_TEXT_TOOLBAR)->insertSeparator();

  lookupLabel = new QLabel(toolBar(ID_TEXT_TOOLBAR), "lookupLabel");
  lookupLabel->setAlignment(AlignCenter);
  lookupLabel->setFrameStyle( QFrame::Box | QFrame::Sunken );
	toolBar(ID_TEXT_TOOLBAR) -> insertWidget (ID_TOOLBAR_LOOKUP_LABEL, 170, lookupLabel, -1);
	
	toolBar(ID_TEXT_TOOLBAR)->insertSeparator();

	lookupCheckBox= new QCheckBox(toolBar(ID_TEXT_TOOLBAR));
	lookupCheckBox->setText(i18n("Instant lookup"));
	toolBar(ID_TEXT_TOOLBAR)->insertWidget(ID_TOOLBAR_LOOKUP_BOX, 150, lookupCheckBox, -1);
	KQuickHelp::add(lookupCheckBox, i18n("Enable this if you want to see browserwidget\nupdated every time if you change the contents."));

	connect(lookupLine, SIGNAL(returnPressed()), this, SLOT(slotLookupReturnPressed()));	
	connect(lookupCheckBox, SIGNAL(clicked()), this, SLOT(slotLookupCheckBoxClicked()));


  ///////////////////////////////////////////////////////////////////
  // CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars by their according number
	// connect for invoking the slot actions
  connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
  // connect for the status help on holding icons pressed with the mouse button
  connect(toolBar(), SIGNAL(pressed(int)), SLOT(statusCallback(int)));
}

void BibleTimeApp::initStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  // STATUSBAR
	// TODO: add your own items you need for displaying current application status.
  statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG );
}

void BibleTimeApp::initDocument()
{
  doc = new BibleTimeDoc(this);
	fontList = doc->setupFontList();
}

void BibleTimeApp::initPrinter()
{
	btPrinter = new BibleTimePrinter(this,"printer");
	verseList = new KeyVerseList;
	verseList->setAutoDelete(true);
	verseList->clear();
}

void BibleTimeApp::initView()
{ 
  ////////////////////////////////////////////////////////////////////
  // create the main widget here that is managed by KTMainWindow's view-region and
  // connect the widget to your document to display document contents.
  view = new BibleTimeView(this);
  doc->addView(view);
  setView(view);	
	setCaption ("BibleTime " VERSION);
	if (fontList != 0)
		view->setBibleFontList( fontList );
	initConnections();
  doc->getModuleList();
  view->setRecentModules();
}

/**  */
void BibleTimeApp::initConnections()
{
  //Global signals
  connect(KApplication::clipboard(),SIGNAL(dataChanged()),
  				this, SLOT(slotClipboardChanged()));

  //doc's signals
	connect (doc,SIGNAL(sigChangeFont(QString, QString)),
  				view, SLOT(slotSetBrowserFont(QString, QString)));
  connect (doc,SIGNAL(sigModuleList(ModuleDefinitionList*)),
  				view, SLOT(slotSetModuleList(ModuleDefinitionList*)));
  connect (doc,SIGNAL(sigCurrentKeyTextChanged(const QString&)),
  				view, SLOT(slotSetCurrentKeyText(const QString&)));  				
  connect (doc,SIGNAL(sigHTMLText(const QString&, const QString&)),
  				view, SLOT(slotSetHTMLText(const QString&, const QString&)));
  connect (doc,SIGNAL(sigSearchResultList(QStrList*)),
  				view, SLOT(slotSetSearchResult(QStrList*)));
  connect (doc,SIGNAL(sigCurrentKeyVerseChanged(const QString&)),
  				this, SLOT(slotSetCurrentKeyVerse(const QString&)));
  connect (doc,SIGNAL(sigCurrentKeyModuleChanged(const QString&, const QString&, const QString&)),
  				this, SLOT(slotSetCurrentKeyModule(const QString&, const QString&, const QString&)));
  connect (doc,SIGNAL(sigCurrentKeyModuleChanged(const QString&, const QString&, const QString&)),
  				view, SLOT(slotSetCurrentKeyModule(const QString&, const QString&, const QString&)));

  //view
  connect(view, SIGNAL(sigNotesEditorSelectionChanged(KWriteView*,bool)),
  				this, SLOT(slotNotesEditorSelectionChanged(KWriteView*,bool)));  				
  connect(view, SIGNAL(sigBrowserSelectionChanged(KHTMLView*, bool)),
  				this, SLOT(slotBrowserSelectionChanged(KHTMLView*, bool)));
  connect (view,SIGNAL(sigModuleChosen(const QString&)),
  				doc, SLOT(slotViewModActivate(const QString&)));
  connect (view,SIGNAL(sigSetCurrentModule(const QString&)),
  				doc, SLOT(slotSetCurrentModule(const QString&)));  				
  connect (view, SIGNAL(sigLookupTextChanged(const char*)),
  				this, SLOT(slotLookupTextChanged(const char*)));  				
  connect (view, SIGNAL(sigCurrentKeyRequested()),
  				doc, SLOT(slotCurrentKeyRequested()));  				
  connect (view, SIGNAL(sigSearchCurrentModule(BibleTimeApp::SearchType, bool, const QString&)),
  				doc, SLOT(slotSearchCurrentModule(BibleTimeApp::SearchType, bool, const QString&)));
  connect (view, SIGNAL(sigSetSearchScope()),	doc, SLOT(slotSetSearchScope()));
  connect (view, SIGNAL(sigSetSearchScope(const QString&, const QString&)),	
  				doc, SLOT(slotSetSearchScope(const QString&, const QString&)));
  connect (view, SIGNAL(sigResetSearchScope()), doc, SLOT(slotResetSearchScope()));  				
  connect (view, SIGNAL(sigResultListSelection(const char*)),
  				doc, SLOT(slotResultListSelectionChanged(const char*)));		
  connect (view, SIGNAL(sigSearchInModule(const QString&, const QString&)),
  				doc, SLOT(slotSearchInModule(const QString&, const QString&)));
  connect (view, SIGNAL(sigPersonalCommentAdd(const QString&, const QString&, const QString&, const QString&)),
  				doc, SLOT(slotPersonalCommentAdd(const QString&, const QString&, const QString&, const QString&)));  				
  connect (view, SIGNAL(sigPersonalCommentRemove(const QString&, const QString&, const QString&)),
  				doc, SLOT(slotPersonalCommentRemove(const QString&, const QString&, const QString&)));
	connect (view, SIGNAL(sigRequestBrowserTextForEditor(const QString&)),
					this, SLOT(slotRequestBrowserTextForEditor(const QString&)));
  								
	//connect the signals emitted by the popupmenu of the resultlist
	connect(view, SIGNAL(sigSaveListToFile(QStrList*, bool)),
					this, SLOT(slotSaveListToFile(QStrList*, bool)));
	connect(view, SIGNAL(sigCopyListToClipboard(QStrList*, bool)),
					this, SLOT(slotCopyListToClipboard(QStrList*, bool)));
	connect(view, SIGNAL(sigPrintList(QStrList*, bool)),
					this, SLOT(slotPrintList(QStrList*, bool)));
	connect(view, SIGNAL(sigPrintText( const QString&, bool, const QString&, bool )),
					this, SLOT(slotPrintText( const QString&, bool, const QString&, bool )));

														
	connect(view, SIGNAL(sigStatusbarMsg(const char*)),
					this, SLOT(slotStatusHelpMsg(const char*)));
	connect(view, SIGNAL(sigTTabChanged(int)),
					this, SLOT(slotTTabChanged(int)));
}


void BibleTimeApp::enableCommand(int id_)
{
  ///////////////////////////////////////////////////////////////////
  // enable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id_,true);
  toolBar()->setItemEnabled(id_,true);
	if (key_accel!=0)
	  key_accel->setItemEnabled(id_,true);
}

void BibleTimeApp::disableCommand(int id_)
{
  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id_,false);
  toolBar()->setItemEnabled(id_,false);
  if (key_accel!=0)
	  key_accel->setItemEnabled(id_,false);
}


BibleTimeDoc* BibleTimeApp::getDocument() const
{
	return doc;
}

void BibleTimeApp::setOptions()
{
	slotSetCurrentTabs();
	
	config->setGroup("General Options");
	bool bViewDisplayStrongs = config->readBoolEntry("Display Strong Numbers", true);
	view_menu->setItemChecked(ID_VIEW_DISPLAY_STRONGS, false);
	if(bViewDisplayStrongs)
		slotViewDisplayStrongs();
		
	config->setGroup("General Options");
	
	bool bViewDisplayFootnotes = config->readBoolEntry("Display Footnotes", true);
	view_menu->setItemChecked(ID_VIEW_DISPLAY_FOOTNOTES, false);
	if(bViewDisplayFootnotes)
		slotViewDisplayFootnotes();
}

void BibleTimeApp::saveOptions()
{	
	config->setGroup("General Options");
	config->writeEntry("Geometry", this->size() );
  config->writeEntry("Show Toolbar", toolBar()->isVisible());
  config->writeEntry("Show Text Toolbar", toolBar(ID_TEXT_TOOLBAR)->isVisible());
  config->writeEntry("Show Statusbar",statusBar()->isVisible());
	config->writeEntry("Display Strong Numbers",view_menu->isItemChecked(ID_VIEW_DISPLAY_STRONGS));
	config->writeEntry("Display Footnotes",view_menu->isItemChecked(ID_VIEW_DISPLAY_FOOTNOTES));
  config->writeEntry("MenuBarPos", (int)menuBar()->menuBarPos());
  config->writeEntry("ToolBarPos",  (int)toolBar()->barPos());
  config->writeEntry("Instant Lookup", lookupCheckBox->isChecked());
  config->writeEntry("Show Notes Editor",view_menu->isItemChecked(ID_VIEW_NOTES_EDITOR));
	config->writeEntry("Big Browser",view_menu->isItemChecked(ID_VIEW_BIG_BROWSER));

	//save the positions of the panners, the current tabs etc.
	view->saveOptions();
}

void BibleTimeApp::readOptions()
{
	config->setGroup("General Options");

	// bar status settings
	bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
	view_menu->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
	if(!bViewToolbar)
		enableToolBar(KToolBar::Hide);

	bool bViewTextToolbar = config->readBoolEntry("Show Text Toolbar", true);
	view_menu->setItemChecked(ID_VIEW_TEXT_TOOLBAR, bViewTextToolbar);
	if(!bViewTextToolbar)
		enableToolBar(KToolBar::Hide, ID_TEXT_TOOLBAR);
	
  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
	view_menu->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
	if(!bViewStatusbar)
		enableStatusBar(KStatusBar::Hide);
	
  bool bViewNotesEditor = config->readBoolEntry("Show Notes Editor", true);
	view_menu->setItemChecked(ID_VIEW_NOTES_EDITOR, bViewNotesEditor);

	bool bViewBigBrowser = config->readBoolEntry("Big Browser", false);
	view_menu->setItemChecked(ID_VIEW_BIG_BROWSER, bViewBigBrowser);

	bool bInstantLookup = config->readBoolEntry("Instant Lookup", false);
	lookupCheckBox->setChecked(bInstantLookup);
	slotLookupCheckBoxClicked();
	
	// bar position settings
	KMenuBar::menuPosition menu_bar_pos;
	menu_bar_pos=(KMenuBar::menuPosition)config->readNumEntry("MenuBarPos", KMenuBar::Top);

  KToolBar::BarPosition tool_bar_pos;
  tool_bar_pos=(KToolBar::BarPosition)config->readNumEntry("ToolBarPos", KToolBar::Top);

	menuBar()->setMenuBarPos(menu_bar_pos);
  toolBar()->setBarPos(tool_bar_pos);
	
  QSize size=config->readSizeEntry("Geometry");
	if(!size.isEmpty())
		resize(size);
	else
		setGeometry(0,0,640,480);
}

bool BibleTimeApp::queryClose()
{
	return true;
}

bool BibleTimeApp::queryExit()
{
	saveOptions();
	return true;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void BibleTimeApp::slotFileNewWindow()
{
  slotStatusMsg(i18n("Opening a new Application window..."));
	this->saveOptions();
	
	BibleTimeApp* new_window= new BibleTimeApp();
	new_window->show();
	
  slotStatusMsg(i18n("Ready."));
}


void BibleTimeApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing file..."));
	
	close();
  slotStatusMsg(i18n("Ready."));
}

void BibleTimeApp::slotFilePrint()
{
  slotStatusMsg(i18n("Printing..."));
	
	btPrinter->slotSetPrintData(verseList);
	if (btPrinter->printQueue())
	{
		verseList->clear();
		disableCommand(ID_FILE_PRINT);
		disableCommand(ID_FILE_EMPTY_PRINT_QUEUE);
	}

  slotStatusMsg(i18n("Ready."));
}

void BibleTimeApp::slotFileEmptyPrintQueue()
{
	verseList->clear();
	disableCommand(ID_FILE_PRINT);
	disableCommand(ID_FILE_EMPTY_PRINT_QUEUE);
}

void BibleTimeApp::slotFileQuit()
{
	// close the first window, the list makes the next one the first again.
	// This ensures that queryClose() is called on each window to ask for closing
	KTMainWindow* w;
	if(memberList){
		for(w=memberList->first(); w; w=memberList->first()){
			// only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
			// the window and the application stay open.
			if(!w->close())
				break;
		}
	}	
}

void BibleTimeApp::slotEditPaste()
{
	if (lookupLine->hasFocus())
		lookupLine->paste();
	else
		view->slotPaste();
};

void BibleTimeApp::slotViewToolBar()
{
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
	if( view_menu->isItemChecked(ID_VIEW_TOOLBAR))
	{
	  view_menu->setItemChecked(ID_VIEW_TOOLBAR, false);
	   enableToolBar(KToolBar::Hide);
	}
	else
	{
		view_menu->setItemChecked(ID_VIEW_TOOLBAR, true);
	  enableToolBar(KToolBar::Show);
	};

  slotStatusMsg(i18n("Ready."));
}

void BibleTimeApp::slotViewTextToolBar()
{
 	if( view_menu->isItemChecked(ID_VIEW_TEXT_TOOLBAR))
	  view_menu->setItemChecked(ID_VIEW_TEXT_TOOLBAR, false);
	else
		view_menu->setItemChecked(ID_VIEW_TEXT_TOOLBAR, true);
		
  enableToolBar(KToolBar::Toggle,ID_TEXT_TOOLBAR);

  slotStatusMsg(i18n("Ready."));
};

void BibleTimeApp::slotViewStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
	if( view_menu->isItemChecked(ID_VIEW_STATUSBAR))
	  view_menu->setItemChecked(ID_VIEW_STATUSBAR, false);
	else
		view_menu->setItemChecked(ID_VIEW_STATUSBAR, true);

  enableStatusBar();

  slotStatusMsg(i18n("Ready."));
}


void BibleTimeApp::slotStatusMsg(const char *text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG );
}


void BibleTimeApp::slotStatusHelpMsg(const char *text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message of whole statusbar temporary (text, msec)
  statusBar()->message(text, 2000);
}


void BibleTimeApp::commandCallback(int id_){
  switch (id_){
    case ID_FILE_NEW_WINDOW:
    	slotFileNewWindow();
	    break;
    case ID_FILE_CLOSE:
    	slotFileClose();
	    break;
		case ID_FILE_PRINT:
			slotFilePrint();
			break;
		case ID_FILE_EMPTY_PRINT_QUEUE:
			slotFileEmptyPrintQueue();
			break;
    case ID_FILE_SEARCH:
    	view->slotSearchText();
    	break;
    case ID_FILE_QUIT:
    	slotFileQuit();
    	break;

    case ID_EDIT_CUT:								
    	view->slotCut();						
    	break;
    case ID_EDIT_COPY:							
	    view->slotCopy();						
	    break;
    case ID_EDIT_PASTE: 						
    	this->slotEditPaste();					
    	break;
  
    case ID_VIEW_TOOLBAR:
    	slotViewToolBar();    	
    	break;
    case ID_VIEW_TEXT_TOOLBAR:
    	slotViewTextToolBar(); 	
    	break;
    case ID_VIEW_STATUSBAR:
    	slotViewStatusBar();		
    	break;
    case ID_VIEW_NOTES_EDITOR:
    	slotViewTNotesEditor();	
    	break;
		case ID_VIEW_BIG_BROWSER:
      slotViewBigBrowser();
      break;    	
		case ID_VIEW_DISPLAY_STRONGS:
      slotViewDisplayStrongs();
      break;    	
		case ID_VIEW_DISPLAY_FOOTNOTES:
      slotViewDisplayFootnotes();
      break;    	

    case ID_SETTINGS_EDITOR:
    	view->notesEditor->slotSettingsEditor();				
    	break;
    case ID_SETTINGS_EDITOR_COLORS:
    	view->notesEditor->slotSettingsEditorColors();	
    	break;
    case ID_SETTINGS_EDITOR_SYHI_DEF:
    	view->notesEditor->slotSettingsSyntaxHighlightingDefaults();	
    	break;
    case ID_SETTINGS_EDITOR_SYHI:
    	view->notesEditor->slotSettingsSyntaxHighlighting();	
    	break;
    case ID_SETTINGS_OPTIONS:
    	slotSettingsOptions();
    	break;

    case ID_HELP_CONTENTS:
    	slotHelpContents();
    	break;
    case ID_HELP_TIP_OF_DAY:
    	slotHelpTipOfDay();
    	break;
    case ID_HELP_BUG_REPORT:
      this->slotHelpBugReport();
      break;
    case ID_HELP_ABOUT:
      this->slotHelpAbout();
      break;

		case ID_TOOLBAR_PREV_VERSE:
			doc->slotMoveToPreviousVerse();
			break;
		case ID_TOOLBAR_NEXT_VERSE:
			doc->slotMoveToNextVerse();
			break;
		case ID_TOOLBAR_PREV_CHAPTER:
			doc->slotMoveToPreviousChapter();
			break;
		case ID_TOOLBAR_NEXT_CHAPTER:
			doc->slotMoveToNextChapter();
			break;
		case ID_TOOLBAR_PREV_BOOK:
			doc->slotMoveToPreviousBook();
			break;
		case ID_TOOLBAR_NEXT_BOOK:
			doc->slotMoveToNextBook();
			break;
		
		case ID_HISTORY_PREV:
			slotHistoryMoveToPrev();
			break;
		case ID_HISTORY_NEXT:
			slotHistoryMoveToNext();
			break;
				
    default: break;
  }
}

void BibleTimeApp::statusCallback(int id_){
  switch (id_){
    case ID_FILE_NEW_WINDOW:
    	slotStatusHelpMsg(i18n("Opens a new application window"));
    	break;
    case ID_FILE_CLOSE:
    	slotStatusHelpMsg(i18n("Closes the actual file"));
    	break;
    case ID_FILE_PRINT:
    	slotStatusHelpMsg(i18n("Prints the current document"));
    	break;
    case ID_FILE_EMPTY_PRINT_QUEUE:
    	slotStatusHelpMsg(i18n("Remove all entries from the print queue"));
    	break;
    case ID_FILE_SEARCH:
    	slotStatusHelpMsg(i18n("Search in the viewed text"));
    	break;
    case ID_FILE_QUIT:
    	slotStatusHelpMsg(i18n("Exits " + kapp->appName() ));
    	break;

    case ID_EDIT_CUT:
    	slotStatusHelpMsg(i18n("Cuts the selected section and puts it to the clipboard"));
    	break;
    case ID_EDIT_COPY:
    	slotStatusHelpMsg(i18n("Copys the selected section to the clipboard"));
    	break;
    case ID_EDIT_PASTE:
    	slotStatusHelpMsg(i18n("Pastes the clipboard contents to actual position"));
    	break;

    case ID_VIEW_TOOLBAR:
    	slotStatusHelpMsg(i18n("Enables / disables the actual Toolbar"));	
    	break;
    case ID_VIEW_STATUSBAR:
     	slotStatusHelpMsg(i18n("Enables / disables the actual Statusbar"));	
     	break;
    case ID_VIEW_NOTES_EDITOR:
    	slotStatusHelpMsg(i18n("Enables / disables the Notes Editor view"));	
    	break;
		case ID_VIEW_BIG_BROWSER:
    	slotStatusHelpMsg(i18n("Maximize / minimize the browsers"));
    	break;
		case ID_VIEW_DISPLAY_STRONGS:
    	slotStatusHelpMsg(i18n("Enable / disable Strongs Numbers where the are present"));	
    	break;
		case ID_VIEW_DISPLAY_FOOTNOTES:
    	slotStatusHelpMsg(i18n("Enable / disable Footnotes where the are present"));	
    	break;
    	
    case ID_SETTINGS_EDITOR:
    	slotStatusHelpMsg(i18n("Customize the notes editor"));
    	break;
    case ID_SETTINGS_EDITOR_COLORS:
    	slotStatusHelpMsg(i18n("Customize editors colors"));
    	break;
    case ID_SETTINGS_EDITOR_SYHI_DEF:
    	slotStatusHelpMsg(i18n("Customize editor defaults"));
    	break;
    case ID_SETTINGS_EDITOR_SYHI:
    	slotStatusHelpMsg(i18n("Customize syntax highlighting"));
    	break;
    case ID_SETTINGS_BROWSER:
    	slotStatusHelpMsg(i18n("Customize the text browser"));
    	break;
    case ID_SETTINGS_OPTIONS:
    	slotStatusHelpMsg(i18n("Customize BibleTime"));
    	break;
		
		case ID_HELP_BUG_REPORT:
			slotStatusHelpMsg(i18n("Opens a window to report a bug"));
			break;
		case ID_HELP_ABOUT:
			slotStatusHelpMsg(i18n("Shows information about BibleTime"));
			break;
			
		case ID_HISTORY_PREV:
			slotStatusHelpMsg(i18n("Moves backwards in history"));
			break;
		case ID_HISTORY_NEXT:
			slotStatusHelpMsg(i18n("Moves forward in history"));
			break;
		
		case ID_TOOLBAR_PREV_VERSE:
			slotStatusHelpMsg(i18n("Shows the previous verse"));
			break;
		case ID_TOOLBAR_NEXT_VERSE:
			slotStatusHelpMsg(i18n("Shows the next verse"));
			break;			
		case ID_TOOLBAR_PREV_CHAPTER:
			slotStatusHelpMsg(i18n("Shows the previous chapter"));			
			break;
		case ID_TOOLBAR_NEXT_CHAPTER:
			slotStatusHelpMsg(i18n("Shows the next chapter"));			
			break;			
		case ID_TOOLBAR_PREV_BOOK:
			slotStatusHelpMsg(i18n("Shows the previous book"));			
			break;		
		case ID_TOOLBAR_NEXT_BOOK:
			slotStatusHelpMsg(i18n("Shows the next book"));			
			break;
		default:
			break;
  }
}

void BibleTimeApp::slotHelpTipOfDay()
{
	KTipofDay *tipdlg= new KTipofDay(this);
	tipdlg->show();
	
	delete tipdlg;
}

void BibleTimeApp::slotSetCurrentTabs()
{
	view->slotSetCurrentTabs();
}

void BibleTimeApp::slotLookupTextChanged(const char *text)
{
	QString t(text);
	if (t.length() > 20)
	{
		QString tmp = t.left(20) + "...";
		lookupLabel->setText(tmp.data());
	}
	else
		lookupLabel->setText(text);
		
	doc->slotLookupTextChanged(t);
}

void BibleTimeApp::slotLookupKeyChanged(const char *text)
{
	QString t(text);
	QString result = doc->getKeyFromAbbrev(t);
	if (result.length() > 20)
	{
		QString tmp = result.left(20) + "...";
		lookupLabel->setText(tmp.data());
	}
	else
		lookupLabel->setText(result);
}

void BibleTimeApp::slotViewTNotesEditor()
{
  if(view_menu->isItemChecked(ID_VIEW_NOTES_EDITOR))
  {
    view_menu->setItemChecked(ID_VIEW_NOTES_EDITOR,false);
		view->slotViewTNotesEditor(false);
  } else {
    view_menu->setItemChecked(ID_VIEW_NOTES_EDITOR,true);
		view->slotViewTNotesEditor(true);
  }
}

void BibleTimeApp::slotViewBigBrowser()
{
  if(view_menu->isItemChecked(ID_VIEW_BIG_BROWSER))
  {
    view_menu->setItemChecked(ID_VIEW_BIG_BROWSER, false);
    view->slotViewBigBrowser(false);
  }
  else
  {
    view_menu->setItemChecked(ID_VIEW_BIG_BROWSER,true);
    view->slotViewBigBrowser(true);
  }
}

void BibleTimeApp::slotViewDisplayStrongs()
{
  if(view_menu->isItemChecked(ID_VIEW_DISPLAY_STRONGS))
  {
    view_menu->setItemChecked(ID_VIEW_DISPLAY_STRONGS, false);
		doc->slotDisplayStrongs(false);
  } else {
    view_menu->setItemChecked(ID_VIEW_DISPLAY_STRONGS,true);
		doc->slotDisplayStrongs(true);
  }
}

void BibleTimeApp::slotViewDisplayFootnotes()
{
  if(view_menu->isItemChecked(ID_VIEW_DISPLAY_FOOTNOTES))
  {
    view_menu->setItemChecked(ID_VIEW_DISPLAY_FOOTNOTES, false);
		doc->slotDisplayFootnotes(false);
  } else {
    view_menu->setItemChecked(ID_VIEW_DISPLAY_FOOTNOTES,true);
		doc->slotDisplayFootnotes(true);
  }
}

void BibleTimeApp::slotSettingsOptions()
{
	COptionsDialog *dlg = new COptionsDialog(0,key_accel, fontList, "optionsDialog");

	connect(dlg, SIGNAL(sigChangeColors()), doc, SLOT(slotChangeColors()));
	connect(dlg, SIGNAL(sigChangeBrowserColors(const QColor&, const QColor&,const QColor&, const QColor&, const bool)),
					view, SLOT(slotChangeBrowserColors(const QColor&, const QColor&,const QColor&, const QColor&, const bool)));	
	connect(dlg, SIGNAL(sigBrowserStandardFont(const char*)), view, SLOT(slotBrowserStandardFont(const char*)));
	connect(dlg, SIGNAL(sigBrowserFixedFont(const char*)), view, SLOT(slotBrowserFixedFont(const char*)));
	connect(dlg, SIGNAL(sigBrowserFontSize(int)), view, SLOT(slotBrowserFontSize(int)));	
	connect(dlg, SIGNAL(sigPrintOptionsChanged()), btPrinter, SLOT(slotSetDefaults()));
  connect(dlg, SIGNAL(sigRefreshLinkLists()), view, SLOT(slotRefreshLinkLists()));
	
	dlg->setCaption(i18n("BibeTime - optionsdialog"));
	
	dlg->exec();
	delete dlg;

	key_accel->readSettings();
	this->setKeyAccels();
}

void BibleTimeApp::slotHelpAbout()
{
	QMessageBox	about(0, "about");
	about.setCaption(i18n("About BibleTime ..."));
	about.setIconPixmap(ICON(kapp->kde_datadir() + "/bibletime/pics/startuplogo-small.bmp"));
	about.setText( i18n("BibleTime version " VERSION "\n\n\
Homepage:\t\thttp://www.bibletime.de\n\
eMail:\t\tBibleTime@bibletime.de\n\n\
\t\t\t\tAuthors:\n\
Bill Allen 	<wallen@our-town.com>\n\
Joachim Ansorg <jockel123@gmx.de>\n\
Jrg Bernau <joerg@memra.teuto.de>\n\
Bob Boyken <rboyken@bloomington.in.us>\n\
Thomas Hagedorn <tom@delix.de>\n\
Jeffrey Hoyt <jeffhoyt@earthlink.net>\n\
Torsten Uhlmann <tuhlmann@gmx.de>\n\n\n\
This program is covered by the GPL version 2 or any later version.\n\
There's really no warranty for this programm.\n\
Use it at your own risk.") );
		
	about.show();
}

void BibleTimeApp::slotHelpBugReport()
{
	TBugReportInfo info;
	info.kde_version = KDE_VERSION_STRING;

	CBugReportDlg dlg(this,"bug",info,"bibletime@gmx.de");
	dlg.show();
}


void BibleTimeApp::slotSetCurrentKeyModule(const QString &t, const QString &m, const QString &type)
{
	this->addToHistory(t, m);
}


void BibleTimeApp::slotSetCurrentKeyVerse(const QString& t)
{
	this->setCaption(t);
	
	view->slotSetCurrentKeyVerse(t);
	if (!lookupCheckBox->isChecked())	
		lookupLine->setText(t.data());
		
	lookupLabel->setText(t.data());
	if (t.length() > 20)
	{
		QString tmp = t.left(20) + "...";
		lookupLabel->setText(tmp.data());
	}
	else
		lookupLabel->setText(t.data());	
};

void BibleTimeApp::slotRequestBrowserTextForEditor(const QString &key)
{
	view->slotRequestBrowserTextForEditor(doc->getTextForKey(key));
}

void BibleTimeApp::slotLookupReturnPressed()
{
	if (!lookupCheckBox->isChecked())
		view->slotActivateBrowserTab();

	this->slotLookupTextChanged(lookupLine->text());
}


void BibleTimeApp::slotLookupCheckBoxClicked()
{
	if (lookupCheckBox->isChecked())
	{
		connect(lookupLine, SIGNAL(textChanged(const char*)), this, SLOT(slotLookupTextChanged(const char*)));
		disconnect(lookupLine,SIGNAL(textChanged(const char*)),this, SLOT(slotLookupKeyChanged(const char*)));
	}	else {
		connect(lookupLine, SIGNAL(textChanged(const char*)), this, SLOT(slotLookupKeyChanged(const char*)));
		disconnect(lookupLine,SIGNAL(textChanged(const char*)),this, SLOT(slotLookupTextChanged(const char*)));
	}
}


QString BibleTimeApp::createHistoryKey(const QString &key, const QString &module)
{
	QString kt = 0;
	kt.sprintf("%s (%s)",key.data(),module.data());
	return kt;
}

void BibleTimeApp::updateHistory()
{
	if (!history_list->isEmpty())		
	{		
		history_prev->clear();
		history_next->clear();

		int cur=history_list->at();
		int count = 0;
		history_list->next();
		while ((history_list->current()) && (count < 15))
		{
			history_prev->insertItem(createHistoryKey(history_list->current()->key,history_list->current()->module));
			history_list->next();
			count++;
		}
		
		history_list->at(cur);
		history_list->prev();
		count = 0;
		while ((history_list->current()) && (count < 15))
		{
			history_next->insertItem(createHistoryKey(history_list->current()->key,history_list->current()->module));
			history_list->prev();
			count++;
		}
		
		history_list->at(cur);
		
		//now check if the items had to be disabled or enabled			
		if (history_prev->count() == 0)	disableCommand(ID_HISTORY_PREV); else enableCommand(ID_HISTORY_PREV);
		if (history_next->count() == 0)	disableCommand(ID_HISTORY_NEXT); else enableCommand(ID_HISTORY_NEXT);
	}
}

void BibleTimeApp::addToHistory ( const QString &key, const QString &mod)
{
	bool 	found = false;
	int 	cur 	= history_list->at();
	history_list->first();
	while ((!found) && (history_list->current()))
		if ((history_list->current()->key == key) && (history_list->current()->module == mod))
			found = true;
		else
			history_list->next();
	
	if (!found)
	{
		HistoryKey *tmpKey = new HistoryKey;
		tmpKey->key = key.copy();
		tmpKey->module = mod.copy();
		if (cur < 0) cur = 0;
		
 		history_list->insert(cur, tmpKey);
		this->updateHistory();
	}
	else
	{
		if (cur >= 0)
			history_list->at(cur);
	}
}

void BibleTimeApp::slotHistoryBack(int id)
{
	QString t 		= history_prev->text(id);
	QString s			= 0;
	int 		cur 	= history_list->at();
	bool 		found = false;
	
	history_list->at(cur); // redundant, but anyway
	while ((!found) && (history_list->current()))
	{
		s = createHistoryKey(history_list->current()->key, history_list->current()->module);
		if ( s == t )
			found = true;
		else
			history_list->next();
	}
	
	if (found)
	{
		QString module 	= history_list->at(history_list->at())->module.copy();
		QString key			= history_list->at(history_list->at())->key.copy();
		this->updateHistory();
		doc->slotViewModActivate(module);
		view->activateModuleAndKey(module, key);
		this->slotLookupTextChanged(key);
	} else
		history_list->at(cur);
}

void BibleTimeApp::slotHistoryNext(int id)
{
	QString t 		= history_next->text(id);
	QString s			= 0;
	int 		cur 	= history_list->at();
	bool 		found = false;
	
	history_list->at(cur); // redundant, but anyway
	while ((!found) && (history_list->current()))
	{
		s = createHistoryKey(history_list->current()->key, history_list->current()->module);
		if ( s == t )
			found = true;
		else
			history_list->prev();
	}
	
	if (found)
	{
		QString module 	= history_list->at(history_list->at())->module.copy();
		QString key			= history_list->at(history_list->at())->key.copy();
		this->updateHistory();
		doc->slotViewModActivate(module);
		view->activateModuleAndKey(module, key);
		this->slotLookupTextChanged(key);
	}
	else
		history_list->at(cur);
}

void BibleTimeApp::slotHistoryMoveToPrev()
{
	int id = history_prev->idAt(0);
	this->slotHistoryBack(id);
}

void BibleTimeApp::slotHistoryMoveToNext()
{
	int id = history_next->idAt(0);
	this->slotHistoryNext(id);
}

/** Toggles the "instant lookup"
* checkbox in the toolbar. */
void BibleTimeApp::slotToggleLookupCheckBox()
{
	lookupCheckBox->setChecked( !lookupCheckBox->isChecked() );
	this->slotLookupCheckBoxClicked();
}

////////////////////////////////////////////////////////
// HELP FUNCTIONS
////////////////////////////////////////////////////////

int BibleTimeApp::executeCommand(const QString &command, KProcess::RunMode rm)
{
  bool success       	= false;
 	// now spawn process
 	process.clearArguments();
  debug("execute: %s",command.data());
 	process << command.data();
	success = process.start(rm, KProcess::AllOutput);
 	if (!success)
	{
  	QMessageBox::information(0,i18n("BibleTime - External Process"),
    	i18n("An external process could not be started!"),
 	  	QMessageBox::Ok | QMessageBox::Default);
 	  return -1;
 	}
	return 0;
}

void BibleTimeApp::slotSendEmail(const char *name, const char *email)
{
	QString command = 0;
	command.sprintf("kmail -s BibleTime %s",email);
	executeCommand(command,KProcess::DontCare);
}

void BibleTimeApp::slotOpenURL(const char *url)
{
	QString command = 0;
	command.sprintf("kfmclient openURL %s",url);
	executeCommand(command,KProcess::DontCare);
}

void BibleTimeApp::slotSaveListToFile(QStrList *list, bool withText)
{
 	QString filename=	KFileDialog::getSaveFileName("","",this,"saveDialog");
 	if (filename == "")
 	{
 		KMsgBox::message(this, "BibleTime - save list", i18n("Sorry, but without a filename I can't save the list."), KMsgBox::INFORMATION, i18n("OK"));
 		return;
 	};
 	 	
	QFile file(filename);
	if (file.exists())
	{
		if (KMsgBox::yesNo(this, i18n("BibleTime - save list"), i18n("The file \"" +filename + "\" does already exist.\nDo you want to overwrite the file?"),\
			 KMsgBox::QUESTION, i18n("Yes"), i18n("No")) == 1) file.remove();
		else return;
	};
	
	if ( file.open(IO_ReadWrite) )
	{
		QTextStream tStream( &file );
		QString s;
		QString currentKey;
		
		if (withText)
		{
			for(uint index=0; index < list->count(); index++)
			{
			 	currentKey=list->at(index);
			 	tStream << currentKey << "\n";
			 	tStream  << doc->getTextForKey(currentKey) << "\n\n";
				list->next();
			}
		}
		else
		{
			for(uint index=0; index < list->count(); index++)
			{
			 	tStream << list->at(index) << "\n";
				list->next();
			}
		};
		file.close();
	}
	else KMsgBox::message(this,"BibleTime",i18n("Can't open the file. Perhaps you have to write access!\nPlease check this!"), 3, i18n("OK"));
};

void BibleTimeApp::slotCopyListToClipboard(QStrList *list, bool withText)
{
		QClipboard *cBoard = kapp->clipboard();
		QString currentKey;
		QString text;
		
		for(uint index=0; index < list->count(); index++)
		{
		 	currentKey=list->at(index);
		 	if (withText)
		 		text.append(currentKey+":\n"+doc->getTextForKey(currentKey)+"\n");
		 	else
		 		text.append(currentKey+"\n");
			list->next();
		}
		
		cBoard->setText(text);
};

void BibleTimeApp::slotPrintList(QStrList *list, bool withText)
{
	QString currentKey;
		
	for(uint index=0; index < list->count(); index++)
	{
	 	currentKey=list->at(index);
	 	KeyVerse *keyVerse = new KeyVerse;
	 	keyVerse->key = currentKey.copy();
	 	keyVerse->printKey = true;
	 	keyVerse->data = 0;
	 	keyVerse->printData = false;
	 	if (withText)
	 	{
	 		keyVerse->data = (doc->getTextForKey(currentKey)).copy();
	 		keyVerse->printData = true;
	 	}
	 		
	 	verseList->append(keyVerse);
		list->next();
	}
	//debug("added to print list");
	enableCommand(ID_FILE_PRINT);
	enableCommand(ID_FILE_EMPTY_PRINT_QUEUE);
}

void BibleTimeApp::slotPrintText( const QString &key, bool pKey, const QString &data, bool pData )
{
	// append one item to the verse list
 	KeyVerse *keyVerse = new KeyVerse;
 	if (pKey)
 		keyVerse->key = key.copy();
 	else
 		keyVerse->key = 0;
 		
 	keyVerse->printKey = pKey;
 	keyVerse->data = 0;
 	keyVerse->printData = pData;
 	if (pData)
 		keyVerse->data = data.copy();
	 		
 	verseList->append(keyVerse);
	enableCommand(ID_FILE_PRINT);
	enableCommand(ID_FILE_EMPTY_PRINT_QUEUE);
}

void BibleTimeApp::slotTTabChanged(int tab_id)
{
	if (tab_id==TAB_BROWSER_LEX)
	{
	  	this->disableCommand(ID_TOOLBAR_PREV_VERSE);
	  	this->disableCommand(ID_TOOLBAR_NEXT_VERSE);

	  	this->disableCommand(ID_TOOLBAR_PREV_BOOK);
	  	this->disableCommand(ID_TOOLBAR_NEXT_BOOK);
	}
	else
	{
	  	this->enableCommand(ID_TOOLBAR_PREV_VERSE);
	  	this->enableCommand(ID_TOOLBAR_NEXT_VERSE);

	  	this->enableCommand(ID_TOOLBAR_PREV_BOOK);	
	  	this->enableCommand(ID_TOOLBAR_NEXT_BOOK);
	};
}

/** Updates the key accels and it's menus */
void BibleTimeApp::setKeyAccels()
{
//file menu
	key_accel->changeMenuAccel( file_menu, ID_FILE_CLOSE, KAccel::Close );
	key_accel->changeMenuAccel(	file_menu, ID_FILE_PRINT, KAccel::Print );
	key_accel->changeMenuAccel(	file_menu, ID_FILE_SEARCH,KAccel::Find );
	key_accel->changeMenuAccel(	file_menu, ID_FILE_QUIT, KAccel::Quit );

//edit menu	
	key_accel->changeMenuAccel( edit_menu, ID_EDIT_COPY, KAccel::Copy);	
	key_accel->changeMenuAccel( edit_menu, ID_EDIT_CUT, KAccel::Cut);	
	key_accel->changeMenuAccel( edit_menu, ID_EDIT_PASTE, KAccel::Paste);	

//view menu
	key_accel->changeMenuAccel( view_menu, ID_VIEW_TEXT_TOOLBAR, "toggle_text_toolbar");
	key_accel->changeMenuAccel( view_menu, ID_VIEW_NOTES_EDITOR, "notes_editor");
	key_accel->changeMenuAccel( view_menu, ID_VIEW_BIG_BROWSER, "browser");
	key_accel->changeMenuAccel( view_menu, ID_VIEW_DISPLAY_STRONGS, "show_strongs");
	key_accel->changeMenuAccel( view_menu, ID_VIEW_DISPLAY_FOOTNOTES, "show_footnotes");

//help menu	
	key_accel->changeMenuAccel( help_menu, ID_HELP_CONTENTS, KAccel::Help);
}

/** Calls the helpprogram of KDE */
void BibleTimeApp::slotHelpContents()
{
	kapp->invokeHTMLHelp("", "index.html");
}

/** Stores the list to send it later to BibleTimeView.
	* We can't connect directly to View because Doc
	* sends the signal before View is created.
	*/
void BibleTimeApp::slotSetBibleFont( QList<TBibleFont>* fontList)
{
	if (fontList !=0)
		this->fontList = fontList;
}

/**Enables or disables Edit->PASTE depending of current state. This function is executed beffore the edit menu popups.*/
void BibleTimeApp::slotClipboardChanged()
{
	if ( kapp->clipboard()->text()!=0 )
		enableCommand( ID_EDIT_PASTE );
	else
		disableCommand( ID_EDIT_PASTE );
}

/**  */
void BibleTimeApp::slotBrowserSelectionChanged(KHTMLView *, bool selected)
{
	if (selected)
	{
		enableCommand(ID_EDIT_COPY);
		browserHasSelection=true;	
	}
	else
	{	
		if ( !notesEditorHasSelection )
			disableCommand(ID_EDIT_COPY);
		browserHasSelection=false;
	};
}

void BibleTimeApp::slotNotesEditorSelectionChanged(KWriteView*, bool selected)
{
	if (selected)
	{
		enableCommand(ID_EDIT_COPY);
		enableCommand(ID_EDIT_CUT);
		notesEditorHasSelection=true;
	}
	else
	{
		if ( !browserHasSelection )
			disableCommand(ID_EDIT_COPY);
		disableCommand(ID_EDIT_CUT);
		notesEditorHasSelection=false;
	}
}

void BibleTimeApp::setCaption(const QString caption)
{
	QString dummy= QString("BibleTime ") + VERSION + QString(" - ") + QString(caption);
	KTMainWindow::setCaption( dummy );
}
