/***************************************************************************
                          bibletimeview.cpp  -  description                              
                             -------------------                                         
    begin                : Tue Jun 15 15:34:26 CEST 1999
                                           
    copyright            : (C) 1999 by Torsten Uhlmann                         
    email                : TUhlmann@gmx.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

//own include files
#include "bibletimeview.h"

#include <qlistview.h>

BibleTimeView::BibleTimeView(QWidget *parent, const char* name) :	KNewPanner(parent, name, KNewPanner::Vertical, KNewPanner::Percent,35)
{
	setBackgroundMode( PaletteBase );
	config=kapp->getConfig();
		
	_moduleList = new ModuleDefinitionList;
	_moduleList->setAutoDelete(true);
	
	recentText 		= 0;
	recentLexicon = 0;
	recentComment = 0;

  fontList = 0;

  initView();
  initConnections();
}

BibleTimeView::~BibleTimeView()
{

}

bool BibleTimeView::highlightTabEntry(KListView *list,const QString &compare, bool setRecent)
{
	unsigned int 	idx 	= 0;
	bool					found	= false;
	QString				tmp 	= 0;
	
	if (!list)
	{
		debug("Internal ERROR (highlightTabEntry): list is NULL!");
		return false;
	}
	
	uint listCount=list->childCount();
	while (idx < listCount)
	{
		tmp.setStr(list->itemAt(idx)->text(0));
		if (tmp.contains(compare))
		{
			list->setCurrentItem(idx);
			found = true;
			break;
		} else
			idx++;
	}
	if ((found) && (setRecent))
	{
		if (list == textTree) recentText.setStr(list->itemAt(idx)->text(0));
		if (list == lexTree) 	recentLexicon.setStr(list->itemAt(idx)->text(0));
		if (list == commTree) recentComment.setStr(list->itemAt(idx)->text(0));
	}
	return found;
}

void BibleTimeView::setRecentModules()
{
	config->setGroup("viewSettings");
	recentText 				= config->readEntry("currentTextModule");
	recentLexicon 		= config->readEntry("currentLexiconModule");
	recentComment 		= config->readEntry("currentCommentModule");
	
	// highlight corresponding tab entries
	if (!recentText.isNull())
	{
		slotFindModule(recentText);
		highlightTabEntry(textTree, recentText);
	}
	// the following produces a QRangeControl error, dont know why
	if (!recentLexicon.isNull())
	{
		slotFindModule(recentLexicon);
		highlightTabEntry(lexTree, recentLexicon);
	}
	if (!recentComment.isNull())
	{
		slotFindModule(recentComment);
		highlightTabEntry(commTree, recentComment);
	}
}

BibleTimeDoc* BibleTimeView::getDocument() const
{
	BibleTimeApp* theApp=(BibleTimeApp*)parentWidget();
	return theApp->getDocument();
}

void BibleTimeView::saveOptions()
{
//save options
	config->setGroup("viewSettings");
	config->writeEntry("leftPannerPosition",		leftPanner->separatorPos());
	
	if (rightPanner->separatorPos() != 100)
		config->writeEntry("rightPannerPosition",	rightPanner->separatorPos());
	else
		config->writeEntry("rightPannerPosition",	rightPannerPosition);
	
	if (this->separatorPos() != 0)
		config->writeEntry("globalPannerPosition",this->separatorPos());
	else
		config->writeEntry("globalPannerPosition",globalPannerPosition);

	config->writeEntry("currentTextTab", 				browserTabView->getCurrentTab());
	config->writeEntry("currentModulesTab",			treeTabView->getCurrentTab());
	config->writeEntry("currentSearchTab",			searchTabView->getCurrentTab());

	config->writeEntry("currentTextModule",			recentText);
	config->writeEntry("currentLexiconModule",	recentLexicon);
	config->writeEntry("currentCommentModule",	recentComment);
}

void BibleTimeView::readOptions()
{
	config->setGroup("General Options");		
  bool viewBigBrowser=config->readBoolEntry("Big Browser", false);
	bool viewNotesEditor=config->readBoolEntry("Show Notes Editor", true);

	config->setGroup("viewSettings");
	
	globalPannerPosition=config->readNumEntry("globalPannerPosition", 35);
	if (globalPannerPosition <= 0 )
		globalPannerPosition=35;
	this->setSeparatorPos(globalPannerPosition);
	if (viewBigBrowser)
 		this->slotViewBigBrowser(true);

	rightPannerPosition=config->readNumEntry("rightPannerPosition", 60 );
	if (rightPannerPosition > 99)
		rightPannerPosition = 60;
	rightPanner->setSeparatorPos(rightPannerPosition);
	if (!viewNotesEditor)		
		this->slotViewTNotesEditor(false);
	
	leftPanner->setSeparatorPos(config->readNumEntry("leftPannerPosition", 50));

	notesEditor->readOptions();
}

void BibleTimeView::slotSetCurrentTabs()
{
	config->setGroup("viewSettings");
	browserTabView->setCurrentTab (config->readNumEntry("currentTextTab",				0));
	treeTabView->setCurrentTab (config->readNumEntry("currentModulesTab",		0));
	searchTabView->setCurrentTab (config->readNumEntry("currentSearchTab",0));
}

void BibleTimeView::initView()
{
  QFont font("Fixed",10);

  leftPanner 		= new KNewPanner(this,"leftPanner",KNewPanner::Horizontal,KNewPanner::Percent,65);
  rightPanner 	= new KNewPanner(this,"rightPanner",KNewPanner::Horizontal,KNewPanner::Percent,76);
	
	// Tree with Modules
  treeTabView = new CTabCtl(leftPanner);
  treeTabView->setFocusPolicy(QWidget::ClickFocus);

  textTree = new KListView(treeTabView,"texts");
  textTree->setFocusPolicy(QWidget::NoFocus);

  lexTree = new KListView(treeTabView,"lexicons");
  lexTree->setFocusPolicy(QWidget::NoFocus);

  commTree = new KListView(treeTabView,"commentary");
  commTree->setFocusPolicy(QWidget::NoFocus);

  treeTabView->addTab(textTree,i18n("Bibles"));
  treeTabView->addTab(lexTree,i18n("Lexicons"));
  treeTabView->addTab(commTree,i18n("Commentary"));

	// Tree with Modules
  searchTabView = new CTabCtl(leftPanner);
  searchTabView->setFocusPolicy(QWidget::ClickFocus);

	// the dialog panel
	searchTextDialog 		= new SearchTextDialog(searchTabView);
	searchScopeDialog		= new SearchScopeDialog(searchTabView);
	searchResultWidget	= new CResultWidget(searchTabView);
  bookmarkTree 				= new BookmarkManager(searchTabView);

  searchResultWidget->setFocusPolicy(QWidget::NoFocus);
  bookmarkTree->setFocusPolicy(QWidget::StrongFocus);

  searchTabView->addTab(searchTextDialog,i18n("Find"));
  searchTabView->addTab(searchScopeDialog,i18n("Scope"));
  searchTabView->addTab(searchResultWidget,i18n("Found"));
  searchTabView->addTab(bookmarkTree,i18n("Bookmarks"));

  ////////////////////////
  // Right main window
  ////////////////////////

  // the tabbar + tabwidgets for browser
  browserTabView = new CTabCtl(rightPanner);
  browserTabView->setFocusPolicy(QWidget::ClickFocus);

  textBrowserWidget = new CDocBrowser(browserTabView,"textbrowser");
  textBrowserWidget->setFocusPolicy(QWidget::StrongFocus);

  lexBrowserWidget = new CDocBrowser(browserTabView,"lexbrowser");
  lexBrowserWidget->setFocusPolicy(QWidget::StrongFocus);

  commBrowserWidget = new CDocBrowser(browserTabView,"commbrowser");
  commBrowserWidget->setFocusPolicy(QWidget::StrongFocus);

  browserTabView->addTab(textBrowserWidget,i18n("&Bible Browser"));
  browserTabView->addTab(lexBrowserWidget,i18n("Le&xica Browser"));
  browserTabView->addTab(commBrowserWidget,i18n("C&ommentary Browser"));

	notesEditor = new CNotesEditor(rightPanner);
	
	rightPanner->activate(browserTabView, notesEditor);
  leftPanner->activate(treeTabView,searchTabView);
  this->activate(leftPanner,rightPanner);
}


/** init sht econnections between SLOTs and SIGNALs */
void BibleTimeView::initConnections()
{
	connect(notesEditor, SIGNAL(sigNotesEditorSelectionChanged(KWriteView*,bool)),
					this, SIGNAL(sigNotesEditorSelectionChanged(KWriteView*,bool)));
	
	// connect KListViews to slotSetModule
	connect (textTree,SIGNAL(singleSelected(int)),	SLOT(slotSetTextModule(int)));
	connect (lexTree,SIGNAL(singleSelected(int)),		SLOT(slotSetLexiconModule(int)));
	connect (commTree,SIGNAL(singleSelected(int)),	SLOT(slotSetCommentModule(int)));

	// connect the searchTextDialog
	connect (searchTextDialog, SIGNAL(sigSearchButtonClicked(BibleTimeApp::SearchType, bool, const QString&)),
					this, SLOT(slotSearchCurrentModule(BibleTimeApp::SearchType, bool, const QString&)));

	//connect the searchResultWidget
	connect(searchResultWidget, SIGNAL(sigSaveListToFile(QStrList*,bool)),
					this, SIGNAL(sigSaveListToFile(QStrList*,bool)));
	connect(searchResultWidget, SIGNAL(sigCopyListToClipboard(QStrList*,bool)),
					this, SIGNAL(sigCopyListToClipboard(QStrList*,bool)));					
	connect(searchResultWidget, SIGNAL(sigPrintList(QStrList*,bool)),
					this, SIGNAL(sigPrintList(QStrList*,bool)));						
	connect (searchResultWidget,SIGNAL(highlighted(const char*)),
					this, SIGNAL(sigResultListSelection(const char*)));

	//connect the browsers										
	connect (textBrowserWidget, SIGNAL(sigActivateModule(const QString&)),
					this, SLOT(activateModule(const QString&)));
	connect (lexBrowserWidget, SIGNAL(sigActivateModule(const QString&)),
					this, SLOT(activateModule(const QString&)));
	connect (commBrowserWidget, SIGNAL(sigActivateModule(const QString&)),
					this, SLOT(activateModule(const QString&)));
	
	connect (textBrowserWidget, SIGNAL(sigSearchInModule(const QString&, const QString&)),
					this, SIGNAL(sigSearchInModule(const QString&, const QString&)));
	connect (lexBrowserWidget, SIGNAL(sigSearchInModule(const QString&, const QString&)),
					this, SIGNAL(sigSearchInModule(const QString&, const QString&)));
	connect (commBrowserWidget, SIGNAL(sigSearchInModule(const QString&, const QString&)),
					this, SIGNAL(sigSearchInModule(const QString&, const QString&)));

	connect (textBrowserWidget, SIGNAL(sigPrintText( const QString&, bool, const QString&, bool )),
					this, SLOT(slotPrintText( const QString&, bool, const QString&, bool )));
	connect (lexBrowserWidget, SIGNAL(sigPrintText( const QString&, bool, const QString&, bool )),
					this, SLOT(slotPrintText( const QString&, bool, const QString&, bool )));
	connect (commBrowserWidget, SIGNAL(sigPrintText( const QString&, bool, const QString&, bool )),
					this, SLOT(slotPrintText( const QString&, bool, const QString&, bool )));
					
	connect(textBrowserWidget, SIGNAL(URLSelected ( KHTMLView *, const char*, int, const char* )),
					this, SLOT(slotURLSelected(KHTMLView*, const char*, int, const char*)));
	connect(lexBrowserWidget, SIGNAL(URLSelected ( KHTMLView *, const char*, int, const char* )),
					this, SLOT(slotURLSelected(KHTMLView*, const char*, int, const char*)));
	connect(commBrowserWidget, SIGNAL(URLSelected ( KHTMLView *, const char*, int, const char* )),
					this, SLOT(slotURLSelected(KHTMLView*, const char*, int, const char*)));
					
	connect(textBrowserWidget, SIGNAL(textSelected(KHTMLView*, bool)),
					this,SIGNAL(sigBrowserSelectionChanged(KHTMLView*, bool)));
	connect(lexBrowserWidget, SIGNAL(textSelected(KHTMLView*, bool)),
					this,SIGNAL(sigBrowserSelectionChanged(KHTMLView*, bool)));
	connect(commBrowserWidget, SIGNAL(textSelected(KHTMLView*, bool)),
					this,SIGNAL(sigBrowserSelectionChanged(KHTMLView*, bool)));
	
	connect (notesEditor, SIGNAL(sigPersonalCommentAdd(const QString&, const QString&, const QString&, const QString&)),
						this, SIGNAL(sigPersonalCommentAdd(const QString&, const QString&, const QString&, const QString&)));
	connect (notesEditor, SIGNAL(sigPersonalCommentRemove(const QString&, const QString&, const QString&)),
						this, SIGNAL(sigPersonalCommentRemove(const QString&, const QString&, const QString&)));
	connect (notesEditor, SIGNAL(sigRequestBrowserTextForEditor(const QString&)),
						this, SIGNAL(sigRequestBrowserTextForEditor(const QString&)));
						
	connect(browserTabView, SIGNAL(tabSelected(int)),	SLOT(slotSelectTTabPage(int)));
	connect(treeTabView, SIGNAL(tabSelected(int)),	SIGNAL(sigTTabChanged(int)));
		
	connect (bookmarkTree, SIGNAL(sigBookmarkChosen(const QString&, const QString&)),
					this, SLOT(activateModuleAndKey(const QString&, const QString&)));
}


void BibleTimeView::slotSetModuleList(ModuleDefinitionList *moList)
{
	_moduleList->clear();

	ModuleDefinition	*m;
	QString						text	= 0;
	int								i			= 0;

	QPixmap textPix 				= ICON(ICON_TEXT);
	QPixmap dictionaryPix 	= ICON(ICON_DICTIONARY);
	QPixmap commentaryPix 	= ICON(ICON_COMMENTARY);
	
	// ////////////////////////////////////////////////////
	// build module tree list, consisting of:
	// + 1st level is ModuleType
	// + 2nd is ModuleName (ModuleDescription)
	// ////////////////////////////////////////////////////
	textTree->clear();
	lexTree->clear();
	commTree->clear();

// put in the 3 groups we have

	
for (moList->first(); moList->current(); moList->next())
	{
		// copy ModuleDefinition
		m = new ModuleDefinition;
		m->name = moList->current()->name.copy();
		m->type = moList->current()->type.copy();
		m->description = moList->current()->description.copy();
		_moduleList->append(m);
		
		text = m->name.copy() + " (" + m->description.copy() + ")";
		m->custom = text.copy();
		
		// which branch does it belong to
		if (m->type.contains("biblical texts",false))
		{
			textTree->addChildItem(text.data(), textPix ,0);
		} else {
			if (m->type.contains("lexicons / dictionaries",false))
			{
				lexTree->addChildItem(text.data(),dictionaryPix,0);
			} else {
				if (m->type.contains("commentaries",false))
				{
					commTree->addChildItem(text.data(),commentaryPix,0);
				} else
					debug("Modul type not found!");
			}
		}
		i++;
	}

	// build the browsers search menus
	textBrowserWidget->buildSearchMenu(_moduleList);
	lexBrowserWidget->buildSearchMenu	(_moduleList);
	commBrowserWidget->buildSearchMenu(_moduleList);
}

void BibleTimeView::slotSearchText()
{
	CDocBrowser *current = 0;
	int id = browserTabView->getCurrentTab();
	
	switch (id)
	{
		case TAB_BROWSER_TEXT: 	current = textBrowserWidget; 	break;
		case TAB_BROWSER_LEX:		current = lexBrowserWidget; 	break;
		case TAB_BROWSER_COMM:	current = commBrowserWidget; 	break;
	}
	current->slotSearchTextBegin();
}

void BibleTimeView::slotFindModule(const QString& t)
{
	bool found = false;
	
	_moduleList->first();
	while ((!found) && (_moduleList->current()))
	{
		//debug("Index %s",_moduleList->current()->custom.data());
		if (_moduleList->current()->custom == t)
			found = true;
		else
			_moduleList->next();
	}
	if (found)
	{
		QString name;
		name.setStr(_moduleList->current()->name.data());
		emit sigModuleChosen(name);	
	} else
		debug("Module not found!");
}

void BibleTimeView::activateModule(const QString &module)
{
	//debug("module is %s",module.data());
	emit sigModuleChosen(module);
	// highlight module in treeTabView
	if (!highlightTabEntry(textTree, module, true))
		if (!highlightTabEntry(lexTree, module, true))
			if (!highlightTabEntry(commTree, module, true))
				debug("Module not found: %s", module.data());
}

void BibleTimeView::activateModuleAndKey(const QString &module, const QString &key)
{
	//debug("module is %s",module.data());
	//debug("key is %s",key.data());
	emit sigModuleChosen(module);
	emit sigLookupTextChanged(key.data());
	// highlight module in treeTabView
	if (!highlightTabEntry(textTree, module, true))
		if (!highlightTabEntry(lexTree, module, true))
			if (!highlightTabEntry(commTree, module, true))
				debug("Module not found: %s", module.data());
}
void BibleTimeView::slotSetHTMLText(const QString& modType, const QString &text)
{
	// decide on modType which widget to write the text to
	CDocBrowser *current = 0;
	
	int cTab = 0;
	if (modType.contains("biblical texts",false))
	{
		current = textBrowserWidget;
		cTab = TAB_BROWSER_TEXT;
	} else {
		if (modType.contains("lexicons / dictionaries",false))
		{
			current = lexBrowserWidget;
			cTab = TAB_BROWSER_LEX;
		}	else {
			if (modType.contains("commentaries",false))
			{
				current = commBrowserWidget;
				cTab = TAB_BROWSER_COMM;
			}	else {
				debug("Modul type not found!");
			}
		}
	}
	if (current)
	{		
		current->begin("");
		current->write(text.data());
		current->end();
		current->parse();
		current->gotoAnchor("Current"); 		//the anchor "Current" mark the current verse
		current->show();
		if (browserTabView->getCurrentTab() != cTab)
			browserTabView->setCurrentTab(cTab);
	}
}

void BibleTimeView::slotSetCurrentKeyText(const QString& t)
{
	searchTextDialog->slotSetCurrentKeyText(t);
}

void BibleTimeView::slotSetCurrentKeyVerse(const QString& t)
{
	notesEditor->slotSetCurrentKeyVerse(t);
	currentKey = t.copy();
}

void BibleTimeView::slotSetCurrentKeyModule(const QString &t, const QString &m, const QString &type)
{
	notesEditor->slotSetCurrentModule(m);
	bookmarkTree->slotSetCurrentKeyModule(t, m, type);
}

void BibleTimeView::slotSetSearchResult(QStrList *list)
{
	searchTextDialog->slotSearchEnd();
//only switch to result-tab if something is found
	if (list->count() == 0) return;
	searchResultWidget->setResult (list);
	searchTabView->setCurrentTab(TAB_SEARCH_RESULT);
}

void BibleTimeView::slotSearchCurrentModule(BibleTimeApp::SearchType st, bool cs, const QString &text)
{
	// get search scope
	BibleTimeApp::SearchScope scope = searchScopeDialog->getSearchScope();
	if (scope == BibleTimeApp::NoScope)
		emit sigResetSearchScope();
	else
	{
		if (scope == BibleTimeApp::LastSearch)
			emit sigSetSearchScope();
		else
			emit sigSetSearchScope(searchScopeDialog->getLowerBound(),searchScopeDialog->getUpperBound());
	}
	emit sigSearchCurrentModule(st, cs, text);
}

void BibleTimeView::slotRequestBrowserTextForEditor(const QString &text)
{
	// get contents from HTML widget and display it whithin editor
	QString cText;
	cText = convertToText(text);
	notesEditor->setText(cText);
}

QString BibleTimeView::convertToText (const QString& HTMLText)
{
  QString newText = HTMLText.copy();

  // convert some tags we need in code
	newText.replace( QRegExp("<BR>\\s*"), "\n" );
	newText.replace( QRegExp("<P>\\s*"), "\n\n" );

	return newText;	
}

void BibleTimeView::slotPrintText( const QString &key, bool pKey, const QString &data, bool pData )
{
	int id 	= browserTabView->getCurrentTab();
	int pos = 0;
	QString key2 = key.copy();
	if (pKey)
	{
		// we need to find out the current Key
		switch (id)
		{
			case TAB_BROWSER_TEXT:
				key2 = currentKey.copy();
				pos = key2.findRev(":");
				if (pos != -1)
				{
					key2.remove(pos,key2.length() - pos);
					key2 = (QString)"Excerpt from " + key2;
				}
				break;
			case TAB_BROWSER_LEX:
				key2 = currentKey;
				break;
			case TAB_BROWSER_COMM:
				key2 = currentKey;
				break;
		}
	}
	emit sigPrintText( key2, pKey, data, pData );
}

void BibleTimeView::slotCopy()
{
	if ( textBrowserWidget->isTextSelected())
		textBrowserWidget->slotCopyText();
	else
	if ( lexBrowserWidget->isTextSelected() )
		lexBrowserWidget->slotCopyText();
	else
	if ( commBrowserWidget->isTextSelected())
		commBrowserWidget->slotCopyText();
	else
	if ( notesEditor->isTextSelected() )
		notesEditor->slotCopy();
}

void BibleTimeView::slotCut()
{
 	if ( notesEditor->isTextSelected() )
		notesEditor->slotCut();
}

void BibleTimeView::slotPaste()
{
  	notesEditor->slotPaste();
}

///////////////////////////////////////////////////////////////////////////////////////
// OPTIONS-Menu functions
///////////////////////////////////////////////////////////////////////////////////////

void	BibleTimeView::slotChangeBrowserColors( const QColor &bg, const QColor &text,
	const QColor &link, const QColor &vlink, const bool underline)
{
	textBrowserWidget->slotDocColorsChanged(bg, text, link, vlink, underline);
	lexBrowserWidget->slotDocColorsChanged(bg, text, link, vlink, underline);
	commBrowserWidget->slotDocColorsChanged(bg, text, link, vlink, underline);
}

void BibleTimeView::slotBrowserFixedFont(const char* font)
{
	textBrowserWidget	->slotDocFixedFont(font);
	lexBrowserWidget	->slotDocFixedFont(font);
	commBrowserWidget	->slotDocFixedFont(font);
};

void BibleTimeView::slotBrowserStandardFont(const char* font)
{
	textBrowserWidget	->slotDocStandardFont(font);
	lexBrowserWidget	->slotDocStandardFont(font);
	commBrowserWidget	->slotDocStandardFont(font);
};

void BibleTimeView::slotBrowserFontSize(int size)
{
	textBrowserWidget	->slotDocFontSize(size);
	lexBrowserWidget	->slotDocFontSize(size);
	commBrowserWidget	->slotDocFontSize(size);
};

void BibleTimeView::slotViewTNotesEditor(bool f)
{
//hide tghe editor and the editor-control
	if (!f)
	{
  	rightPannerPosition = rightPanner->separatorPos();
    rightPanner->setSeparatorPos(100);
  }
//show the commentary editor and the editor-control
  else
 	{
		rightPanner->setSeparatorPos(rightPannerPosition);
  }

	QRect rMainGeom= rightPanner->geometry();
	rightPanner->resize(rMainGeom.width(),rMainGeom.height()+1);
	rightPanner->resize(rMainGeom.width(),rMainGeom.height());
}

//////////////////////////////////////////////////////////////////
//Useful functions
//////////////////////////////////////////////////////////////////

void BibleTimeView::slotSelectTTabPage(int id)
{
	//switch to the fitting page of treeTabView
//	KHTMLView *current = 0;
//	QRect 		rMainGeom= rightPanner->geometry();
	QString 	m = 0;
	
	treeTabView->setCurrentTab(id);
	
	switch (id)
	{
		case TAB_BROWSER_TEXT: 	
//			current = textBrowserWidget;
			m.setStr("Biblical Texts");
			break;
		case TAB_BROWSER_LEX: 	
//			current = lexBrowserWidget; 	
			m.setStr("Lexicons / Dictionaries");
			break;
		case TAB_BROWSER_COMM:	
//			current = commBrowserWidget;
			m.setStr("Commentaries");
			break;
	}
//	current->show();
//	current->resize(rMainGeom.width(),rMainGeom.height());
	emit sigSetCurrentModule(m);
	emit sigCurrentKeyRequested();
};

void BibleTimeView::slotSelectSTabPage(int id)
{
	//switch to the fitting page of browserTabView
//	KHTMLView *current = 0;
//	QRect 		rMainGeom= rightPanner->geometry();
	QString 	m = 0;
	
	browserTabView->setCurrentTab(id);
	
	switch (id)
	{
		case TAB_BROWSER_TEXT: 	
//			current = textBrowserWidget;
			m.setStr("Biblical Texts");
			break;
		case TAB_BROWSER_LEX: 	
//			current = lexBrowserWidget; 	
			m.setStr("Lexicons / Dictionaries");
			break;
		case TAB_BROWSER_COMM:	
//			current = commBrowserWidget;
			m.setStr("Commentaries");
			break;
	}
//	current->show();
//	current->resize(rMainGeom.width(),rMainGeom.height());
	emit sigSetCurrentModule(m);
	emit sigCurrentKeyRequested();
};

void BibleTimeView::slotViewBigBrowser(bool f)
{
	if (f)
	{
		globalPannerPosition = this->separatorPos();
		this->setSeparatorPos(0);
	}
	else
	{
		this->setSeparatorPos(globalPannerPosition);
  }

  this->resizeEvent(0);
}

void BibleTimeView::slotURLSelected(KHTMLView* browserWidget, const char* url,int, const char*)
{
	this->showURL(url);
}

/** Activates the browsetttab.*/
void BibleTimeView::slotActivateBrowserTab()
{
	switch (treeTabView->getCurrentTab())
	{
		case TAB_BROWSER_TEXT: textBrowserWidget->setFocus(); break;
		case TAB_BROWSER_LEX:  lexBrowserWidget->setFocus(); 	break;
		case TAB_BROWSER_COMM: commBrowserWidget->setFocus(); break;
	};
}

/** Implmentation of the resizeEvent. Handles
	* the different parts and correct their sizes if they
	* are partially hidden. */
void BibleTimeView::resizeEvent( QResizeEvent *event )
{
	KNewPanner::resizeEvent( event );
}

/** Sets the font of the
	* HTML widget (e.g. hebrew).
	*/
void BibleTimeView::setBibleFontList( QList<TBibleFont> *fontList )
{
	this->fontList = fontList;
}

/** Sets the font of the bible-browser */
void BibleTimeView::slotSetBrowserFont(QString module, QString fontname)
{
  if (fontname == "")
	{	
		config->setGroup( "DocBrowserAppearance" );
  	fontname = config->readEntry( "StandardFont");	
  }
  else
	{
  	config->setGroup("Foreign Fonts");
  	fontname = config->readEntry( module, fontname);	
  };
  	
	textBrowserWidget->slotDocStandardFont( fontname );
}


/** Shows the URL in the curret browserwidget */
void BibleTimeView::showURL(const char* url)
{
	CDocBrowser *current = 0;
	int id = browserTabView->getCurrentTab();

	switch (id)
	{
		case TAB_BROWSER_TEXT:
			current = textBrowserWidget;
			break;
		case TAB_BROWSER_LEX:
			current = lexBrowserWidget;
			break;
		case TAB_BROWSER_COMM:
			current = commBrowserWidget;
			break;
	}
	current->showURL(url);
}

/**  */
void BibleTimeView::slotRefreshLinkLists()
{
	textBrowserWidget->slotRefreshLinkLists();
	lexBrowserWidget->slotRefreshLinkLists();
	commBrowserWidget->slotRefreshLinkLists();
}

/**  */
void BibleTimeView::slotSetTextModule( int index )
{
	QString t=textTree->itemAt(index)->text(0);
		
	recentText = t.copy();
	slotFindModule(t);
}

/**  */
void BibleTimeView::slotSetLexiconModule( int index )
{
	QString t=lexTree->itemAt(index)->text(0);
		
	recentLexicon = t.copy();
	slotFindModule(t);
}

/**  */
void BibleTimeView::slotSetCommentModule( int index )
{
	QString t=commTree->itemAt(index)->text(0);
			
	recentComment = t.copy();
	slotFindModule(t);
}
