/* This may look like C, but it's really -*- C++ -*- */

/*
    kdb - A Simple but fast database editor for the KDE
    Copyright (C) 1998  Thorsten Niedzwetzki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "memorydb.h"


/*
 * Destructor calls clear() to remove all data from memory.
 */
TMemoryDB::~TMemoryDB()
{
  clear();
}


/*
 * Remove field names and data.
 */
void
TMemoryDB::clear()
{
  if (fieldNames) {
    fieldNames->clear();
    delete fieldNames;
    fieldNames = 0;
  }
  if (data) {
    data->setAutoDelete(true);
    while (!data->isEmpty()) {
      data->first()->setAutoDelete(true);
      data->remove();
    }
    data->clear();
    delete data;
    data = 0;
  }
}


/*
 * Set or reset field names list.
 */
bool
TMemoryDB::setFieldNames(QStrList *fieldNames)
{
  if (this->fieldNames != NULL) {
    if (this->fieldNames->count() == fieldNames->count()) {
      this->fieldNames->clear();
      delete this->fieldNames;
    } else {
      return false;
    }
  }
  this->fieldNames = fieldNames;
  this->fieldNames->setAutoDelete(true);
  return true;
}


/*
 * Return number of fields or 0 on empty table.
 */
inline unsigned int
TMemoryDB::getFieldCount() const
{
  return (fieldNames ? fieldNames->count() : 0);
}


/*
 * Return number of records or 0 on empty table.
 */
inline unsigned int
TMemoryDB::getRecordCount() const
{
  return (data ? data->count() : 0);
}


/*
 * return name of specified field.
 */
inline char*
TMemoryDB::getFieldName(unsigned int fieldNo) const
{
  return fieldNames->at(fieldNo);
}


/*
 * Return field names list.
 */
QStrList*
TMemoryDB::getFields() const
{
  return fieldNames;
}
  

/*
 * Return specified item of a record.
 */
char*
TMemoryDB::getItemAsText(int record, int field) const
{
  return data->at(record)->at(field);
}


/*
 * Append record given as a string list.
 */
bool
TMemoryDB::appendRecordAsText(QStrList *record)
{
  if (!data)  // create data list on the fly
    data = new QList<QStrList>;
  while (record->count() < getFieldCount())
    // replace non-existing fields by empty fields
    record->append("");
  data->append(record);
  return true;
}

bool TMemoryDB::replaceRecordAsText(unsigned int recNo, QStrList *record) {
  if (recNo < data->count()) {
    while (record->count() < getFieldCount())
      // replace non-existing fields by empty fields
      record->append("");
    data->remove(recNo);
    data->insert(recNo, record);
    return true;
  } else {
    return false;
  }
}


/*
 * Return record from the data list.
 */
QStrList* TMemoryDB::getRecord(int record) const
{
  return data->at(record);
}


/*
 * Remove record from the data list.
 */
bool TMemoryDB::removeRecord(int record)
{
  return data->remove(record);
}
