/***************************************************************************
                          map.cpp  -  description
                             -------------------
    begin                : Fri Dec 10 1999
    copyright            : (C) 1999 by Ralf-Christian Juergensen
    email                : ralf-christian.juergensen@stud.fh-flensburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "map.h"

Map::Map(void)
{
    width = 0;
    height = 0;
    fields = NULL;
}

Map::Map(const char* mapFileName)
{
    fields = NULL;
    load(mapFileName);
}

Map::Map(int width, int height)
{
    this->width = width;
    this->height = height;
    
    fields = new Field[width * height];
    
    for(int x=0; x < width; x++)
    {
        for(int y=0; y < height; y++)
        {
            setField(x, y, Field(NONE, EMPTY));
        }
    }
}

Map::Map(const Map& map)
{
    width = map.width;
    height = map.height;
    	
    fields = new Field[width * height];
    	
    for(int x=0; x < width; x++)
    {
        for(int y=0; y < height; y++)
        {
            fields[coords(x, y)] = map.fields[coords(x, y)];
        }
    }
}

Map::~Map(void)
{
    deleteFields();
}

void Map::load(const char* mapFileName)
{
    // create backup of map
    Field* backupFields = fields;
    int backupWidth = width;
    int backupHeight = height;
    	
    fields = NULL;
    	
    ifstream mapFile(mapFileName);

    Token* tokenList = new Token[LAST_TOKEN];
    
    tokenList[WIDTH]        .setToken("width"  , true );
    tokenList[HEIGHT]       .setToken("height" , true );
    tokenList[NO_PLAYER]    .setToken("none"   , false);
    tokenList[PLAYER_ONE]   .setToken("one"    , true );
    tokenList[PLAYER_TWO]   .setToken("two"    , true );
    tokenList[OFF_FIELD]    .setToken("off"    , true );
    tokenList[ON_FIELD]     .setToken("on"     , true );
    tokenList[EMPTY_FIELD]  .setToken("empty"  , false);
    tokenList[PENGUIN_FIELD].setToken("penguin", false);
    		
    try
    {
        parseDefines(mapFile, tokenList);        
        		
        // 'save' values and allocate memory for fields
        width  = *(int*)tokenList[WIDTH].getValue();
        height = *(int*)tokenList[HEIGHT].getValue();
        fields = new Field[width * height];

        parseMapLines(mapFile, tokenList);        
        
        delete[] tokenList;
    }
    catch(char* errorMsg)
    {
        deleteFields();
        		
        // restore old map
        fields = backupFields;
        width = backupWidth;
        height = backupHeight;
        		
        throw errorMsg;
    }
    
    if(backupFields != NULL)
        delete[] backupFields;
}

void Map::deleteFields(void)
{
    if(fields != NULL)
    {
        delete[] fields;
        fields = NULL;
    }
}

void Map::and(int x, int y, bool bit)
{
    checkBounds(x, y);
    fields[coords(x, y)].and(bit);
}

void Map::or(int x, int y, bool bit)
{
    checkBounds(x, y);
    fields[coords(x, y)].or(bit);
}

void Map::xor(int x, int y, bool bit)
{
    checkBounds(x, y);
    fields[coords(x, y)].xor(bit);
}

int Map::checkFields(Player player, Value value)
{
    int noFields = 0;
            
    for(int x=0; x < width; x++)
    {
        for(int y=0; y < height; y++)
        {
             if((getFieldPlayer(x, y) == player) && (getFieldValue(x, y) == value))
             {
                 noFields++;
             }
        }
    }
            
    return noFields;
}

Value Map::getFieldValue(int x, int y)
{
    checkBounds(x, y);
    return fields[coords(x, y)].getValue();
}

Player Map::getFieldPlayer(int x, int y)
{
    checkBounds(x, y);
    return fields[coords(x, y)].getPlayer();
}

void Map::setFieldValue(int x, int y, Value value)
{
    checkBounds(x, y);
    fields[coords(x, y)].setValue(value);
}

void Map::setFieldPlayer(int x, int y, Player player)
{
    checkBounds(x, y);
    fields[coords(x, y)].setPlayer(player);
}

void Map::setField(int x, int y, Field field)
{
    checkBounds(x, y);
    fields[coords(x, y)] = field;
}

Field Map::getField(int x, int y)
{
    checkBounds(x, y);
    return fields[coords(x, y)];
}

int Map::getWidth(void)
{
    return width;
}

int Map::getHeight(void)
{
    return height;
}

void Map::parseDefines(ifstream& mapFile, Token* tokenList)
{
    static char line[80];
    char* errorStr = line;    // Not that good, but saves memory...
                              // And if there's an error we do not need the line anymore...
    int i;
    
    memset(line, 0, 80);      // clear line-string, b'coz it's static!!
    
    // Parse the definition-part
    while(!mapFile.eof() && strstr(line, "start:") == NULL)
    {
        mapFile.getline(line, sizeof(line));
        
        line[79] = 0;
        
        for(i=WIDTH; i < LAST_TOKEN; i++)
        {
            if(tokenList[i].getValue() == 0)
            {
                switch(i)
                {
                case WIDTH:
                case HEIGHT:
                    tokenList[i].setValue((void*)readInt(line, tokenList[i].getTokenStr()));
                    break;
                default: // all other tokens have a char-value!!
                    tokenList[i].setValue((void*)readChar(line, tokenList[i].getTokenStr()));
                    break;
                }
            }
        }
    }
    
    for(i=WIDTH; i < LAST_TOKEN; i++)
    {
        if(!tokenList[i].isValid())
        {
            strcpy(errorStr, "\'");
            strcpy(errorStr+1, tokenList[i].getTokenStr());
            strcpy(errorStr+strlen(errorStr), "\' definition not found!");
            
            throw errorStr;
        }
    }
}

int* Map::readInt(const char* string, const char* token)
{
    char* foundString;  // contains '<token>=...' or NULL
    char valueStr[9];   // the string of the value max 8 digits plus NULL
    int pos;            // actual position in foundString
    int* value;
    	
    foundString = strstr(string, token);
    	
    if(foundString == NULL)
        return NULL;
    	
    // step over '<token>'
    foundString += strlen(token);
    	
    // accept only '=' after <token>
    if(*foundString != '=')
        return NULL;
    	
    foundString++;
    
    // step over blanks (space tab) after =
    while(isspace(int(*foundString)))
        foundString++;
        
    // check for digits	
    for(pos=0; (pos < 8) && isdigit(int(foundString[pos])); pos++)
        valueStr[pos] = foundString[pos];
    	
    // when there is no ';' after value
    if(foundString[pos] != ';')
        return NULL;
    
    // terminate string
    valueStr[pos] = 0;
    
    value = new int;
    *value = atoi(valueStr);
    return value;
}

char* Map::readChar(const char* string, const char* token)
{
    char* foundString;  // contains '<token>=...' or NULL
    char* c;
    	
    foundString = strstr(string, token);
    	
    if(foundString == NULL)
        return NULL;
    	
    // step over '<token>'
    foundString += strlen(token);
    
    // accept only '=' after <token>
    if(*foundString != '=')
        return NULL;
    	
    foundString++;
    
    // step over blanks (space tab) after =
    while(isspace(int(*foundString)))
        foundString++;
        
    // when there is no ';' after char
    if(foundString[1] != ';')
        return NULL;
    		
    c = new char;
    *c = *foundString;
    return c;
}

void Map::parseMapLines(ifstream& mapFile, Token* tokenList)
{
    static char players[80];    // contains the current players line
    static char values[80];     // contains the current values line
    	
    char* posPlayer = NULL;     // pointer to current position in players
    char* posValue = NULL;      // pointer to current position in values
    	
    int currentX = 0;           // current x-position in map
    int currentY = 0;           // current y-position in map

    int i;
    bool charOk;    
    
    for(currentY = 0; !mapFile.eof() && currentY < height; currentY++)
    {
        mapFile.getline(players, sizeof(players));
        mapFile.getline(values, sizeof(values));

        players[79] = values[79] = 0;
                    
        posPlayer = strchr(players, '\"');
        posValue  = strchr(values,  '\"');
            			
        if(posPlayer == NULL)
            throw "missing player-line";
        if(posValue == NULL)
            throw "missing value-line";
            			
        posPlayer++;
        posValue++;
            
        for(currentX = 0;
            currentX < width && *posPlayer != '\"' && *posPlayer != 0 && *posValue != '\"' && *posValue != 0;
            currentX++, posPlayer++, posValue++)
        {
            // Check the player-line            				
            for(i=NO_PLAYER, charOk = false; i < OFF_FIELD; i++)
            {
                if(*posPlayer == *(char*)tokenList[i].getValue())
                {
                    setFieldPlayer(currentX, currentY, Player(i-NO_PLAYER));
                    charOk = true;
                }
            }
            if(!charOk)
            {
                throw "Unknown char in player-line.";
            }
    
            // Check the value-line
            for(i=OFF_FIELD, charOk = false; i < LAST_TOKEN; i++)
            {
                if(*posValue == *(char*)tokenList[i].getValue())
                {
                    setFieldValue(currentX, currentY, Value(i-OFF_FIELD));
                    charOk = true;
                }
            }
            if(!charOk)
                throw "Unknown char in value-line.";
        }
        
        if(currentX < width)
            throw "missing characters in map lines.";
        if(*posPlayer == 0)
            throw "missing \" in player-line.";
        if(*posValue == 0)    
            throw "missing \" in value-line.";
        if(*posPlayer != '\"')
        {
            if(*posValue == '\"')
                throw "missing \" in player-line.";
        }
        else
        {
            if(*posValue != '\"')
                throw "missing \" in value-line.";
        }
    }
    
    if(currentY < height)
        throw "missing map lines.";
}
