#include <score.h>
#include <score.moc>

#include <stdlib.h>
#include <ctype.h>

#if (KDE_VERSION_MAJOR >= 1 && KDE_VERSION_MINOR >= 1)
#include <kaccel.h>
#else
#include <kkeyconf.h>
#endif

#include <kapp.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qdstream.h>
#include <qkeycode.h>
#include <qtimer.h>

#include "bitfont.h"

Score::Score(QWidget *parent, const char *name, Bitfont *font) : QWidget(parent, name)
{
    setFocusPolicy(QWidget::StrongFocus);

    paused = FALSE;

    initKeys();
    initTiming();

    bitfont = font;

    QString libDir;
    libDir.setStr(KApplication::kde_datadir());
    libDir.append("/kpacman/");
    file.setName((const char *) (libDir+"highScore"));
    read();
    lastScore = -1;
    lastPlayer = -1;

    cursor.x = -1;
    cursor.y = -1;
    cursor.on = FALSE;
    cursor.chr = QString("?");

    for (int p = 0; p < maxPlayer; p++) {
        playerScore[p] = 0;
        playerName[p] = getenv("LOGNAME");
        if (playerName[p].length() < minPlayerNameLength)
            playerName[p].setExpand(minPlayerNameLength-1, ' ');

        for (uint i = 0; i < playerName[p].length(); i++)
            if ((uchar) playerName[p].at(i) < bitfont->firstChar() ||
                (uchar) playerName[p].at(i) > bitfont->lastChar())
                playerName[p].at(i) = toupper(playerName[p].at(i));
    }
}

Score::~Score()
{
    write();
}

void Score::timerEvent( QTimerEvent * )
{
    if (paused)
        return;

    cursor.on = !cursor.on;
    repaint(rect(cursor.x, cursor.y*1.25, cursor.chr), FALSE);
    scrollRepeat = FALSE;
}

void Score::paintEvent( QPaintEvent *e)
{
    if (rect(1, 0, i18n("  1UP ")).intersects(e->rect())) {
	    QPixmap pix = bitfont->text(i18n("  1UP "), WHITE, BLACK);
        bitBlt(this, x(1), y(0), &pix);
    }

    if (rect(8, 0, i18n(" HIGH SCORE ")).intersects(e->rect())) {
	    QPixmap pix = bitfont->text(i18n(" HIGH SCORE "), WHITE, BLACK);
        bitBlt(this, x(8), y(0), &pix);
    }
/*
    if (rect(21, 0, i18n("  2UP ")).intersects(e->rect())) {
	    QPixmap pix = bitfont->text(i18n("  2UP "), WHITE, BLACK);
        bitBlt(this, x(21), y(0), &pix);
    }
*/
    QString s;

    s.sprintf("%6d0", playerScore[0]/10);
    if (rect(0, 1, s).intersects(e->rect())) {
	    QPixmap pix = bitfont->text(s, WHITE, BLACK);
	    bitBlt(this, x(0), y(1), &pix);
    }

    s.sprintf("%8d0", HighScore/10);
    if (rect(8, 1, s).intersects(e->rect())) {
	    QPixmap pix = bitfont->text(s, WHITE, BLACK);
	    bitBlt(this, x(8), y(1), &pix);
    }

    if (lastScore >= 0) {
        if (rect(1, 4*1.25, i18n("     CONGRATULATIONS      ")).intersects(e->rect())) {
            QPixmap pix = bitfont->text(i18n("     CONGRATULATIONS      "), YELLOW, BLACK);
            bitBlt(this, x(1), y(4*1.25), &pix);
        }
        if (rect(1, 6*1.25, i18n("    YOU HAVE ARCHIEVED    ")).intersects(e->rect())) {
            QPixmap pix = bitfont->text(i18n("    YOU HAVE ARCHIEVED    "), CYAN, BLACK);
            bitBlt(this, x(1), y(6*1.25), &pix);
        }
        if (rect(1, 7*1.25, i18n("  A SCORE IN THE TOP 10.  ")).intersects(e->rect())) {
            QPixmap pix = bitfont->text(i18n("  A SCORE IN THE TOP 10.  "), CYAN, BLACK);
            bitBlt(this, x(1), y(7*1.25), &pix);
        }
        if (rect(1, 8*1.25, i18n("                          ")).intersects(e->rect())) {
            QPixmap pix = bitfont->text(i18n("                          "), CYAN, BLACK);
            bitBlt(this, x(1), y(8*1.25), &pix);
        }
    }

    if (rect(1, 9.5*1.25, i18n("RNK   SCORE  NAME   DATE")).intersects(e->rect())) {
	    QPixmap pix = bitfont->text(i18n("RNK   SCORE  NAME   DATE"), WHITE, BLACK);
        bitBlt(this, x(1), y(9.5*1.25), &pix);
    }

    for (int i = 0; i < 10; i++) {
        s.sprintf("%2d%9d  %-3.3s  %-8.8s",
            i+1, hallOfFame[i].points, hallOfFame[i].name,
            formatDate(hallOfFame[i].moment.date()).data());
        if (rect(1, (11+i)*1.25, s).intersects(e->rect())) {
            QPixmap pix = bitfont->text(s, (i == lastScore) ? YELLOW : WHITE, BLACK);
            bitBlt(this, x(1), y((11+i)*1.25), &pix);
        }
    }

    if (cursor.x != -1 && cursor.y != -1 && cursor.on) {
        if (rect(cursor.x, (cursor.y*1.25), cursor.chr).intersects(e->rect())) {
            QPixmap pix = bitfont->text(cursor.chr, BLACK, YELLOW);
            bitBlt(this, x(cursor.x), y(cursor.y*1.25), &pix);
        }
    }

    if (paused) {

        QPixmap pix = bitfont->text(i18n("PAUSED"), RED, BLACK);
        QRect r = bitfont->rect(i18n("PAUSED"));
        r.moveCenter(QPoint(this->width()/2, this->height()/2));

        bitBlt(this, r.x(), r.y(), &pix);
    }

}

void Score::keyPressEvent( QKeyEvent *k )
{
    if (lastScore < 0 || lastPlayer < 0 || lastPlayer >= maxPlayer || paused) {
	k->ignore();
        return;
    }

    int x = cursor.x;
    int y = cursor.y;

    uint key = k->key();

    if (scrollRepeat && (key == UpKey || key == Key_Up || key == DownKey || key == Key_Down)) {
        k->ignore();
        return;
    }

    if (key != Key_Return) {
        if (key == RightKey || key == Key_Right)
            if (++cursor.x > 16)
                cursor.x = 14;
        if (key == LeftKey || key == Key_Left)
            if (--cursor.x < 14)
                cursor.x = 16;
        if (key == UpKey || key == Key_Up)
            if ((uchar) ++cursor.chr.at(0) > bitfont->lastChar())
                cursor.chr.at(0) = bitfont->firstChar();
        if (key == DownKey || key == Key_Down)
            if ((uchar) --cursor.chr.at(0) < bitfont->firstChar())
                cursor.chr.at(0) = (char) bitfont->lastChar();

        if (cursor.x == x && cursor.y == y &&
            cursor.chr.at(0) == playerName[lastPlayer].at(cursor.x-14)) {
            uint ascii = k->ascii();

            if (ascii < bitfont->firstChar() || ascii > bitfont->lastChar())
                ascii = toupper(ascii);

            if (ascii >= bitfont->firstChar() && ascii <= bitfont->lastChar()) {
                cursor.chr.at(0) = (char) ascii;
                playerName[lastPlayer].at(cursor.x-14) = cursor.chr.at(0);
                if (++cursor.x > 16)
                    cursor.x = 14;
            }
        }
    }

    if (key == Key_Return) {
        hallOfFame[lastScore].name = playerName[lastPlayer].data();
        write();
        read();
        lastScore = -1;
        cursor.x = -1;
        cursor.y = -1;
        killTimers();
	emit toggleNew();
        end();
    }

    if (x != cursor.x || y != cursor.y) {
        if (cursor.x != -1)
            cursor.chr = playerName[lastPlayer].mid(cursor.x-14,1);
        scrollRepeat = FALSE;
        repaint(rect(x, y*1.25, cursor.chr), FALSE);
    } else
        playerName[lastPlayer].at(cursor.x-14) = cursor.chr.at(0);

    if (key == UpKey || key == Key_Up || key == DownKey || key == Key_Down)
       scrollRepeat = TRUE;
    else
       repaint(rect(cursor.x, cursor.y*1.25, cursor.chr), FALSE);
}

void Score::initKeys()
{
    KConfig *conf = kapp->getConfig();

    QString up("Up");
    up = conf->readEntry("upKey", (const char*) up);
    UpKey    = stringToKey(up);

    QString down("Down");
    down = conf->readEntry("downKey", (const char*) down);
    DownKey  = stringToKey(down);

    QString left("Left");
    left = conf->readEntry("leftKey", (const char*) left);
    LeftKey  = stringToKey(left);

    QString right("Right");
    right = conf->readEntry("rightKey", (const char*) right);
    RightKey = stringToKey(right);
}

void Score::initTiming()
{
    KConfig *conf = kapp->getConfig();

    cursorBlinkMS = conf->readNumEntry("CursorBlinkMS", 250);
    hallOfFameMS = conf->readNumEntry("HallOfFameMS", 7000);

    afterPauseMS = conf->readNumEntry("AfterPauseMS", 1000);
}

void Score::set(int score)
{
    set(score, 0);
}

void Score::set(int score, int player)
{
    if (player < 0 || player >= maxPlayer)
        return;

    lastPlayer = player;
    playerScore[lastPlayer] = score;

    QString s;

    s.sprintf("%6d0", playerScore[lastPlayer]/10);
    repaint(rect(0, 1, s), FALSE);

    if (score > HighScore) {
	    HighScore = score;
	    s.sprintf("%8d0", HighScore/10);
	    repaint(rect(8, 1, s), FALSE);
    }
}

/*
 * Set the score for player after the game if over. If the score is in the
 * high scores then the hall of fame is updated (shifted) and the scoreboard
 * is shown.
 */
void Score::setScore(int level, int player)
{
    lastScore = -1;

    if (player < 0 || player >= maxPlayer || level == 0) {
        if (level != 0)
            emit toggleNew();
        QTimer::singleShot(hallOfFameMS, this, SLOT(end()));
        return;
    }

    lastPlayer = player;

    for (int i = 0; i < 10; i++)
        if ( playerScore[lastPlayer] > hallOfFame[i].points) {
            lastScore = i;
            break;
        }

    if (lastScore < 0) {
        emit toggleNew();
        QTimer::singleShot(hallOfFameMS, this, SLOT(end()));
        return;
    }

    for (int i = 9; i > lastScore && i > 0; i--)
        hallOfFame[i] = hallOfFame[i-1];

    hallOfFame[lastScore].points = playerScore[lastPlayer];
    hallOfFame[lastScore].levels = level;
    hallOfFame[lastScore].moment = QDateTime::currentDateTime();
    hallOfFame[lastScore].name = playerName[lastPlayer].data();

    cursor.x = 14;
    cursor.y = 11+lastScore;
    cursor.chr = playerName[lastPlayer].mid(cursor.x-14,1);

    startTimer(cursorBlinkMS);
    setFocus();
}

void Score::setFont(Bitfont *font)
{
    bitfont = font;

    for (int p = 0; p < maxPlayer; p++)
        for (uint i = 0; i < playerName[p].length(); i++)
            if ((uchar) playerName[p].at(i) < bitfont->firstChar() ||
                (uchar) playerName[p].at(i) > bitfont->lastChar())
                playerName[p].at(i) = toupper(playerName[p].at(i));

    for (int i = 0; i < 10; i++)
        for (uchar *name = (uchar *) hallOfFame[i].name; *name; name++)
            if (*name < bitfont->firstChar() || *name > bitfont->lastChar())
                *name = toupper(*name);

    if ((uchar) cursor.chr.at(0) < bitfont->firstChar() ||
        (uchar) cursor.chr.at(0) > bitfont->lastChar())
        cursor.chr.at(0) = toupper(cursor.chr.at(0));
}

/*
 * Read the highscores, if no file or a file shorter than 4 bytes (versions before 0.2.4 stores only
 * the points of one highscore) exists - the highscores were initialized with default values.
 */
void Score::read()
{
    if (file.exists() && file.size() > 4) {
        if (file.open(IO_ReadOnly)) {
            QDataStream s(&file);
	    for (int i = 0; i < 10; i++)
	        s >> hallOfFame[i].points >> hallOfFame[i].levels >> hallOfFame[i].duration >>
	             hallOfFame[i].moment >> hallOfFame[i].name;
	    file.close();
        }
    } else {
        for (int i = 0; i < 10; i++) {
	    hallOfFame[i].points = 5000;
	    hallOfFame[i].levels = 0;
	    hallOfFame[i].duration = QTime();
	    hallOfFame[i].moment = QDateTime();
	    hallOfFame[i].name = "???";
	}
	write();
    }

    for (int i = 0; i < 10; i++)
        for (uchar *name = (uchar *) hallOfFame[i].name; *name; name++)
            if (*name < bitfont->firstChar() || *name > bitfont->lastChar())
                *name = toupper(*name);

    HighScore = hallOfFame[0].points;
}

void Score::write()
{
    if (file.open(IO_WriteOnly)) {
	QDataStream s(&file);
	for (int i = 0; i < 10; i++)
	    s << hallOfFame[i].points << hallOfFame[i].levels << hallOfFame[i].duration <<
                 hallOfFame[i].moment << hallOfFame[i].name;
	file.close();
    }
}

void Score::pause()
{
    paused = !paused;

    QRect r = bitfont->rect(i18n("PAUSED"));
    r.moveCenter(QPoint(this->width()/2, this->height()/2));
    repaint(r, TRUE);
}

void Score::end()
{
    if (paused) {
        QTimer::singleShot(afterPauseMS, this, SLOT(end()));
        return;
    }

    emit forcedHallOfFame(FALSE);
}

/*
 * Return the date in a formatted QString. The format can be changed using internationalization
 * of the string "YY/MM/DD". Invalid QDate's where returned as "00/00/00".
 */
QString Score::formatDate(QDate date)
{
    QString s = i18n("@YY@/@MM@/@DD@");

    QString dd;
    dd.sprintf("%02d", date.isValid() ? date.year() % 100 : 0);
    s.replace(QRegExp("@YY@"), dd);
    dd.sprintf("%02d", date.isValid() ? date.month() : 0);
    s.replace(QRegExp("@MM@"), dd);
    dd.sprintf("%02d", date.isValid() ? date.day() : 0);
    s.replace(QRegExp("@DD@"), dd);

    return s;
}

QRect Score::rect(int col, float row, QString str, int align)
{
    QRect r = bitfont->rect(str);
    r.moveBy(x(col), y(row));

    int dx = 0;
    int dy = 0;

    if (align & AlignLeft || align & AlignRight) {
        dx = (str.length()-1) * (bitfont->width()/2);
        if (align & AlignRight)
            dx *= -1;
    }

    if (align & AlignTop || align & AlignBottom) {
        dy = bitfont->height()/2;
        if (align & AlignBottom)
            dy *= -1;
    }

    if (dx != 0 || dy != 0)
        r.moveBy(dx, dy);

    return r;
}

int Score::x(int col)
{
    return col*bitfont->width();
}

int Score::y(float row)
{
    return (int) (row*(bitfont->height()+bitfont->height()/4));
}
