
#include "main.h"

#include "kwormwidget.h"
#include "playground.h"
#include "maze.h"
#include "d_highscore.h"

#include <qpainter.h>
#include <qkeycode.h>
#include <qtimer.h>
#include <qfont.h>
#include <qapplication.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// Delay between two worm steps
#define TIMER_DELAY (_parent->speed())
#define NEW_LIFE_SCORE 2500

CWormPlayground::CWormPlayground(QWidget* parent,CTimeWidget* tb) : QWidget(parent)
{
	_parent = (KWormWidget*) parent;
	setBackgroundMode(PaletteMid);

	connect(this,SIGNAL(sigStart()),this,SLOT(slotStart()));
	connect(this,SIGNAL(sigRestart()),this,SLOT(slotRestart()));
	connect(this,SIGNAL(sigChangeScore(unsigned int)),this,SLOT(slotChangeScore(unsigned int)));
	connect(this,SIGNAL(sigDie(char)),this,SLOT(slotTerminate(char)));

	_timerBar = tb;

	_worm = new CWorm(this);
	//	connect(_worm,SIGNAL(sigAccessField(int,int)),this,SLOT(slotWormAccess(int,int)));
	connect(_worm,SIGNAL(sigUpdate(const QRect&)),this,SLOT(update(const QRect&)));
	connect(_worm,SIGNAL(sigSpecialUpdate(QRect)),this,SLOT(slotSpecialUpdate(QRect)));

	_timer = new QTimer(this);
	connect(_timer,SIGNAL(timeout()),this,SLOT(slotTimer()));

	_restartTimer = new QTimer(this);
	connect(_restartTimer,SIGNAL(timeout()),this,SLOT(slotDoRestart()));

	_timerCount = -1;
	connect(parent,SIGNAL(sigNewGame()),this,SLOT(slotDoStart()));
	connect(parent,SIGNAL(sigStopGame(char)),this,SLOT(slotTerminate(char)));
	connect(parent,SIGNAL(sigPauseGame(bool)),this,SLOT(slotDoPause(bool)));
	connect(parent,SIGNAL(sigSpeedChanged(int)),this,SLOT(slotChangeSpeed(int)));

	_maze = new CMaze(this);
	connect(_maze,SIGNAL(sigEatBall(char)),this,SLOT(slotEatBall(char)));
	connect(_maze,SIGNAL(sigUpdate(const QRect&)),this,SLOT(update(const QRect&)));
	
	connect(_maze,SIGNAL(sigEatBall(char)),_worm,SLOT(slotEatBall(char)));
	connect(_maze,SIGNAL(sigOpenExit()),_timerBar,SLOT(slotStop()));

	setLives(0);

	setFocusPolicy(QWidget::StrongFocus);
	setFocus();
}

CWormPlayground::~CWormPlayground()
{
	if (QApplication::closingDown()) return;
	delete _worm;
	delete _maze;
}

void CWormPlayground::paintEvent(QPaintEvent* ev)
{
	char datadir[50]; // For pixmaps
	char temp[50];
	getDatadir(datadir);
	strcpy(temp,datadir);
	bool __status;
	QPixmap pxm;
	QPainter painter;
	QRect drawRect;
	painter.begin(this);
	QFont fnt("Helvetica",14,QFont::Bold);
	//	const QFont& old = painter.font();
	switch(runMode()) {
	case RM_NOT_STARTED:
// 		fatal("CWormPlayground::paintEvent called with RM_NOT_STARTED");
// 		__status = pxm.load("kworm1.xpm");
// 		if (!__status) fatal("Keine Pixmap");
// 		painter.drawPixmap(QPoint(1,1),pxm);
		break;
	case RM_RESTART_STALLED:
		__status = pxm.load(strcat(temp,"restart.xpm"));
		if (!__status) fatal("Keine Pixmap");

		drawRect.setRect((width() - pxm.width()) / 2,(height() - pxm.height()) / 2,pxm.width(),
				pxm.height());
		bitBlt(this,drawRect.topLeft(),&pxm);

		strcpy(temp,datadir);
		__status = pxm.load(strcat(temp,"f3.xpm"));
		if (!__status) fatal("Keine Pixmap");
		bitBlt(this,drawRect.bottomLeft() + QPoint(0,10),&pxm);
		break;
	case RM_RESTARTING:
		__status = pxm.load(strcat(temp,"restart.xpm"));
		if (!__status) fatal("Keine Pixmap");

		drawRect.setRect((width() - pxm.width()) / 2,(height() - pxm.height()) / 2,pxm.width(),
				pxm.height());
		bitBlt(this,drawRect.topLeft(),&pxm);
		break;
	case RM_PAUSED:
		__status = pxm.load(strcat(temp,"pause.xpm"));
		if (!__status) fatal("Keine Pixmap");

		drawRect.setRect((width() - pxm.width()) / 2,(height() - pxm.height()) / 2,pxm.width(),
				pxm.height());
		bitBlt(this,drawRect.topLeft(),&pxm);
		
		strcpy(temp,datadir);
		__status = pxm.load(strcat(temp,"f3.xpm"));
		if (!__status) fatal("Keine Pixmap");
		bitBlt(this,drawRect.bottomLeft() + QPoint(0,10),&pxm);
		break;
	default:
		_maze->draw(&painter);
		_worm->draw(&painter);
	}
	painter.end();

}

void CWormPlayground::slotDoStart()
{
	emit sigStart();
}

void CWormPlayground::slotDoRestart()
{
	emit sigRestart();
}

void CWormPlayground::slotDoPause(bool force)
{
	if (force) {
		_timer->stop();
// 		if (_runMode != RM_PAUSED) {
// 			_runMode = RM_PAUSED;
// 			update();
// 		}
		switch (_runMode) {
		case RM_PAUSED:
		case RM_RESTART_STALLED:
		case RM_NOT_STARTED: break;
		case RM_RUNNING:
			_runMode = RM_PAUSED;
 			update();
			break;
		case RM_RESTARTING:
			_restartTimer->stop();
			_runMode = RM_RESTART_STALLED;
			update();
			break;
		}
		return;
	}
	switch(_runMode) {
	case RM_PAUSED:
		_timer->start(TIMER_DELAY);
		_runMode = RM_RUNNING;
		break;
	case RM_RUNNING:
		_timer->stop();
		_runMode = RM_PAUSED;
		break;
	case RM_RESTARTING:
		_restartTimer->stop();
		_runMode = RM_RESTART_STALLED;
		update();
		break;
	case RM_RESTART_STALLED:
		emit sigRestart();
		break;
	}
	update();
}

void CWormPlayground::slotStart()
{
	setScore(0);
	setLives(3); 
	_timer->start(TIMER_DELAY);
	_runMode = RM_RUNNING;
	_maze->load(_levelSet,_levelNumber);
	_bgPixmap = _parent->getBackgroundPixmap();
	setBackgroundPixmap(*_bgPixmap);
	update();
}

void CWormPlayground::slotRestart() 
{
	_timer->start(TIMER_DELAY);
	_runMode = RM_RUNNING;
	_bgPixmap = _parent->getBackgroundPixmap();
	setBackgroundPixmap(*_bgPixmap);
	update();
}

void CWormPlayground::slotEatBall(char sc)
{
	setScore(score() + sc);
	((KWormWidget*) parentWidget())->resetTimer();
}

void CWormPlayground::slotSpecialUpdate(QRect r)
{
	QSize __a(r.width() * FDWIDTH,r.height() * FDHEIGHT);
	QPixmap* np = new QPixmap(__a);
	
	for(int i = r.left();i <= r.right();i++)
		for(int j = r.top();j <= r.bottom();j++) {
			QPoint pixmapLoc = LPtoDP(QPoint(i - r.left(),j - r.top()));
			bitBlt(np,pixmapLoc,_bgPixmap,QRect(0,0,FDWIDTH,FDHEIGHT),CopyROP);
			maze()->drawAt(QPoint(i,j),np,pixmapLoc);
			worm()->drawAt(QPoint(i,j),np,pixmapLoc);
		}
	// np now contains picture to draw
	bitBlt(this,LPtoDP(r.topLeft()),np,np->rect(),CopyROP,true);
}

void CWormPlayground::slotTimer()
{
	_timerCount++;
	if (_timerCount == 30240) _timerCount = 0;
	emit sigTimer(_timerCount);
}

void CWormPlayground::slotTerminate(char reason)
{
	_timer->stop();
	switch (reason) {
	case DieWon: QMessageBox::information(0,"Cool","You may enter the next level now",
					      QMessageBox::Ok | QMessageBox::Default);
		_runMode = RM_RESTARTING;
		update();
		_restartTimer->start(3000,true);
		return;
	case DieTerminateCommand: 
		// Here might be a message box
		break;
	case DieRestartCommand: 
		// To avoid aforesaid message box
		break;
	default: QMessageBox::information(0,"You die","You die",QMessageBox::Ok | QMessageBox::Default);
		if (loseLife()) break; 
		_runMode = RM_RESTARTING;
		_restartTimer->start(3000,true);
		return;
	}
	// End of game
       	_runMode = RM_NOT_STARTED;
        emit sigTerminate();
	// Highscores
	showHighscoreDialog();
}

void CWormPlayground::slotChangeScore(unsigned int s)
{
	if ((score() / NEW_LIFE_SCORE) != (s / NEW_LIFE_SCORE)) {
		if (livesCount() < 6) setLives(livesCount() + 1);
	}
}

void CWormPlayground::slotChangeSpeed(int msec)
{
	if (_timer->isActive()) {
		_timer->changeInterval(msec);
	}
}

void CWormPlayground::fwdKey(QKeyEvent* e)
{
	switch (e->key()) 
		{
		case Key_Left: emit sigNewDir(WormLeft); break;
		case Key_Right: emit sigNewDir(WormRight); break;
		case Key_Down: emit sigNewDir(WormDown); break;
		case Key_Up: emit sigNewDir(WormUp); break;
		case Key_E: emit sigDie(DieTerminateCommand); break;	
		default: e->ignore();
		}

}

QPoint CWormPlayground::LPtoDP(QPoint pt)
{
	QPoint t(pt.x() * FDWIDTH,pt.y() * FDHEIGHT);
	return t;
}

QRect CWormPlayground::LPtoDP(QRect rct)
{
	QPoint tl = rct.topLeft();
	tl = LPtoDP(tl);
	QPoint br = rct.bottomRight();
	br = LPtoDP(br);
	return QRect(tl,br);
}

QPoint CWormPlayground::DPtoLP(QPoint pt)
{
	QPoint t(pt.x() / FDWIDTH,pt.y() / FDHEIGHT);
	return t;
}

QRect CWormPlayground::DPtoLP(QRect rct)
{
	QPoint tl = rct.topLeft();
	tl = DPtoLP(tl);
	QPoint br = rct.bottomRight();
	br = DPtoLP(br);
	return QRect(tl,br);
}

int CWormPlayground::fieldState(QPoint pt,char dir)
{
	if (_worm->covers(pt)) return FieldWorm;
	return (_maze->accessible(pt,dir));

}

void CWormPlayground::winningLevel()
{
	// Activate next level
	_levelNumber++;
	if (!_maze->load(_levelSet,_levelNumber)) {
		_levelNumber = 1;
		if (!_maze->load(_levelSet,_levelNumber)) fatal("Error loading level"); //TODO
	}

	emit sigDie(DieWon);
	setScore(score() + 200); // TODO
}

bool CWormPlayground::wormAccess(int x,int y)
{
	int fs;
	if (maze()->accessField(x,y)) return true;
	if ((fs = fieldState(QPoint(x,y),_worm->dir())) > 0) {
		emit sigDie(fs);
		return true;
	}
	return false;
}

void CWormPlayground::showHighscoreDialog()
{
	CHighscoreDialog dlg(_parent,score());
	dlg.initializeDialog();
	dlg.exec();

}
