/*-
 * neri.c - Neri's face for xlock, the X Window System lockscreen.
 *
 * Copyright (c) 1988 by Sun Microsystems
 *
 * See xlock.c for copying information.
 *
 * Revision History:
 * 07-Oct-98: Neri, a NeriLock mode ported to KDE
 *            (David.Banz@smail.inf.fh-rhein-sieg.de)
 * 18-Sep-95: 5 bats now in color (patol@info.isbiel.ch)
 * 20-Sep-94: 5 bats instead of bouncing balls, based on bounce.c
 *            (patol@info.isbiel.ch)
 * 2-Sep-93: bounce version (David Bagley bagleyd@hertz.njit.edu)
 * 1986: Sun Microsystems
 */

/* 
 * original copyright
 * **************************************************************************
 * Copyright 1988 by Sun Microsystems, Inc. Mountain View, CA.
 *
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of Sun or MIT not be used in advertising
 * or publicity pertaining to distribution of the software without specific
 * prior written permission. Sun and M.I.T. make no representations about the
 * suitability of this software for any purpose. It is provided "as is"
 * without any express or implied warranty.
 *
 * SUN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * IN NO EVENT SHALL SUN BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * ***************************************************************************
 */


/* original "bat" mode ported to kscreensave:
   July 1997, Emanuel Pirker <epirker@edu.uni-klu.ac.at>
   Contact me if something doesn't work correctly!
   Last revised: 11-Jul-97
*/

// layout management added 1998/04/19 by Mario Weilguni <mweilguni@kde.org>

#include "xlock.h"
#include <math.h>

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#define MINSPEED 0
#define MAXSPEED 100
#define DEFSPEED 50
#define MINBATCH 0
#define MAXBATCH 20
#define DEFBATCH 5

#if HAVE_XPM
#if 1
#include <X11/xpm.h>
#else
#include <xpm.h>
#endif
#include "pixmaps/neri-0.xpm"
#endif

#include "bitmaps/neri-0.xbm"

#define MAX_STRENGTH 24
#define FRICTION 15
#define PENETRATION 0.4
#define SLIPAGE 4
#define TIME 32

#define ORIENTS 8
#define ORIENTCYCLE 32
#define CCW 1
#define CW (ORIENTS-1)
#define DIR(x)	(((x)>=0)?CCW:CW)
#define SIGN(x)	(((x)>=0)?1:-1)
#define ABS(x)	(((x)>=0)?x:-(x))

//ModeSpecOpt neri_opts = {0, NULL, NULL, NULL};

static XImage bimages[] =
{
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1}
};
static XImage *images[ORIENTS / 2 + 1];

typedef struct {
	int         x, y, xlast, ylast;
	int         spincount, spindelay, spindir, orient;
	int         vx, vy, vang;
	int         mass, size, sizex, sizey;
	unsigned long color;
} neristruct;

typedef struct {
	int         width, height;
	int         nneris;
	int         restartnum;
	neristruct   neris[MAXBATCH];
} bouncestruct;

static bouncestruct bounces[MAXSCREENS];

static void checkCollision(int a_neri);
static void drawaneri(Window win, neristruct * neri);
static void moveneri(neristruct * neri);
static void flapneri(neristruct * neri, int dir, int *vel);
static int  collide(int a_neri);
static void XEraseImage(Display * display, Window win, GC gc, int x, int y, int xlast, int ylast, int xsize, int ysize);

static      first = 1;

static unsigned char *bits[] =
{
	neri0_bits, neri0_bits, neri0_bits, neri0_bits, neri0_bits
};

#if HAVE_XPM
static char **pixs[] =
{
	neri0, neri0, neri0, neri0, neri0
};

#endif

static void
init_images()
{
	int         i;

#if HAVE_XPM
	int         xpm_ret = 0;

	if (!mono && Scr[screen].npixels > 2)
		for (i = 0; i <= ORIENTS / 2; i++)
			xpm_ret += XpmCreateImageFromData(dsp, pixs[i], &(images[i]),
				   (XImage **) NULL, (XpmAttributes *) NULL);
	if (mono || Scr[screen].npixels <= 2 || xpm_ret != 0)
#endif
		for (i = 0; i <= ORIENTS / 2; i++) {
			bimages[i].data = (char *) bits[i];
			bimages[i].width = neri0_width;
			bimages[i].height = neri0_height;
			bimages[i].bytes_per_line = (neri0_width + 7) / 8;
			images[i] = &(bimages[i]);
		}
}

void
initneri(Window win)
{
	bouncestruct *bp = &bounces[screen];
	int         i;
	XWindowAttributes xwa;

	if (first) {
		init_images();
		first = 0;
	}
	XGetWindowAttributes(dsp, win, &xwa);
	bp->width = xwa.width;
	bp->height = xwa.height;
	bp->restartnum = TIME;

	bp->nneris = batchcount;
	if (bp->nneris < 1)
		bp->nneris = 1;
	//if (!bp->neris)
	//	bp->neris = (neristruct *) malloc(bp->nneris * sizeof (neristruct));
	i = 0;
	while (i < bp->nneris) {
		if (neri0_width > bp->width / 2 || neri0_height > bp->height / 2) {
			bp->neris[i].sizex = 7;
			bp->neris[i].sizey = 3;
			bp->neris[i].size = (bp->neris[i].sizex + bp->neris[i].sizey) / 2;
		} else {
			bp->neris[i].sizex = neri0_width;
			bp->neris[i].sizey = neri0_height;
			bp->neris[i].size = (bp->neris[i].sizex + bp->neris[i].sizey) / 2;
		}
		bp->neris[i].vx = ((LRAND() & 1) ? -1 : 1) * (LRAND() % MAX_STRENGTH + 1);
		bp->neris[i].x = (bp->neris[i].vx >= 0) ? 0 : bp->width - bp->neris[i].sizex;
		bp->neris[i].y = LRAND() % (bp->height / 2);
		if (i == collide(i)) {
			if (!mono && Scr[screen].npixels > 2)
				bp->neris[i].color = Scr[screen].pixels[LRAND() % Scr[screen].npixels];
			else
				bp->neris[i].color = WhitePixel(dsp, screen);
			bp->neris[i].xlast = -1;
			bp->neris[i].ylast = 0;
			bp->neris[i].spincount = 1;
			bp->neris[i].spindelay = 1;
			bp->neris[i].vy = ((LRAND() & 1) ? -1 : 1) * (LRAND() % MAX_STRENGTH);
			bp->neris[i].spindir = 0;
			bp->neris[i].vang = 0;
			bp->neris[i].orient = LRAND() % ORIENTS;
			i++;
		} else
			bp->nneris--;
	}
	XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
	XFillRectangle(dsp, win, Scr[screen].gc, 0, 0, bp->width, bp->height);
}

static void
checkCollision(int a_neri)
{
	bouncestruct *bp = &bounces[screen];
	int         i, amount, spin, d, size;
	double      x, y;

	for (i = 0; i < bp->nneris; i++) {
		if (i != a_neri) {
			x = (double) (bp->neris[i].x - bp->neris[a_neri].x);
			y = (double) (bp->neris[i].y - bp->neris[a_neri].y);
			d = (int) sqrt(x * x + y * y);
			size = (bp->neris[i].size + bp->neris[a_neri].size) / 2;
			if (d > 0 && d < size) {
				amount = size - d;
				if (amount > PENETRATION * size)
					amount = (int)(PENETRATION * size);
				bp->neris[i].vx += (int)(amount * x / d);
				bp->neris[i].vy += (int)(amount * y / d);
				bp->neris[i].vx -= bp->neris[i].vx / FRICTION;
				bp->neris[i].vy -= bp->neris[i].vy / FRICTION;
				bp->neris[a_neri].vx -= (int)(amount * x / d);
				bp->neris[a_neri].vy -= (int)(amount * y / d);
				bp->neris[a_neri].vx -= bp->neris[a_neri].vx / FRICTION;
				bp->neris[a_neri].vy -= bp->neris[a_neri].vy / FRICTION;
				spin = (bp->neris[i].vang - bp->neris[a_neri].vang) /
					(2 * size * SLIPAGE);
				bp->neris[i].vang -= spin;
				bp->neris[a_neri].vang += spin;
				bp->neris[i].spindir = DIR(bp->neris[i].vang);
				bp->neris[a_neri].spindir = DIR(bp->neris[a_neri].vang);
				if (!bp->neris[i].vang) {
					bp->neris[i].spindelay = 1;
					bp->neris[i].spindir = 0;
				} else
					bp->neris[i].spindelay = (int)(M_PI * bp->neris[i].size /
						(ABS(bp->neris[i].vang)) + 1);
				if (!bp->neris[a_neri].vang) {
					bp->neris[a_neri].spindelay = 1;
					bp->neris[a_neri].spindir = 0;
				} else
					bp->neris[a_neri].spindelay = (int)(M_PI * bp->neris[a_neri].size /
						(ABS(bp->neris[a_neri].vang)) + 1);
				return;
			}
		}
	}
}

void
drawneri(Window win)
{
	bouncestruct *bp = &bounces[screen];
	int         i;

	for (i = 0; i < bp->nneris; i++) {
		drawaneri(win, &bp->neris[i]);
		moveneri(&bp->neris[i]);
	}
	for (i = 0; i < bp->nneris; i++)
		checkCollision(i);
	if (!(LRAND() % TIME))	/* Put some randomness into the time */
		bp->restartnum--;
	if (!bp->restartnum)
		initneri(win);
}

static void
drawaneri(Window win, neristruct * neri)
{
	if (neri->sizex < neri0_width) {
		if (neri->xlast != -1) {
			XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
			XFillRectangle(dsp, win, Scr[screen].gc,
			     neri->xlast, neri->ylast, neri->sizex, neri->sizey);
		}
		XSetForeground(dsp, Scr[screen].gc, neri->color);
		XFillRectangle(dsp, win, Scr[screen].gc,
			       neri->x, neri->y, neri->sizex, neri->sizey);
	} else {
		XSetForeground(dsp, Scr[screen].gc, neri->color);
		XPutImage(dsp, win, Scr[screen].gc,
			  images[(neri->orient > ORIENTS / 2) ? ORIENTS - neri->orient : neri->orient],
			  0, 0, neri->x, neri->y, neri->sizex, neri->sizey);
		if (neri->xlast != -1) {
			XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
			XEraseImage(dsp, win, Scr[screen].gc,
				    neri->x, neri->y, neri->xlast, neri->ylast, neri->sizex, neri->sizey);
		}
	}
}

static void
moveneri(neristruct * neri)
{
	bouncestruct *bp = &bounces[screen];

	neri->xlast = neri->x;
	neri->ylast = neri->y;
	neri->x += neri->vx;
	if (neri->x > (bp->width - neri->sizex)) {
		/* Bounce off the right edge */
		neri->x = 2 * (bp->width - neri->sizex) - neri->x;
		neri->vx = -neri->vx + neri->vx / FRICTION;
		flapneri(neri, 1, &neri->vy);
	} else if (neri->x < 0) {
		/* Bounce off the left edge */
		neri->x = -neri->x;
		neri->vx = -neri->vx + neri->vx / FRICTION;
		flapneri(neri, -1, &neri->vy);
	}
	neri->vy++;
	neri->y += neri->vy;
	if (neri->y >= (bp->height + neri->sizey)) {	/* Don't see neri bounce */
		/* Bounce off the bottom edge */
		neri->y = (bp->height - neri->sizey);
		neri->vy = -neri->vy + neri->vy / FRICTION;
		flapneri(neri, -1, &neri->vx);
	}			/* else if (neri->y < 0) { */
	/* Bounce off the top edge */
	/*neri->y = -neri->y;
	   neri->vy = -neri->vy + neri->vy / FRICTION;
	   flapneri(neri, 1, &neri->vx);
	   } */
	if (neri->spindir) {
		neri->spincount--;
		if (!neri->spincount) {
			neri->orient = (neri->spindir + neri->orient) % ORIENTS;
			neri->spincount = neri->spindelay;
		}
	}
}

static void
flapneri(neristruct * neri, int dir, int *vel)
{
	*vel -= (int)((*vel + SIGN(*vel * dir) * neri->spindelay * ORIENTCYCLE /
		 (M_PI * neri->size)) / SLIPAGE);
	if (*vel) {
		neri->spindir = DIR(*vel * dir);
		neri->vang = *vel * ORIENTCYCLE;
		neri->spindelay = (int)(M_PI * neri->size / (ABS(neri->vang)) + 1);
	} else
		neri->spindir = 0;
}

static int
collide(int a_neri)
{
	bouncestruct *bp = &bounces[screen];
	int         i, d, x, y;

	for (i = 0; i < a_neri; i++) {
		x = (bp->neris[i].x - bp->neris[a_neri].x);
		y = (bp->neris[i].y - bp->neris[a_neri].y);
		d = (int) sqrt((double) (x * x + y * y));
		if (d < (bp->neris[i].size + bp->neris[a_neri].size) / 2)
			return i;
	}
	return i;
}

/* This stops some flashing, could be more efficient */
static void
XEraseImage(Display * display, Window win, GC gc, int x, int y, int xlast, int ylast, int xsize, int ysize)
{
	if (ylast < y) {
		if (y < ylast + ysize)
			XFillRectangle(display, win, gc, xlast, ylast, xsize, y - ylast);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	} else if (ylast > y) {
		if (y > ylast - ysize)
			XFillRectangle(display, win, gc, xlast, y + ysize, xsize, ylast - y);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	}
	if (xlast < x) {
		if (x < xlast + xsize)
			XFillRectangle(display, win, gc, xlast, ylast, x - xlast, ysize);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	} else if (xlast > x) {
		if (x > xlast - xsize)
			XFillRectangle(display, win, gc, x + xsize, ylast, xlast - x, ysize);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	}
}

// --------------------------------------------------------------------

#include <qpushbt.h>
#include <qchkbox.h>
#include <qcolor.h>
#include <qmsgbox.h>
#include <qlayout.h>
#include <kbuttonbox.h>
#include "kslider.h"
#include "helpers.h"

#include "neri.h"

#include "neri.moc"

// this refers to klock.po. If you want an extra dictionary, 
// create an extra KLocale instance here.
extern KLocale *glocale;

static kNeriSaver *saver = NULL;

void startScreenSaver( Drawable d )
{
	if ( saver )
		return;
	saver = new kNeriSaver( d );
}

void stopScreenSaver()
{
	if ( saver )
		delete saver;
	saver = NULL;
}

int setupScreenSaver()
{
	kNeriSetup dlg;

	return dlg.exec();
}

const char *getScreenSaverName()
{
	return glocale->translate("Neri");
}

//-----------------------------------------------------------------------------

kNeriSaver::kNeriSaver( Drawable drawable ) : kScreenSaver( drawable )
{
	readSettings();

	colorContext = QColor::enterAllocContext();

	batchcount = maxLevels;

	initXLock( gc );
	initneri( d );

	timer.start( speed );
	connect( &timer, SIGNAL( timeout() ), SLOT( slotTimeout() ) );
}

kNeriSaver::~kNeriSaver()
{
	timer.stop();
	QColor::leaveAllocContext();
	QColor::destroyAllocContext( colorContext );
}

void kNeriSaver::setSpeed( int spd )
{
	timer.stop();
	speed = MAXSPEED - spd;
	timer.start( speed );
}

void kNeriSaver::setLevels( int l )
{
	batchcount = maxLevels = l;
	initneri( d );
}

void kNeriSaver::readSettings()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString str;

	str = config->readEntry( "Speed" );
	if ( !str.isNull() )
		speed = MAXSPEED - atoi( str );
	else
		speed = DEFSPEED;

	str = config->readEntry( "MaxLevels" );
	if ( !str.isNull() )
		maxLevels = atoi( str );
	else
		maxLevels = DEFBATCH;

}

void kNeriSaver::slotTimeout()
{
    drawneri( d );
}

//-----------------------------------------------------------------------------

kNeriSetup::kNeriSetup( QWidget *parent, const char *name )
	: QDialog( parent, name, TRUE )
{
	speed = 50;

	readSettings();

	setCaption( glocale->translate("Setup KNeri") );

	QLabel *label;
	QPushButton *button;
	KSlider *slider;
	
	QVBoxLayout *tl = new QVBoxLayout(this, 10);
	QHBoxLayout *tl1 = new QHBoxLayout;
	tl->addLayout(tl1);

	QVBoxLayout *tl11 = new QVBoxLayout(5);
	tl1->addLayout(tl11);
	label = new QLabel( glocale->translate("Speed:"), this );
	min_size(label);
	tl11->addWidget(label);

	slider = new KSlider( KSlider::Horizontal, this );
	slider->setFixedHeight(20);
	slider->setMinimumWidth(90);
	slider->setRange( MINSPEED, MAXSPEED );
	slider->setSteps( (MAXSPEED-MINSPEED)/4, (MAXSPEED-MINSPEED)/2 );
	slider->setValue( speed );
	connect( slider, SIGNAL( valueChanged( int ) ), 
		 SLOT( slotSpeed( int ) ) );
	tl11->addWidget(slider);
	tl11->addSpacing(15);

	label = new QLabel( glocale->translate("Number of pics:"), this );
	min_size(label);
	tl11->addWidget(label);

	slider = new KSlider( KSlider::Horizontal, this );
	slider->setFixedHeight(20);
	slider->setMinimumWidth(90);
	slider->setRange( MINBATCH, MAXBATCH );
	slider->setSteps( (MAXBATCH-MINBATCH)/4, (MAXBATCH-MINBATCH)/2 );
	slider->setValue( maxLevels );
	connect( slider, SIGNAL( valueChanged( int ) ), 
		 SLOT( slotLevels( int ) ) );
	tl11->addWidget(slider);
	tl11->addStretch(1);

	preview = new QWidget( this );
	preview->setFixedSize( 220, 170 );
	preview->setBackgroundColor( black );
	preview->show();    // otherwise saver does not get correct size
	saver = new kNeriSaver( preview->winId() );
	tl1->addWidget(preview);

	KButtonBox *bbox = new KButtonBox(this);	
	button = bbox->addButton( glocale->translate("About"));
	connect( button, SIGNAL( clicked() ), SLOT(slotAbout() ) );
	bbox->addStretch(1);

	button = bbox->addButton( glocale->translate("Ok"));	
	connect( button, SIGNAL( clicked() ), SLOT( slotOkPressed() ) );

	button = bbox->addButton(glocale->translate("Cancel"));
	connect( button, SIGNAL( clicked() ), SLOT( reject() ) );
	bbox->layout();
	tl->addWidget(bbox);

	tl->freeze();
}

void kNeriSetup::readSettings()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString str;

	str = config->readEntry( "Speed" );
	if ( !str.isNull() )
		speed = atoi( str );

	if ( speed > MAXSPEED )
		speed = MAXSPEED;
	else if ( speed < MINSPEED )
		speed = MINSPEED;

	str = config->readEntry( "MaxLevels" );
	if ( !str.isNull() )
		maxLevels = atoi( str );
	else
		maxLevels = DEFBATCH;

}

void kNeriSetup::slotSpeed( int num )
{
	speed = num;

	if ( saver )
		saver->setSpeed( speed );
}

void kNeriSetup::slotLevels( int num )
{
	maxLevels = num;

	if ( saver )
		saver->setLevels( maxLevels );
}

void kNeriSetup::slotOkPressed()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString sspeed;
	sspeed.setNum( speed );
	config->writeEntry( "Speed", sspeed );

	QString slevels;
	slevels.setNum( maxLevels );
	config->writeEntry( "MaxLevels", slevels );

	config->sync();
	accept();
}

void kNeriSetup::slotAbout()
{
	QMessageBox::message(glocale->translate("About Neri"), 
			     glocale->translate("Neri\n\nCopyright (c) 1986 by Sun Microsystems,\nDavid Banz and J.M. Shiff Productions\n\nPorted to kscreensave by David Banz."), 
			     glocale->translate("Ok"));
}


