/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** cmesh.cpp
*/

#include "cmesh.h" 
#include "view.h" 

#include "graph.h" 


cmesh::cmesh(base *p,char *n,texture *t) :
	csgobj(p,n,t)
{
	numtype = NUM_CMESH;

	gptr = 0;

	if(parent)
		addToParent(parent);
}


cmesh::cmesh(base *p,char *n,Vector3 &v1,Vector3 &v2,Vector3 &v3):
	csgobj(p,n)
{
	if(parent)
		addToParent(parent);

	gptr = 0;

	addSegment(addKnode(v1),addKnode(v2),addKnode(v3));
}

cmesh::cmesh(base *p,char *n,texture *t,Vector3 &v1,Vector3 &v2,Vector3 &v3) :
	csgobj(p,n,t)
{
	if(parent)
		addToParent(parent);

	gptr = 0;

	addSegment(addKnode(v1),addKnode(v2),addKnode(v3));
}

cmesh::cmesh(base *p,cmesh *mc) :
	csgobj(p,mc)
{
	int		i,s0,s1,s2;
	Vector3		v0,v1;

	gptr = 0;

	if(parent)
		addToParent(parent);

	for(i = 0;i < mc->vectorlist.length();i++)
	{
		if(mc->getKnode(i,v0,v1) >= 0)
			addKnode(v0,v1);
	}
	for(i = 0;i < mc->trianglelist.length();i++)
	{
		if(mc->getSegment(i,s0,s1,s2) >= 0)
			addSegment(s0,s1,s2);
	}
}

cmesh::cmesh(mesh &m) : 	// Typumwandlung
	csgobj(m.getParent(),"Unnamed")
{
	gptr = 0;

	printf("Typecast not implemented\n");
}


cmesh::~cmesh()
{
	if(gptr)
		delete gptr;
}

base		*cmesh::copy(base *p)
{
	return new cmesh(p,this);
}

int		cmesh::addKnode(double x, double y, double z,double nx, double ny, double nz)
{
	return addKnode(Vector3(x,y,z),Vector3(nx,ny,nz));
}

int		cmesh::addKnode(Vector3 v,Vector3 n)
{
	dvec		dv;
	int		i;

	dv.position= v;
	dv.direction = n;

	if((i = vectorlist.append(dv)) != 0)
		return i;

	if(gptr)
	{
		delete gptr;
		gptr = 0;
	}

	return vectorlist.length() - 1;
}

int	cmesh::addSegment(int p1,int p2,int p3)
{
	seg		s;
	int		i;

	if(p1 < 0 || p1 >= vectorlist.length())
		return -1; // p1 out of range
	if(p2 < 0 || p2 >= vectorlist.length())
		return -2; // p2 out of range
	if(p3 < 0 || p3 >= vectorlist.length())
		return -3; // p3 out of range

	s.pointnum[0] = p1;
	s.pointnum[1] = p2;
	s.pointnum[2] = p3;

	if((i = trianglelist.append(s)) != 0)
		return i;

	if(gptr)
	{
		delete gptr;
		gptr = 0;
	}

	return trianglelist.length() - 1;
}

int		cmesh::getKnode(int n,Vector3 &v,Vector3 &nv)
{
	if(n < 0 || n > vectorlist.length())
		return -1;

	v = vectorlist[n].direction;
	nv = vectorlist[n].position;

	return 0;
}

Vector3		&cmesh::getPoint(int n)
{
	return vectorlist[n].position;
}

Vector3		&cmesh::getNormal(int n)
{
	return vectorlist[n].direction;
}

int		cmesh::getSegment(int n,int &s0,int &s1,int &s2)
{
	if(n < 0 || n > trianglelist.length()) 
		return  -1;

	s0 = trianglelist[n].pointnum[0]; 
	s1 = trianglelist[n].pointnum[1]; 
	s2 = trianglelist[n].pointnum[2];

	return 0;
}

int		cmesh::removeSegment(int segnum)
{
	if(gptr)
	{
		delete gptr;
		gptr = 0;
	}

	return trianglelist.deleteAt(segnum);
}

int		cmesh::removeKnode(int knodenum)
{
	int	i,s[3];

	if(gptr)
	{
		delete gptr;
		gptr = 0;
	}

	for(i = 0;i < trianglelist.length();i++)
	{
		getSegment(i,s[0],s[1],s[2]);

		if(knodenum == s[0] || knodenum == s[1] || knodenum == s[2])
			return -1;
	}

	return vectorlist.deleteAt(knodenum);
}


graph		*cmesh::createGraph()
{
	int		i,s1,s2,s3;

	if(gptr)
		return gptr;

	gptr = new graph();
	for(i = 0;i < vectorlist.length();i++)
	{
		new knode(gptr,vectorlist[i].position);
	}
	for(i = 0;i < trianglelist.length();i++)
	{
		s1 = trianglelist[i].pointnum[0];
		s2 = trianglelist[i].pointnum[1];
		s3 = trianglelist[i].pointnum[2];
		new edge(gptr,gptr->getKnode(s1),gptr->getKnode(s2),gptr->numEdges() + 1);
		new edge(gptr,gptr->getKnode(s2),gptr->getKnode(s3),gptr->numEdges() + 1);
		new edge(gptr,gptr->getKnode(s3),gptr->getKnode(s1),gptr->numEdges() + 1);
	}

	return gptr;
}


int		cmesh::createFromGraph(graph *g)
{
	int		i,t,c;
	Vector3		v;
	list<edge*>	del;
	edge		*e1,*e2,*e3;
	knode		*k;
	graph		*gt;

	if(!g)
		return 0;

	vectorlist.empty();
	trianglelist.empty();

	gt = gptr;
	gptr = 0;

	for(i = 0;i < g->numKnodes();i++)
	{
		k = g->getKnode(i);
		v = k->vector();
		addKnode(v);
	}
	for(i = 0;i < g->numEdges();i++)
	{
		del += g->getEdge(i);
	}
	for(i = 0;i < g->numEdges();i++)
	{
		del += g->getEdge(i);
	}

	for(c = 0;c < del.length();c++)
	{
		e1 = del[c % del.length()];
		for(i = 0;i < e1->knode2()->numEdges();i++)
		{
			e2 = e1->knode2()->getEdge(i);
			if(e1 == e2 || del.find(e2) < 0 || e1->knode2() != e2->knode1())
				continue;
			for(t = 0;t < e2->knode2()->numEdges();t++)
			{
				e3 = e2->knode2()->getEdge(t);
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->knode1()->vector()[0],e1->knode1()->vector()[1],e1->knode1()->vector()[2],
					e2->knode1()->vector()[0],e2->knode1()->vector()[1],e2->knode1()->vector()[2],
					e3->knode1()->vector()[0],e3->knode1()->vector()[1],e3->knode1()->vector()[2]);
*/
				if(e2->knode2() != e3->knode1() || e3->knode2() != e1->knode1())
					continue;
				if(addSegment(e1,e2,e3))
				{
					del.deleteAt(del.find(e1));
					del.deleteAt(del.find(e2));
					del.deleteAt(del.find(e3));
					e1 = del[c % del.length()];
				}
			}
			for(t = 0;t < e2->knode2()->numEdges();t++)
			{
				e3 = e2->knode2()->getEdge(t)->swap();
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->knode1()->vector()[0],e1->knode1()->vector()[1],e1->knode1()->vector()[2],
					e2->knode1()->vector()[0],e2->knode1()->vector()[1],e2->knode1()->vector()[2],
					e3->knode1()->vector()[0],e3->knode1()->vector()[1],e3->knode1()->vector()[2]);
*/
				if(e2->knode2() != e3->knode1() || e3->knode2() != e1->knode1())
					continue;
				if(addSegment(e1,e2,e3))
				{
					del.deleteAt(del.find(e1));
					del.deleteAt(del.find(e2));
					del.deleteAt(del.find(e3));
					e1 = del[c % del.length()];
				}
			}
		}
		for(i = 0;i < e1->knode2()->numEdges();i++)
		{
			e2 = e1->knode2()->getEdge(i)->swap();
			if(e1 == e2 || del.find(e2) < 0 || e1->knode2() != e2->knode1())
				continue;
			for(t = 0;t < e2->knode2()->numEdges();t++)
			{
				e3 = e2->knode2()->getEdge(t);
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->knode1()->vector()[0],e1->knode1()->vector()[1],e1->knode1()->vector()[2],
					e2->knode1()->vector()[0],e2->knode1()->vector()[1],e2->knode1()->vector()[2],
					e3->knode1()->vector()[0],e3->knode1()->vector()[1],e3->knode1()->vector()[2]);
*/
				if(e2->knode2() != e3->knode1() || e3->knode2() != e1->knode1())
					continue;
				if(addSegment(e1,e2,e3))
				{
					del.deleteAt(del.find(e1));
					del.deleteAt(del.find(e2));
					del.deleteAt(del.find(e3));
					e1 = del[c % del.length()];
				}
			}
			for(t = 0;t < e2->knode2()->numEdges();t++)
			{
				e3 = e2->knode2()->getEdge(t)->swap();
				if(e1 == e3 || e2 == e3 || del.find(e3) < 0)
					continue;
/*
				printf("seg(%i,%i,%i) <%g,%g,%g><%g,%g,%g><%g,%g,%g>\n",e1->id(),e2->id(),e3->id(),
					e1->knode1()->vector()[0],e1->knode1()->vector()[1],e1->knode1()->vector()[2],
					e2->knode1()->vector()[0],e2->knode1()->vector()[1],e2->knode1()->vector()[2],
					e3->knode1()->vector()[0],e3->knode1()->vector()[1],e3->knode1()->vector()[2]);
*/
				if(e2->knode2() != e3->knode1() || e3->knode2() != e1->knode1())
					continue;
				if(addSegment(e1,e2,e3))
				{
					del.deleteAt(del.find(e1));
					del.deleteAt(del.find(e2));
					del.deleteAt(del.find(e3));
					e1 = del[c % del.length()];
				}
			}
		}
//		sleep(1);
	}

	if(gt)
		delete gt;

	return 1;
}

int	cmesh::addSegment(edge *e1,edge *e2,edge *e3)
{
	int		n,s1,s2,s3;

	if(e1 != e2 && e1 != e3 && e2 != e3
		&& e1->knode2() == e2->knode1()
		&& e2->knode2() == e3->knode1()
		&& e3->knode2() == e1->knode1())
	{
		s1 = s2 = s3 = -1;
		for(n = 0;n < vectorlist.length();n++)
		{
			if(vectorlist[n].position == e1->knode1()->vector())
				s1 = n;
			if(vectorlist[n].position == e1->knode2()->vector())
				s2 = n;
			if(vectorlist[n].position == e2->knode2()->vector())
				s3 = n;
		}
		addSegment(s1,s2,s3);
		printf("add seg (%i,%i,%i)\n",s1,s2,s3);

		return 1;
	}

	return 0;
}


int		cmesh::isClose()
{
	graph		*g;

	g = createGraph();

	return g->isClose();
}


list<edge*>		cmesh::getOpenPolygone(list<edge*> el)
{
	graph		*g;

	g = createGraph();

	return g->getOpenPolygone(el);
}

int		cmesh::closeMesh()
{
	graph		*g;

	g = createGraph();


	printf("graph has %i points and %i edges\n",g->numKnodes(),g->numEdges());

/*
	g->removeDoubleKnodes();
	g->removeDoubleEdges();
*/
	if(g->isClose())
	{
		printf("is close\n");
		return 1;
	}

	printf("try to close\n");


	if(g->close())
	{
		if(g->isClose())
		{
			printf("is close\n");
		}
		else
		{
			printf("is not close\n");
		}

		printf("graph has %i points and %i edges\n",g->numKnodes(),g->numEdges());
		return createFromGraph(g);
	}

	printf("graph has %i points and %i edges\n",g->numKnodes(),g->numEdges());

	createFromGraph(g);

	return 0;
}

int		cmesh::close()
{
	return closeMesh();
}

int		cmesh::isConvex()
{
	return 0;
}

int		cmesh::simplify(double grade)
{
	// check all 2 neighbourtriangles
	// if there are some in a plane, simplify

	return 0;
}

int		cmesh::subdivide()
{
	// take  every triangle and divide it into 2

	return 0;
}

int		cmesh::subdivideClever(int base,double grade)
{
	// take every tiangle and divide it into base * base
	// triangles
	// take advantage of the normalvectors
	
	return 0;
}

int		cmesh::addMesh(cmesh&)
{
	return -1;
}

int		cmesh::addMesh(cmesh*)
{
	return -1;
}


int		cmesh::subtractMesh(cmesh&)
{
	return -1;
}

int		cmesh::subtractMesh(cmesh*)
{
	return -1;
}



int		cmesh::intersectMesh(cmesh&)
{
	return -1;
}


int		cmesh::intersectMesh(cmesh*)
{
	return -1;
}



int		cmesh::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int		cmesh::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

void		cmesh::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("cmesh: %s\n",name);
}

int		cmesh::exportPOVMesh(FILE *fp,int tab,int tabsize,int anim)
{
	int		i;

	printTab(fp,tab); 
	fprintf(fp,"mesh\n"); 
	printTab(fp,tab); 
	fprintf(fp,"{\n"); 
	for(i = 0;i < trianglelist.length();i++)
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"triangle\n");
		printTab(fp,tab + tabsize);
		fprintf(fp,"{\n");
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>,\n",
			vectorlist[trianglelist[i].pointnum[0]].position[0],
			vectorlist[trianglelist[i].pointnum[0]].position[1],
			vectorlist[trianglelist[i].pointnum[0]].position[2]);
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>,\n",
			vectorlist[trianglelist[i].pointnum[1]].position[0],
			vectorlist[trianglelist[i].pointnum[1]].position[1],
			vectorlist[trianglelist[i].pointnum[1]].position[2]);
		printTab(fp,tab + tabsize + tabsize); 
		fprintf(fp,"<%g, %g, %g>\n",
			vectorlist[trianglelist[i].pointnum[2]].position[0],
			vectorlist[trianglelist[i].pointnum[2]].position[1],
			vectorlist[trianglelist[i].pointnum[2]].position[2]);

		printTab(fp,tab + tabsize);
		fprintf(fp,"} \n");
	}
	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	dim::exportPOV(fp,tab + tabsize,tabsize,anim);

	printTab(fp,tab);
	fprintf(fp,"}\n\n");

	return 0;
}

int		cmesh::exportPOVCSG(FILE *fp,int tab,int tabsize,int anim)
{
	return -1;
}

int		cmesh::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	if(isFlag(NO_EXPORT)) return 0;

	if(trianglelist.length() > 0)
	{
		printTab(fp,tab);
		fprintf(fp,"// Objectname = %s\n",name);
		printTab(fp,tab);
		fprintf(fp,"// Objecttype = cmesh\n");

		if(!isClose())
		{
			close();
		}

		if(!isClose() || !isConvex())
		{
			exportPOVMesh(fp,tab,tabsize,anim);
		}
		else
		{
			exportPOVCSG(fp,tab,tabsize,anim);
		}

	}

	return 0;
}

int		cmesh::save(media *m)
{
	int	i;

	if(!m)
		return -1;

	setMedia(m);

	writeChunk("CMSH");
	writeNameChunk(name);

	saveFlags(m);
	anim::save(m);
	dim::save(m);

	writeInt(vectorlist.length());
	for(i = 0;i < vectorlist.length();i++)
	{
		writeVector(vectorlist[i].position);
		writeVector(vectorlist[i].direction);
	}
	writeInt(trianglelist.length());
	for(i = 0;i < trianglelist.length();i++)
	{
		writeInt(trianglelist[i].pointnum[0]);
		writeInt(trianglelist[i].pointnum[1]);
		writeInt(trianglelist[i].pointnum[2]);
	}

	saveTexture(m);

	writeChunkLen();

	return 0;
}

int		cmesh::load(media *m,int l)
{
	int		i,n,s0,s1,s2;
	int		pos;
	Vector3		v,vn;

	setMedia(m);

	pos = m->tell();

	loadFlags(m,l);

	anim::load(m,l - (m->tell() - pos));
	dim::load(m,l - (m->tell() - pos));

	n = readInt();
	for(i = 0;i < n;i++)
	{
		v = readVector();
		vn = readVector();

		addKnode(v,vn);
	}

	n = readInt();
	for(i = 0;i < n;i++)
	{
		s0 = readInt(); 
		s1 = readInt(); 
		s2 = readInt(); 
		addSegment(s0,s1,s2); 
	}

	if(l - (m->tell() - pos) > 0)
	{
		loadTexture(m,l - (m->tell() - pos));
	}

	return 0;
}


int		cmesh::draw(view *v,Matrix44 m,int anim)
{
	int		i;
	seg		s;
	Vector3		v0,v1,v2;

	if(isFlag(HIDE)) return 0;

	dimMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
	{
		for(i = 0;i < trianglelist.length();i++)
		{
			s = trianglelist[i];
			v0 = vectorlist[s.pointnum[0]].position;
			v1 = vectorlist[s.pointnum[1]].position;
			v2 = vectorlist[s.pointnum[2]].position;
			v->drawLine(v0,v1,m,anim);
			v->drawLine(v1,v2,m,anim);
			v->drawLine(v2,v0,m,anim);
		}
	}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);


	return 0;
}

int		cmesh::calculate(int)
{
	int		i;
	double		x,y,z;

	vmin = vmax = vectorlist[0].position;
	for(i = 1;i < vectorlist.length();i++)
	{
		x = vectorlist[i].position[0];
		y = vectorlist[i].position[1];
		z = vectorlist[i].position[2];
		vmin = Vector3(MIN(x,vmin[0]),MIN(y,vmin[1]),MIN(z,vmin[2]));
		vmax = Vector3(MAX(x,vmax[0]),MAX(y,vmax[1]),MAX(z,vmax[2]));
	}

	return 0;
}

int		cmesh::addChild(box *b)
{
	vectorlist.empty();
	trianglelist.empty();

	addKnode(-0.5,-0.5,-0.5);
	addKnode( 0.5,-0.5,-0.5);
	addKnode(-0.5, 0.5,-0.5);
	addKnode( 0.5, 0.5,-0.5);
	addKnode(-0.5,-0.5, 0.5);
	addKnode( 0.5,-0.5, 0.5);
	addKnode(-0.5, 0.5, 0.5);
	addKnode( 0.5, 0.5, 0.5);

	addSegment(0,1,3);
	addSegment(0,3,2);
	addSegment(0,4,6);
	addSegment(0,6,2);
	addSegment(0,1,5);
	addSegment(0,4,5);
	addSegment(7,6,4);
	addSegment(7,4,5);
	addSegment(7,3,1);
	addSegment(7,1,5);
	addSegment(7,6,2);
	addSegment(7,2,3);

	return 0;
}

int		cmesh::addChild(cone *c)
{
	int		i,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a;

	vectorlist.empty();
	trianglelist.empty();


	addKnode(0,0,0.5);
	addKnode(0,0,-0.5);
	for(i = 0;i < s;i++)
	{
		a = (double)i / (double)s * 2 * PI;
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			0.5);
		addKnode(v);
		v = Vector3(c->capRadius() * sin(a),
			c->capRadius() * cos(a),
			-0.5);
		addKnode(v);
	}

	for(i = 0;i < s;i++)
	{
		p1 = i * 2 + 2;
		p2 = i * 2 + 1 + 2;
		p3 = (i * 2 + 2) % (s * 2) + 2;
		p4 = (i * 2 + 3) % (s * 2) + 2;
		addSegment(0,p1,p3);
		addSegment(p1,p2,p3);
		addSegment(p1,p3,p4);
		addSegment(1,p2,p4);
	}

	return 0;
}

int		cmesh::addChild(cylinder *c)
{
	int		i,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a;

	vectorlist.empty();
	trianglelist.empty();


	addKnode(0,0,0.5);
	addKnode(0,0,-0.5);
	for(i = 0;i < s;i++)
	{
		a = (double)i / (double)s * 2 * PI;
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			0.5);
		addKnode(v);
		v = Vector3(0.5 * sin(a),
			0.5 * cos(a),
			-0.5);
		addKnode(v);
	}

	for(i = 0;i < s;i++)
	{
		p1 = i * 2 + 2;
		p2 = i * 2 + 1 + 2;
		p3 = (i * 2 + 2) % (s * 2) + 2;
		p4 = (i * 2 + 3) % (s * 2) + 2;
		addSegment(0,p1,p3);
		addSegment(p1,p2,p3);
		addSegment(p1,p3,p4);
		addSegment(1,p2,p4);
	}

	return 0;
}

int		cmesh::addChild(sphere *c)
{
	int		i,t,s = 32,p4,p1,p2,p3;
	Vector3		v;
	double		a,b;

	vectorlist.empty();
	trianglelist.empty();


	addKnode(0,0.5,0);
	addKnode(0,-0.5,0);
	for(t = 1;t < (s / 2);t++)
	{
		b = (double)t / (double)s * 2 * PI;
		for(i = 0;i < s;i++)
		{
			a = (double)i / (double)s * 2 * PI;
			v = Vector3(0.5 * sin(a) * sin(b),
				0.5 * cos(b),
				0.5 * cos(a) * sin(b));
			addKnode(v);
		}
	}

	for(i = 0;i < s;i++)
	{
		p1 = i + 2;
		p2 = (i + 1) % s + 2;
		addSegment(0,p1,p2);
	}
	for(t = 0;t < (s / 2) - 1;t++)
	{
		for(i = 0;i < s;i++)
		{
			p1 = t * s + i + 2;
			p2 = (t + 1) * s + i + 2;
			p3 = t * s + (i + 1) % s + 2;
			p4 = (t + 1) * s + (i + 1) % s + 2;
			addSegment(p1,p2,p4);
			addSegment(p1,p3,p4);
		}
	}
	for(i = 0;i < s;i++)
	{
		p1 = i + s * (s / 2 - 2) + 2;
		p2 = (i + 1) % s + s * (s / 2 - 2) + 2;
		addSegment(1,p1,p2);
	}

	return 0;
}

int		cmesh::addChild(text *c)
{
	int			i;

	vectorlist.empty();
	trianglelist.empty();

	if((i = c->getMesh(this)) != 0)
		return i;

/*
	printf("Got %i knodes and %i edges\n",vectorlist.length(),trianglelist.length() * 3);

	for(i = 0;i < vectorlist.length();i++)
		printf("%i. <%g,%g,%g>\n",i,vectorlist[i].position[0],vectorlist[i].position[1],vectorlist[i].position[2]);
	for(i = 0;i < trianglelist.length();i++)
		printf("%i. <%i,%i,%i>\n",i,trianglelist[i].pointnum[0],trianglelist[i].pointnum[1],trianglelist[i].pointnum[2]);
*/

	close();

	return 0;
}

int		cmesh::addChild(csgobj *co)
{
	if(gptr)
		delete gptr;
	gptr = 0;

	switch(co->type())
	{
		case NUM_BOX:
			addChild((box*)co);
		break;
		case NUM_CONE:
			addChild((cone*)co);
		break;
		case NUM_CYLINDER:
			addChild((cylinder*)co);
		break;
		case NUM_SPHERE:
			addChild((sphere*)co);
		break;
		case NUM_TEXT:
			addChild((text*)co);
		break;
		default:
			return -1;
	}

	return -1;
}

int		cmesh::addChild(blobobj *co)
{
	return addChild((csgobj*)co);
}

