/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** sphere.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sphere.h"
#include "blob.h"
#include "num.h"
#include "view.h"


sphere::sphere(base *p,char *n,double st,texture *t) :
	blobobj(p,n,st,t)
{ 
	numtype = NUM_SPHERE;
	strength = st;

	if(parent) addToParent(parent);
}

sphere::sphere(base *p,char *n,double st,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	blobobj(p,n,st,vs,vr,vt)
{
	numtype = NUM_SPHERE;
	strength = st;

	if(parent) addToParent(parent);
}

sphere::sphere(base *p,char *n,double st,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	blobobj(p,n,st,t,vs,vr,vt)
{ 
	numtype = NUM_SPHERE;
	strength = st;

	if(parent) addToParent(parent);
}

sphere::sphere(base *p,sphere *sc) :
	blobobj(p,(blobobj*)sc)
{ 
	numtype = NUM_SPHERE;
	strength = sc->strength;

	if(parent) addToParent(parent);
}

sphere::~sphere()
{
	removeFromParent();
}

base	*sphere::copy(base *p)
{
	return new sphere(p,this);
}

int	sphere::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int	sphere::removeFromParent() 
{ 
	if(!parent) return -2; 
 
	return parent->removeChild(this); 
} 
 
int	sphere::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = sphere\n");

	printTab(fp,tab);
	fprintf(fp,"sphere\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"<0,0,0>,0.5");

	if (!isBlobObject())
		fprintf(fp,"\n"); 
	else
		fprintf(fp,",%f\n",strength);
	
	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	dim::exportPOV(fp,tab + tabsize,tabsize,anim);

	if(isFlag(HOLLOW_ON))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow on\n");
	}
	else if(isFlag(HOLLOW_OFF))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow off\n");
	}
	
	printTab(fp,tab);
	fprintf(fp,"}\n\n");

	return 0;
}

void	sphere::dumpNames(int tab,int)
{
	printTab(stderr,tab);
	printf("sphere: %s\n",name);
}

int	sphere::save(media *m)
{
	if(!m) return -1;

	setMedia(m);
	writeChunk("SPHR");
	writeNameChunk(name);

	saveFlags(m);

	writeDouble(strength);
	anim::save(m);
	dim::save(m);
	saveTexture(m);
	writeChunkLen();

	return 0;
}

int	sphere::load(media *m,int l)
{
	int	pos = m->tell();

	loadFlags(m,l);

	strength = readDouble();

	anim::load(m,l - (m->tell() - pos));
	dim::load(m,l - (m->tell() - pos));

 	if(l > m->tell() - pos)
	{
		loadTexture(m,l - (m->tell() - pos));
	}

	return 0;
}

int	sphere::draw(view *v,Matrix44 m,int anim)
{
	Vector3	p,vx,vy,vz;
	double	rr,t;

	//printf("Sphere\n");

	if(isFlag(HIDE)) return 0;

	p = Vector3(0,0,0);
	vx = Vector3(0.5,0,0);
	vy = Vector3(0,0.5,0);
	vz = Vector3(0,0,0.5);

	dimMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
	{
		v->drawSphere(p,vx,vy,vz,m,anim);

		if(isBlobObject())
		{
			t = ((blob*)parent)->thresholdValue();
			rr = sqrt(1 - sqrt(t / strength));

			v->drawBlobSphere(p,vx * rr,vy * rr,vz * rr,m,anim);
		}
	}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	sphere::calculate(int)
{
	vmin = Vector3(-0.5,-0.5,-0.5);
	vmax = Vector3(0.5,0.5,0.5);

	return 0;
}

double	sphere::volume(int anim)
{
	Vector3	v;

	v = getVScale();

	return (v[0] * v[1] * v[2]) / 6 * PI;
}




