/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** text.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "text.h"
#include "num.h"
#include "view.h"

#include "cmesh.h"


text::text(base *p,const char *n,const char *txt,const char *fnt,double th,texture *t) :
	csgobj(p,n,t)
{
	numtype = NUM_TEXT;
	fontname = strdup(fnt);
	context = strdup(txt);
	thickness = th;
	offset = Vector2(0,0);

	fontptr = new font(fnt,fnt);
	fontcache = 0;

	if(parent) addToParent(parent);
}

text::text(base *p,const char *n,const char *txt,const char *fnt,double th,Vector2 &offs,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,t,vs,vr,vt)
{
	numtype = NUM_TEXT;
	fontname = strdup(fnt);
	context = strdup(txt);
	thickness = th;
	offset = offs;

	fontptr = new font(fnt,fnt);
	fontcache = 0;

	if(parent) addToParent(parent);
}

text::text(base *p,const char *n,const char *txt,const char *fnt,double th,Vector2 &offs,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,vs,vr,vt)
{
	numtype = NUM_TEXT;
	fontname = strdup(fnt);
	context = strdup(txt);
	thickness = th;
	offset = offs;

	fontptr = new font(fnt,fnt);
	fontcache = 0;

	if(parent) addToParent(parent);
}

text::text(base *p,text *tc) :
	csgobj(p,(csgobj*)tc)
{
	numtype = NUM_TEXT;
	fontname = strdup(tc->fontname);
	context = strdup(tc->context);
	thickness = tc->thickness;
	offset = tc->offset;

	fontptr = new font(*tc->fontptr); // kopie erstellen
	fontcache = 0;

	if(parent) addToParent(parent);
}

text::~text()
{
	if(fontname)
		free(fontname);
	if(context)
		free(context);

	if(fontcache) free(fontcache);
	delete fontptr;
}

base	*text::copy(base *p)
{
	return new text(p,this);
}

int	text::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int	text::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

void	text::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("text: %s\n",name);
}

int	text::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{ 
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab); 
	fprintf(fp,"// Objectname = %s\n",name); 
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = text\n");
	printTab(fp,tab);
	fprintf(fp,"text\n");
	printTab(fp,tab); 
	fprintf(fp,"{\n"); 
	printTab(fp,tab + tabsize); 
	fprintf(fp,"ttf \"%s\", \n",fontname); 
	printTab(fp,tab + tabsize);
	fprintf(fp,"\"%s\",\n", context);
	printTab(fp,tab + tabsize);
	fprintf(fp,"%g <%g,%g>\n",thickness, offset(0),offset(1));
	
	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	dim::exportPOV(fp,tab + tabsize,tabsize,anim);

	if(isFlag(HOLLOW_ON))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow on\n");
	}
	else if(isFlag(HOLLOW_OFF))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow off\n");
	}
	
	printTab(fp,tab);
	fprintf(fp,"}\n");

	return 0;
}

int	text::save(media *m)
{
	if(!m) return -1;

	setMedia(m);
	writeChunk("TEXT");
	writeNameChunk(name);

	saveFlags(m);

	writeName(fontname);
	writeName(context);
	writeDouble(thickness);
	writeVector(offset);

	anim::save(m);
	dim::save(m);

	saveTexture(m);

	writeChunkLen();

	return 0;
}

int	text::load(media *m,int l)
{
	int	pos = m->tell();

	loadFlags(m,l);

	fontname = readName();
	context = readName();
	thickness = readDouble();
	offset = readVector2();

	anim::load(m,l - (m->tell() - pos));
	dim::load(m,l - (m->tell() - pos));

 	if(l > (m->tell() - pos))
	{
		loadTexture(m,l - (m->tell() - pos));
	}

	if(fontptr)
		fontptr->setFileName(fontname);

	return 0;
}

int	text::draw(view *v,Matrix44 m,int anim)
{
	char		*cp;
	fontglyph	*gp = 0;
	int		cc,sp,ep,pn,poc;
	Vector2		p[2048];
	double		x,y;

//	printf("Text: %s\n",context);

	if(isFlag(HIDE)) return 0;

	dimMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
	{
		x = 0;
		y = 0;

		for(cp = context;*cp != '\0';cp++)
		{
			gp = fontptr->getGlyph(*cp);
			if(gp)
			{
				if(*cp != ' ')
				{
					for(cc = 0;cc < gp->nContours();cc++)
					{
						sp = gp->getContourStartPoint(cc);
						ep = gp->getContourEndPoint(cc);
						//printf("(%i,%i)(%g,%g)\n",sp,ep,x,y);
						poc = 0;
						p[poc] = gp->getPoint(sp);
						poc++;
						for(pn = sp + 1;pn <= ep;pn++)
						{
							if(gp->isOnPoint(pn))
							{
								p[poc] = gp->getPoint(pn);
								poc++;
								drawSpline(v,m,p,poc,x,y,anim);
								poc = 0;
								p[poc] = gp->getPoint(pn);
								poc++;
							}
							else
							{
								p[poc] = gp->getPoint(pn);
								poc++;
							}
						}
						p[poc] = gp->getPoint(sp);
						poc++;
						drawSpline(v,m,p,poc,x,y,anim);
					}
				}
				x += gp->space();
			}
			x += offset(0);
			y += offset(1);
		}
	}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	text::drawSpline(view *v,Matrix44 m,Vector2 *p,int pn,double x,double y,int anim)
{
	Vector3		pfa3,pfb3,pba3,pbb3;
	int		n;

	n = pn - 1;

/*
	int		ti,i,n;
	double		t,tc;
	Vector2		b;

	if(pn > 1)
	{
		pfb3 = Vector3(p[0](0) + x,p[0](1) + y,0);
		pbb3 = Vector3(p[0](0) + x,p[0](1) + y,thickness);
		for(ti = 1;ti <= pn * 3;ti++)
		{
			t = (double)ti / (double)(pn * 3);
			if(t <= 0.5)
			{
				tc = t / (1 - t);
				b = p[n];
				for(i = 1;i <= pn;i++)
				{
					b = b * ((double)i / (double)(pn - i + 1)) * tc + p[pn - i];
				}
				b = b * pow(1 - t,pn);
			}
			else
			{
				tc = (1 - t) / t;
				b = p[0];
				for(i = 1;i <= pn;i++)
				{
					b = b * ((double)i / (double)(pn - i + 1)) * tc + p[i - 1];
				}
				b = b * pow(t,pn);
			}
			pfa3 = pfb3;
			pba3 = pbb3;
			pfb3 = Vector3(b(0) + x,b(1) + y,0);
			pbb3 = Vector3(b(0) + x,b(1) + y,-thickness);
			v->drawLine(pfa3,pfb3,m);
			v->drawLine(pba3,pbb3,m);
			v->drawLine(pfa3,pba3,m);
		}
	}
*/

	pfa3 = Vector3(p[0](0) + x,p[0](1) + y,0);
	pfb3 = Vector3(p[n](0) + x,p[n](1) + y,0);

	pba3 = Vector3(p[0](0) + x,p[0](1) + y,thickness);
	pbb3 = Vector3(p[n](0) + x,p[n](1) + y,thickness);

	v->drawLine(pfa3,pfb3,m,anim);
	v->drawLine(pba3,pbb3,m,anim);
	v->drawLine(pfa3,pba3,m,anim);


	return 0;
}

int	text::calculate(int)
{
	char		*cp;
	fontglyph	*gp = 0;
	double		x,y,h,m;

	x = 0;
	y = 0;
	h = 0;
	m = 0;

	for(cp = context;*cp != '\0';cp++)
	{
		gp = fontptr->getGlyph(*cp);
		if(gp)
		{
			x += gp->width();
			h = MAX(gp->ascend() + y,h);
			m = MIN(gp->descend() + y,m);
		}
		x += offset[0];
		y += offset[1];
	}

	vmin = Vector3(0,m,0);
	vmax = Vector3(x,h,thickness);

	return 0;
}

char	*text::getText()
{
	return context;
}

void	text::setText(const char *txt)
{
	if(txt)
	{
		if(context) free(context);
		context = strdup(txt);
	}
}

char	*text::getFontName()
{
	return fontname;
}

void	text::setFontName(const char *fn)
{
	if(fn)
	{
		if(fontname) free(fontname);
		fontname = strdup(fn);
	}
	if(fontptr) fontptr->setFileName(fontname);
}

double	text::getThickness()
{
	return thickness;
}

void	text::setThickness(double t)
{
	thickness = t;
}

Vector2	text::getOffset()
{
	return offset;
}

void	text::setOffset(Vector2 &o)
{
	offset = o;
}

void	text::setFontPath(const char*)
{
}

double	text::volume(int anim)
{
	Vector3		v;

	v = getVScale();

	// aproximate

	return 1;
}

int	text::getMesh(cmesh *mo)
{
	char			*cp;
	fontglyph		*gp = 0;
	int			cc,sp,ep,pn;
	simplelist<Vector2>	pl;
	double			x,y;

	x = 0;
	y = 0;

	for(cp = context;*cp != '\0';cp++)
	{
		gp = fontptr->getGlyph(*cp);
		if(gp)
		{
			if(*cp != ' ')
			{
				for(cc = 0;cc < gp->nContours();cc++)
				{
					sp = gp->getContourStartPoint(cc);
					ep = gp->getContourEndPoint(cc);
					//printf("(%i,%i)(%g,%g)\n",sp,ep,x,y);
					pl += gp->getPoint(sp);
					for(pn = sp + 1;pn <= ep;pn++)
					{
						if(gp->isOnPoint(pn))
						{
							pl += gp->getPoint(pn);
							addMeshSpline(mo,pl,x,y);
							pl.empty();
							pl += gp->getPoint(pn);
						}
						else
						{
							pl += gp->getPoint(pn);
						}
					}
					pl += gp->getPoint(sp);
					addMeshSpline(mo,pl,x,y);
				}
			}
			x += gp->width();
		}
		x += offset[0];
		y += offset[1];
	}

	return 0;
}

int	text::addMeshSpline(cmesh *mo,simplelist<Vector2> pl,double x,double y)
{
	int		p0,p1,p2,p3;

	p0 = mo->addKnode(pl[0][0] + x,pl[0][1] + y,0);
	p1 = mo->addKnode(pl.getLast()[0] + x,pl.getLast()[1] + y,0);

	p2 = mo->addKnode(pl[0][0] + x,pl[0][1] + y,thickness);
	p3 = mo->addKnode(pl.getLast()[0] + x,pl.getLast()[1] + y,thickness);

	mo->addSegment(p0,p1,p2);
	mo->addSegment(p2,p3,p1);

	return 0;
}






