/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** animwidget.cpp
*/

#include "animwidget.h"
#include "animwidget.moc"

#include <qslider.h>

AnimControlWidget::AnimControlWidget(QWidget *parent,char *name) :
	QWidget(parent,name)
{
	lvalue = 0;
	rvalue = 1000;
	stepsize = 40;
	now = 0;
	running = 0;

	timeslider = new QSlider(this);
	timeslider->setOrientation(QSlider::Horizontal);
	timeslider->setTracking(FALSE);
	timeslider->setRange(lvalue,rvalue);
	timeslider->setTickInterval(40);
	timeslider->setTickmarks(QSlider::Below);

	llabel = new QLabel(this);
	llabel->setText("0:00.00.000");
	llabel->setAlignment(AlignLeft);
	rlabel = new QLabel(this);
	rlabel->setText("0:00.01.000");
	rlabel->setAlignment(AlignLeft);

	timelabel = new QLabel(this);
	timelabel->setText("Interval:");

	starttime = new QLineEdit(this);
	starttime->setText("0");
	connect(starttime,SIGNAL(textChanged(const char*)),this,SLOT(changedLValue(const char*)));
	endtime = new QLineEdit(this);
	endtime->setText("1");
	connect(endtime,SIGNAL(textChanged(const char*)),this,SLOT(changedRValue(const char*)));

	stepvalue = new QLineEdit(this);
	stepvalue->setText("40");
	connect(stepvalue,SIGNAL(textChanged(const char*)),this,SLOT(changedStepsize(const char*)));
	steplabel = new QLabel(this);
	steplabel->setText("Stepsize (ms):");

	exportform = new QLineEdit(this);
	exportform->setText("data/pov/anim-%0.8i.pov");
	exportpov = new QCheckBox(this);
	exportlabel = new QLabel(this);
	exportlabel->setText("Exportform:");

	playbt = new QPushButton(this);
	playbt->setText("play");
	connect(playbt,SIGNAL(pressed()),this,SLOT(startPressed()));
	stopbt = new QPushButton(this);
	stopbt->setText("stop");
	connect(stopbt,SIGNAL(pressed()),this,SLOT(stopPressed()));

	layout();	
}

AnimControlWidget::~AnimControlWidget()
{
}

world	*AnimControlWidget::getWorld()
{
	return _world;
}

void	AnimControlWidget::paintEvent(QPaintEvent*)
{
	layout();
}

void	AnimControlWidget::layout()
{
	int	ww,wh;
	int	x,y,w,h;

	ww = width();
	wh = height();

	x = 5;
	y = 5;
	w = ww - 10 - rlabel->sizeHint().width();
	h = 25;
	timeslider->setGeometry(x,y,w,h);

	x = 5 + 3;
	y += h + 5;
	w = llabel->sizeHint().width();
	h = MAX(llabel->sizeHint().height(),rlabel->sizeHint().height());
	llabel->setGeometry(x,y,w,h);
	w = rlabel->sizeHint().width();
	x = ww - w - 5 - 3;
	rlabel->setGeometry(x,y,w,h);

	x = 5;
	y += h + 5;
	w = 80;
	h = 25;
	timelabel->setGeometry(x,y,w,h);
	x += w + 5;
	starttime->setGeometry(x,y,w,h);
	x += w + 5;
	endtime->setGeometry(x,y,w,h);

	x = 5;
	y += h + 5;
	w = 80;
	steplabel->setGeometry(x,y,w,h);
	x += w + 5;
	stepvalue->setGeometry(x,y,w,h);

	x += w + 5;
	w = 80;
	exportlabel->setGeometry(x,y,w,h);
	x += w + 5;
	w = 16;
	exportpov->setGeometry(x,y,w,h);
	x += w + 5;
	w = 160;
	exportform->setGeometry(x,y,w,h);

	x = 5;
	y += h + 5;
	w = 80;
	playbt->setGeometry(x,y,w,h);
	x += w + 5;
	stopbt->setGeometry(x,y,w,h);

	setMaximumSize(8192,y + h + 5);
	setMinimumSize(360,y + h + 5);
}

void	AnimControlWidget::hide()
{
	setMaximumSize(0,0);
	setMinimumSize(0,0);
	QWidget::hide();
}

void	AnimControlWidget::show()
{
	layout();
	QWidget::show();
}

void	AnimControlWidget::changedStepsize(int nss)
{
	int	ticks,ws;

	if(nss > 0 && nss < rvalue - lvalue)
		stepsize = nss;

	ws = timeslider->width() - 8;

	ticks = (rvalue - lvalue) / stepsize;
	while(ticks > ws / 4)
		ticks /= 10;

	timeslider->setTickInterval((rvalue - lvalue) / ticks);
}

void	AnimControlWidget::changedStepsize(const char *s)
{
	int	nss;
	QString	str;

	str = s;

	nss = str.toInt();

	changedStepsize(nss);
}

void	AnimControlWidget::changedInterval(int nlv,int nrv)
{
	QString		str;

	if(nlv < nrv)
	{
		lvalue = nlv;
		rvalue = nrv;
	}

	timeslider->setRange(lvalue,rvalue);

	str.sprintf("%i:%0.2i.%0.2i.%0.3i",
		lvalue / 3600000,
		(lvalue / 60000) % 60,
		(lvalue / 1000) % 60,
		lvalue % 1000);

	llabel->setText(str);

	str.sprintf("%i:%0.2i.%0.2i.%0.3i",
		rvalue / 3600000,
		(rvalue / 60000) % 60,
		(rvalue / 1000) % 60,
		rvalue % 1000);

	rlabel->setText(str);

	changedStepsize(stepsize);
}

void	AnimControlWidget::changedLValue(const char *s)
{
	int	nlv;
	QString	str;

	str = s;

	nlv = str.toInt() * 1000;

	changedInterval(nlv,rvalue);
}

void	AnimControlWidget::changedRValue(const char *s)
{
	int	nrv;
	QString	str;

	str = s;

	nrv = str.toInt() * 1000;

	changedInterval(lvalue,nrv);
}

void	AnimControlWidget::startPressed()
{
	int	i;
	char	str[2048];

	if(running) return;

	i = timeslider->value();

	if(i >= rvalue)
	{
		i = lvalue;

		timeslider->setValue(i);
	}

	c = new Comm();

	startTimer(stepsize);
	running = 1;

	sprintf(str,"rm %s/*.sh","tmp");
	system(str);

	_world->callscript("main","tmp",c,i,lvalue,rvalue,stepsize);

	if(exportpov->isChecked())
	{
		char	str[2048];

		sprintf(str,exportform->text(),i);
		printf("export \"%s\" (pov)\n",str);

		emit exportPOVanim(str);
	}

	emit drawanim();
}

void	AnimControlWidget::stopPressed()
{
	if(!running) return;

	killTimers();
	running = 0;

	emit redraw();

	delete c;
	c = 0;
}

void	AnimControlWidget::timerEvent(QTimerEvent*)
{
	int		i;
	
	i = timeslider->value();
	i += stepsize;

	if(i > rvalue)
	{
		stopPressed();

		return;
	}
	timeslider->setValue(i);

	_world->callscript("main","tmp",c,i,lvalue,rvalue,stepsize);

	if(exportpov->isChecked())
	{
		char	str[2048];

		sprintf(str,exportform->text(),i);
		printf("export \"%s\" (pov)\n",str);

		emit exportPOVanim(str);
	}

	emit drawanim();
}

void	AnimControlWidget::setWorld(world *w)
{
	if(w)
		_world = w;
}



