/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** imagemap.cpp
*/

#include "imagemap.h"

#include <stdio.h>
#include <stdlib.h>

#include "../classes/imageconverter.h"

imageMap::imageMap(textureBase *p)
	: textureBase(IMAGEMAP,p)
{
 	Filename = 0;
	Filetype = 0;
	FilterValue = 0;
	TransmitValue = 0;

	realFileName = 0;
	realFileType = 0;

	setSubItems(false);
	setName("Imagemap");
	setInfo("No file");

	picData=NULL;     	
	picLen=0;

	addAddableItem(IMAGEMAP,false);
	addAddableItem(BITMAPMODIFIER,false);
}

void imageMap::updateInfo()
{
char *buffer=(char *)malloc(512);
	sprintf(buffer,"Filename:%s",getImageMapFilename());
	setInfo(buffer);
	free(buffer);
}

imageMap::~imageMap()
{
char buffer[1024];
	if (realFileName) {
		sprintf(buffer,"rm %s",realFileName);
		system(buffer);
	}
	if (Filename!=NULL) {free(Filename);}
	if (Filetype!=NULL) {free(Filetype);}
}

bitmapmodifier *imageMap::getModifier()
{
	return &modifier;
}

const char *imageMap::getImageMapFilename()
{
	return Filename;
}

const char *imageMap::getImageMapFiletype()
{
	return Filetype;
}

bool imageMap::isEqual(textureBase *t)
{
imageMap *i;

	if (!t) return false;
	i=(imageMap*)t;

#ifdef DEBUG
	printf("Comparing Imagemap...\n");
#endif
	if (textureBase::isEqual(t)==false) return false;
	if (t->getType()!=IMAGEMAP) return false;
	if ((i->getImageMapFilename()==NULL)&&(getImageMapFilename()==NULL)) return true;
	if ((i->getImageMapFiletype()==NULL)&&(getImageMapFiletype()==NULL)) return true;
	if ((i->getImageMapFilename()==NULL)||(getImageMapFilename()==NULL)) return false;
	if ((i->getImageMapFiletype()==NULL)||(getImageMapFiletype()==NULL)) return false;
	if (
		(!(strcmp(i->getImageMapFilename(),getImageMapFilename()) )) &&
		(!(strcmp(i->getImageMapFiletype(),getImageMapFiletype()) )) &&
		(i->getImageMapFilterValue()==getImageMapFilterValue() ) &&
		(i->getImageMapTransmitValue()==getImageMapTransmitValue() ) &&
		(modifier.isEqual(i->getModifier() )==true)
	) {
#ifdef DEBUG
	printf("Comparing Imagemap...equal !\n");
#endif
		return true;
	}
	return false;

}

void imageMap::add(textureBase *t)
{
	if (!t) return;
	if (t->getType()==BITMAPMODIFIER) {
		modifier.add( (bitmapmodifier*)t );
		return;
	}
	if (t->getType()==IMAGEMAP) {
		imageMap *i=(imageMap *)t;
		setImageMapFilename(i->getImageMapFilename() );
		setImageMapFiletype(i->getImageMapFiletype() );
		setImageMapFilterValue(i->getImageMapFilterValue() );
		setImageMapTransmitValue(i->getImageMapTransmitValue() );
		modifier.add( i->getModifier() );
	}
	updateInfo();
	return;
}


void imageMap::setImageMapFilename(const char *n)
{
	if (!n) return;
	if (Filename) {free(Filename);}
	Filename = strdup(n);
	updateInfo();
}

void imageMap::setImageMapFiletype(const char *n)
{
	if (!n) return;
	if (Filetype!=NULL) { free(Filetype); }
	Filetype = strdup(n);
	updateInfo();
}


double imageMap::getImageMapFilterValue()
{
	return FilterValue;
}

void imageMap::setImageMapFilterValue(double n)
{
	FilterValue=n;
	updateInfo();
}


double imageMap::getImageMapTransmitValue()
{
	return TransmitValue;
}

void imageMap::setImageMapTransmitValue(double n)
{
	TransmitValue=n;
	updateInfo();
}

int imageMap::save(media *m)
{
chunk ch;
	if (!m) return -1;
	#ifdef DEBUG
		printf("saving ImageMap\n");
	#endif
	setMedia(m);

	writeChunk("IMGM");
	
	textureBase::save(m);
	
	ch.setMedia(m);
	ch.writeChunk("DATA");

		writeName(Filename);
		writeName(Filetype);	

	       	m->write(&FilterValue,sizeof(double) );
       		m->write(&TransmitValue,sizeof(double) );
	ch.writeChunkLen();

	if (picData) {
		ch.writeChunk("PDAT");
	       	m->write(picData,picLen );	
		ch.writeChunkLen();
		ch.writeChunkLen();
	}

	modifier.save(m);//  Schreiben Chunk: BITM
	
	writeChunkLen();

	#ifdef DEBUG
		printf("ImageMap saved\n");
	#endif
	return 0;
}

int imageMap::load(media *m, int l)
{
	int pos=m->tell();
	int len;
	char	chunk[4];
	bool	read;

	if (!m) return -1;
	#ifdef DEBUG
		printf("loading ImageMap\n");
	#endif

	setMedia(m);

	if (picData) free(picData);
	picData=NULL;

	do {
		m->read(chunk,4);
		m->read(&len,4);
		read=false;

		if(strncmp(chunk,"BITM",4) == 0) {
			modifier.load(m,len);			
			read=true;
		}
		if(strncmp(chunk,"BASE",4) == 0) {
			textureBase::load(m,len);
			read=true;
		}
		if(strncmp(chunk,"DATA",4) == 0) {
			setImageMapFilename(readName());
			Filetype=readName();
		       	m->read(&FilterValue,sizeof(double) );
		       	m->read(&TransmitValue,sizeof(double) );	
			read=true;
		}
		if(strncmp(chunk,"PDAT",4) == 0) {
			picData=(char*)malloc(len);
		       	m->read(&picData,len );
			read=true;
		}
		
		if (read==false) { m->seek(len,SEEK_CUR); }
	} while (m->tell()< (pos+l) );

	m->seek(pos+l,SEEK_SET); // Man weiss ja nie...

	#ifdef DEBUG
		printf("Imagemap loaded\n");
	#endif
	updateInfo();
	return 0;
}

int imageMap::exportPOV(FILE *fp, int tab, int tabsize, int a, bool dummy)
{
	char buffer[1024];
	
	if(realFileName)
	{
		sprintf(buffer,"rm %s",realFileName);
		system(buffer);
	}

	#ifdef DEBUG
		printf("exporting IMageMap (POV)\n");
	#endif
	
	if(picData)
	{
	}

	if(Filename != 0)
	{
		printTab(fp,tab);
		fprintf(fp,"image_map //%s\n",getName());
		printTab(fp,tab);
		fprintf(fp,"{\n");

		printTab(fp,tab + tabsize);
	
	        if(getImageConverter())
	        {
			if(strncmp(getImageConverter()->getType(Filename),"???",3) == 0)
			{
				fprintf(fp,"%s \"%s\"\n",getImageConverter()->getExtension(),Filename);
			}
			else if((strncmp(getImageConverter()->getType(Filename),"PNG",3) != 0)
				&&(strncmp(getImageConverter()->getType(Filename),"JPEG",4) != 0))
			{
				realFileName = getImageConverter()->convert(Filename,0,0);
				realFileType = getImageConverter()->getExtension();
				fprintf(fp,"%s \"%s\"\n",realFileType,realFileName);
			}
			else
			{
				fprintf(fp,"%s \"%s\"\n",Filetype,Filename);
			}
		}
		else
		{
			fprintf(fp,"%s \"%s\"\n",Filetype,Filename);
		}


		if (FilterValue!=0.0)
		{
			printTab(fp,tab+tabsize);
			fprintf(fp,"filter all %f\n",FilterValue);
		}
		if (TransmitValue!=0.0)
		{
			printTab(fp,tab+tabsize);
			fprintf(fp,"transmit all %f\n",TransmitValue);
		}
		modifier.exportPOV(fp,tab,tabsize,a);

		textureBase::exportPOV(fp,tab,tabsize,a,dummy);

		printTab(fp,tab);
		fprintf(fp,"}\n");
	}
	else
	{
		return 1;
	}
	#ifdef DEBUG
		printf("ImageMap exported (POV)\n");
	#endif
	
	//if (fn) free(fn);
	//if (ft) free(ft);

	return 0;
}


