/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** map2mesh.cpp
*/
#include "map2mesh.h"
#include "map2mesh.moc"

#include "../misc/preferences.h"

#include <stdio.h>
#include <kapp.h>
#include <qimage.h>
#include <qcolor.h>

#include <evaluate.h>


Map2MeshDialog::Map2MeshDialog(base *bp,QWidget *pw) :
	QDialog(pw,"Map2Mesh",TRUE)
{
	parent = bp;
	
	w = 0;
	h = 0;
	p = 0;
	meshobj = 0;
	
	projection = new QButtonGroup(this);
	projection->setTitle(i18n("Projection"));
	projection->setExclusive(TRUE);
	
	spherical = new QRadioButton(projection);
	spherical->setText(i18n("Spherical"));
	projection->insert(spherical,0);
	
	cylindrical = new QRadioButton(projection);
	cylindrical->setText(i18n("Cylindrical"));
	projection->insert(cylindrical,1);
	
	torical = new QRadioButton(projection);
	torical->setText(i18n("Torish"));
	projection->insert(torical,2);
	torical->setEnabled(FALSE);
	
	boxed = new QRadioButton(projection);
	boxed->setText(i18n("Boxish"));
	projection->insert(boxed,3);
	boxed->setEnabled(FALSE);
	
	planar = new QRadioButton(projection);
	planar->setText(i18n("Planar"));
	projection->insert(planar,4);
	
	projection->setButton(0);
	
	connect(projection,SIGNAL(clicked(int)),this,SLOT(setProjection(int)));
	
	lfn = new QLabel(this);
	lfn->setText(i18n("Function:"));
	
	efn = new QLineEdit(this);
	efn->setText("0");
	
	gfw = new GetFileWidget(this,i18n("Heightmap:"));
	
	lsize = new QLabel(this);
	lsize->setText(i18n("Size of grid:"));
	
	ewidth = new QLineEdit(this);
	ewidth->setText("32");
	eheight = new QLineEdit(this);
	eheight->setText("32");
	
	lmul = new QLabel(this);
	lmul->setText(i18n("Multiplier:"));
	
	emul = new QLineEdit(this);
	emul->setText("1.0");
	
	ok = new QPushButton(this);
	ok->setText(i18n("OK"));
	connect(ok,SIGNAL(clicked()),SLOT(accept()));
	
	cancel = new QPushButton(this);
	cancel->setText(i18n("Cancel"));
 	connect(cancel,SIGNAL(clicked()),SLOT(reject()));
	
 	resize(400,340);
 	
	layout();
}

Map2MeshDialog::~Map2MeshDialog()
{
}

void	Map2MeshDialog::layout()
{
	int	x,y,w,h,x2,y2,w2,h2,w3;
	
	setCaption(i18n("Map2Mesh"));
	
	x = 5;
	y = 5;
	w = width() - 10;
	h = MAX(25 + 5 * 25 + 4 * 1,projection->sizeHint().height());

	projection->setGeometry(x,y,w,h);
	
	x2 = 15;
	y2 = 20;
	w2 = w - 30;
	h2 = 25;
	
	spherical->setGeometry(x2,y2,w2,h2);
	y2 += 1 + h2;
	cylindrical->setGeometry(x2,y2,w2,h2);
	y2 += 1 + h2;
	torical->setGeometry(x2,y2,w2,h2);
	y2 += 1 + h2;
	boxed->setGeometry(x2,y2,w2,h2);
	y2 += 1 + h2;
	planar->setGeometry(x2,y2,w2,h2);
	
	w2 = MAX(lfn->sizeHint().width(),lsize->sizeHint().width());
	w2 = MAX(w2,gfw->getLabelWidthHint());
	w2 = MAX(w2,lmul->sizeHint().width());
	
	x = 5;
	y += h + 5;
	w = w2;
	h = 25;
	
	lfn->setGeometry(x,y,w,h);
	
	x += w + 5;
	w = width() - 15 - w2;
	
	efn->setGeometry(x,y,w,h);

        x = 5;
        y += h + 5;
        w = width() - 10;

        gfw->setLabelWidth(w2);
        gfw->setGeometry(x,y,w,h);

	x = 5;
        y += h + 5;
        w = w2;

        lsize->setGeometry(x,y,w,h);

        x += w + 5;
        w3 = width() - 20 - w2;
        w = MAX(50,MIN(80,w3 / 2));
        w3 = w;

        ewidth->setGeometry(x,y,w,h);

        x += w + 5;

        eheight->setGeometry(x,y,w,h);

 	x = 5;
        y += h + 5;
        w = w2;

        lmul->setGeometry(x,y,w,h);

        x += w + 5;
        w = w3;

        emul->setGeometry(x,y,w,h);

	w2 = MAX(ok->sizeHint().width(),cancel->sizeHint().width());

        x = width() - 10 - 2 * w2;
        y = height() - 30;
        w = w2;
        h = 25;

        ok->setGeometry(x,y,w,h);

        x += w + 5;
        cancel->setGeometry(x,y,w,h);
}

smesh	*Map2MeshDialog::getObject()
{
	smesh	*mp;
	
	if(meshobj)
	{
		mp = meshobj;
	}
	else
	{
		mp = new smesh(parent,"SMesh");
	}

	if(mp)
	{
		// save extradata to the mesh
		QString		qstr;
		const char	*fn,*filename;
		QImage		qi;
		Preferences	prefs;
			
		fn = efn->text();
       		if(mp->setData("Map2Mesh","fn",fn) >= 0)
       			printf("xtradata fn ok!\n");
       		filename = gfw->text();
       		if(mp->setData("Map2Mesh","mapfile",filename) >= 0)
       			printf("xtradata mapfile ok!\n");
       			
       		qstr = ewidth->text();
       		w = qstr.toInt();
       		if(mp->setData("Map2Mesh","width",w) >= 0)
			printf("xtradata width ok!\n");
			
      		qstr = eheight->text();
       		h = qstr.toInt();
       		if(mp->setData("Map2Mesh","height",h) >= 0)
			printf("xtradata height ok!\n");
       		if(mp->setData("Map2Mesh","projection",p) >= 0)
			printf("xtradata projection ok!\n");
		
      		qstr = emul->text();
       		multiplier = qstr.toDouble();
       		if(mp->setData("Map2Mesh","multiplier",multiplier) >= 0)
			printf("xtradata multiplier ok!\n");
			
		// now do something with the mesh
		mp->empty();
		
		qstr = prefs.getPath(filename);
		
		if(!qi.load(qstr))
		{
			printf("Cannot load image [%s]\n",filename);
			printf("Cannot load image [%s]\n",(const char *)qstr);
		}
			
		switch(p)
		{
			case 0: // sphere
			{
				int		x,y,p4,p1,p2,p3;
				Vector3		v;
				double		a,b,radius;
				
				radius = calculateRadius(fn,0,0);
				radius += calculateImageOffset(qi,0,0);
				mp->addNode(0,radius,0);
				radius = calculateRadius(fn,w - 1,h - 1);
				radius += calculateImageOffset(qi,w - 1,h - 1);
				mp->addNode(0,-radius,0);
				for(y = 1;y < h;y++)
				{
					b = (double)y / (double)h * PI;
					for(x = 0;x < w;x++)
					{
						radius = calculateRadius(fn,x,y);
						radius += calculateImageOffset(qi,x,y);
						a = (double)x / (double)w * 2 * PI;
						v = Vector3(radius * sin(a) * sin(b),
							radius * cos(b),
							radius * cos(a) * sin(b));
						mp->addNode(v);
					}
				}

				for(x = 0;x < h;x++)
				{
					p1 = x + 2;
					p2 = (x + 1) % w + 2;
					mp->addSegment(0,p1,p2);
				}
				for(y = 0;y < h - 1;y++)
				{
					for(x = 0;x < w;x++)
					{
						p1 = x + y * w + 2;
						p2 = (x + 1) % w + y * w + 2;
						p3 = x + ((y + 1) % h) * w + 2;
						p4 = (x + 1) + ((y + 1) % h) * w + 2;
						mp->addSegment(p1,p2,p4);
						mp->addSegment(p1,p3,p4);
					}
				}
				for(x = 0;x < h;x++)
				{
					p1 = x + w * (h - 2) + 2;
					p2 = (x + 1) % w + w * (h - 2) + 2;
					mp->addSegment(1,p1,p2);
				}
			}
			break;
			case 1: // cylinder
			{
				int		x,y,p4,p1,p2,p3;
				Vector3		v;
				double		a,b,radius;

				for(y = 0;y < h;y++)
				{
					for(x = 0;x < w;x++)
					{
						a = (double)x / (double)w * 2 * PI;
						b = (double)y / (double)(h - 1);
						radius = calculateRadius(fn,x,y);
						radius += calculateImageOffset(qi,x,y);
						v = Vector3(radius * sin(a),
							radius * cos(a),
							0.5 - b);
						mp->addNode(v);
					}
				}

				mp->addNode(0,0,0.5);
				mp->addNode(0,0,-0.5);
			
				for(y = 0;y < h - 1;y++)
				{
					for(x = 0;x < w;x++)
					{
						p1 = x + y * w;
						p2 = (x + 1) % w + y * w;
						p3 = x + (y + 1) * w;
						p4 = (x + 1) % w + (y + 1) * w;
						mp->addSegment(p1,p2,p4);
						mp->addSegment(p1,p3,p4);
					}
				}
       				for(x = 0;x < w;x++)
       				{
       					p1 = x;
       					p2 = (x + 1) % w;
       					p3 = w * h;
       					mp->addSegment(p1,p2,p3);
       				}
       				for(x = 0;x < w;x++)
       				{
 					p1 = x + (h - 1) * w;
					p2 = (x + 1) % w + (h - 1) * w;
      					p3 = w * h + 1;
       					mp->addSegment(p1,p2,p3);
       				}
			}
			break;
			case 2: // torus
			break;
			case 3: // box
			break;
			case 4: // plane
			{
				int		x,y,p4,p1,p2,p3;
				Vector3		v;
				double		a,b,radius;

				for(y = 0;y < h;y++)
				{
					for(x = 0;x < w;x++)
					{
						a = (double)x / (double)(w - 1);
						b = (double)y / (double)(h - 1);
						radius = calculateRadius(fn,x,y);
						radius += calculateImageOffset(qi,x,y);
						v = Vector3(0.5 - a,radius,0.5 - b);
						mp->addNode(v);
					}
				}

				for(y = 0;y < h - 1;y++)
				{
					for(x = 0;x < w - 1;x++)
					{
						p1 = x + y * w;
						p2 = x + 1 + y * w;
						p3 = x + (y + 1) * w;
						p4 = x + 1 + (y + 1) * w;
						mp->addSegment(p1,p2,p4);
						mp->addSegment(p1,p3,p4);
					}
				}
			}
			break;
		}
	}
		
	return mp;
}

void	Map2MeshDialog::setObject(smesh *mp)
{
	if(mp)
	{
		meshobj = mp;
		if(mp->hasDataCollection("Map2Mesh") >= 0)
		{
			const char	*txt;
			
			meshobj = mp;
			
			// Fill the dialog
			if(mp->getData("Map2Mesh","fn",txt) >= 0)
			{
				efn->setText(txt);
	       			printf("xtradata fn ok!\n");
			}
			if(mp->getData("Map2Mesh","mapfile",txt) >= 0)
			{
				gfw->setFilename(txt);
	 			printf("xtradata mapfile ok!\n");
			}
			if(mp->getData("Map2Mesh","width",w) >= 0)
			{
				QString		qstr;
				
				qstr.setNum(w);
				ewidth->setText(qstr);
				printf("xtradata width ok!\n");
			}
			if(mp->getData("Map2Mesh","height",h) >= 0)
			{
				QString		qstr;
				
				qstr.setNum(h);
				eheight->setText(qstr);
				printf("xtradata height ok!\n");
			}
			if(mp->getData("Map2Mesh","projection",p) >= 0)
			{
				projection->setButton(p);
				printf("xtradata projection ok!\n");
			}
			if(mp->getData("Map2Mesh","multiplier",multiplier) >= 0)
			{
				QString		qstr;
				
				qstr.setNum(multiplier);
				emul->setText(qstr);
				printf("xtradata multiplier ok!\n");
			}
		}
	}
	else
	{
		meshobj = 0;
	}
}

void	Map2MeshDialog::setProjection(int np)
{
	p = np;
}

void	Map2MeshDialog::paintEvent(QPaintEvent*)
{
	layout();
}

double	Map2MeshDialog::calculateRadius(const char *term,int x,int y)
{
	char		*calcstr;
	ntuple_t	*ntuple = 0;
	paralist_t	plist = {0,0};
	double		retval;

	calcstr = strdup(term);

	add_param(&plist,"X",(double)x);
	add_param(&plist,"x",(double)x);
	add_param(&plist,"Y",(double)y);
	add_param(&plist,"y",(double)y);

/*
	add_param(&plist,"W",w);
	add_param(&plist,"w",w);
	add_param(&plist,"H",h);
	add_param(&plist,"h",h);
*/

	add_param(&plist,"PI",PI);

	ntuple = eval_term(calcstr,&plist);

	retval = ntuple->tuple[0];

	free(calcstr);
	free(ntuple->tuple);
	free(ntuple);

	return retval;
}

double	Map2MeshDialog::calculateImageOffset(QImage image,int x,int y)
{
	int	xi,yi;
	double	retval;
	
	xi = x * image.width() / w;
	yi = y * image.height() / h;
	
	if(!image.valid(xi,yi))
		return 0;
		
	if(image.depth() > 8)
	{
		QRgb	rgb;
		
		rgb = image.pixel(xi,yi);
		retval = (double)(qRed(rgb) * 65536 + qGreen(rgb) * 255 + qBlue(rgb)) / (double)0xfffff;
	}
	else
	{
		retval = (double)image.pixelIndex(xi,yi) / 255.0;
	}
	
	retval *= multiplier;
	
	return retval;
}




