/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** evaluate.c
*/
#include "evaluate.h"


int	eval_number(const char *term,double *result)
{
	int		t;
	int		is_zero,is_neg;

	is_zero = 1;
	is_neg = 0;
	if(term[0] == '+')
	{
		term++;
	}
	if(term[0] == '-')
	{
		is_neg = 1;
		term++;
	}

	for(t = 0;t < (int)strlen(term);t++)
	{
		if((term[t] >= '0' && term[t] <= '9') || term[t] == '.')
		{
			if(term[t] != '0' && term[t] != '.') is_zero = 0;
		}
		else return 0;
	}
			
	if(is_zero)
	{
		*result = 0;
		return 1;
	}

	sscanf(term,"%lf",result);

	if(is_neg) *result = -*result;

	return (*result == 0) == is_zero;
}
			
static char	*term_left;
static char	*term_right;

static char	minusstr[] = "-";
static char	plusstr[] = "+";
static char	mulstr[] = "*";
static char	divstr[] = "/";
static char	powstr[] = "^";
static char	nullstr[] = "";
static char	andstr[] = "&";
static char	orstr[] = "|";
static char	equalstr[] = "=";
static char	smalerstr[] = "<";
static char	greaterstr[] = ">";
static char	diffstr[] = "#";
static char	modstr[] = "%";

char	*get_next_operator(char *term,char *ops)
{
	int		t,cb,i,sign;
	char		*op = 0;

	cb = 0;
	term_left = 0;
	term_right = 0;
	sign = 0;
	for(t = strlen(term) - 1;t >= 0;t--)
	{
		if(term[t] == ')') cb++;
		if(term[t] == '(') cb--;
		if(cb == 0 && strchr(ops,term[t]))
		{
			if(term[t] == '-')
			{
				for(i = t - 1;i >= 0 && strchr(" \t+-*/^&|<>=#%",term[i]);i--)
				{
					if(strchr("*/-+^&|<>=#%",term[i]) != 0)
					{
						sign = 1;
						break;
					}
				}
			}
			if(term[t] == '+')
			{
				for(i = t - 1;i >= 0 && strchr(" \t+-*/^&|<>=#%",term[i]);i--)
				{
					if(strchr("*/-+^&|<>=#%",term[i]) != 0)
					{
						sign = 1;
						break;
					}
				}
			}
			if(sign) continue;
			if(term[t] == '-') op = minusstr;
			if(term[t] == '+') op = plusstr;
			if(term[t] == '*') op = mulstr;
			if(term[t] == '/') op = divstr;
			if(term[t] == '^') op = powstr;
			if(term[t] == '&') op = andstr;
			if(term[t] == '|') op = orstr;
			if(term[t] == '=') op = equalstr;
			if(term[t] == '#') op = diffstr;
			if(term[t] == '<') op = smalerstr;
			if(term[t] == '>') op = greaterstr;
			if(term[t] == '%') op = modstr;
			term[t] = '\0';
			term_right = term + t + 1;
			term_left = term;
			if(t == 0) term_left = nullstr;

			return op;
		}
	}

	return 0;
}

char	*get_left_term(void)
{
	return term_left;
}

char	*get_right_term(void)
{
	return term_right;
}

static char	*term_sub;

char	*get_function_name(char *term)
{
	int		t;
	char		*func;


	term_sub = 0;
	t = 0;
	while(term[t] == ' ') t++;
	if(term[t] >= '0' && term[t] <= '9') return 0;
	func = term + t;
	if(t > 0) term[t - 1] = '\0';
	t = 0;
	while((func[t] != '(') && (t < (int)strlen(func))) t++;
	if(func[t] == '(')
	{
		term_sub = func + t + 1;
		func[t] = '\0';
		for(t = strlen(term_sub) - 1;t >= 0 && term_sub[t] != ')';t--);
		if(term_sub[t] != ')')
		{
			if(t == 0)
			{
				term_sub  = 0;
				return 0;
			}
		}
		if(t + 1 < (int)strlen(term_sub)) term_sub[t + 1] = '\0';
	}
	else return 0;

/*
	printf("Function: %s SubTerm: %s\n",func,term_sub);
*/
	func = strdup(func);
	term_sub--;
	term_sub[0] = '(';

	return func;
}

char	*get_subterm(void)
{
	return term_sub;
}

char	*format_term(char *term)
{
	int	 	t,len,bo[3];

	bo[0] = bo[1] = bo[2] = 0;
	len = strlen(term);
	for(t = 0;t < len;t++)
	{
		if(term[t] == '(') bo[0]++;
		if(term[t] == ')') bo[0]--;
		if(term[t] == '{') bo[1]++;
		if(term[t] == '}') bo[1]--;
		if(term[t] == '[') bo[2]++;
		if(term[t] == ']') bo[2]--;
	}
	
	if(bo[0] || bo[1] || bo[2]) return 0;

	while(*term == ' ') term++;
	len = strlen(term);
	for(t = len - 1;t >= 0 && term[t] == ' ';t--);
	if(t >= 0 && term[t + 1] == ' ') term[t + 1] = '\0';

	return term;
}

int	add_tuple(ntuple_t *tuple,double d)
{
	tuple->ntuple++;
	tuple->tuple = (double*)realloc(tuple->tuple,sizeof(double) * tuple->ntuple);
	tuple->tuple[tuple->ntuple - 1] = d;

	return 0;
}

char	**get_simple_terms(char *term)
{
	int		t,bo,nt,simple,len;
	char		**terms;
	char		*cp;

	if(term[0] != '(')
	{
		terms = (char**)malloc(sizeof(char*) * 2);
		terms[0] = term;
		terms[1] = 0;

		return terms;
	}
	if(term[strlen(term) - 1] != ')')
	{
		terms = (char**)malloc(sizeof(char*) * 2);
		terms[0] = term;
		terms[1] = 0;

		return terms;
	}

	bo = 0;
	simple = 1;
	for(t = 0;t < (int)strlen(term);t++)
	{
		if(term[t] == '(') bo++;
		if(term[t] == ')') bo--;
		if(t < (int)strlen(term) - 1 && bo == 0) simple = 0;
	}

	nt = 2;
	terms = (char**)malloc(sizeof(char*) * nt);
	terms[0] = term;
	terms[1] = 0;

	if(!simple) return terms;

	bo = 0;
	len = strlen(term);
	for(t = 1;t < len - 1;t++)
	{
		if(term[t] == '(') bo++;
		if(term[t] == ')') bo--;
		if(bo == 0 && term[t] == ',')
		{
			nt++;
			terms = (char**)realloc(terms,sizeof(char*) * nt);
			terms[nt - 2] = term + t + 1;
			terms[nt - 1] = 0;
			term[t] = '\0';
		}
	}
	cp = strrchr(terms[nt - 2],')');
	if(cp) *cp = '\0';
	terms[0]++;

	return terms;
}

int	check_equal(char *equal,paralist_t *plist)
{
	ntuple_t	*r;
	int		vbool;

	vbool = 0;

	r = eval_term(equal,plist);
	if(r)
	{
		if(r->tuple)
		{
			vbool = (r->tuple[0] != 0);
			free(r->tuple);
		}
		free(r);
	}

	return vbool;
}

ntuple_t	*eval_part(char *term,paralist_t *plist)
{
	int		bo,po,t,len;
	char		*equal,*subterm;
//	ntuple_t	*result;

	if(term[0] != '{') return 0;
	if(term[strlen(term) - 1] != '}') return 0;

/*
	result = (ntuple_t*)malloc(sizeof(ntuple_t));
	result->ntuple = 0;
	result->tuple = 0;
*/

	term[strlen(term) - 1] = '\0';
	term++;

	bo = 0;
	po = 0;

	equal = term;
	subterm = 0;

	len = strlen(term);

	for(t = 0;t < len;t++)
	{
		if(term[t] == '{') po++;
		if(term[t] == '}') po--;

		if(term[t] == '(') bo++;
		if(term[t] == ')') bo--;

		if(bo != 0) continue;
		if(po != 0) continue;

		if(term[t] == ':')
		{
			subterm = term + t + 1;
			term[t] = '\0';
		}
		if(term[t] == ';')
		{
			term[t] = '\0';
			if(equal && subterm)
			{
				if(check_equal(equal,plist))
				{
					return eval_term(subterm,plist);
				}
			}
			else return 0;
			equal = term + t + 1;
			subterm = 0;
		}
	}

	if(equal && subterm)
	{
		if(check_equal(equal,plist))
		{
			return eval_term(subterm,plist);
		}
	}
	return 0;
}

ntuple_t	*eval_term(char *term_unformated,paralist_t *plist)
{
	char		*term,*term_formated;
	char		**terms;
	char		*o,*subterm,*func;
	char		*term_left,*term_right;
	int		i,t;
	ntuple_t	*result,*rl,*rr,*r;
	double		number;

	result = (ntuple_t*)malloc(sizeof(ntuple_t));
	result->ntuple = 0;
	result->tuple = 0;

	if(term_unformated == 0)
	{
		add_tuple(result,0);
		return result;
	}

	term_formated = format_term(term_unformated);

	if(!term_formated) return result;

	terms = get_simple_terms(term_formated);

	term = terms[0];
	t = 1;
	
	while(term)
	{
		if(term[0] == '{')
		{
			r = eval_part(term,plist);
			if(r)
			{
				if(r->tuple)
				{
					add_tuple(result,r->tuple[0]);
					free(r->tuple);
				}
				else add_tuple(result,0);
				free(r);
			}
			else add_tuple(result,0);
			term = terms[t];
			t++;
			continue;
		}
		if(term[0] == '\0')
		{
			add_tuple(result,0);
			term = terms[t];
			t++;
			continue;
		}
		if(eval_number(term,&number))
		{
			add_tuple(result,number);
			term = terms[t];
			t++;
			continue;
		}

		for(i = 0;i < plist->nparam;i++)
		{
			if(strcmp(term,plist->param[i].name) == 0)
			{
				add_tuple(result,plist->param[i].value);
			}
		}

		if((o = get_next_operator(term,"&|")))
		{
			int		vbool,bl,br;
			
			vbool = 0;

			term_left = get_left_term();
			term_right = get_right_term();
			rl = eval_term(term_left,plist);
			rr = eval_term(term_right,plist);
			bl = 0;
			br = 0;
			if(rl)
			{
				if(rl->tuple)
				{
					bl = (rl->tuple[0] != 0);
					free(rl->tuple);
				}
				free(rl);
			}
			if(rr)
			{
				if(rr->tuple)
				{
					br = (rr->tuple[0] != 0);
					free(rr->tuple);
				}
				free(rr);
			}
			switch(*o)
			{
				case '&':
					vbool = bl && br;
				break;
				case '|':
					vbool = bl || br;
				break;
			}
			if(vbool) add_tuple(result,1);
			else add_tuple(result,0);
		}
		else if((o = get_next_operator(term,"<=>#")))
		{
			int		vbool;
			double		dl,dr;
			
			vbool = 0;

			term_left = get_left_term();
			term_right = get_right_term();
			rl = eval_term(term_left,plist);
			rr = eval_term(term_right,plist);
			dl = 0;
			dr = 0;
			if(rl)
			{
				if(rl->tuple)
				{
					dl = rl->tuple[0];
					free(rl->tuple);
				}
				free(rl);
			}
			if(rr)
			{
				if(rr->tuple)
				{
					dr = rr->tuple[0];
					free(rr->tuple);
				}
				free(rr);
			}
			switch(*o)
			{
				case '=':
					vbool = (dl == dr);
				break;
				case '<':
					vbool = (dl < dr);
				break;
				case '>':
					vbool = (dl > dr);
				break;
				case '#':
					vbool = (dl != dr);
				break;
			}
			if(vbool) add_tuple(result,1);
			else add_tuple(result,0);
		}
		else if((o = get_next_operator(term,"+-")))
		{
			double		dl,dr;

			term_left = get_left_term();
			term_right = get_right_term();
			rl = eval_term(term_left,plist);
			rr = eval_term(term_right,plist);
			dl = dr = 0;
			if(rl)
			{
				if(rl->tuple)
				{
					dl = rl->tuple[0];
					free(rl->tuple);
				}
				free(rl);
			}
			if(rr)
			{
				if(rr->tuple)
				{
					dr = rr->tuple[0];
					free(rr->tuple);
				}
				free(rr);
			}
			if(*o == '+') add_tuple(result,dl + dr);
			if(*o == '-') add_tuple(result,dl - dr);
		}
		else if((o = get_next_operator(term,"*/%")))
		{
			double		dr,dl;

			term_left = get_left_term();
			term_right = get_right_term();
			rl = eval_term(term_left,plist);
			rr = eval_term(term_right,plist);
			dl = dr = 0;
			if(rl)
			{
				if(rl->tuple)
				{
					dl = rl->tuple[0];
					free(rl->tuple);
				}
				free(rl);
			}
			if(rr)
			{
				if(rr->tuple)
				{
					dr = rr->tuple[0];
					free(rr->tuple);
				}
				free(rr);
			}
			if(*o == '*') add_tuple(result,dl * dr);
			if(*o == '/') add_tuple(result,dl / dr);
			if(*o == '%') add_tuple(result,fmod(dl,dr));
		}
		else if((o = get_next_operator(term,"^")))
		{
			double		dl,dr;

			term_left = get_left_term();
			term_right = get_right_term();
			rl = eval_term(term_left,plist);
			rr = eval_term(term_right,plist);
			dl = dr = 0;
			if(rl)
			{
				if(rl->tuple)
				{
					dl = rl->tuple[0];
					free(rl->tuple);
				}
				free(rl);
			}
			if(rr)
			{
				if(rr->tuple)
				{
					dr = rr->tuple[0];
					free(rr->tuple);
				}
				free(rr);
			}
			if(*o == '^') add_tuple(result,pow(dl,dr));
		}
		else
		{
			func = get_function_name(term);
			subterm = get_subterm();
			if(subterm)
			{
				r = eval_term(subterm,plist);
				if(func)
				{
					if(strncmp(func,"sin",3) == 0) add_tuple(result,sin(r->tuple[0]));
					else if(strncmp(func,"cos",3) == 0) add_tuple(result,cos(r->tuple[0]));
					else if(strncmp(func,"tan",3) == 0) add_tuple(result,tan(r->tuple[0]));
					else if(strncmp(func,"asin",4) == 0) add_tuple(result,asin(r->tuple[0]));
					else if(strncmp(func,"acos",4) == 0) add_tuple(result,acos(r->tuple[0]));
					else if(strncmp(func,"atan",4) == 0) add_tuple(result,atan(r->tuple[0]));
					else if(strncmp(func,"sinh",4) == 0) add_tuple(result,sinh(r->tuple[0]));
					else if(strncmp(func,"cosh",4) == 0) add_tuple(result,cosh(r->tuple[0]));
					else if(strncmp(func,"tanh",4) == 0) add_tuple(result,tanh(r->tuple[0]));
					else if(strncmp(func,"asinh",5) == 0) add_tuple(result,asinh(r->tuple[0]));
					else if(strncmp(func,"acosh",5) == 0) add_tuple(result,acosh(r->tuple[0]));
					else if(strncmp(func,"atanh",5) == 0) add_tuple(result,atanh(r->tuple[0]));
					else if(strncmp(func,"logn",4) == 0) add_tuple(result,log(r->tuple[0]));
					else if(strncmp(func,"logd",4) == 0) add_tuple(result,log10(r->tuple[0]));
					else if(strncmp(func,"ceil",4) == 0) add_tuple(result,ceil(r->tuple[0]));
					else if(strncmp(func,"floor",5) == 0) add_tuple(result,floor(r->tuple[0]));
					else if(strncmp(func,"abs",3) == 0) add_tuple(result,fabs(r->tuple[0]));
					else if(strncmp(func,"fabs",4) == 0) add_tuple(result,fabs(r->tuple[0]));
					else if(strncmp(func,"sqrt",4) == 0) add_tuple(result,sqrt(r->tuple[0]));
					else if(strncmp(func,"sqr",3) == 0) add_tuple(result,r->tuple[0] * r->tuple[0]);
					else if(strncmp(func,"pow",3) == 0) add_tuple(result,pow(r->tuple[0],r->tuple[1]));
					else if(strncmp(func,"mod",3) == 0) add_tuple(result,fmod(r->tuple[0],r->tuple[1]));
					else if(strncmp(func,"isnan",5) == 0) add_tuple(result,isnan(r->tuple[0]));
					/*else if(strncmp(func,"not",3) == 0) add_tuple(result,((r->tuple[0] == 0)?1:0));*/
					else if(strncmp(func,"not",3) == 0) add_tuple(result,BOOL(r->tuple[0]));
					else if(strncmp(func,"and",3) == 0) add_tuple(result,BOOL(r->tuple[0]) && BOOL(r->tuple[1]));
					else if(strncmp(func,"or",2) == 0) add_tuple(result,BOOL(r->tuple[0]) || BOOL(r->tuple[1]));
					else if(strncmp(func,"xor",3) == 0) add_tuple(result,BOOL(r->tuple[0]) ^ BOOL(r->tuple[1]));
/*
					else if(strncmp(func,"nor",3) == 0) add_tuple(result,BOOL(r->tuple[0]) ^ BOOL(r->tuple[1]));
					else if(strncmp(func,"nand",4) == 0) add_tuple(result,BOOL(r->tuple[0]) ^ BOOL(r->tuple[1]));
*/
					else if(strncmp(func,"drand",5) == 0) add_tuple(result,(double)rand() / RAND_MAX);				
					else if(strncmp(func,"rand",4) == 0) add_tuple(result,(double)rand() / RAND_MAX);				
					else if(strncmp(func,"seedrand",8) == 0)
					{
						srand((unsigned int)r->tuple[0]);
						add_tuple(result,(double)rand() / RAND_MAX);
					}				
					else if(strncmp(func,"max",3) == 0) add_tuple(result,MAX(r->tuple[0],r->tuple[1]));				
					else if(strncmp(func,"min",3) == 0) add_tuple(result,MIN(r->tuple[0],r->tuple[1]));				
					else if(strncmp(func,"degree",6) == 0) add_tuple(result,DEGREE(r->tuple[0]));				
					else if(strncmp(func,"deg",3) == 0) add_tuple(result,DEGREE(r->tuple[0]));				

					free(func);
				}
				else add_tuple(result,r->tuple[0]);
				if(r->tuple) free(r->tuple);
				free(r);
			}
		}
		term = terms[t];
		t++;
	}

	free(terms);
	
	return result;
}
	
void	add_param(paralist_t *list,const char *name,double value)
{
	list->nparam++;
	list->param = (param_t*)realloc(list->param,sizeof(param_t) * list->nparam);
	list->param[list->nparam - 1].name = strdup(name);
	list->param[list->nparam - 1].value = value;
}

void	set_param(paralist_t *list,const char *name,double value)
{
	int		t;

	for(t = 0;t < list->nparam;t++)
	{
		if(strcmp(list->param[t].name,name) == 0)
		{
			list->param[t].value = value;
			return;
		}
	}

	add_param(list,name,value);
}

int	exist_param(paralist_t *list,const char *name)
{
	int		t;

	for(t = 0;t < list->nparam;t++)
	{
		if(strcmp(list->param[t].name,name) == 0)
		{
			return 1;
		}
	}

	return 0;
}

double	get_param(paralist_t *list,const char *name)
{
	int		t;

	for(t = 0;t < list->nparam;t++)
	{
		if(strcmp(list->param[t].name,name) == 0)
		{
			return list->param[t].value;
		}
	}

	return 0;
}

void	copy_paralist(paralist_t *slist,paralist_t *dlist)
{
	int		t;

	for(t = 0;t < slist->nparam;t++)
	{
		set_param(dlist,slist->param[t].name,slist->param[t].value);
	}
}

void	kill_paralist(paralist_t *list)
{
	int		t;

	for(t = 0;t < list->nparam;t++)
	{
		if(list->param[t].name)
		{
			free(list->param[t].name);
		}
	}
	
	if(list->param)
		free(list->param);
	list->param = 0;
	list->nparam = 0;
}


#ifdef TEST

void	test(const char *term,paralist_t *plist)
{
	char		*termcopy;
	int		t;
	ntuple_t	*ntuple = 0;

	termcopy = strdup(term);
	printf("eval_term(%s",termcopy);
	for(t = 0;t < plist->nparam;t++)
	{
		printf(",%5.2f",plist->param[t].value);
	}
	printf(") = (");
	ntuple = eval_term(termcopy,plist);
	if(!ntuple) return;
	for(t = 0;t < ntuple->ntuple;t++)
	{
		printf("%5.2f",ntuple->tuple[t]);
		if(t < ntuple->ntuple - 1) printf(",");
	}
	printf(")\n");
	free(termcopy);
}

void	main(int argc,char **argv)
{
	paralist_t	plist = {0,0};

	printf("Test eval_term:\n\n");

	add_param(&plist,"Zahl",50);
	test("{Zahl = 50 & Zahl < 100: 0 ; 1 : 1}",&plist);
	test("{Zahl > 0 & Zahl < 100: 0 ; 1 : 1}",&plist);
	test("{Zahl < 100: 0 ; 1 : 1}",&plist);
	test("{Zahl > 100: 0 ; 1 : 1}",&plist);
	test("1 + 2",&plist);
	test("1 + 2 + 3",&plist);
	test("1 + 2 + 3 * 4",&plist);
	test("1 + 2 + 3 * 4 + 5",&plist);
	test("1 + (2 + 3) * (4 + 5)",&plist);
	add_param(&plist,"param",45);
	test("param",&plist);
	test("param * 2",&plist);
	test("abs(-2)",&plist);
	test("abs(2)",&plist);
	test("sqrt(9)",&plist);
	test("sqrt(param)",&plist);
	test("5 * sin(param / 180 * 3.1415)",&plist);
	test("pow(3,3)",&plist);
	test("(3 * param,0,0,1)",&plist);
	add_param(&plist,"y",0);
	add_param(&plist,"z",0);
	test("(3 * param,y,z,1)",&plist);
	test("(3 * -param,y,z,1)",&plist);

	exit(0);
}

#endif
