/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <Background.h>
#include <SFColor.h>
#include <SFRenderInfo.h>
#include <SFString.h>
#include <Children.h>
#include <SFEvent.h>

/* default image dimensions */
#ifndef BackDims
#   define BackDims 64
#endif

/*************************************************************************/
void Background::Solid(SFImage &img, SFColor &C)
{

  /* allocate a new image */
  img.alloc(BackDims, BackDims);

  /* visit each pixel */
  for (int j = 0; j < BackDims; j++) {
    for (int i = 0; i < BackDims; i++) {  
      img(i, j, 0) = (unsigned char)(C[0]() * 255);
      img(i, j, 1) = (unsigned char)(C[1]() * 255);
      img(i, j, 2) = (unsigned char)(C[2]() * 255);
      img(i, j, 3) = 255;
    }
  }
}
/*************************************************************************/

/*************************************************************************/
void Background::InterpolateSkyToGround(SFImage &img)
{

  /* do interpolation based on the sky colors */
  float r = 0.0, g = 0.0, b = 0.0;
  for (int j = 0; j < BackDims; j++) {
    
    /* compute the color for the whole row - true backgrounds should
    ** be spherical but I am using linear across the whole row. */
    if (skyColor.num() != 0) {
      r = skyColor[0][0];
      g = skyColor[0][1];
      b = skyColor[0][2];
    } else {
      r = 0.0;
      g = 0.0;
      b = 0.0;
    }
    
    /* only do interpolations if appropriate */
    if (skyAngle.isValid()) {
      
      /* compute the radians factor */
      float rad = float(j) / float(BackDims - 1); 
      
      /* find the point where the radians value falls between the
      ** two associated angles */
      int interp = -1;
      for (interp = 0; interp < skyAngle.num(); interp++) {
	if (rad < (skyAngle[interp] / M_PI)) break;
      }

      /* do the interpolation when we have the first angle */
      float fac = 0.0, r1 = 0.0, r2 = 0.0, g1 = 0.0, g2 = 0.0, b1 = 0.0, b2 = 0.0;
      if (interp == 0) {
	fac = rad / (skyAngle[0] / M_PI);
	r1  = skyColor[0][0];
	g1  = skyColor[0][1];
	b1  = skyColor[0][2];
	r2  = skyColor[1][0];
	g2  = skyColor[1][1];
	b2  = skyColor[1][2];
	r   = r1 * (1.0 - fac) + r2 * fac;
	g   = g1 * (1.0 - fac) + g2 * fac;
	b   = b1 * (1.0 - fac) + b2 * fac;

      } else if (interp == skyAngle.num()) {
	r  = skyColor[skyAngle.num()][0];
	g  = skyColor[skyAngle.num()][1];
	b  = skyColor[skyAngle.num()][2];

      } else {
	float a1 = skyAngle[interp - 1] / M_PI;
	float a2 = skyAngle[interp] / M_PI;
	fac = (rad - a1) / (a2 - a1);
	r1  = skyColor[interp][0];
	g1  = skyColor[interp][1];
	b1  = skyColor[interp][2];
	r2  = skyColor[interp + 1][0];
	g2  = skyColor[interp + 1][1];
	b2  = skyColor[interp + 1][2];
	r   = r1 * (1.0 - fac) + r2 * fac;
	g   = g1 * (1.0 - fac) + g2 * fac;
	b   = b1 * (1.0 - fac) + b2 * fac;

      }
    }
    
    /* fill in the whole row with that value */
    for (int i = 0; i < BackDims; i++) {
      img(i, BackDims - j - 1, 0) = (unsigned char)(r * 255);
      img(i, BackDims - j - 1, 1) = (unsigned char)(g * 255);
      img(i, BackDims - j - 1, 2) = (unsigned char)(b * 255);
      img(i, BackDims - j - 1, 3) = 255;
    }
  }

}
/*************************************************************************/

/*************************************************************************/
void Background::InterpolateGroundToSky(SFImage &img)
{

  /* do not interpolate upwards if not needed */
  if ((groundColor.num() == 0) || (groundAngle.num() == 0)) return;

  /* do interpolation based on the ground colors */
  float r = 0.0, g = 0.0, b = 0.0;
  for (int j = 0; j < BackDims; j++) {
    
    /* compute the color for the whole row - true backgrounds should
    ** be spherical but I am using linear across the whole row. */
    if (groundColor.num() != 0) {
      r = groundColor[0][0];
      g = groundColor[0][1];
      b = groundColor[0][2];
    } else {
      r = 0.0;
      g = 0.0;
      b = 0.0;
    }
    
    /* only do interpolations if appropriate */
    if ((groundAngle.num() != 0) && (groundAngle.num() > 0)) {
      
      /* compute the radians factor */
      float rad = float(j) / float(BackDims - 1); 
      
      /* find the point where the radians value falls between the
      ** two associated angles */
      int interp = -1;
      for (interp = 0; interp < groundAngle.num(); interp++) {
	if (rad < (groundAngle[interp] / M_PI)) break;
      }

      /* do the interpolation when we have the first angle */
      float fac = 0.0, r1 = 0.0, r2 = 0.0, g1 = 0.0, g2 = 0.0, b1 = 0.0, b2 = 0.0;
      if (interp == 0) {
	fac = rad / (groundAngle[0] / M_PI);
	r1  = groundColor[0][0];
	g1  = groundColor[0][1];
	b1  = groundColor[0][2];
	r2  = groundColor[1][0];
	g2  = groundColor[1][1];
	b2  = groundColor[1][2];
	r   = r1 * (1.0 - fac) + r2 * fac;
	g   = g1 * (1.0 - fac) + g2 * fac;
	b   = b1 * (1.0 - fac) + b2 * fac;

      } else if (interp == groundAngle.num()) {
	continue;

      } else {
	float a1 = groundAngle[interp - 1] / M_PI;
	float a2 = groundAngle[interp] / M_PI;
	fac = (rad - a1) / (a2 - a1);
	r1  = groundColor[interp][0];
	g1  = groundColor[interp][1];
	b1  = groundColor[interp][2];
	r2  = groundColor[interp + 1][0];
	g2  = groundColor[interp + 1][1];
	b2  = groundColor[interp + 1][2];
	r   = r1 * (1.0 - fac) + r2 * fac;
	g   = g1 * (1.0 - fac) + g2 * fac;
	b   = b1 * (1.0 - fac) + b2 * fac;

      }
    }
    
    /* fill in the whole row with that value */
    for (int i = 0; i < BackDims; i++) {
      img(i, j, 0) = (unsigned char)(r * 255);
      img(i, j, 1) = (unsigned char)(g * 255);
      img(i, j, 2) = (unsigned char)(b * 255);
      img(i, j, 3) = 255;
    }
  }

}
/*************************************************************************/

/*************************************************************************/
void Background::Interpolate(SFImage &img)
{

  /* allocate a new image */
  img.alloc(BackDims, BackDims);

  /* do the sky to ground interpolation */
  InterpolateSkyToGround(img);

  /* then do the ground to sky */
  InterpolateGroundToSky(img);

}
/*************************************************************************/

/*************************************************************************/
void Background::ComputeInterpolatedImage(SFImage &img, SFUrl &url)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Background::ComputeInterpolatedImage\n";
#endif 

  /* create the image */
  if (! url.isValid()) Interpolate(img);
  else                 img.LoadImage(url(0));

}
/*************************************************************************/

/*************************************************************************/
void Background::ComputeSolidImage(SFImage &img, SFUrl &url, SFColor &C)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Background::ComputeSolidImage\n";
#endif 

  /* if no url exists, load the associated image */
  if  (! url.isValid()) Solid(img, C);
  else                  img.LoadImage(url(0));

}
/*************************************************************************/

/*************************************************************************/
void Background::prepare (SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Background::prepare (" << DEF << ")\n";
#endif 
  
  /* the default solid background color */
  SFColor c;

  /* create the four sides of the background box */
  ComputeInterpolatedImage(backImage, backUrl);
  ComputeInterpolatedImage(leftImage, leftUrl);
  ComputeInterpolatedImage(rightImage, rightUrl);
  ComputeInterpolatedImage(frontImage, frontUrl);

  /* create the top image if it does not already exist */
  if (skyColor.isValid()) {
    ComputeSolidImage(topImage, topUrl, skyColor[0]);
  } else {
    ComputeSolidImage(topImage, topUrl, c);
  }
  
  /* create the bottom image if it does not already exist */
  if (groundColor.isValid()) {
    ComputeSolidImage(botImage, bottomUrl, groundColor[0]);
  } else {
    ComputeSolidImage(botImage, bottomUrl, c);
  }

  /* prepare all of the images for rendering */
  backImage.prepare(min, max);
  frontImage.prepare(min, max);
  leftImage.prepare(min, max);
  rightImage.prepare(min, max);
  topImage.prepare(min, max);
  botImage.prepare(min, max);

}
/*************************************************************************/

/*************************************************************************/
void Background::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Background::eventIn (" << DEF << ")\n";
#endif 

  /* Determine what action to take based on incoming event */
  if        ((eventType == "groundColor")    || (eventType == "set_groundColor")) {
    MFColor *n = (MFColor *) eventData;
    groundColor = *n;
    
  } else if ((eventType == "groundAngle") || (eventType == "set_groundAngle")) {
    MFFloat *n = (MFFloat *) eventData;
    groundAngle = *n;

  } else if ((eventType == "skyColor")    || (eventType == "set_skyColor")) {
    MFColor *n = (MFColor *) eventData;
    skyColor = *n;

  } else if ((eventType == "skyAngle")    || (eventType == "set_skyAngle")) {
    MFFloat *n = (MFFloat *) eventData;
    skyAngle = *n;

  } else if ((eventType == "frontUrl")    || (eventType == "set_frontUrl")) {
    SFUrl *n = (SFUrl *) eventData;
    frontUrl = *n;

  } else if ((eventType == "leftUrl")    || (eventType == "set_leftUrl")) {
    SFUrl *n = (SFUrl *) eventData;
    leftUrl = *n;

  } else if ((eventType == "rightUrl")    || (eventType == "set_rightUrl")) {
    SFUrl *n = (SFUrl *) eventData;
    rightUrl = *n;

  } else if ((eventType == "backUrl")    || (eventType == "set_backUrl")) {
    SFUrl *n = (SFUrl *) eventData;
    backUrl = *n;

  } else if ((eventType == "topUrl")    || (eventType == "set_topUrl")) {
    SFUrl *n = (SFUrl *) eventData;
    topUrl = *n;

  } else if ((eventType == "bottomUrl")    || (eventType == "set_bottomUrl")) {
    SFUrl *n = (SFUrl *) eventData;
    bottomUrl = *n;

  } else if (eventType == "set_bind") {
    SFBool *n = (SFBool *) eventData;
    isBound = *n;
    SFString out = "isBound";    
    eventOut(out, eventData);

  }

}
/*************************************************************************/

/*************************************************************************/
void Background::eventOut(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Background::eventOut (" << DEF << ")\n";
#endif 

  /* if the event queue is empty, no one cares */
  if (eventQueue() == (Children *) 0) return;

  /* generate any appropriate cascading events */
  for (Children *C = eventQueue(); C != (Children *) 0; C = C->next()) {

    /* do not process incomplete nodes */
    if (C->data() == (SFNode *) 0) continue;

    /* determine the event */
    SFEvent *event = (SFEvent *) C->data();

    /* determine the appropriate event to generate */
    if (event->to() == (SFNode *) 0) continue;

    /* generate the event */
    if (event->eventFromName() == "isBound") {
      event->to()->eventIn(eventType, eventData);

    }
  }
}
/*************************************************************************/

/*************************************************************************/
void Background::render(SFRenderInfo &SFRI)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Background::render (" << DEF << ")\n";
#endif 

  /* if the node is not bound, do nothing */
  if (isBound() == false) return;

  /* Store the direction vector in the proper position in image plane */
  float fovo2  = SFRI.fov() / 2.0;
  float AzPose = fovo2 * float(SFRI.w()) / float(SFRI.h());
  float ElPose = fovo2;
  float nx = -sin(AzPose) * cos(ElPose);
  float ny =  sin(ElPose);
  float nz = -cos(AzPose) * cos(ElPose);

  /* compute the far clipping plane */
  float px = -nx * SFRI.farPlane();
  float py =  ny * SFRI.farPlane();
  float pz =  nz * SFRI.farPlane();

  /* store the polygon flags for restoration */
  glPushAttrib(GL_ALL_ATTRIB_BITS);

  /* do not do lighting */
  glDisable(GL_LIGHTING);

  /* Store the matrix stack */
  glMatrixMode(GL_MODELVIEW);  glPushMatrix();

  /* get this matrix */
  GLdouble m[16];
  glGetDoublev (GL_MODELVIEW_MATRIX,  m);
  m[12] = 0.0;
  m[13] = 0.0;
  m[14] = 0.0;
  glLoadMatrixd(m);

  /* disable depth buffering */
  glDisable(GL_DEPTH_TEST);

  /* set up the background texture map */
  backImage.render(SFRI);
  glBegin(GL_QUADS); {
    glTexCoord2f(0.01, 0.99); glVertex3f(-px,  py, -pz);
    glTexCoord2f(0.99, 0.99); glVertex3f( px,  py, -pz);
    glTexCoord2f(0.99, 0.01); glVertex3f( px, -py, -pz);
    glTexCoord2f(0.01, 0.01); glVertex3f(-px, -py, -pz);
  } glEnd();

  /* set up the background texture map */
  frontImage.render(SFRI);
  glBegin(GL_QUADS); {
    glTexCoord2f(0.01, 0.01); glVertex3f(-px, -py, pz);
    glTexCoord2f(0.99, 0.01); glVertex3f( px, -py, pz);
    glTexCoord2f(0.99, 0.99); glVertex3f( px,  py, pz);
    glTexCoord2f(0.01, 0.99); glVertex3f(-px,  py, pz);
  } glEnd();

  /* set up the top texture map */
  topImage.render(SFRI);
  glBegin(GL_QUADS); {
    glTexCoord2f(0.01, 0.99); glVertex3f(-px,  py,  pz);
    glTexCoord2f(0.99, 0.99); glVertex3f( px,  py,  pz);
    glTexCoord2f(0.99, 0.01); glVertex3f( px,  py, -pz);
    glTexCoord2f(0.01, 0.01); glVertex3f(-px,  py, -pz);
  } glEnd();

  /* set up the bottom texture map */
  botImage.render(SFRI);
  glBegin(GL_QUADS); {
    glTexCoord2f(0.01, 0.01); glVertex3f(-px,  -py, -pz);
    glTexCoord2f(0.99, 0.01); glVertex3f( px,  -py, -pz);
    glTexCoord2f(0.99, 0.99); glVertex3f( px,  -py,  pz);
    glTexCoord2f(0.01, 0.99); glVertex3f(-px,  -py,  pz);
  } glEnd();

  /* set up the left texture map */
  leftImage.render(SFRI);
  glBegin(GL_QUADS); {
    glTexCoord2f(0.01, 0.99); glVertex3f(px,   py, -pz);
    glTexCoord2f(0.99, 0.99); glVertex3f(px,   py,  pz);
    glTexCoord2f(0.99, 0.01); glVertex3f(px,  -py,  pz);
    glTexCoord2f(0.01, 0.01); glVertex3f(px,  -py, -pz);
  } glEnd();

  /* set up the right texture map */
  rightImage.render(SFRI);
  glBegin(GL_QUADS); {
    glTexCoord2f(0.01, 0.01); glVertex3f(-px,  -py, -pz);
    glTexCoord2f(0.99, 0.01); glVertex3f(-px,  -py,  pz);
    glTexCoord2f(0.99, 0.99); glVertex3f(-px,   py,  pz);
    glTexCoord2f(0.01, 0.99); glVertex3f(-px,   py, -pz);
  } glEnd();

  /* restore the matrix stack */
  glPopMatrix();

  /* save the state before rendering */
  glPopAttrib();

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

/*************************************************************************/
void Background::parse(char *header, istream &InFile)
{

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);
    
#ifdef DEBUG_PARSE    
    cout << "\tBackground: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "groundColor") {
      groundColor.parse(header, InFile);

    } else if (Token == "groundAngle") {
      groundAngle.parse(header, InFile);

    } else if (Token == "skyColor") {
      skyColor.parse(header, InFile);

    } else if (Token == "skyAngle") {
      skyAngle.parse(header, InFile);

    } else if (Token == "frontUrl") {
      frontUrl.parse(header, InFile);

    } else if (Token == "backUrl") {
      backUrl.parse(header, InFile);

    } else if (Token == "rightUrl") {
      rightUrl.parse(header, InFile);

    } else if (Token == "leftUrl") {
      leftUrl.parse(header, InFile);

    } else if (Token == "topUrl") {
      topUrl.parse(header, InFile);

    } else if (Token == "bottomUrl") {
      bottomUrl.parse(header, InFile);

    } else if (Token == "Background") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
  
}
/*************************************************************************/
