/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <Extrusion.h>

/*************************************************************************/
static void gl_rotation_matrix(GLfloat angle, 
			       GLfloat x, 
			       GLfloat y, 
			       GLfloat z,
			       GLdouble m[])
{

  /* This function was stolen directly from mesa 2.6. Here is his
   * disclaimer:
   * Mesa 3-D graphics library Version: 2.6 Copyright (C)
   * 1995-1997 Brian Paul */

   /* This function contributed by Erich Boleyn (erich@uruk.org) */
   GLfloat mag, s, c;
   GLfloat xx, yy, zz, xy, yz, zx, xs, ys, zs, one_c;
   GLfloat Identity[16] = {
     1.0, 0.0, 0.0, 0.0,
     0.0, 1.0, 0.0, 0.0,
     0.0, 0.0, 1.0, 0.0,
     0.0, 0.0, 0.0, 1.0
   };       

   s = sin( angle * M_PI / 180.0);
   c = cos( angle * M_PI / 180.0);

   mag = sqrt( x*x + y*y + z*z );

   /* generate an identity matrix and return */
   if (mag == 0.0) {
      memcpy(m, Identity, sizeof(GLfloat)*16);
      return;
   }

   x /= mag;
   y /= mag;
   z /= mag;

#define M(row,col)  m[col*4+row]

   /*
    *     Arbitrary axis rotation matrix.
    *
    *  This is composed of 5 matrices, Rz, Ry, T, Ry', Rz', multiplied
    *  like so:  Rz * Ry * T * Ry' * Rz'.  T is the final rotation
    *  (which is about the X-axis), and the two composite transforms
    *  Ry' * Rz' and Rz * Ry are (respectively) the rotations necessary
    *  from the arbitrary axis to the X-axis then back.  They are
    *  all elementary rotations.
    *
    *  Rz' is a rotation about the Z-axis, to bring the axis vector
    *  into the x-z plane.  Then Ry' is applied, rotating about the
    *  Y-axis to bring the axis vector parallel with the X-axis.  The
    *  rotation about the X-axis is then performed.  Ry and Rz are
    *  simply the respective inverse transforms to bring the arbitrary
    *  axis back to it's original orientation.  The first transforms
    *  Rz' and Ry' are considered inverses, since the data from the
    *  arbitrary axis gives you info on how to get to it, not how
    *  to get away from it, and an inverse must be applied.
    *
    *  The basic calculation used is to recognize that the arbitrary
    *  axis vector (x, y, z), since it is of unit length, actually
    *  represents the sines and cosines of the angles to rotate the
    *  X-axis to the same orientation, with theta being the angle about
    *  Z and phi the angle about Y (in the order described above)
    *  as follows:
    *
    *  cos ( theta ) = x / sqrt ( 1 - z^2 )
    *  sin ( theta ) = y / sqrt ( 1 - z^2 )
    *
    *  cos ( phi ) = sqrt ( 1 - z^2 )
    *  sin ( phi ) = z
    *
    *  Note that cos ( phi ) can further be inserted to the above
    *  formulas:
    *
    *  cos ( theta ) = x / cos ( phi )
    *  sin ( theta ) = y / sin ( phi )
    *
    *  ...etc.  Because of those relations and the standard trigonometric
    *  relations, it is pssible to reduce the transforms down to what
    *  is used below.  It may be that any primary axis chosen will give the
    *  same results (modulo a sign convention) using thie method.
    *
    *  Particularly nice is to notice that all divisions that might
    *  have caused trouble when parallel to certain planes or
    *  axis go away with care paid to reducing the expressions.
    *  After checking, it does perform correctly under all cases, since
    *  in all the cases of division where the denominator would have
    *  been zero, the numerator would have been zero as well, giving
    *  the expected result.
    */

   xx = x * x;
   yy = y * y;
   zz = z * z;
   xy = x * y;
   yz = y * z;
   zx = z * x;
   xs = x * s;
   ys = y * s;
   zs = z * s;
   one_c = 1.0F - c;

   M(0,0) = (one_c * xx) + c;
   M(0,1) = (one_c * xy) - zs;
   M(0,2) = (one_c * zx) + ys;
   M(0,3) = 0.0F;

   M(1,0) = (one_c * xy) + zs;
   M(1,1) = (one_c * yy) + c;
   M(1,2) = (one_c * yz) - xs;
   M(1,3) = 0.0F;

   M(2,0) = (one_c * zx) - ys;
   M(2,1) = (one_c * yz) + xs;
   M(2,2) = (one_c * zz) + c;
   M(2,3) = 0.0F;

   M(3,0) = 0.0F;
   M(3,1) = 0.0F;
   M(3,2) = 0.0F;
   M(3,3) = 1.0F;

#undef M
} 
/*************************************************************************/

/*************************************************************************/
void Extrusion::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Extrusion::prepare (" << DEF << ")\n";
#endif 

  /* Create a matrix. These will be used to scale, rotate and
  ** translate each piece of the spine. */
  GLdouble m[16];

  /* if we are to use the default extrusion spine create it */
  if (spine.num() == 0) {
    spine.alloc(2);
    spine[0][0] = 0.0;
    spine[0][1] = 0.0;
    spine[0][2] = 0.0;
    spine[1][0] = 0.0;
    spine[1][1] = 1.0;
    spine[1][2] = 0.0;
  }

  /* if we are to use the default extrusion crossSection create it */
  if (crossSection.num() == 0) {
    crossSection.alloc(4);
    crossSection[0][0] =  1.0;
    crossSection[0][1] =  1.0;
    crossSection[1][0] =  1.0;
    crossSection[1][1] = -1.0;
    crossSection[2][0] = -1.0;
    crossSection[2][1] = -1.0;
    crossSection[3][0] = -1.0;
    crossSection[3][1] =  1.0;
  }

  /* if we are to use the default extrusion scale create it */
  if (scale.num() == 0) {
    scale.alloc(1);
    scale[0][0] =  1.0;
    scale[0][1] =  1.0;
  }

  /* if we are to use the default extrusion orientation create it */
  if (orientation.num() == 0) {
    orientation.alloc(1);
    orientation[0][0] = 0.0;
    orientation[0][1] = 0.0;
    orientation[0][2] = 1.0;
    orientation[0][3] = 0.0;
  }

  /* if the points making up the object are not already created, do so
  ** now.*/
  if (! coord.isValid()) {

    /* compute the number of points that will make up the extrusion. */
    int NumPoints = spine.num() * crossSection.num();

    /* allocate the coordinates */
    coord.alloc(NumPoints);

    /* now visit each point in the extrusion and compute a valid 3d
    ** point. */
    for (int i = 0; i < spine.num(); i++) {
      for (int j = 0; j < crossSection.num(); j += 1) {

	/* compute a scale */
	float sx = 1.0, sz = 1.0;
	if (scale.num() == 1) {
	  sx = scale[0][0];
	  sz = scale[0][1];
	} else {
	  sx = scale[i][0];
	  sz = scale[i][1];
	}
	
	/* first scale the cross section */
	float ix = crossSection[j][0] * sx;
	float iy = 0.0;
	float iz = crossSection[j][1] * sz;

	/* next we need to orient this knot */

	/* figure out the contribution of the orientation */
	if (orientation.num() == 1) {
	  gl_rotation_matrix(orientation[0][3],
			     orientation[0][0],
			     orientation[0][1],
			     orientation[0][2],
			     m);
	} else {
	  gl_rotation_matrix(orientation[i][3],
			     orientation[i][0],
			     orientation[i][1],
			     orientation[i][2],
			     m);
	}

	/* apply the orient field */
	float ox = (m[0] * ix) + (m[4] * iy) + (m[8]  * iz);
	float oy = (m[1] * ix) + (m[5] * iy) + (m[9]  * iz);
	float oz = (m[2] * ix) + (m[6] * iy) + (m[10] * iz);

	/* next compute the translation */
	ox += spine[i][0];
	oy += spine[i][1];
	oz += spine[i][2];
	
	/* store the point back into the coordinate info */
	coord[i * crossSection.num() + j][0] = ox;
	coord[i * crossSection.num() + j][1] = oy;
	coord[i * crossSection.num() + j][2] = oz;

      }
    }
  }

  /* Compute the normal at each point */
  if (normal.num() == 0) {

    /* compute the number of points that will make up the extrusion. */
    int NumPoints = spine.num() * crossSection.num();

    /* allocate the vectors */
    normal.alloc(NumPoints);

    /* the normal at each point */
    float x1 = 0.0, x2 = 0.0, x3 = 0.0;
    float y1 = 0.0, y2 = 0.0, y3 = 0.0;
    float z1 = 0.0, z2 = 0.0, z3 = 0.0;
    for (int i = 0; i < spine.num(); i++) {
      for (int j = 0, idx = 0; j < crossSection.num(); j += 1) {

	/* the first point to use is just the point */
	idx = i * crossSection.num() + j;
	x1 = coord[idx][0];
	y1 = coord[idx][1];
	z1 = coord[idx][2];

	/* the second point is the neighbor upwards */
	if (i == (spine.num() - 1)) {
	  idx = (i - 1) * crossSection.num() + j;
	} else {
	  idx = (i + 1) * crossSection.num() + j;
	}
	x2 = coord[idx][0];
	y2 = coord[idx][1];
	z2 = coord[idx][2];

	/* the thirs point is the neighbor to the right  */
	if (j == (crossSection.num() - 1)) {
	  idx = i * crossSection.num() + (j - 1);
	} else {
	  idx = i * crossSection.num() + (j + 1);
	}
	x3 = coord[idx][0];
	y3 = coord[idx][1];
	z3 = coord[idx][2];
 
	/* compute two vectors between these two points. */
	float v1x = x3 - x1; 
	float v1y = y3 - y1; 
	float v1z = z3 - z1; 
	float v2x = x2 - x1; 
	float v2y = y2 - y1; 
	float v2z = z2 - z1; 

	/* compute the normal */
	float nx = v1y * v2z - v1z * v2y;
	float ny = v1z * v2x - v1x * v2z;
	float nz = v1x * v2y - v1y * v2x;

	/* this normal may be backwards */

	/* store the normal */
	normal[i * crossSection.num() + j][0] = nx;
	normal[i * crossSection.num() + j][1] = ny;
	normal[i * crossSection.num() + j][2] = nz;

	/* normaliee */
	normal[i * crossSection.num() + j].normalize();

      }
    }
  }

  /* get the modelview matrix */
  glGetDoublev (GL_MODELVIEW_MATRIX,  m);

  /* determine the max and min */
  for (int i = 0; i < coord.num(); i++) {
    float ix = coord[i][0];
    float iy = coord[i][1];
    float iz = coord[i][2];
    float ox = (m[0] * ix) + (m[4] * iy) + (m[8]  * iz) + m[12];
    float oy = (m[1] * ix) + (m[5] * iy) + (m[9]  * iz) + m[13];
    float oz = (m[2] * ix) + (m[6] * iy) + (m[10] * iz) + m[14];
    if (min[0] > ox) min[0] = ox;
    if (min[1] > oy) min[1] = oy;
    if (min[2] > oz) min[2] = oz;
    if (max[0] < ox) max[0] = ox;
    if (max[1] < oy) max[1] = oy;
    if (max[2] < oz) max[2] = oz;
  }

  /* free the call list if one already exists */
  if (CallNum != 0) glDeleteLists(CallNum, 1);

  /* create the call list */
  CallNum = glGenLists(1);
  glNewList(CallNum, GL_COMPILE_AND_EXECUTE);

  /* set the attributes about vertex ordering */
  if (ccw) glFrontFace(GL_CCW);
  else     glFrontFace(GL_CW);

  /* set the attributes about solid */
  if (solid) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  else       glPolygonMode(GL_FRONT, GL_FILL);

  /* for now do not cull */
  glDisable(GL_CULL_FACE);

  /* store the current transformation */
  glPushMatrix();

  /* we are going to draw each extrusion as a trinagle connecting the
  ** current row to the next one.*/  
  for (int i = 0; i < (spine.num() - 1); i++) {

    /* draw each transformed point of the cross section */
    glBegin(GL_TRIANGLES);
    for (int j = 0; j < (crossSection.num() - 1); j += 1) {

      /* store the top point of the triangle and the normal at each point */
      int idx = i * crossSection.num() + j;
      glNormal3f(normal[idx][0], normal[idx][1], normal[idx][2]);
      glVertex3f(coord[idx][0], coord[idx][1], coord[idx][2]);		 

      idx = i * crossSection.num() + j + 1;
      glNormal3f(normal[idx][0], normal[idx][1], normal[idx][2]);
      glVertex3f(coord[idx][0], coord[idx][1], coord[idx][2]);		 

      idx = (i + 1) * crossSection.num() + j;
      glNormal3f(normal[idx][0], normal[idx][1], normal[idx][2]);
      glVertex3f(coord[idx][0], coord[idx][1], coord[idx][2]);		 

      /* store the other triangle and the normal at each point */
      idx = i * crossSection.num() + j + 1;
      glNormal3f(normal[idx][0], normal[idx][1], normal[idx][2]);
      glVertex3f(coord[idx][0], coord[idx][1], coord[idx][2]);		 

      idx = (i + 1) * crossSection.num() + j + 1;
      glNormal3f(normal[idx][0], normal[idx][1], normal[idx][2]);
      glVertex3f(coord[idx][0], coord[idx][1], coord[idx][2]);		 

      idx = (i + 1) * crossSection.num() + j;
      glNormal3f(normal[idx][0], normal[idx][1], normal[idx][2]);
      glVertex3f(coord[idx][0], coord[idx][1], coord[idx][2]);		 

    }

    /* finish up drawing */
    glEnd();

  }

  /* restore the matrix stack */
  glPopMatrix();

  /* finish up the list */
  glEndList();

}
/*************************************************************************/

/*************************************************************************/
void Extrusion::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Extrusion::eventsIn (" << DEF << ")\n";
#endif 

  /* Determine what action to take based on incoming event */
  if ((eventType == "spine")    || (eventType == "set_spine")) {
    MFVec3f *n = (MFVec3f *) eventData;
    spine = *n;

  } else if ((eventType == "crossSection") || (eventType == "set_crossSection")) {
    MFVec2f *n = (MFVec2f *) eventData;
    crossSection = *n;

  } else if ((eventType == "scale") || (eventType == "set_scale")) {
    MFVec2f *n = (MFVec2f *) eventData;
    scale = *n;

  } else if ((eventType == "orientation")|| (eventType == "set_orientation")) {
    MFRotation *n = (MFRotation *) eventData;
    orientation = *n;

  }

  /* re-generate the call list */
  SFVec3f min, max;
  prepare(min, max);

}
/*************************************************************************/

/*************************************************************************/
void Extrusion::render(SFRenderInfo &)
{
    
#ifdef DEBUG_RENDER
  cout << "\nEntered Extrusion::render()\n";  
#endif

  /* call the render function */
  glCallList(CallNum);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif
 
}
/*************************************************************************/

/*************************************************************************/
void Extrusion::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);
    
#ifdef DEBUG_PARSE    
    cout << "\tExtrusion: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "spine") {
      spine.parse(header, InFile);

    } else if (Token == "crossSection") {
      crossSection.parse(header, InFile);

    } else if (Token == "scale") {
      scale.parse(header, InFile);

    } else if (Token == "orientation") {
      orientation.parse(header, InFile);

    } else if (Token == "beginCap") {
      beginCap.parse(header, InFile);

    } else if (Token == "endCap") {
      endCap.parse(header, InFile);

    } else if (Token == "ccw") {
      ccw.parse(header, InFile);

    } else if (Token == "solid") {
      solid.parse(header, InFile);

    } else if (Token == "convex") {
      convex.parse(header, InFile);

    } else if (Token == "creaseAngle") {
      creaseAngle.parse(header, InFile);

    } else if (Token == "Extrusion") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/



