/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <SFString.h>
#include <SFEvent.h>
#include <TimeSensor.h>
#include <Children.h>

/*************************************************************************/
void TimeSensor::eventIn (SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered TimeSensor::eventsIn (" << DEF << ")\n";
#endif 

  /* Determine what action to take based on incoming event */
  if        ((eventType == "cycleInterval")    || (eventType == "set_cycleInterval")) {
    SFTime *n = (SFTime *) eventData;
    cycleInterval = *n;

  } else if ((eventType == "enabled")    || (eventType == "set_enabled")) {
    SFBool *n = (SFBool *) eventData;
    enabled = *n;

  } else if ((eventType == "loop")    || (eventType == "set_loop")) {
    SFBool *n = (SFBool *) eventData;
    loop = *n;

  } else if ((eventType == "startTime")    || (eventType == "set_startTime")) {
    SFTime *n = (SFTime *) eventData;
    startTime = *n;

  } else if ((eventType == "stopTime")    || (eventType == "set_stopTime")) {
    SFTime *n = (SFTime *) eventData;
    stopTime = *n;

  }

  /* update the time */
  time += 0.5;

  /* if may be the case that the timer is not enabled */
  if (enabled == false) return;

  /* determine if the time is in the correct interval */
  if ((time > stopTime) && (! loop)) return;

  /* determine if the time is in the correct interval */
  if (time < startTime) return;

  /* generate the fraction changed events */
  fractionChanged = fmod(time - startTime, cycleInterval) / cycleInterval;

  /* cascade */
  SFString out = "set_fraction";
  eventOut(out, &(fractionChanged));

  /* update the time */
  time += 0.1;

}
/*************************************************************************/

/*************************************************************************/
void TimeSensor::eventOut(SFString &, SFNode *)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered TimeSensor::eventOut (" << DEF << ")\n";
#endif 

  /* if no events need to be generated we are done */
  if (eventQueue() == (Children *) 0) return;

  /* generate any appropriate cascading events */
  for (Children *C = eventQueue(); C != (Children *) 0; C = C->next()) {

    /* do not process incomplete nodes */
    if (C->data() == (SFNode *) 0) continue;

    /* determine the event */
    SFEvent *event = (SFEvent *) C->data();

    /* determine the appropriate event to generate */
    if (event->to() == (SFNode *) 0) continue;

    /* generate the event */
    if (event->eventFromName() == "fraction_changed") {
      event->to()->eventIn(event->eventToName(), &(fractionChanged));
    }
  }
  
}
/*************************************************************************/

/*************************************************************************/
void TimeSensor::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tTimeSensor: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "cycleInterval") {
      cycleInterval.parse(header, InFile);

    } else if (Token == "enabled") {
      enabled.parse(header, InFile);

    } else if (Token == "loop") {
      loop.parse(header, InFile);

    } else if (Token == "startTime") {
      startTime.parse(header, InFile);

    } else if (Token == "stopTime") {
      stopTime.parse(header, InFile);

    } else if (Token == "TimeSensor") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/

