/* 
 * kmolcalc.h
 *
 * Copyright (C) 1999 Tomislav Gountchev <tomi@socrates.berkeley.edu>
 */


#ifndef KMOLCALC_H
#define KMOLCALC_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif 

#include "kmolelements.h"
#include <qdict.h>
#include <qlist.h>
#include <qstring.h>
#include <qfile.h>
#include <qtextstream.h>
#include <kapp.h>

/**
 * KMOLCALC is the calculation engine. It knows about a hashtable of user defined atomic
 * weights and group definitions ELSTABLE, and the currently processed formula, stored
 * as a list of elements and their coefficients, ELEMENTS. 
 */
class KMolCalc {

public:

  KMolCalc();
  ~KMolCalc();
  
  /** 
   * Parse a string S and construct the ElementList ELEMENTS, representing the composition
   * of S. Returns 0 if successful, or an error message if parsing failed.
   * The elements is S must be valid element or group symbols, as stored in ELSTABLE.
   * See help files for correct formula syntax.
   */
  QString* readFormula(QString s);
  
  /**
   * Calculate and return the molecular weight of the current chemical formula.
   */
  double getWeight();
  
  /**
   * Return the elemental composition of the current formula, as a string of tab-separated
   * element - percentage pairs, separated by newlines.
   */
  QString getEA();


  /**
   * Return the empirical formula of the current compound as a QString.
   */
  QString getEmpFormula();

  /**
   * Add a new element name - atomic weight record to the ELSTABLE hashtable. 
   * Assumes NAME has valid syntax.
   */
  void defineElement(QString name, double weight);

  /**
   * Add a new group definition to the ELSTABLE. Returns 0 if OK, error message
   * if parsing FORMULA fails. Assumes the syntax of NAME is correct.
   */
  QString* defineGroup(QString name, QString formula);

  /**
   * Remove a group or element definition from ELSTABLE.
   */
  void undefineGroup(QString name);

  /**
   * Save the element definitions file.
   */
  void writeElstable();

  /**
   * Read the element definitions file and construct ELSTABLE.
   */
  void readElstable();

  /**
   * The element dictionary.
   */
  QDict<SubUnit>* elstable;

  QString mwfile;

 private:
  double weight;

  QString* readGroup(QString s, ElementList* els);
  void readMwfile(QString mwfile);
  ElementList* elements;
};
    
#endif





