// KreateCD - CD recording software for the K desktop environment
//
// 1999 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "AudioOptions.h"
#include "CDTrack.h"
#include "AudioFile.h"

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>
#include <kslider.h>
#include <kiconloader.h>


#include <qwindowdefs.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qlabel.h>

#include <strings.h>
#include <stdio.h>

extern KLocale *locale;

AudioOptions::AudioOptions( QWidget *parent, const char *name,CDTrack *track):QWidget(parent,name)
 {
  KConfig *config;
  QRect DefaultRect(100,50,500,300);
  QVBoxLayout *mainlayout;
  QHBoxLayout *controllayout;
  QLabel *ql1,*ql2;
  QPushButton *okbutton;

  ThisTrack=track;
  ThisAudio=track->getAudioFile();
  absSize=ThisAudio->getAbsoluteDuration();

  this->setCaption(locale->translate("KreateCD - audio options"));

  config=kapp->getConfig();

  mainlayout=new QVBoxLayout(this);

  mainlayout->addStretch(1);
  PlaySlider=new KSlider(0,absSize,60*75,0,KSlider::Horizontal,this,0);
  PlaySlider->setSteps(5*75,60*75);
  PlaySlider->setTracking(TRUE);
   {
    QSize qs;
    qs=PlaySlider->sizeHint();
    qs.setHeight(qs.height()+4);
    PlaySlider->setFixedHeight(qs.height());
   }
  mainlayout->addWidget(PlaySlider,1);
  mainlayout->addStretch(1);

  controllayout=new QHBoxLayout();
  mainlayout->addLayout(controllayout,1);
  mainlayout->addStretch(1);


  controllayout->addStretch(3);
  SetStartButton=new QPushButton(this);
  SetStartButton->setPixmap(Icon("set_start.xpm"));
  SetStartButton->setFixedSize(SetStartButton->sizeHint());
  controllayout->addWidget(SetStartButton,1,AlignCenter);

  controllayout->addStretch(2);
  BackwardButton=new QPushButton(this);
  BackwardButton->setPixmap(Icon("backward.xpm"));
  BackwardButton->setFixedSize(BackwardButton->sizeHint());
  controllayout->addWidget(BackwardButton,1,AlignCenter);

  controllayout->addStretch(1);
  PlayBeginButton=new QPushButton(this);
  PlayBeginButton->setPixmap(Icon("play_begin.xpm"));
  PlayBeginButton->setFixedSize(PlayBeginButton->sizeHint());
  controllayout->addWidget(PlayBeginButton,1,AlignCenter);

  controllayout->addStretch(1);
  PlayButton=new QPushButton(this);
  PlayButton->setPixmap(Icon("play.xpm"));
  PlayButton->setFixedSize(PlayButton->sizeHint());
  controllayout->addWidget(PlayButton,1,AlignCenter);

  controllayout->addStretch(1);
  StopButton=new QPushButton(this);
  StopButton->setPixmap(Icon("stop.xpm"));
  StopButton->setFixedSize(StopButton->sizeHint());
  controllayout->addWidget(StopButton,1,AlignCenter);

  controllayout->addStretch(1);
  ForwardButton=new QPushButton(this);
  ForwardButton->setPixmap(Icon("forward.xpm"));
  ForwardButton->setFixedSize(ForwardButton->sizeHint());
  controllayout->addWidget(ForwardButton,1,AlignCenter);
  controllayout->addStretch(2);

  SetStopButton=new QPushButton(this);
  SetStopButton->setPixmap(Icon("set_stop.xpm"));
  SetStopButton->setFixedSize(SetStopButton->sizeHint());
  controllayout->addWidget(SetStopButton,1);
  controllayout->addStretch(3);

  mainlayout->addStretch(1);
  ql1=new QLabel(this);
  ql1->setPixmap(Icon("balance.xpm"));
  ql1->setFixedSize(ql1->sizeHint());
  mainlayout->addWidget(ql1,1,AlignCenter);
  BalanceSlider=new QSlider(0,200,10,0,QSlider::Horizontal,this,0);
  BalanceSlider->setTracking(TRUE);
   {
    QSize qs;
    qs=BalanceSlider->sizeHint();
    qs.setHeight(qs.height());
    BalanceSlider->setFixedHeight(qs.height());
   }
  mainlayout->addWidget(BalanceSlider,1);
  mainlayout->addStretch(1);

  ql2=new QLabel(this);
  ql2->setPixmap(Icon("volume.xpm"));
  ql2->setFixedSize(ql2->sizeHint());
  mainlayout->addWidget(ql2,1,AlignCenter);
  VolumeSlider=new QSlider(0,1000,2,0,QSlider::Horizontal,this,0);
  VolumeSlider->setTracking(TRUE);
   {
    QSize qs;
    qs=VolumeSlider->sizeHint();
    qs.setHeight(qs.height());
    VolumeSlider->setFixedHeight(qs.height());
   }
  mainlayout->addWidget(VolumeSlider,1);
  mainlayout->addStretch(1);
  okbutton=new QPushButton(locale->translate("OK"),this);
  okbutton->setFixedSize(okbutton->sizeHint());
  mainlayout->addWidget(okbutton,1);

  BalanceSlider->setValue((int)((ThisAudio->getBalance()*100)+100));
  VolumeSlider->setValue((int)((ThisAudio->getBoost()*100)));

  connect(PlayButton,SIGNAL(clicked()),this,SLOT(doPlay()));
  connect(PlayBeginButton,SIGNAL(clicked()),this,SLOT(doPlayBegin()));
  connect(StopButton,SIGNAL(clicked()),ThisAudio,SLOT(stopAudio()));
  connect(ForwardButton,SIGNAL(clicked()),this,SLOT(doForward()));
  connect(BackwardButton,SIGNAL(clicked()),this,SLOT(doBackward()));
  connect(SetStartButton,SIGNAL(clicked()),this,SLOT(setStart()));
  connect(SetStopButton,SIGNAL(clicked()),this,SLOT(setStop()));
  connect(ThisAudio,SIGNAL(audioPlayPosition(int)),
          this,SLOT(positionChanged(int)));
  connect(PlaySlider,SIGNAL(valueChanged(int)),
          this,SLOT(sliderMove(int)));
  connect(PlaySlider,SIGNAL(sliderPressed(void)),
          this,SLOT(sliderPress(void)));
  connect(PlaySlider,SIGNAL(sliderReleased(void)),
          this,SLOT(sliderRelease(void)));
  connect(BalanceSlider,SIGNAL(valueChanged(int)),
          this,SLOT(changeBalance(int)));
  connect(VolumeSlider,SIGNAL(valueChanged(int)),
          this,SLOT(changeBoost(int)));
  connect(okbutton,SIGNAL(clicked()),this,SLOT(hide()));


  isPressed=0;
  lastPosition=0;

  mainlayout->activate();

  config->setGroup("Geometry");
  this->setGeometry(config->readRectEntry("AudioOptWindowGeo",&DefaultRect));
 }


AudioOptions::~AudioOptions(void)
 {
  KConfig *config;
  QRect SaveRect(this->pos(),this->size());
  config=kapp->getConfig();
  config->setGroup("Geometry");
  config->writeEntry("AudioOptWindowGeo",SaveRect);
  ThisAudio->stopAudio();
 }

void AudioOptions::hide()
 {
  QWidget::hide();
  emit closed();
 }

void AudioOptions::positionChanged(int frame)
 {
  if (isPressed) return;
  if (frame>absSize) return;
  lastPosition=frame;
  PlaySlider->setValue(frame);
 }

void AudioOptions::sliderPress(void)
 {
  isPressed=1;
 }

void AudioOptions::sliderRelease(void)
 {
  isPressed=0;
  ThisAudio->setAudioPosition(lastPosition);
 }


void AudioOptions::sliderMove(int frame)
 {
  lastPosition=frame;
 }

void AudioOptions::doPlay(void)
 {
  if (lastPosition==absSize) lastPosition=-1;
  ThisAudio->playAudio(lastPosition);
 }

void AudioOptions::doPlayBegin(void)
 {
  ThisAudio->playAudio(-1);
 }

void AudioOptions::doForward(void)
 {
  if ((lastPosition+15*75)>=absSize) return;
  lastPosition+=15*75;
  PlaySlider->setValue(lastPosition);
  ThisAudio->setAudioPosition(lastPosition);
 }

void AudioOptions::doBackward(void)
 {
  if ((lastPosition<15*75))
   {
    lastPosition=0;
   }
   else
   {
    lastPosition-=15*75;
   }
  PlaySlider->setValue(lastPosition);
  ThisAudio->setAudioPosition(lastPosition);
 }

void AudioOptions::changeBalance(int balance)
 {
  float rbalance;
  rbalance=(float)(balance-100)/100;
  ThisAudio->setAudioBalance(rbalance);
  ThisAudio->setBalance(rbalance);
  emit(needUpdate());
 }

void AudioOptions::changeBoost(int boost)
 {
  float rboost;
  rboost=(float) boost/100;
  ThisAudio->setAudioBoost(rboost);
  ThisAudio->setBoost(rboost);
  emit(needUpdate());
 }


void AudioOptions::setStart(void)
 {
  ThisAudio->setStartPos(lastPosition*(44100/75));
  emit(needUpdate());
 }

void AudioOptions::setStop(void)
 {
  ThisAudio->setEndPos(lastPosition*(44100/75));
  emit(needUpdate());
 }