// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "IsoImage.h"
#include "IsoFile.h"
#include "ProgressDialog.h"
#include "uidcontrol.h"
#include "pathconfig.h"

#include <kapp.h>
#include <kconfig.h>
#include <kprocess.h>
#include <klocale.h>


#include <qmessagebox.h>
#include <qstring.h>
#include <qstrlist.h>

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>

#define OPTION_DEFAULTS 4

extern KLocale *locale;

static const bool option_defaults[OPTION_DEFAULTS][10]=
      {
      /*otdot ndeep long  leado nover  rr   anorr trans joli  all*/
       {false,false,true ,true ,false,false,true ,false,false,true }, /* UNIX RockRidge*/
       {false,true ,true ,true ,false,false,true ,false,true ,true }, /* UNIX + WIN */
       {false,true ,true ,true ,false,false,false,false,true ,true }, /* WIN */
       {false,false,false,false,false,false,false,false,false,true }  /* MS-DOS*/
      };



ISOImage::ISOImage(void)
 {
  KConfig *config;
  QString qs("/");
  rootObject=new ISOFile(ISOFile::ISO_ISODir,&qs);
  fileTree=0;

#if QT_VERSION >= 200
  config=kapp->config();
#else
  config=kapp->getConfig();
#endif
  config->setGroup("ISOOptions");
  loadConfig(config);
 }

void ISOImage::loadConfig(KConfig *config)
 {
  QString ts;
  ts=config->readEntry("ApplicationID","KreateCD");
  strcpy(applicationID,ts.data());
  ts=config->readEntry("PublisherID","No publisher");
  strcpy(publisherID,ts.data());
  ts=config->readEntry("PreparerID","No preparer");
  strcpy(preparerID,ts.data());
  ts=config->readEntry("VolumeID","UNNAMED");
  strcpy(volumeID,ts.data());
  ts=config->readEntry("SystemID","LINUX");
  strcpy(systemID,ts.data());

  omitTrailingDot=config->readBoolEntry("NoTrailingDots",false);
  noDeepRelocation=config->readBoolEntry("NoDeepRelocation",false);
  longISONames=config->readBoolEntry("LongISONames",true);
  allowLeadingDot=config->readBoolEntry("LeadingDots",true);
  omitVersions=config->readBoolEntry("OmitVersions",false);
  withRockRidge=config->readBoolEntry("RockRidge",false);
  withAnonymousRockRidge=config->readBoolEntry("AnonymousRockRidge",true);
  makeTransTab=config->readBoolEntry("TransTab",false);
  withJoliet=config->readBoolEntry("Joliet",false);
  allFiles=config->readBoolEntry("AllFiles",true);
 }

void ISOImage::saveConfig(KConfig *config)
 {
  config->writeEntry("ApplicationID",applicationID);
  config->writeEntry("PublisherID",publisherID);
  config->writeEntry("PreparerID",preparerID);
  config->writeEntry("VolumeID",volumeID);
  config->writeEntry("SystemID",systemID);

  config->writeEntry("NoTrailingDots",omitTrailingDot);
  config->writeEntry("NoDeepRelocation",noDeepRelocation);
  config->writeEntry("LongISONames",longISONames);
  config->writeEntry("LeadingDots",allowLeadingDot);
  config->writeEntry("OmitVersions",omitVersions);
  config->writeEntry("RockRidge",withRockRidge);
  config->writeEntry("AnonymousRockRidge",withAnonymousRockRidge);
  config->writeEntry("TransTab",makeTransTab);
  config->writeEntry("Joliet",withJoliet);
  config->writeEntry("AllFiles",allFiles);
 }

ISOImage::~ISOImage(void)
 {
  if (fileTree!=0) clearImageTree();
  delete(rootObject);
 }


ISOFile *ISOImage::imageRoot(void)
 {
  return(rootObject);
 }

void ISOImage::setImageRoot(ISOFile *root)
 {
  if (fileTree!=0) clearImageTree();
  delete(rootObject);
  rootObject=new ISOFile(*root); 
 }

bool ISOImage::buildImageTree(void)
 {
  KConfig *config;
  QString tempdir;
  int i=0,r;
  char numbersuf[16];

  if (fileTree!=0) clearImageTree();

#if QT_VERSION >= 200
  config=kapp->config();
#else
  config=kapp->getConfig();
#endif
  config->setGroup("Path");

  while (i<500)
   {
    r=rand()%1000;
    sprintf(numbersuf,"/isotree_%d",r);
    tempdir=config->readEntry("PathTemporary","/tmp");
    tempdir+=numbersuf;
    if (mkdir(tempdir.data(),0700)==0) break;
    ++i;
   }
   if (i==500)
    {
     return(false);
    }

  if (!rootObject->createTree(tempdir.data()))
   {
    rootObject->deleteTree(tempdir.data());
    rmdir(tempdir.data());
    return(false);
   }
  fileTree=new QString(tempdir.data());
  return(true);
 }

bool ISOImage::clearImageTree(void)
 {
  if (fileTree==0) return(true);
  if (!rootObject->deleteTree(fileTree->data()))
   {
    delete(fileTree);
    fileTree=0;
    return(false);
   }

  if (rmdir(fileTree->data())!=0)
   {
    delete(fileTree);
    fileTree=0;
    return(false);
   }

  delete(fileTree);
  fileTree=0;
  return(true);
 }

void ISOImage::setOptionsDefault(ISOImage::ISOType type)
 {
  int itype;
  if (type==CustomType) return;
  itype=type-1;
  omitTrailingDot=option_defaults[itype][0];
  noDeepRelocation=option_defaults[itype][1];
  longISONames=option_defaults[itype][2];
  allowLeadingDot=option_defaults[itype][3];
  omitVersions=option_defaults[itype][4];
  withRockRidge=option_defaults[itype][5];
  withAnonymousRockRidge=option_defaults[itype][6];
  makeTransTab=option_defaults[itype][7];
  withJoliet=option_defaults[itype][8];
  allFiles=option_defaults[itype][9];
  emit(imageChanged());
 }

ISOImage::ISOType ISOImage::getOptionsDefault(void)
 {
  int itype;
  for (itype=0;itype<OPTION_DEFAULTS;++itype)
   {
    if (omitTrailingDot!=option_defaults[itype][0]) continue;
    if (noDeepRelocation!=option_defaults[itype][1]) continue;
    if (longISONames!=option_defaults[itype][2]) continue;
    if (allowLeadingDot!=option_defaults[itype][3]) continue;
    if (omitVersions!=option_defaults[itype][4]) continue;
    if (withRockRidge!=option_defaults[itype][5]) continue;
    if (withAnonymousRockRidge!=option_defaults[itype][6]) continue;
    if (makeTransTab!=option_defaults[itype][7]) continue;
    if (withJoliet!=option_defaults[itype][8]) continue;
    if (allFiles!=option_defaults[itype][9]) continue;
    return( (ISOImage::ISOType) (itype+1) );
   }
  return(CustomType);
 }

void ISOImage::setOmitTrailingDot(bool b)
 {
  omitTrailingDot=b;
  emit(imageChanged());
 }

void ISOImage::setNoDeepRelocation(bool b)
 {
  noDeepRelocation=b;
  emit(imageChanged());
 }

void ISOImage::setLongISONames(bool b)
 {
  longISONames=b;
  emit(imageChanged());
 }

void ISOImage::setAllowLeadingDot(bool b)
 {
  allowLeadingDot=b;
  emit(imageChanged());
 }

void ISOImage::setOmitVersions(bool b)
 {
  omitVersions=b;
  emit(imageChanged());
 }

void ISOImage::setWithRockRidge(bool b)
 {
  withRockRidge=b;
  if ( withRockRidge && withAnonymousRockRidge) withAnonymousRockRidge=false;
  emit(imageChanged());
 }

void ISOImage::setWithAnonymousRockRidge(bool b)
 {
  withAnonymousRockRidge=b;
  if ( withRockRidge && withAnonymousRockRidge) withRockRidge=false;
  emit(imageChanged());
 }

void ISOImage::setMakeTransTab(bool b)
 {
  makeTransTab=b;
  emit(imageChanged());
 }

void ISOImage::setWithJoliet(bool b)
 {
  withJoliet=b;
  emit(imageChanged());
 }

void ISOImage::setAllFiles(bool b)
 {
  allFiles=b;
  emit(imageChanged());
 }

static void scopy(char *dest,const char *source,int i)
 {
  while ( (i>0) && (*source!=0) )
   {
    *(dest++)=*(source++);
    i--;
   }
  *dest='\0';
 }

void ISOImage::setApplicationID(const QString &id)
 {
  scopy(applicationID,id.data(),128);
  emit(imageChanged());
 }

void ISOImage::setPreparerID(const QString &id)
 {
  scopy(preparerID,id.data(),128);
  emit(imageChanged());
 }

void ISOImage::setPublisherID(const QString &id)
 {
  scopy(publisherID,id.data(),128);
  emit(imageChanged());
 }

void ISOImage::setVolumeID(const QString &id)
 {
  scopy(volumeID,id.data(),32);
  emit(imageChanged());
 }

void ISOImage::setSystemID(const QString &id)
 {
  scopy(systemID,id.data(),32);
  emit(imageChanged());
 }

void ISOImage::setApplicationID(const char *id)
 {
  scopy(applicationID,id,128);
  emit(imageChanged());
 }

void ISOImage::setPreparerID(const char *id)
 {
  scopy(preparerID,id,128);
  emit(imageChanged());
 }

void ISOImage::setPublisherID(const char *id)
 {
  scopy(publisherID,id,128);
  emit(imageChanged());
 }

void ISOImage::setVolumeID(const char *id)
 {
  scopy(volumeID,id,32);
  emit(imageChanged());
 }

void ISOImage::setSystemID(const char *id)
 {
  scopy(systemID,id,32);
  emit(imageChanged());
 }

bool ISOImage::getOmitTrailingDot(void)
 {
  return(omitTrailingDot);
 }

bool ISOImage::getNoDeepRelocation(void)
 {
  return(noDeepRelocation);
 }

bool ISOImage::getLongISONames(void)
 {
  return(longISONames);
 }

bool ISOImage::getAllowLeadingDot(void)
 {
  return(allowLeadingDot);
 }

bool ISOImage::getOmitVersions(void)
 {
  return(omitVersions);
 }

bool ISOImage::getWithRockRidge(void)
 {
  return(withRockRidge);
 }

bool ISOImage::getWithAnonymousRockRidge(void)
 {
  return(withAnonymousRockRidge);
 }

bool ISOImage::getMakeTransTab(void)
 {
  return(makeTransTab);
 }

bool ISOImage::getWithJoliet(void)
 {
  return(withJoliet);
 }

bool ISOImage::getAllFiles(void)
 {
  return(allFiles);
 }

const char *ISOImage::getApplicationID(void)
 {
  return(applicationID);
 }

const char *ISOImage::getPreparerID(void)
 {
  return(preparerID);
 }

const char *ISOImage::getPublisherID(void)
 {
  return(publisherID);
 }

const char *ISOImage::getSystemID(void)
 {
  return(systemID);
 }

const char *ISOImage::getVolumeID(void)
 {
  return(volumeID);
 }


bool ISOImage::makeImage(const char *filename)
 {
  int retval;
  KConfig *config;


  prepareProcess();
#if QT_VERSION >= 200
  config=kapp->config();
#else
  config=kapp->getConfig();
#endif
  config->setGroup("Path");
  if ROOT_CALLED
   {
    *this<<config->readEntry("PathMkisofs",PATH_MKISOFS);
   }
   else
   {
    *this<<PATH_MKISOFS;
   }
  
  if (allFiles) *this<<"-no-bak";
  if (omitTrailingDot) *this<<"-d";
  if (noDeepRelocation) *this<<"-D";
  if (longISONames) *this<<"-l";
  if (allowLeadingDot) *this<<"-L";
  if (omitVersions) *this<<"-N";
  if (withRockRidge) *this<<"-R";
  if (withAnonymousRockRidge) *this<<"-r";
  if (makeTransTab) *this<<"-T";
  if (withJoliet) *this<<"-J";

  *this<<"-o"<<filename;
  *this<<"-f"; // always follow symbolic links

  if (applicationID[0]!=0) *this<<"-A"<<applicationID;
  if (publisherID[0]!=0) *this<<"-P"<<publisherID;
  if (preparerID[0]!=0) *this<<"-p"<<preparerID;
  if (volumeID[0]!=0) *this<<"-V"<<volumeID;
  if (systemID[0]!=0) *this<<"-sysid"<<systemID;


  setWorkText(locale->translate("Creating directory tree..."));
  showDialog();

  if (!buildImageTree())
   {
#if QT_VERSION >= 200
    QMessageBox::critical(0,QString::null,locale->translate("Unable to build directory tree!"));
#else
    QMessageBox::critical(0,0,locale->translate("Unable to build directory tree!"));
#endif
    closeProcess();
    return(false);
   }
  setWorkText(locale->translate("Creating ISO 9660 image..."));
  *this<<(fileTree->data());
  isoflag=0;
  retval=startProcess(false);   // 1=ripping was OK  -1=abort -2=ripping error  0=window hard closed

  setWorkText(locale->translate("Cleaning up directory tree..."));
  if (!clearImageTree())
   {
#if QT_VERSION >= 200
    QMessageBox::critical(0,QString::null,locale->translate("Unable to delete directory tree!"));
#else
    QMessageBox::critical(0,0,locale->translate("Unable to delete directory tree!"));
#endif
   }

  if (retval!=1) remove(filename);
  if (retval==-2)
   {
#if QT_VERSION >= 200
    QMessageBox::critical(0,QString::null,locale->translate("Unable to create ISO image!"));
#else
    QMessageBox::critical(0,0,locale->translate("Unable to create ISO image!"));
#endif
   }

  closeProcess();

  if (retval!=1) return(false);

  return(true);
 }

int ISOImage::processExited(void)
 {
  return(isoflag?1:-2);
 }

bool ISOImage::processStderrLine(char *linebuffer)
 {
   if (strstr(linebuffer,"done, estimate finish")!=0)
    {
     unsigned long fin;
     fin=strtoul(linebuffer,0,10);
     setProgress(fin,100,true);
     return(true);
    }

   if (strstr(linebuffer,"extents written")!=0)
    {
     isoflag=1;
     return(true);
    }
  return(true);
 }

void ISOImage::saveImage(KConfig *config)
 {
  saveConfig(config);
  initConfigPutter(config);
  putISOFile(rootObject);  
 }

ISOImage::ISOImage(KConfig *config)
 {
  char entryName[40];
  int index;
  ISOFile *cparent=0;
  QStrList xlist;

  xlist.setAutoDelete(true);

  rootObject=0;
  fileTree=0;

  loadConfig(config);
  index=0;
  
  while ( (index==0) || (cparent!=0) )
   {
    int narg;
    char type;
    QString qs1,qs2;
    char *tp;
    sprintf(entryName,"ISOTree%d",index);
    narg=config->readListEntry(entryName,xlist,'*');
    if (narg==0)
     {
      if (index==0) 
       {
        QString qs("/");
        rootObject=new ISOFile(ISOFile::ISO_ISODir,&qs);
       }
      break;
     }
    type=xlist.first()[0];
    tp=xlist.next();
    qs1=QString(tp);
    tp=xlist.next();
    qs2=QString(tp);
    switch (type)
     {
      case 'I':
       {
        ISOFile *newfile;
        newfile=new ISOFile(ISOFile::ISO_ISODir,&qs1);
        if (cparent==0)
         {
          rootObject=newfile;
         }
         else
         {
          cparent->addObject(newfile);
         }
        cparent=newfile;
        break;
       }
      case 'D':
       {
        ISOFile *newfile;
        newfile=new ISOFile(ISOFile::ISO_RealDir,&qs1,&qs2);
        cparent->addObject(newfile);
        break;
       }
      case 'F':
       {
        ISOFile *newfile;
        newfile=new ISOFile(ISOFile::ISO_RealFile,&qs1,&qs2);
        cparent->addObject(newfile);
        break;
       }
      case 'X':
       {
        cparent=cparent->getParent();
       }
     }
    index++;
   }
 }

void ISOImage::initConfigPutter(class KConfig *config)
 {
  cputConfig=config;
  cputIndex=0;
 }

void ISOImage::putConfigObject(const char *string1,const QString *string2,const QString *string3)
 { 
  QStrList slist=QStrList();
  char entryName[40];
  
  slist.append(string1);
  if (string2!=0) slist.append(string2->data());
  if (string3!=0) slist.append(string3->data());
  slist.setAutoDelete(true);
  sprintf(entryName,"ISOTree%d",cputIndex);
  cputConfig->writeEntry(entryName,slist,'*');
  cputIndex++;
 }

void ISOImage::putISOFile(ISOFile *ifil)
 {
  char tstr[2];
  ISOFile::ISOType type;
  type=ifil->type();
  tstr[1]=0;
  switch (type)
   {
    case ISOFile::ISO_RealFile:
      tstr[0]='F';
      break;
    case ISOFile::ISO_RealDir:
      tstr[0]='D';
      break;
    case ISOFile::ISO_ISODir:
      tstr[0]='I';
      break;
   }
  putConfigObject((const char *) tstr,(const QString*) ifil->name(),(const QString *) ifil->reference());
  if (type==ISOFile::ISO_ISODir)
   {
    ISOFile *walker=0;
    while ( (walker=ifil->getChildren(walker))!=0)
     {
      putISOFile(walker);
     }
    putConfigObject("X");
   }
 }
