	/*

	Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

		      (C) 1999 Martin Lorenz
			           lorenz@ch.tum.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "synthmodule.h"
#include <math.h>

class Synth_STD_EQUALIZER :public SynthModule
{
	// inputs:
	enum { INVALUE, LOW, MID, HIGH, FREQUENCY, Q, MAX };

	// outputs:
	enum { OUTVALUE };

	float p[MAX], oldp[MAX], a1, a2, b0, b1, b2, x_0, x_1, x_2, y_1, y_2;

	unsigned long all;
public:
	void Initialize();
	void Calculate() { assert(false); };
	void CalculateBlock(unsigned long samples);
	void CalculateParams();
	string getParams() { return("invalue,low,mid,high,frequency,q;outvalue"); }
	static void *Creator() { return new Synth_STD_EQUALIZER; };
};

ModuleClient MC_Synth_STD_EQUALIZER(SynthModule::get_MS,
						"Synth_STD_EQUALIZER",Synth_STD_EQUALIZER::Creator);

void Synth_STD_EQUALIZER::Initialize()
{
	haveCalculateBlock = true;

	int i;
	for(i=0;i<MAX;i++) { p[i]=-42.7; oldp[i]=-42.8; }
	x_0 = x_1 = x_2 = y_1 = y_2 = 0.0;

	all = 0;
}

void Synth_STD_EQUALIZER::CalculateBlock(unsigned long samples)
{
	all += samples;

	if(all > 1024)
	{
		/* The _problem_:  (observed on a PII-350)
		 *
		 * I am not quite sure what happens here, but it seems to be like that:
		 * 
		 * If an ordinary signal (a mp3 for instance) is sent through the
		 * equalizer, and then no more input is given (zeros as input),
		 * the y_1 and y_2 values oscillate for some time, coming closer and
		 * close to zero.
		 *
		 * But before the reach zero, they reach the smallest negative number
		 * (or smallest positive, or whatever), and stay there
		 * (because 0.005*smallest_negative will remain smallest_negative).
		 *
		 * Since then, the CPU usage for all operations on these floats
		 * increases, (since handling of smallest_negative seems to be a rare
		 * case).
		 *
		 * The _fix_:
		 *
		 * We observe the value of y_1. If it's very close to zero (may be as
		 * well smallest_positive/smallest_negative), we set it to zero,
		 * together with y_2. This shouldn't significantly influence
		 * correctness of the filter, but effectively solves the problem.
		 *
		 * If you don't believe me, try without this fix and tell me what
		 * happens on your computer.
		 */
		const float zero_lower =-0.00000001;
		const float zero_upper = 0.00000001;
		all = 0;

		if(zero_lower < y_1 && y_1 < zero_upper)
			y_1 = y_2 = 0.0;
	}
	p[LOW] = in[LOW][0];
	p[MID] = in[MID][0];
	p[HIGH] = in[HIGH][0];
	p[FREQUENCY] = in[FREQUENCY][0];
	p[Q] = in[Q][0];

	if(memcmp(p,oldp,sizeof(float)*MAX) != 0) CalculateParams();
	
	unsigned long i;
	float tmp;
	for(i=0;i<samples;i++)
	{
		x_0 = in[INVALUE][i];
		tmp = x_0*b0 + x_1*b1 + x_2*b2 - y_1*a1 - y_2*a2;
		x_2=x_1; x_1 = x_0; y_2 = y_1; y_1=tmp;
		out[OUTVALUE][i] = tmp;
	}
}

void Synth_STD_EQUALIZER::CalculateParams()
{
	/*
	 * LOW,MID,HIGH are in dB, transform them to
	 *
	 * -6dB => 0.5 ; 0dB => 1 ; 6dB = 2.0 ; ...
	 */

	float low = exp(p[LOW]*0.115524530093324);		// exp(p[LOW]*ln(2)/6)
	float mid = exp(p[MID]*0.115524530093324);
	float high = exp(p[HIGH]*0.115524530093324);

	// FREQUENCY is given in Hz, we need the w-value (and do clipping if
	// it exceeds SR/2)
	const float SAMPLING_RATE = 44100.0; // well, could have a better solution

	float frequency = p[FREQUENCY];
	if(frequency > SAMPLING_RATE/2.01) frequency = SAMPLING_RATE/2.01;

	float w = 2*M_PI*frequency/SAMPLING_RATE;

	// Q is given as it is used
	float q = p[Q];

	// Calculations:
	float t = 1/tan(w/2);
	float tq = t/q;
	float t2 = t*t;


	float a0=1+tq+t2;
	float a0r=1/a0;

	// and now the real filter values:
	a1=(2-2*t2)*a0r;
	a2=(1-tq+t2)*a0r;
	b0=(low+mid*tq+high*t2)*a0r;
	b1=(2*low-2*high*t2)*a0r;
	b2=(low-mid*tq+high*t2)*a0r;

	memcpy(oldp, p, sizeof(float)*MAX);
}
