/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include "timerDevice.h"
#include "../config.h"

TimerDevice::TimerDevice( QWidget *parent=0, const char *name=0 ):
  QWidget( parent, name ) {

  lcd = new LCDBitmap();
  timeString = new Buffer(10);

  setTimeMode(_TIMEMODE_PLAYED);
  setMinimumSize( sizeHint() );

  streamInfoDevice=NULL;
  timerId        = 0;
  currentSeconds = -10;
  repaint(true);
}

TimerDevice::~TimerDevice(){
  if (streamInfoDevice != 0)
    disconnect(streamInfoDevice->getEventQueue(),NULL,this,NULL);
  streamInfoDevice = NULL;
  delete lcd;
  delete timeString;
}


void TimerDevice::attachTo(AudioStreamInfoDevice* infoDevice) {
  if (streamInfoDevice != NULL) {
    debugOutput( cout << "error:  connecting to already connected AlarmTimer!"<<endl );
    exit(1);
  }
  streamInfoDevice=infoDevice;
  connect(streamInfoDevice->getEventQueue(),SIGNAL(processEvent(char)),
	  this,SLOT(playerStatusChanged(char)));
}


void TimerDevice::detach() {
  disconnect(streamInfoDevice->getEventQueue(),NULL,this,NULL);
  streamInfoDevice=NULL;
}


int  TimerDevice::getTimeMode() {  return mode_cfg; }
void TimerDevice::setTimeMode(int mode) { mode_cfg=mode; }
void TimerDevice::toggleTimeMode() { 
  if (mode_cfg == _TIMEMODE_PLAYED)
    mode_cfg = _TIMEMODE_REMAIN;
  else 
    mode_cfg = _TIMEMODE_PLAYED;
  emit repaint();
}


char* TimerDevice::getTimeString() {
  currentSeconds = streamInfoDevice->getTimeInfo()->getTotalSec();
  createTimeFormat(currentSeconds,
		   streamInfoDevice->getMusicInfo()->getLen(),
		   timeString);
  return timeString->getData();
}


void TimerDevice::createTimeFormat(int played,int len,Buffer* dest) {
  char *format="";
  int   size=dest->getSize();
  int   time=played;
  int   min;
  int   sec;

  if (mode_cfg == _TIMEMODE_REMAIN) time=len-played;

  min=time/60;
  sec=time%60;
  if (sec < 10) {
    // && (time != 0)
    if ((mode_cfg == _TIMEMODE_REMAIN) ) format="-%d:0%d"; 
    else format="%d:0%d";
  } 
  else {
    if (mode_cfg == _TIMEMODE_REMAIN) format="-%d:%d";
    else format="%d:%d";
  }
  dest->clear();
  snprintf(dest->getData(),size,format,min,sec);
}            

void TimerDevice::stopRepaintTimer(){
  if (timerId)  {
    killTimer(timerId);
    timerId = 0;
    if (mode_cfg == _TIMEMODE_REMAIN)
      repaintTime("-0:00");
    else 
      repaintTime("0:00");

    currentSeconds = -1;
  }
}

void TimerDevice::startRepaintTimer(){
  if (!timerId)  timerId = startTimer(250);
}
    
void TimerDevice::mousePressEvent ( QMouseEvent* mouseEvent) {
  if (mouseEvent->button() ==  LeftButton) {
    if (mode_cfg == _TIMEMODE_PLAYED) 
      mode_cfg = _TIMEMODE_REMAIN;
    else 
      mode_cfg = _TIMEMODE_PLAYED;
  }
  currentSeconds=-1; /* force a repaint */
  emit repaint();
}


QSize TimerDevice::sizeHint () {
  int w;
  int h;
  QPixmap* pixmap=lcd->getPixmap(_LCD_DIGIT_8);
  w=pixmap->size().width();
  h=pixmap->size().height();
  QSize size(w*4+w/2,h);
  return size;
}

void TimerDevice::playerStatusChanged(char mode){ 
  if (mode == _AUDIOSTREAMINFO_STATUS_CHANGE){
    if( streamInfoDevice->getStatusInfo()->getStatus() == _STATUS_STOPPED) 
      stopRepaintTimer();
    else 
      startRepaintTimer();
  }
}

void TimerDevice::timerEvent( QTimerEvent * ){   
  if(streamInfoDevice->getTimeInfo()->getTotalSec()!=currentSeconds)
    repaintTime(); 
}

void TimerDevice::paintEvent ( QPaintEvent * ){
  if (currentSeconds < 0){
    if (mode_cfg == _TIMEMODE_REMAIN)
      repaintTime("-0:00");
    else 
      repaintTime("0:00");
  }
  else 
    repaintTime();
}

void TimerDevice::repaintTime(){
  repaintTime(getTimeString());
}
  
void TimerDevice::repaintTime(char* time) {
  QSize size=sizeHint();
  int   i;
  int   x=size.width();
  int   y=0;
  char* timeString=time;
  char  currentBst;
  int   len=strlen(timeString);
  QPixmap* pixmapCurrent;
  QPainter paint;

  paint.begin( this );
  for(i=len-1;i>=0;i--) {
    currentBst=timeString[i];
    pixmapCurrent=lcd->getPixmap(_LCD_BLANK);
    if (currentBst == '-') pixmapCurrent=lcd->getPixmap(_LCD_MINUS);
    if (currentBst == ' ') pixmapCurrent=lcd->getPixmap(_LCD_BLANK);
    if (('0' <= currentBst) && (currentBst<= '9')) pixmapCurrent=lcd->getPixmap(_LCD_DIGIT_0+((int)currentBst-(int)'0'));
    //    if (('A' <= currentBst) && (currentBst<= 'F')) pixmapCurrent=lcd->getPixmap(_LCD_A+((int)currentBst-(int)'A'));
    if (currentBst == ':') pixmapCurrent=lcd->getPixmap(_LCD_DOTS);
    x=x-pixmapCurrent->width();
    paint.drawPixmap(x, y, *pixmapCurrent);
  }

  while(x > 0) {
    pixmapCurrent=lcd->getPixmap(_LCD_COL_RIGHT);
    x=x-pixmapCurrent->width();   
    paint.drawPixmap(x, y, *pixmapCurrent);
  }

  paint.end();  
}


void TimerDevice::resizeEvent(QResizeEvent *){
  currentSeconds = -10;
  repaintTime("0:00");
}

