/*
 *	      ConfigDLG - configuration dialog for kmikmod
 *	(C)1998, 1999 Roope Anttinen - roope.anttinen@ntc.nokia.com
 *	
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#include <qframe.h>
#include <qlayout.h>
#include <qgrpbox.h>
#include <qbuttongroup.h>
#include <qtooltip.h>
#include <qlabel.h>
#include <kapp.h>	// needed for i18n macro
#include <qstring.h>
#include "config.h"
#include "ConfigDLG.h"

ConfigDLG::ConfigDLG(QWidget *parent, const char *name)
	 :QTabDialog(parent, name)
{
	color_changed = false;	
	setOkButton(i18n("OK"));
	setCancelButton(i18n("Cancel"));
	setApplyButton(i18n("Apply"));
	setDefaultButton(i18n("Defaults"));
	addTab(soundConf(), i18n("&Sound"));
	addTab(driverFrame(), i18n("&Driver"));
	addTab(colorFrame(), i18n("&Colors"));
	addTab(aboutFrame(), i18n("&About"));
	setCaption(i18n("Configure KMikMod"));
	resize(sizeHint());
	connect (KB_Time,
		 SIGNAL(changed(const QColor &)),
		 SLOT(colorChangedSlot(const QColor &)));
	connect (KB_Info,
		 SIGNAL(changed(const QColor &)),
		 SLOT(colorChangedSlot(const QColor &)));
	connect (this,
		 SIGNAL(defaultButtonPressed()),
		 SLOT(defaults()));
}

ConfigDLG::~ConfigDLG()
{

}

QWidget *ConfigDLG::soundConf()
{
	QFrame *base = new QFrame (this);
	QButtonGroup *grpkHZ = new QButtonGroup(i18n("Frequency"), base);
	QButtonGroup *grpBits = new QButtonGroup(i18n("Sample size"), base);
	QButtonGroup *grpOptions = new QButtonGroup(i18n("Options"), base);
	
	kHZ11 = new QRadioButton ("11025Hz", grpkHZ);
	kHZ22 = new QRadioButton ("22050Hz", grpkHZ);
	kHZ44 = new QRadioButton ("44100Hz", grpkHZ);
	kHZ48 = new QRadioButton ("48000Hz", grpkHZ);

	bits8 = new QRadioButton ("8b", grpBits);
	bits16 = new QRadioButton ("16b", grpBits);
	
	stereo = new QCheckBox ("Stereo", grpOptions);
	interpolation = new QCheckBox ("Interpolate", grpOptions);
	hqmixer = new QCheckBox ("HQ Mixer", grpOptions);
	surround = new QCheckBox ("Surround", grpOptions);
	fadeout = new QCheckBox ("Fadeout", grpOptions);
	{
		int height = (kHZ48->height()-5);
		kHZ11->setFixedHeight(height);
		kHZ22->setFixedHeight(height);
		kHZ44->setFixedHeight(height);
		kHZ48->setFixedHeight(height);
		bits8->setFixedHeight(height);
		bits16->setFixedHeight(height);
		stereo->setFixedHeight(height);
		interpolation->setFixedHeight(height);
		hqmixer->setFixedHeight(height);
		surround->setFixedHeight(height);
		fadeout->setFixedHeight(height);
	}
 	grpkHZ->setMinimumHeight((4*stereo->height()));
 	grpBits->setMinimumHeight((2*stereo->height()));
 	grpOptions->setMinimumHeight((5*stereo->height()));

	use_rt = new QCheckBox (i18n("Use realtime scheduler"), base);
	use_rt->setMinimumHeight (use_rt->height());
	SP_FQ = new QSpinBox (0, 4000, 100, base);
	SP_FQ->setFixedSize(SP_FQ->sizeHint());
	
	QLabel *LBL_FQ = new QLabel(i18n("Player sleep time"), base);
	LBL_FQ->setFixedSize(LBL_FQ->sizeHint());

	QToolTip::add(grpkHZ, i18n("Right value dependent on your soundcard"));
	QToolTip::add(grpBits, i18n("Right value dependent on your soundcard"));
	QToolTip::add(grpOptions, i18n("MikMod options"));
	QToolTip::add(use_rt, i18n("This may lockup your computer combined with\n"
			      "too short sleep time. Use with care!"));
	QToolTip::add(LBL_FQ, i18n("Smaller value takes more CPU time"));

	QBoxLayout *vbox1 = new QVBoxLayout(base,10);
	QBoxLayout *hbox1 = new QHBoxLayout(5);
	QBoxLayout *vbox2 = new QVBoxLayout(grpkHZ, 10);
	QBoxLayout *vbox3 = new QVBoxLayout(grpBits, 10);
	QBoxLayout *vbox4 = new QVBoxLayout(grpOptions, 10);
	QBoxLayout *hbox3 = new QHBoxLayout(5);

	vbox1->addLayout(hbox1);
	vbox1->addWidget(use_rt);
	vbox1->addLayout(hbox3);
	vbox1->addSpacing(40);

 	hbox1->addWidget(grpkHZ);
 	hbox1->addWidget(grpBits);
 	hbox1->addWidget(grpOptions);

	vbox2->addSpacing(10);
	vbox2->addWidget(kHZ11);
	vbox2->addWidget(kHZ22);
	vbox2->addWidget(kHZ44);
	vbox2->addWidget(kHZ48);
	vbox2->addStretch();
	
	vbox3->addSpacing(10);
	vbox3->addWidget(bits8);
	vbox3->addWidget(bits16);
	vbox3->addStretch();
	
	vbox4->addSpacing(10);
	vbox4->addWidget(stereo);
	vbox4->addWidget(interpolation);
	vbox4->addWidget(hqmixer);
	vbox4->addWidget(surround);
	vbox4->addWidget(fadeout);
	vbox4->addStretch();

	hbox3->addWidget(SP_FQ);
	hbox3->addWidget(LBL_FQ);
	hbox3->addStretch();

	vbox1->activate();
	return base;
}

QWidget *ConfigDLG::aboutFrame()
{
	char version[32]="";
	snprintf (version,32, "libmikmod-%i.%i.%i", LIBMIKMOD_VERSION_MAJOR,
						    LIBMIKMOD_VERSION_MINOR,
						    LIBMIKMOD_REVISION);
	QFrame *base = new QFrame(this);
	QLabel *info = new QLabel(base);
	QBoxLayout *vbox1 = new QVBoxLayout (base);
	QString Info = "KMikMod ";
	Info += VERSION;
	Info +=	"\n\nModule player for KDE Desktop Environment\n\n"
		"(c)1998,1999 Roope Anttinen\n"
		"anttiner@myy.helia.fi\n\n"
		"French translation by Jrme Brevire\n"
		"jbreviere@mail.dotcom.fr\n"
		"German translation by Martin Baumung\n"
		"uo6a@rz.uni-karlsruhe.de\n\n"
		"playercode: ";
	Info += version;
	Info +=	"\n\nThis is free software and comes with no warranty\n"
		"See GNU General Public License for details.";
	info->setText(Info);
	info->setFixedSize(info->sizeHint());
	vbox1->addWidget(info);
	vbox1->activate();
	return base;
}

QWidget *ConfigDLG::colorFrame()
{
	QFrame *base = new QFrame(this);
	QLabel *L_time = new QLabel(i18n("Big display base color"), base);
	QLabel *L_info = new QLabel(i18n("Small leds base color"), base);
	KB_Time = new KColorButton(base);
	KB_Info = new KColorButton(base);
	L_time ->setFixedSize(L_time->sizeHint());
	L_info ->setFixedSize(L_info->sizeHint());
	KB_Time->setFixedSize(40,40);
	KB_Info->setFixedSize(40,40);
	QBoxLayout *vbox = new QVBoxLayout (base, 10);
	vbox->addWidget(L_time);
	vbox->addWidget(KB_Time);
	vbox->addWidget(L_info);
	vbox->addWidget(KB_Info);
	vbox->activate();
	return base;
}

QWidget *ConfigDLG::driverFrame()
{
	QFrame *base = new QFrame(this);
	LB_Driver = new QListBox(base);
	LB_Driver->setMinimumHeight(200);
	QBoxLayout *box = new QVBoxLayout(base, 10);
	box->addWidget(LB_Driver);
	box->activate();
	return base;
}

void ConfigDLG::setConfig (const configuration &ref)
{
	conf = ref;
	stereo->setChecked(conf.md_mode&DMODE_STEREO);
	interpolation->setChecked(conf.md_mode&DMODE_INTERP);
	hqmixer->setChecked(conf.md_mode&DMODE_HQMIXER);
	surround->setChecked(conf.md_mode&DMODE_SURROUND);
	conf.md_mode&DMODE_16BITS?bits16->setChecked(true):bits8->setChecked(true);
	switch (conf.md_mixfreq) {
		case 11025:
			kHZ11->setChecked(true);
			break;
		case 22050:
			kHZ22->setChecked(true);
			break;
		case 44100:
			kHZ44->setChecked(true);
			break;
		case 48000:
			kHZ48->setChecked(true);
			break;
	}
	fadeout->setChecked(conf.fadeout);
#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
	use_rt->setChecked(conf.use_rt);
#else
	use_rt->setChecked(false);
#endif // _POSIX_THREAD_PRIORITY_SCHEDULING
	SP_FQ->setValue(conf.sleep_time);
	LB_Driver->setSelected(conf.md_device, true);
}

void ConfigDLG::setDriverInfo(const char *ref)
{
	static int initialized = false;
	QString tmp, tmp2;

	if (initialized) return;

	LB_Driver->insertItem (" 0 Autodetect");
	tmp = ref;
	while (tmp.find('\n',0)>0)
	{
		tmp2 = tmp;
		tmp.remove(0,tmp2.find('\n',0)+1);
		tmp2.truncate(tmp2.find('\n', 0));
		LB_Driver->insertItem(tmp2);
	}
	LB_Driver->insertItem(tmp);
	initialized = true;
}

void ConfigDLG::setColors(const QColor &T, const QColor &I)
{
	KB_Time->setColor (T);
	KB_Info->setColor (I);
}

const configuration &ConfigDLG::getConfig ()
{
	if (kHZ11->isChecked()) conf.md_mixfreq = 11025;
	else if (kHZ22->isChecked()) conf.md_mixfreq = 22050;
	else if (kHZ44->isChecked()) conf.md_mixfreq = 44100;
	else if (kHZ48->isChecked()) conf.md_mixfreq = 48000;

	conf.md_mode = DMODE_SOFT_MUSIC;
	if (bits16->isChecked()) conf.md_mode 	     |= DMODE_16BITS;
	if (stereo->isChecked()) conf.md_mode 	     |= DMODE_STEREO;
	if (interpolation->isChecked()) conf.md_mode |= DMODE_INTERP;
	if (hqmixer->isChecked()) conf.md_mode	     |= DMODE_HQMIXER;
	if (surround->isChecked()) conf.md_mode	     |= DMODE_SURROUND;
	conf.fadeout = fadeout->isChecked();
	
	conf.sleep_time = SP_FQ->value();
#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
	conf.use_rt = use_rt->isChecked();
#endif
	conf.md_device = LB_Driver->currentItem();
	
	return conf;
}

void ConfigDLG::colorChangedSlot (const QColor &)
{
	color_changed = true;
}
