/*
  reads input data
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "bufferInputStream.h"



BufferInputStream::BufferInputStream(int size,int minlinSize,char* name):
                   InputStream(name) {
  ringBuffer=new SimpleRingBuffer(size,minlinSize);
  leof=false;
  this->name=name;
  bytePos=0;
  fillgrade=0;

  pthread_mutex_init(&writeInMut,NULL);
  pthread_mutex_init(&changeMut,NULL);
  pthread_cond_init(&changeCond,NULL);
}


BufferInputStream::~BufferInputStream() {
  delete ringBuffer;
}


int BufferInputStream::open(char* name) {
  leof=false;
  ringBuffer->setCanWaitForSpace(true);
  setUrl(name);
  return true;
}

void BufferInputStream::close() {
  leof=true;
  clear();
  ringBuffer->setCanWaitForSpace(false);
  setUrl(NULL);
}

int BufferInputStream::eof() {
  return leof;
}


void BufferInputStream::setEof(int leof) {
  this->leof=leof;
}


int BufferInputStream::read(char* ptr,int size) {
  int i=0;
  int n=size;
  int canCopy=n;
  char* readPtr;
  while((eof()==false) && (n > 0)) {
    canCopy=n;
    ringBuffer->getReadArea(readPtr,canCopy);
    if (canCopy <= 0){
      ringBuffer->waitForData(1);
      continue;
    }
    if (n < canCopy) {
      canCopy=n;
    }
    memcpy((char*)ptr+i,readPtr,canCopy);
    i=i+canCopy;
    n=n-canCopy;
    ringBuffer->forwardReadPtr(canCopy);
    ringBuffer->forwardLockPtr(canCopy);
  }
  lockBuffer();
  bytePos+=i;
  fillgrade-=i;
  unlockBuffer();
  return i;
}


int BufferInputStream::write(char* ptr,int elements) {
  int i=0;
  int n=elements;
  int canWrite=n;
  char* writePtr;
  
  while((eof()==false) && (n > 0)) {
    canWrite=n;
    ringBuffer->getWriteArea(writePtr,canWrite);
    if (canWrite <= 0){
      ringBuffer->waitForSpace(1);
      continue;
    }
    if (canWrite > n) {
      canWrite=n;
    }
    memcpy(writePtr,(char*)ptr+i,canWrite);
    i=i+canWrite;
    n=n-canWrite;
    ringBuffer->forwardWritePtr(canWrite);
  }
  lockBuffer();
  fillgrade+=elements;
  unlockBuffer();
  return i;
}



long BufferInputStream::getByteLength() {
  return 0;
}

int BufferInputStream::getFillgrade() {
  return ringBuffer->getFillgrade();
}


int BufferInputStream::getFreeRead() {
  return ringBuffer->getFreeRead();
}


int BufferInputStream::getFreeWrite() {
  return ringBuffer->getFreeWrite();
}



long BufferInputStream::getBytePosition() {
  return bytePos;
}

void BufferInputStream::setBytePosition(long bytePos) {
  this->bytePos=bytePos;
}


int BufferInputStream::seek(long) {
  return false;
}


void BufferInputStream::clear() {
  lockBuffer();
  ringBuffer->emptyBuffer();
  ringBuffer->exitWaitForData();
  ringBuffer->exitWaitForSpace();
  timeStampArray->clear();
  bytePos=0;
  fillgrade=0;
  unlockBuffer();
 
}
 
void BufferInputStream::insertTimeStamp(TimeStamp* stamp,int len) {
  lockBuffer();
  long key;
  key=bytePos+fillgrade;
  InputStream::insertTimeStamp(stamp,key,len);
  unlockBuffer();
}


// remote read extension
int BufferInputStream::readRemote(char* &ptr,int size) {
  int n=0;
  char* readPtr;

  while((eof()==false)) {

    n=size;

    ringBuffer->getReadArea(readPtr,n);

    if (n < size){
      ringBuffer->waitForData(size);
      if (ringBuffer->getCanWaitForData()==false) {
	cout << "cant wait:"<<n<<endl;
	break;
      }
      continue;
    }
    break;
  }
  ptr=readPtr;
  return n;
}


void BufferInputStream::forwardReadPtr(int bytes) {
  lockBuffer();
  
  ringBuffer->forwardReadPtr(bytes);
  ringBuffer->forwardLockPtr(bytes);

  bytePos+=bytes;
  fillgrade-=bytes;
  unlockBuffer();
}
 

void BufferInputStream::setCanWaitForData(int lBlock) {
  ringBuffer->setCanWaitForData(lBlock);
}


void BufferInputStream::lockBuffer() {
  pthread_mutex_lock(&changeMut);
  pthread_mutex_lock(&writeInMut);
}


void BufferInputStream::unlockBuffer() {
  pthread_mutex_unlock(&changeMut);
  pthread_cond_signal(&changeCond);
  pthread_mutex_unlock(&writeInMut); 
}
 
