/*
  management for memChunks
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#ifndef __MEMMANAGER_H
#define __MEMMANAGER_H

#include <pthread.h>
#include <deviceConfig/mem/memChunk.h>

/**
  A MemManager must take care of memChunks. Often it is to time consuming
  to make things safe by memcpy all stream data. Eg. You
  have video streams. This means you have many MB each seconds
  which must be send through all connected devices. But often
  it is not necessary to copy the data. But sometime you must
  store the memory even if the delivering thread has exited.
  (because you wait for data of another thread)
  Then you can lock the memory(instead of memcpy it)
  But you cannot lock the memory for all the time. Eg. a ringbuffer
  needs the memory sometimes and it will block until the
  memory is unlocked.
  <p>
  If you can be sure that the memory is unlocked "shortly" after
  the lock this is a very good method to avoid unecessary memcpy
  calls.<p>
  If you permanently need the memory you can memcpy it nevertheless.
*/


class MemManager {

  pthread_mutex_t managerMut;

 public:
  MemManager();
  virtual ~MemManager();

  void lockMemManager();
  void unlockMemManager();

  virtual int overflowDanger();

  virtual void lockMemChunk(struct MemChunk* memChunk);
  virtual void unlockMemChunk(struct MemChunk* memChunk);
  virtual void lockViolation(struct MemChunk* memChunk); 
  // should be called segfault ? :-)

};
#endif
