/*
  implements a node (n sources, m devices)
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */




#ifndef __NODEDEVICE_H
#define __NODEDEVICE_H
#include <pthread.h>

#include <deviceConfig/deviceConfigArray.h>
#include <graph/yafGraph.h>


extern "C" {
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>

	   }

/**
   Nodedevices are the heart of the filtergraph. You can never make
   an instance of it. This devices takes care that you
   can build graphs with the addListener Methods or destroy them
   with the removeListener method.
   Then it adds basic locking functions for a device, but this
   is only important if you write your own device. The "hello world"
   programm which uses the addListener call is:
   <pre>
   AudioDevice* audioDevice= new AudioDevice("/dev/dsp");
   GenericPlayer* player=Amplifier::createPlayer(_MP3_DECODER);

   player->addListener(audioDevice);
   player->open("xyz.mp3");
   </pre>
   This program plays xyz.mp3 to /dev/dsp.
*/
  

class NodeDevice {

  struct YafGraph* yafGraph;
  struct Edges* sources;
  struct Edges* listeners;

 public:
  NodeDevice();
  virtual ~NodeDevice();

  void addListener(struct NodeDevice *);
  void removeListener(struct NodeDevice *);
  int hasListener();

  struct Edges* getListeners();
  struct Edges* getSources();

  virtual void writeIn(struct NodeDevice* source,struct DeviceConfig* config);
  virtual void writeOut(struct DeviceConfigArray* buf);
  
  virtual char* getNodeName()=0;  //pure virtual

  /** 
      devices which write in this device must request this lock
      until they are ready
  */
  void writeInLock();
  void writeInUnlock();
  
  /**
    this mut is locked when the thread is active (if there is any)
  */
  void writeOutLock();
  void writeOutUnlock();

  /* If I would include dasiyChain.h the compiler returns an error
     because auf a recursive inclusion.
     the void pointer can safley casted to a dasiyChain
  */
  static void* getDaisyChain();        

  static void lockDeviceTree();
  static void unlockDeviceTree();

  static void doSleep(int msec);

 protected:
  virtual void addSource(struct NodeDevice* source);
  virtual void removeSource(struct NodeDevice* source); 
  virtual void wakeUpThread(int lLockInProgress);

  void deliverToListener(DeviceConfig* config);
  

  pthread_mutex_t writeInMut;
  pthread_mutex_t changeMut;

  pthread_cond_t changeCond;

  pthread_mutex_t writeOutMut;

};

#endif
  
  



