/*
  the yaf generator interface works together with yaf backend players
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



#include <producer/yaf/yafGenerator.h>



static int instance=0;

YafGenerator::YafGenerator(int bufferSize) {
  config=new DeviceConfig();
  
  unmutexStatus=new StatusInfo();

  AudioStream* audioStream=config->getAudioStream();
  audioInfo=audioStream->getAudioInfo();
  musicInfo=audioStream->getMusicInfo();
  audioBuffer=audioStream->getAudioBuffer();
  timeInfo=audioStream->getTimeInfo();
  audioStatus=audioStream->getStatusInfo();
  audioBuffer->setPreferredFillGrade(4096);
  control=new YafXPlayerControl();
  stream=new YafStream(this,bufferSize);

  control->addYafListener(this);
  eventQueue=new EventQueue("yafGenerator");
  eventQueue->addListener(this);
  eventQueue->setNotifyMode(_NOTIFY_ALL);

  lPlayRequest=false;

  lClearRequest=false;
  clearCmdNr=-1;
  fullUpdate=false;
  eofBytes=-1;
  threadBlock=false;
  instanceCnt=instance;
  instance++;

  unmutexStatus->setStatus(_STATUS_STOPPED);
  dataUpdate();
}


YafGenerator::~YafGenerator() {
  stream->close();
  kill();

  control->removeYafListener(this);
  delete control;
  delete stream;
  delete config;
  delete eventQueue;
  
}

/**
   Here we create new data for the stream.
   i)  We create statusData. 
   ii) We can create audio/pcm data or controlMessages.
   iii) we take care to shutdown the thread, if "underrun"
*/
int YafGenerator::updateDeviceConfig(DeviceConfig* newConfig) {
  AudioStream* audioStream=newConfig->getAudioStream();
  AudioBuffer* newAudioBuffer=audioStream->getAudioBuffer();
  TimeInfo* timeInfoStream=audioStream->getTimeInfo();
  StatusInfo* statusInfoThread=newConfig->getStatusInfo();
  StatusInfo* statusInfoAudio=audioStream->getStatusInfo();
  

 

  if (audioStatus->getChange()) {
    audioStream->setStatusInfo(audioStatus);
    audioStatus->setChange(false);
    int overallState=audioStream->getStatusInfo()->getStatus();
    statusInfoThread->setStatus(overallState);
    if (overallState != _STATUS_PLAYING) {
      return _THREAD_DELIVER_THEN_SLEEP;
    }
  }

  if (statusInfoThread->getStatus() != _STATUS_PLAYING) {
    return _THREAD_SLEEP;
  }
 
  int back=_THREAD_DELIVER;

  // status playing or no change==play

  if (stream->fillBuffer(newAudioBuffer) == false) {
    return _THREAD_NO_DELIVER;
  }

  if (audioInfo->getChange()) {
    audioStream->setAudioInfo(audioInfo);
    audioInfo->setChange(false);
  }
  if (timeInfo->getChange()) {
    audioStream->setTimeInfo(timeInfo);
    timeInfo->setChange(false);
  }
  if (musicInfo->getChange()) {
    audioStream->setMusicInfo(musicInfo);
    musicInfo->setChange(false);
  }
  
  int bytes=newAudioBuffer->getReadBytes();
  // now we calculate the time.
  timeInfoStream->setSec(audioInfo->calculateTime(bytes));
  // if we read all bytes we make sure that the next
  // statusmessage is "stopped"

  if ((eofBytes > -1) && (bytes >= eofBytes)) {
    eofBytes=-1;
    newAudioBuffer->setMemChunk(NULL);
    eventQueue->sendEvent(_EMIT_EOF_EVENT);
    back=_THREAD_DELIVER_THEN_SLEEP;
    return back;
  }

  return back;
}


int YafGenerator::open(char* filename ) {
  int back;
  close();
  back=control->open(filename);
  pause();
  return back;
}


int YafGenerator::close() {

  pause();

  clearStream();
  control->close();


  return true;
}


int YafGenerator::play() {

  control->play();
  return true;
}


int YafGenerator::pause() {
  control->pause();
  return true;
}


int YafGenerator::jump(int sec) {
  int lPlay=lPlayRequest;
  pause();
  clearStream();
  timeInfo->setTimeOffset(sec);

  control->jump(sec);
  if (lPlay) {
    play();
  }
  return true;
}


void YafGenerator::kill() {
  stream->setReaderBlock(false);
  control->send("quit\n");
}


void YafGenerator::wakeUpThread(int lLockInProgress) {
  if (lLockInProgress) {
    threadBlock=stream->getReaderBlock();
    stream->setReaderBlock(false);
  } else {
    stream->setReaderBlock(threadBlock);
  }

}

void YafGenerator::processRuntimeCommand(int cmd,char* args) {
  int val;
  int point;


  if (cmd == _PLAYER_RUN_STREAMINFO_START) {
    audioInfo->setValid(false);
    dataUpdate();
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_CHANNELS) {
    sscanf(args,"%d %d",&val,&point);
    audioInfo->setStereo(0);
    if (val == 2) {
      audioInfo->setStereo(1);
    }
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_SAMPLESIZE) {
    sscanf(args,"%d %d",&val,&point);
    audioInfo->setSampleSize(val);
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_SPEED) {
    sscanf(args,"%d %d",&val,&point);
    audioInfo->setSpeed(val);
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_END) {
    audioInfo->setValid(true);
    dataUpdate();
  }

  if (cmd == _PLAYER_RUN_MUSICINFO_START) {
    musicInfo->setValid(false);
    dataUpdate();
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_SONG_FILENAME) {
    musicInfo->setFileName(args);
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_SONG_NAME) {
    musicInfo->setName(args);
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_SONG_LEN) {
    sscanf(args,"%d",&val);
    musicInfo->setLen(val);
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_SONG_JUMPS) {
    sscanf(args,"%d",&val);
    musicInfo->setJumps(val);
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_END) {
    musicInfo->setValid(true);
    dataUpdate();
  }


  if (cmd == _PLAYER_RUN_FILEOPEN) {
    if (strcmp(args,"after")==0) {
    }
  }
   
  if (cmd == _PLAYER_RUN_PLAYER_STATUS) {

    if (strlen(args) >= 3) {
      if ( strncmp("off",args,3) == 0 ) {
	long allWrite;
	sscanf(args,"off %ld %ld",&eofBytes,&allWrite);
	//cout << "******eofPos:"<<eofBytes<<" allWrite:"<<allWrite<<endl;
	stream->setEOF(true);
      }
    }
    if ( strncmp("on",args,3) == 0 ) {
      eofBytes=-1;
      stream->setEOF(false);
    }
    dataUpdate();
    return;
    
  }

  if (cmd == _YAF_RUN_EXIT) {
    setThreadState(_STOP_WORKING_BECAUSE_OF_SHUTDOWN);
    setThreadState(_GENERATOR_CRASHED);
  }

}


void YafGenerator::processReturnCommand(int cmdNr,int cmdId,
				       char* ret,char* args) {

  if (cmdId == _PLAYER_CLEAR){
    setThreadState(_NOT_ALLOW_GENERATOR_ENTER);
    stream->clear();
    stream->open();
    control->setOnline(true);
    clearCmdNr=-1;
  }
  if (cmdId == _PLAYER_CLOSE) {
    audioInfo->setValid(false);
    musicInfo->setValid(false);
    timeInfo->setTimeOffset(0);
    unmutexStatus->setStatus(_STATUS_STOPPED);
    lClearRequest=false;
    dataUpdate();
  }
  if (cmdId == _PLAYER_JUMP) {
    int pos;

    lClearRequest=false;
  
    sscanf(args,"%d",&pos);
    timeInfo->setTimeOffset(pos);
    dataUpdate();
  }
  if (cmdId == _PLAYER_OPEN) {
    timeInfo->setTimeOffset(0);
    unmutexStatus->setStatus(_STATUS_PLAYING);
    lClearRequest=false;
    dataUpdate();
  }  
  if (cmdId == _PLAYER_PLAY) {
    lPlayRequest=true;
    unmutexStatus->setStatus(_STATUS_PLAYING);
    lClearRequest=false;
    dataUpdate();
  }
  if (lClearRequest) {
    return;
  }
  if (cmdId == _PLAYER_PAUSE) {
    lPlayRequest=false;
    unmutexStatus->setStatus(_STATUS_PAUSED);
    dataUpdate();
  }



}

 
void YafGenerator::addArgument(char* arg) {
  control->addArgument(arg);
}


void YafGenerator::startDecoder() {
  control->startDecoder();
  control->send(stream->getFifoCommand());
}


void YafGenerator::bufferFilledNotify(int filled) {
  audioBuffer->setBufferFilled(filled);
  dataUpdate();
}



// If this method is called it must be sure
// that the thread is at least "paused"

void YafGenerator::clearStream() {
  if (lClearRequest == false) {
    lClearRequest=true;
    stream->setReaderBlock(false);
    stream->close();
    setThreadState(_NOT_ALLOW_GENERATOR_ENTER);
    control->pause();
    control->send("clear\n");
    control->setOnline(false);
  }
}




void YafGenerator::dataUpdate() {
  int lCanPlay=true;

  setThreadState(_NOT_ALLOW_GENERATOR_ENTER);
  unmutexStatus->copyTo(audioStatus);

  if (audioInfo->getValid() == false) {
    lCanPlay=false;
  }
  if (musicInfo->getValid() == false) {
    lCanPlay=false;
  }
  if (audioStatus->getStatus() != _STATUS_PLAYING) {
    lCanPlay=false;
  }
 

  if (stream->getEOF()==false) {
    stream->setReaderBlock(true);
  } else {
    stream->setReaderBlock(false);
  }
  if (lCanPlay) {
    audioStatus->setStatus(_STATUS_PLAYING);
  } 
  setThreadState(_ALLOW_GENERATOR_ENTER);

  return;

}


void YafGenerator::setThreadState(int msg) {
  ThreadNotifier* client;
  client=getThreadNotifier();
  if (client == NULL) {
    return;
  }

    
  client->sendSyncMessage(msg);
}
    

void YafGenerator::processEvent(char msgId) {

  switch(msgId) {
  case _EMIT_EOF_EVENT: {
    unmutexStatus->setStatus(_STATUS_STOPPED);
    audioInfo->setValid(false);
    musicInfo->setValid(false);
    stream->setEOF(false);
    eofBytes=-1;

    dataUpdate();
    processEvent(_EMIT_UPDATE_THREAD_EVENT);

    while(audioStatus->getChange()==true) {

      struct timeval time;
      time.tv_sec=0;
      time.tv_usec=5000;
      
      select(0,NULL,NULL,NULL,&time);
    }

    setThreadState(_STOP_WORKING_BECAUSE_OF_EOF);
    break;
  }
  case _EMIT_UPDATE_THREAD_EVENT: {
    setThreadState(_ALLOW_GENERATOR_ENTER);
    break;
  }

  default:
    cout << "unknown msg: "<<msgId<<" in YafGenerator::processEvent"<<endl;
  }
}

