/***************************************************************************
                          csample.cpp  -  description
                             -------------------
    begin                : Sun Dec 26 1999
    copyright            : (C) 1999 by Tobias Wollgam
    email                : Tobias.Wollgam@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "csample.h"
#include "carray.h"
#include "machine.c"

#define SWAPBYTES(a)	((((a) & 0xff00) >> 8) + (((a) &0x00ff) << 8))

CSample::CSample()
{
	len = 0;
	rate = 44100;
	name = 0;
	loop[0] = 0;
	loop[1] = 0;
	loop[2] = 0;
	sample_mode = 0;
	data.resize(0);
	addChannel(1);

	audio_dev[0] = "/dev/dsp";
	audio[0] = -1;
	audio_dev[1] = "/dev/dsp1";
	audio[1] = -1;

	buffer_size = 65536;
	m_writesamples = 32768;
}

CSample::CSample(uint c)
{
	len = 0;
	rate = 0;
	name = NULL;
	loop[0] = 0;
	loop[1] = 0;
	loop[2] = 0;
	sample_mode = 0;
	data.resize(0);
	addChannel(c);

	audio_dev[0] = "/dev/dsp";
	audio[0] = -1;
	audio_dev[1] = "/dev/dsp1";
	audio[1] = -1;

	buffer_size = 65536;
	m_writesamples = 32768;
}

CSample::CSample(uint c,uint mode,uint size)
{
	len = size;
	rate = 0;
	name = NULL;
	loop[0] = 0;
	loop[1] = 0;
	loop[2] = 0;
	sample_mode = mode;
	data.resize(0);
	addChannel(c);

	audio_dev[0] = "/dev/dsp";
	audio[0] = -1;
	audio_dev[1] = "/dev/dsp1";
	audio[1] = -1;

	buffer_size = 65536;
	m_writesamples = 32768;
}

CSample::CSample(CSample *s)
{
	len = s->len;
	rate = s->rate;
	if(s->name) name = strdup(s->name);
	else name = NULL;
	loop[0] = s->loop[0];
	loop[1] = s->loop[1];
	loop[2] = s->loop[2];
	sample_mode = s->sample_mode;
	data.resize(0);
	addChannel(s->data.size());
	set(0,s->len,s);
	
	audio_dev[0] = "/dev/dsp";
	audio[0] = -1;
	audio_dev[1] = "/dev/dsp1";
	audio[1] = -1;

	buffer_size = 65536;
	m_writesamples = 32768;
}

CSample::CSample(CSample &s)
{
	len = s.len;
	rate = s.rate;
	if(s.name)
		name = strdup(s.name);
	else
		name = 0;
	loop[0] = s.loop[0];
	loop[1] = s.loop[1];
	loop[2] = s.loop[2];
	sample_mode = s.sample_mode;
	data.resize(0);
	addChannel(s.data.size());
	set(0,s.len,&s);
	
	audio_dev[0] = "/dev/dsp";
	audio[0] = -1;
	audio_dev[1] = "/dev/dsp1";
	audio[1] = -1;

	buffer_size = 65536;
	m_writesamples = 32768;
}

CSample::~CSample()
{
	unsigned int	t;

	if(name)
		free(name);
	name = 0;

	for(t = 0;t < data.size();t++)
	{
		delete data[t].d8;
	}
	data.resize(0);
}

void CSample::setName(char *n)
{
	name = strdup(n);
}

char*	CSample::getName()
{
	return name;
}

void	CSample::setRate(int r)
{
	rate = r;
}

int	CSample::getRate()
{
	return rate;
}

void	CSample::setPitch(ushort p)
{
	pitch = p;
}

ushort	CSample::getPitch()
{
	return pitch;
}

void	CSample::setPitchRange(ushort pmin,ushort pmax)
{
	pitch_min = pmin;
	pitch_max = pmax;
}

ushort	CSample::getPitchMin()
{
	return pitch_min;
}

ushort	CSample::getPitchMax()
{
	return pitch_max;
}

unsigned int CSample::getLength()
{
	return len;
}

int	CSample::getNChannel()
{
	return data.size();
}

int	CSample::addChannel()
{
	uint		t;

	data.resize(data.size() + 1);
	//if(data == NULL) return 0;
	if(isType8())
	{
		data[data.size() - 1].d8 = new CArray<char>();
		data[data.size() - 1].d8->resize(len);
		for(t = 0;t < len;t++) data[data.size() - 1].d8->at(t) = 0;
	}
	if(isType16())
	{
		data[data.size() - 1].d16 = new CArray<short>();
		data[data.size() - 1].d16->resize(len);
		for(t = 0;t < len;t++) data[data.size() - 1].d16->at(t) = 0;
	}
	if(isType32())
	{
		data[data.size() - 1].d32 = new CArray<int>();
		data[data.size() - 1].d32->resize(len);
		for(t = 0;t < len;t++) data[data.size() - 1].d32->at(t) = 0;
	}
	if(isTypeFL())
	{
		data[data.size() - 1].dfl = new CArray<float>();
		data[data.size() - 1].dfl->resize(len);
		for(t = 0;t < len;t++) data[data.size() - 1].dfl->at(t) = 0;
	}
	if(isTypeDB())
	{
		data[data.size() - 1].ddb = new CArray<double>();
		data[data.size() - 1].ddb->resize(len);
		for(t = 0;t < len;t++) data[data.size() - 1].ddb->at(t) = 0;
	}
	return !0;
}

int	CSample::addChannel(uint n)
{
	uint		t;

	for(t = 0;t < n;t++)
	{
		if(!addChannel()) return 0;
	}

	return n;
}

int	CSample::delChannel(uint n)
{
	uint		t;

	if(n >= data.size()) return 0;

	delete data[n].d8;
	//data.del(n);
	//data.cut(n,1);
	for(t = n;t < data.size() - 1;t++)
	{
		data[t] = data[t + 1];
	}
	data.resize(data.size() - 1);

	return !0;
}
 
int	CSample::clear()
{
	uint		t;

	for(t = data.size();t > 0;t--) delChannel(0);

	if(getNChannel() == 0)
	{
		len = 0;
		data.resize(0);
	
		return !0;
	}

	return 0;
}

bool	CSample::isMono()
{
	return (getNChannel() == 1);
}

bool	CSample::isStereo()
{
	return (getNChannel() == 2);
}

bool	CSample::isType8()
{
	return ((sample_mode & SMPFLAGS_TYPEMASK) == SMPFLAGS_TYPE_8);
}

bool	CSample::isType16()
{
	return ((sample_mode & SMPFLAGS_TYPEMASK) == SMPFLAGS_TYPE_16);
}

bool	CSample::isType32()
{
	return ((sample_mode & SMPFLAGS_TYPEMASK) == SMPFLAGS_TYPE_32);
}

bool	CSample::isTypeFL()
{
	return ((sample_mode & SMPFLAGS_TYPEMASK) == SMPFLAGS_TYPE_FL);
}

bool	CSample::isTypeDB()
{
	return ((sample_mode & SMPFLAGS_TYPEMASK) == SMPFLAGS_TYPE_DB);
}

void	CSample::convertToTypeDB()
{
	uint		c,t;

	if(isTypeDB()) return;
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_DB;
		return;
	}

	if(isType8())
	{
		char		*i8;

		if((i8 = (char*)malloc(sizeof(char) * len)) == NULL) return;

	 	for(c = 0;c < data.size();c++)
		{
			get(c,0,len,i8);
			data[c].ddb->resize(len);
			for(t = 0;t < len;t++)
			{
				data[c].ddb->at(t) = CHAR2D(i8[t]);
			}
		}

		free(i8);
		sample_mode = SMPFLAGS_TYPE_DB;
	}
	if(isType16())
	{
		short		*i16;

		if((i16 = (short*)malloc(sizeof(short) * len)) == NULL) return;

	 	for(c = 0;c < data.size();c++)
		{
			get(c,0,len,i16);
			data[c].ddb->resize(len);
			for(t = 0;t < len;t++)
			{
				data[c].ddb->at(t) = SHORT2D(i16[t]);
			}
		}

		free(i16);
		sample_mode = SMPFLAGS_TYPE_DB;
	}
	if(isType32())
	{
		int		*i32;

		if((i32 = (int*)malloc(sizeof(int) * len)) == NULL) return;

	 	for(c = 0;c < data.size();c++)
		{
			get(c,0,len,i32);
			data[c].ddb->resize(len);
			for(t = 0;t < len;t++)
			{
				data[c].ddb->at(t) = INT2D(i32[t]);
			}
		}

		free(i32);
		sample_mode = SMPFLAGS_TYPE_DB;
	}
	if(isTypeFL())
	{
		float		*fl;

		if((fl = (float*)malloc(sizeof(float) * len)) == NULL) return;

	 	for(c = 0;c < data.size();c++)
		{
			get(c,0,len,fl);
			data[c].ddb->resize(len);
			for(t = 0;t < len;t++)
			{
				data[c].ddb->at(t) = FLOAT2D(fl[t]);
			}
		}

		free(fl);
		sample_mode = SMPFLAGS_TYPE_DB;
	}
	
	
}

void	CSample::convertTypeDBToType8()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB()) return;
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_8;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].d8->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].d8->at(t) = D2CHAR(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_8;
}

void	CSample::convertTypeDBToType16()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB()) return;
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_16;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].d16->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].d16->at(t) = D2SHORT(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_16;
}

void	CSample::convertTypeDBToType32()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB()) return;
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_32;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].d32->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].d32->at(t) = D2INT(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_32;
}

void	CSample::convertTypeDBToTypeFL()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB()) return;
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_FL;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].dfl->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].dfl->at(t) = D2FLOAT(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_FL;
}

void	CSample::convertToType8()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB())
	{
		convertToTypeDB();
		if(!isTypeDB())
			return;
	}
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_8;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].d8->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].d8->at(t) = D2CHAR(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_8;
}

void	CSample::convertToType16()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB())
	{
		convertToTypeDB();
		if(!isTypeDB())
			return;
	}
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_16;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].d16->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].d16->at(t) = D2SHORT(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_16;
}

void	CSample::convertToType32()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB())
	{
		convertToTypeDB();
		if(!isTypeDB())
			return;
	}
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_32;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].d32->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].d32->at(t) = D2INT(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_32;
}

void	CSample::convertToTypeFL()
{
	uint		c,t;
	double		*d;

	if(!isTypeDB())
	{
		convertToTypeDB();
		if(!isTypeDB())
			return;
	}
	if(len == 0)
	{
		sample_mode = SMPFLAGS_TYPE_FL;
		return;
	}

	if((d = (double*)malloc(sizeof(double) * len)) == NULL) return;

 	for(c = 0;c < data.size();c++)
	{
		get(c,0,len,d);
		data[c].dfl->resize(len);
		for(t = 0;t < len;t++)
		{
			data[c].dfl->at(t) = D2FLOAT(d[t]);
		}
	}

	free(d);
	sample_mode = SMPFLAGS_TYPE_FL;
}

int	CSample::resize(uint size)
{
	uint		t;

	if(size == len) return !0;
	for(t = 0;t < data.size();t++)
	{
		if(isType8())
		{
			data[t].d8->resize(size);
		}
		if(isType16())
		{
			data[t].d16->resize(size);
		}
		if(isType32())
		{
			data[t].d32->resize(size);
		}
		if(isTypeFL())
		{
			data[t].dfl->resize(size);
		}
		if(isTypeDB())
		{
			data[t].ddb->resize(size);
		}
	}
	len = size;

	return !0;
}

int	CSample::get(uint c,uint pos,uint l,char *d8)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			d8[t] = data[c].d8->at(t + pos);
		}
		if(isType16())
		{
			d8[t] = D2CHAR(SHORT2D(data[c].d16->at(t + pos)));
		}
		if(isType32())
		{
			d8[t] = D2CHAR(INT2D(data[c].d32->at(t + pos)));
		}
		if(isTypeFL())
		{
			d8[t] = D2CHAR(FLOAT2D(data[c].dfl->at(t + pos)));
		}
		if(isTypeDB())
		{
			d8[t] = D2CHAR(data[c].ddb->at(t + pos));
		}
	}

	return !0;
}

int	CSample::get(uint c,uint pos,uint l,short *d16)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			d16[t] = D2SHORT(CHAR2D(data[c].d8->at(t + pos)));
		}
		if(isType16())
		{
			d16[t] = data[c].d16->at(t + pos);
		}
		if(isType32())
		{
			d16[t] = D2SHORT(INT2D(data[c].d32->at(t + pos)));
		}
		if(isTypeFL())
		{
			d16[t] = D2SHORT(FLOAT2D(data[c].dfl->at(t + pos)));
		}
		if(isTypeDB())
		{
			d16[t] = D2SHORT(data[c].ddb->at(t + pos));
		}
	}

	return !0;
}

int	CSample::get(uint c,uint pos,uint l,int *d32)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			d32[t] = D2INT(CHAR2D(data[c].d8->at(t + pos)));
		}
		if(isType16())
		{
			d32[t] = D2INT(SHORT2D(data[c].d16->at(t + pos)));
		}
		if(isType32())
		{
			d32[t] = D2INT(INT2D(data[c].d32->at(t + pos)));
		}
		if(isTypeFL())
		{
			d32[t] = D2INT(FLOAT2D(data[c].dfl->at(t + pos)));
		}
		if(isTypeDB())
		{
			d32[t] = D2INT(data[c].ddb->at(t + pos));
		}
	}

	return !0;
}

int	CSample::get(uint c,uint pos,uint l,float *dfl)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			dfl[t] = D2FLOAT(CHAR2D(data[c].d8->at(t + pos)));
		}
		if(isType16())
		{
			dfl[t] = D2FLOAT(SHORT2D(data[c].d16->at(t + pos)));
		}
		if(isType32())
		{
			dfl[t] = D2FLOAT(INT2D(data[c].d32->at(t + pos)));
		}
		if(isTypeFL())
		{
			dfl[t] = D2FLOAT(FLOAT2D(data[c].dfl->at(t + pos)));
		}
		if(isTypeDB())
		{
			dfl[t] = D2FLOAT(data[c].ddb->at(t + pos));
		}
	}

	return !0;
}

int	CSample::get(uint c,uint pos,uint l,double *ddb)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			ddb[t] = CHAR2D(data[c].d8->at(t + pos));
		}
		if(isType16())
		{
			ddb[t] = SHORT2D(data[c].d16->at(t + pos));
		}
		if(isType32())
		{
			ddb[t] = INT2D(data[c].d32->at(t + pos));
		}
		if(isTypeFL())
		{
			ddb[t] = FLOAT2D(data[c].dfl->at(t + pos));
		}
		if(isTypeDB())
		{
			ddb[t] = data[c].ddb->at(t + pos);
		}
	}

	return !0;
}

int	CSample::get(uint c,uint pos,uint l,CSample *s)
{
	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	s->clear();
	s->resize(l);
	s->addChannel();
	if(s->isType8())
	{
		get(c,pos,l,s->data[c].d8->data());
	}
	if(s->isType16())
	{
		get(c,pos,l,s->data[c].d16->data());
	}
	if(s->isType32())
	{
		get(c,pos,l,s->data[c].d32->data());
	}
	if(s->isTypeFL())
	{
		get(c,pos,l,s->data[c].dfl->data());
	}
	if(s->isTypeDB())
	{
		get(c,pos,l,s->data[c].ddb->data());
	}

	return !0;
}

int	CSample::get(uint c,uint pos,uint l,CSample &s)
{
	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	s.clear();
	s.resize(l);
	s.addChannel();
	if(s.isType8())
	{
		get(c,pos,l,s.data[c].d8->data());
	}
	if(s.isType16())
	{
		get(c,pos,l,s.data[c].d16->data());
	}
	if(s.isType32())
	{
		get(c,pos,l,s.data[c].d32->data());
	}
	if(s.isTypeFL())
	{
		get(c,pos,l,s.data[c].dfl->data());
	}
	if(s.isTypeDB())
	{
		get(c,pos,l,s.data[c].ddb->data());
	}

	return !0;
}

int	CSample::get(uint pos,uint l,CSample *s)
{
	uint		t;

	// if(s->data.size() > data.size()) return 0;
	if(pos + l > len) return 0;
	s->clear();
	s->resize(l);
	s->addChannel(data.size());
	for(t = 0;t < data.size();t++)
	{
		if(s->isType8())
		{
			get(t,pos,l,s->data[t].d8->data());
		}
		if(s->isType16())
		{
			get(t,pos,l,s->data[t].d16->data());
		}
		if(s->isType32())
		{
			get(t,pos,l,s->data[t].d32->data());
		}
		if(s->isTypeFL())
		{
			get(t,pos,l,s->data[t].dfl->data());
		}
		if(s->isTypeDB())
		{
			get(t,pos,l,s->data[t].ddb->data());
		}
	}

	return !0;
}

int	CSample::get(uint pos,uint l,CSample &s)
{
	uint		t;

	// if(s->data.size() > data.size()) return 0;
	if(pos + l > len) return 0;
	s.clear();
	s.resize(l);
	s.addChannel(data.size());
	for(t = 0;t < data.size();t++)
	{
		if(s.isType8())
		{
			get(t,pos,l,s.data[t].d8->data());
		}
		if(s.isType16())
		{
			get(t,pos,l,s.data[t].d16->data());
		}
		if(s.isType32())
		{
			get(t,pos,l,s.data[t].d32->data());
		}
		if(s.isTypeFL())
		{
			get(t,pos,l,s.data[t].dfl->data());
		}
		if(s.isTypeDB())
		{
			get(t,pos,l,s.data[t].ddb->data());
		}
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,char *d8)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			data[c].d8->at(t + pos) = D2CHAR(CHAR2D(d8[t]));
		}
		if(isType16())
		{
			data[c].d16->at(t + pos) = D2SHORT(CHAR2D(d8[t]));
		}
		if(isType32())
		{
			data[c].d32->at(t + pos) = D2INT(CHAR2D(d8[t]));
		}
		if(isTypeFL())
		{
			data[c].dfl->at(t + pos) = D2FLOAT(CHAR2D(d8[t]));
		}
		if(isTypeDB())
		{
			data[c].ddb->at(t + pos) = CHAR2D(d8[t]);
		}
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,short *d16)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			data[c].d8->at(t + pos) = D2CHAR(SHORT2D(d16[t]));
		}
		if(isType16())
		{
			data[c].d16->at(t + pos) = D2SHORT(SHORT2D(d16[t]));
		}
		if(isType32())
		{
			data[c].d32->at(t + pos) = D2INT(SHORT2D(d16[t]));
		}
		if(isTypeFL())
		{
			data[c].dfl->at(t + pos) = D2FLOAT(SHORT2D(d16[t]));
		}
		if(isTypeDB())
		{
			data[c].ddb->at(t + pos) = SHORT2D(d16[t]);
		}
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,int *d32)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			data[c].d8->at(t + pos) = D2CHAR(INT2D(d32[t]));
		}
		if(isType16())
		{
			data[c].d16->at(t + pos) = D2SHORT(INT2D(d32[t]));
		}
		if(isType32())
		{
			data[c].d32->at(t + pos) = D2INT(INT2D(d32[t]));
		}
		if(isTypeFL())
		{
			data[c].dfl->at(t + pos) = D2FLOAT(INT2D(d32[t]));
		}
		if(isTypeDB())
		{
			data[c].ddb->at(t + pos) = INT2D(d32[t]);
		}
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,float *dfl)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			data[c].d8->at(t + pos) = D2CHAR(FLOAT2D(dfl[t]));
		}
		if(isType16())
		{
			data[c].d16->at(t + pos) = D2SHORT(FLOAT2D(dfl[t]));
		}
		if(isType32())
		{
			data[c].d32->at(t + pos) = D2INT(FLOAT2D(dfl[t]));
		}
		if(isTypeFL())
		{
			data[c].dfl->at(t + pos) = D2FLOAT(FLOAT2D(dfl[t]));
		}
		if(isTypeDB())
		{
			data[c].ddb->at(t + pos) = INT2D(dfl[t]);
		}
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,double *ddb)
{
	uint		t;

	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < l;t++)
	{
		if(isType8())
		{
			data[c].d8->at(t + pos) = D2CHAR(ddb[t]);
		}
		if(isType16())
		{
			data[c].d16->at(t + pos) = D2SHORT(ddb[t]);
		}
		if(isType32())
		{
			data[c].d32->at(t + pos) = D2INT(ddb[t]);
		}
		if(isTypeFL())
		{
			data[c].dfl->at(t + pos) = D2FLOAT(ddb[t]);
		}
		if(isTypeDB())
		{
			data[c].ddb->at(t + pos) = ddb[t];
		}
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,CSample *s)
{
	if(s == 0) return 0;
	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	if(s->isType8())
	{
		set(c,pos,l,s->data[0].d8->data());
	}
	if(s->isType16())
	{
		set(c,pos,l,s->data[0].d16->data());
	}
	if(s->isType32())
	{
		set(c,pos,l,s->data[0].d32->data());
	}
	if(s->isTypeFL())
	{
		set(c,pos,l,s->data[0].dfl->data());
	}
	if(s->isTypeDB())
	{
		set(c,pos,l,s->data[0].ddb->data());
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,CSample &s)
{
	if(c > data.size()) return 0;
	if(pos + l > len) return 0;
	if(s.isType8())
	{
		set(c,pos,l,s.data[0].d8->data());
	}
	if(s.isType16())
	{
		set(c,pos,l,s.data[0].d16->data());
	}
	if(s.isType32())
	{
		set(c,pos,l,s.data[0].d32->data());
	}
	if(s.isTypeFL())
	{
		set(c,pos,l,s.data[0].dfl->data());
	}
	if(s.isTypeDB())
	{
		set(c,pos,l,s.data[0].ddb->data());
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,CSample *s,uint sc)
{
	if(c > data.size()) return 0;
	if(sc > s->data.size()) return 0;
	if(pos + l > len) return 0;
	if(s->isType8())
	{
		set(c,pos,l,s->data[sc].d8->data());
	}
	if(s->isType16())
	{
		set(c,pos,l,s->data[sc].d16->data());
	}
	if(s->isType32())
	{
		set(c,pos,l,s->data[sc].d32->data());
	}
	if(s->isTypeFL())
	{
		set(c,pos,l,s->data[sc].dfl->data());
	}
	if(s->isTypeDB())
	{
		set(c,pos,l,s->data[sc].ddb->data());
	}

	return !0;
}

int	CSample::set(uint c,uint pos,uint l,CSample &s,uint sc)
{
	if(c > data.size()) return 0;
	if(sc > s.data.size()) return 0;
	if(pos + l > len) return 0;
	if(s.isType8())
	{
		set(c,pos,l,s.data[sc].d8->data());
	}
	if(s.isType16())
	{
		set(c,pos,l,s.data[sc].d16->data());
	}
	if(s.isType32())
	{
		set(c,pos,l,s.data[sc].d32->data());
	}
	if(s.isTypeFL())
	{
		set(c,pos,l,s.data[sc].dfl->data());
	}
	if(s.isTypeDB())
	{
		set(c,pos,l,s.data[sc].ddb->data());
	}

	return !0;
}

int	CSample::set(uint pos,uint l,CSample *s)
{
	uint		t;

	if(data.size() != s->data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < data.size();t++)
	{
		if(s->isType8())
		{
			set(t,pos,l,s->data[t].d8->data());
		}
		if(s->isType16())
		{
			set(t,pos,l,s->data[t].d16->data());
		}
		if(s->isType32())
		{
			set(t,pos,l,s->data[t].d32->data());
		}
		if(s->isTypeFL())
		{
			set(t,pos,l,s->data[t].dfl->data());
		}
		if(s->isTypeDB())
		{
			set(t,pos,l,s->data[t].ddb->data());
		}
	}

	return !0;
}

int	CSample::set(uint pos,uint l,CSample &s)
{
	uint		t;

	if(data.size() != s.data.size()) return 0;
	if(pos + l > len) return 0;
	for(t = 0;t < data.size();t++)
	{
		if(s.isType8())
		{
			set(t,pos,l,s.data[t].d8->data());
		}
		if(s.isType16())
		{
			set(t,pos,l,s.data[t].d16->data());
		}
		if(s.isType32())
		{
			set(t,pos,l,s.data[t].d32->data());
		}
		if(s.isTypeFL())
		{
			set(t,pos,l,s.data[t].dfl->data());
		}
		if(s.isTypeDB())
		{
			set(t,pos,l,s.data[t].ddb->data());
		}
	}

	return !0;
}

int	CSample::move(uint dest,uint src,int l)
{
	uint		c;

	if(l <= 0)
	{
		printf("CSample: move: l out of range (%i)!\n",l);
		return 0;
	}
	if(dest + l > len)
	{
		printf("CSample: move: dest + l out of range (%i,%i)!\n",dest + l,len);
		return 0;
	}
	if(src + l > len)
	{
		printf("CSample: move: src + l out of range!\n");
		return 0;
	}

	if(isType8())
	{
		for(c = 0;c < data.size();c++)
			data[c].d8->move(dest,src,(uint)l);
	}
	else if(isType16())
	{
		for(c = 0;c < data.size();c++)
			data[c].d16->move(dest,src,(uint)l);
	}
	else if(isType32())
	{
		for(c = 0;c < data.size();c++)
			data[c].d32->move(dest,src,(uint)l);
	}
	else if(isTypeFL())
	{
		for(c = 0;c < data.size();c++)
			data[c].dfl->move(dest,src,(uint)l);
	}
	else if(isTypeDB())
	{
		for(c = 0;c < data.size();c++)
			data[c].ddb->move(dest,src,(uint)l);
	}

	return !0;
}

int	CSample::insert(uint pos,uint l,CSample *s)
{
	uint		t;

	if(s->data.size() != data.size()) return 0;
	if(pos > len) return 0;
	//if(l > s->len) return 0;
	printf("resize: %i to %i\n",len,l + len);
	resize(len + l);
	printf("move: %i,%i,%i\n",l + pos,pos,len - l - pos - 1);
	//move(pos + l,pos,len - l - pos - 1);
	move(pos + l,pos,len - l - pos - 1);
	for(t = 0;t < data.size();t++)
	{
		printf("set: %i,%i\n",pos,l);
		set(t,pos,l,s,t);
	}

	return !0;
}

int	CSample::insert(uint pos,CSample *s)
{
	uint		t,l;

	if(s == NULL) return 0;
	l = s->getLength();
	if(s->data.size() != data.size()) return 0;
	if(pos > len) return 0;
	//if(l > s->len) return 0;
	printf("resize: %i to %i\n",len,l + len);
	resize(len + l);
	printf("move: %i,%i,%i\n",l + pos,pos,len - l - pos);
	//move(pos + l,pos,len - l - pos - 1);
	move(pos + l,pos,len - l - pos);
	for(t = 0;t < data.size();t++)
	{
		printf("set: %i,%i\n",pos,l);
		set(t,pos,l,s,t);
	}

	return !0;
}

int	CSample::insert(uint pos,CSample &s)
{
	uint		t,l;

	l = s.getLength();
	if(s.data.size() != data.size()) return 0;
	if(pos > len) return 0;
	//if(l > s->len) return 0;
	printf("resize: %i to %i\n",len,l + len);
	resize(len + l);
	printf("move: %i,%i,%i\n",l + pos,pos,len - l - pos);
	//move(pos + l,pos,len - l - pos - 1);
	move(pos + l,pos,len - l - pos);
	for(t = 0;t < data.size();t++)
	{
		printf("set: %i,%i\n",pos,l);
		set(t,pos,l,s,t);
	}

	return !0;
}

int	CSample::cut(uint pos,uint l,CSample *s = NULL)
{
	if(pos + l > len) return 0;
	if(s != NULL)
	{
		if(s->data.size() != data.size()) return 0;
		if(!get(pos,l,s)) return 0;
	}
	move(pos,pos + l,len - (pos + l + 1));
	resize(len - l);

	return !0;
}

int	CSample::cut(uint pos,uint l,CSample &s)
{
	if(pos + l > len) return 0;
	if(s.data.size() != data.size()) return 0;
	if(!get(pos,l,s)) return 0;
	move(pos,pos + l,len - (pos + l + 1));
	resize(len - l);

	return !0;
}

int CSample::loadWave(const char* namepath)
{
	FILE		*fp;
	char		buffer[256];
	uint		l,encoding,cn;

	if((fp = fopen(namepath,"rb")) == 0) return -1;
	fread(buffer,8,1,fp);
	if(strncmp(buffer,"RIFF",4) != 0)
	{
		fclose(fp);
		return -2;
	}
	fread(buffer,4,1,fp);
	if(strncmp(buffer,"WAVE",4) != 0)
	{
		fclose(fp);
		return -3;
	}
	if(fread(buffer,8,1,fp) == 0) return -4;
	l = make_le4(*(int*)(buffer + 4));
	if(strncmp(buffer,"fmt ",4) == 0)
	{
		fread(buffer,l,1,fp);
		rate = make_le4(*(int*)(buffer + 4));
		if(make_le2(*(short*)(buffer + 14)) == 8)
		{
			sample_mode = SMPFLAGS_TYPE_8;
		}
		else
		{
			sample_mode = SMPFLAGS_TYPE_16;
		}
		clear();
		cn = make_le2(*(short*)(buffer + 2));
		addChannel(cn);
		encoding = make_le2(*(short*)buffer);
	}
	else return -4;
	if(fread(buffer,8,1,fp) == 0) return -4;
	l = make_le4(*(int*)(buffer + 4));

		switch(encoding)
		{
			case 0x0001: // WAVE_FORMAT_PCM
				printf("WAVE_FORMAT_PCM\n");
			break;
			case 0x0002: // WAVE_FORMAT_ADPCM
				printf("WAVE_FORMAT_ADPCM\n");
			break;
			case 0x0006: // WAVE_FORMAT_ALAW
				printf("WAVE_FORMAT_ALAW\n");
			break;
			case 0x0007: // WAVE_FORMAT_MULAW
				printf("WAVE_FORMAT_MULAW\n");
			break;
			case 0x0010: // WAVE_FORMAT_OKI_ADPCM
				printf("WAVE_FORMAT_OKI_ADPCM\n");
			break;
			case 0x0015: // WAVE_FORMAT_DIGISTD
				printf("WAVE_FORMAT_DIGISTD\n");
			break;
			case 0x0016: // WAVE_FORMAT_DIGIFIX
				printf("WAVE_FORMAT_DIGIFIX\n");
			break;
			case 0x0101: // IBM_FORMAT_MULAW
				printf("IBM_FORMAT_MULAW\n");
			break;
			case 0x0102: // IBM_FORMAT_ALAW
				printf("IBM_FORMAT_ALAW\n");
			break;
			case 0x0103: // IBM_FORMAT_ADPCM
				printf("IBM_FORMAT_ADPCM\n");
			break;
			default:
				printf("Unknown Coding! Please report to tobias.wollgam@gmx.de\nCode: %i\n",encoding);
		}

	if(strncmp(buffer,"fact",4) == 0)
	{
		fseek(fp,l,SEEK_CUR);
		if(fread(buffer,8,1,fp) == 0) return -4;
		l = make_le4(*(int*)(buffer + 4));
	}
	if(strncmp(buffer,"data",4) == 0)
	{
		if(isType8())
		{
			char	c;
			uint	i;

			resize(l / cn);

			for(i = 0;i < l;i++)
			{
				fread(&c,1,1,fp);
				switch(encoding)
				{
					case 0x0001: // WAVE_FORMAT_PCM
					case 0x0002: // WAVE_FORMAT_ADPCM
					case 0x0006: // WAVE_FORMAT_ALAW
					case 0x0007: // WAVE_FORMAT_MULAW
					case 0x0010: // WAVE_FORMAT_OKI_ADPCM
					case 0x0015: // WAVE_FORMAT_DIGISTD
					case 0x0016: // WAVE_FORMAT_DIGIFIX
					case 0x0101: // IBM_FORMAT_MULAW
					case 0x0102: // IBM_FORMAT_ALAW
					case 0x0103: // IBM_FORMAT_ADPCM
						c = 127 - c;
					break;
					default:
						c = 127 - c;
				}
				set(i % cn,i / cn,1,&c);
			}
		}
		if(isType16())
		{
			short	s;
			uint	i;

			resize(l / 2 / cn);

			for(i = 0;i < l / 2;i++)
			{
				fread(&s,2,1,fp);
				switch(encoding)
				{
					case 0x0001: // WAVE_FORMAT_PCM
					case 0x0002: // WAVE_FORMAT_ADPCM
					case 0x0006: // WAVE_FORMAT_ALAW
					case 0x0007: // WAVE_FORMAT_MULAW
					case 0x0010: // WAVE_FORMAT_OKI_ADPCM
					case 0x0015: // WAVE_FORMAT_DIGISTD
					case 0x0016: // WAVE_FORMAT_DIGIFIX
					case 0x0101: // IBM_FORMAT_MULAW
					case 0x0102: // IBM_FORMAT_ALAW
					case 0x0103: // IBM_FORMAT_ADPCM
						s = s;
					break;
					default:
						s = s;
				}
				set(i % cn,i / cn,1,&s);
			}
		}
	}
	else return -4;

	fclose(fp);

	return 0;
}

int CSample::saveWave(const char *namepath)
{
	FILE		*fp;
	char		buffer[256];
	uint		i,t,l,encoding = 0x0001;

	if((fp = fopen(namepath,"wb")) == 0) return -1;
	sprintf(buffer,"RIFF");
	l = getLength() * getNChannel();
	if(isType16()) l *= 2;
	l = l + 12 + 8 + 16 + 8;
	writestr4(buffer + 4,make_le4(l));
	sprintf(buffer + 8,"WAVE");
	sprintf(buffer + 12,"fmt ");
	writestr4(buffer + 16,make_le4(16));
	writestr2(buffer + 20,make_le2(encoding));
	writestr2(buffer + 22,getNChannel());
	writestr4(buffer + 24,make_le4(getRate()));
	l = getNChannel();
	if(isType16()) l *= 2;
	writestr4(buffer + 28,make_le4(getRate() * l));
	if(isType8()) writestr2(buffer + 32,make_le2(1));
	if(isType16()) writestr2(buffer + 32,make_le2(2));
	if(isType8()) writestr2(buffer + 34,make_le2(8));
	if(isType16()) writestr2(buffer + 34,make_le2(16));
	sprintf(buffer + 36,"data");
	l = getLength() * getNChannel();
	if(isType16()) l *= 2;
	l = (l & 1 ? l + 1 : l);
	writestr4(buffer + 40,make_le4(l));
	fwrite(buffer,44,1,fp);
	for(t = 0;t < getLength();t++)
	{
		for(i = 0;i < (unsigned int)getNChannel();i++)
		{
			if(isType8())
			{
				char	d;

				d = data[i].d8->at(t);
				switch(encoding)
				{
					case 0x0001: // WAVE_FORMAT_PCM
					case 0x0002: // WAVE_FORMAT_ADPCM
					case 0x0006: // WAVE_FORMAT_ALAW
					case 0x0007: // WAVE_FORMAT_MULAW
					case 0x0010: // WAVE_FORMAT_OKI_ADPCM
					case 0x0015: // WAVE_FORMAT_DIGISTD
					case 0x0016: // WAVE_FORMAT_DIGIFIX
					case 0x0101: // IBM_FORMAT_MULAW
					case 0x0102: // IBM_FORMAT_ALAW
					case 0x0103: // IBM_FORMAT_ADPCM
						d = 127 - d;
					break;
					default:
						d = 127 - d;
				}
				fwrite(&d,sizeof(char),1,fp);
			}
			if(isType16())
			{
				short	d;

				d = data[i].d16->at(t);
				switch(encoding)
				{
					case 0x0001: // WAVE_FORMAT_PCM
					case 0x0002: // WAVE_FORMAT_ADPCM
					case 0x0006: // WAVE_FORMAT_ALAW
					case 0x0007: // WAVE_FORMAT_MULAW
					case 0x0010: // WAVE_FORMAT_OKI_ADPCM
					case 0x0015: // WAVE_FORMAT_DIGISTD
					case 0x0016: // WAVE_FORMAT_DIGIFIX
					case 0x0101: // IBM_FORMAT_MULAW
					case 0x0102: // IBM_FORMAT_ALAW
					case 0x0103: // IBM_FORMAT_ADPCM
						d = d;
					break;
					default:
						d = d;
				}
				fwrite(&d,sizeof(short),1,fp);
			}
		}
	}
	if((getLength() & 1) && isType8())
	{
		char	d;

		d = 127 - data[0].d8->at(t);
		fwrite(&d,sizeof(char),1,fp);
	}
	fclose(fp);

	return 0;
}

void	CSample::setSample(CSample *s)
{
	if(this == s)
	{
		printf("same sample\n");
		return;
	}
	clear();
	len = s->len;
	rate = s->rate;
	if(s->name) name = strdup(s->name);
	else name = NULL;
	loop[0] = s->loop[0];
	loop[1] = s->loop[1];
	loop[2] = s->loop[2];
	sample_mode = s->sample_mode;
	addChannel(s->data.size());
	set(0,s->len,s);
}

void	CSample::equalize(CSample *s)
{
	if(this == s)
	{
		printf("same sample\n");
		return;
	}
	clear();
	//len = s->len;
	rate = s->rate;
	if(name) free(name);
	if(s->name) name = strdup(s->name);
	else name = NULL;
	loop[0] = s->loop[0];
	loop[1] = s->loop[1];
	loop[2] = s->loop[2];
	sample_mode = s->sample_mode;
	addChannel(s->data.size());
}

void	CSample::equalize(CSample &s)
{
	if(this == &s)
	{
		printf("same sample\n");
		return;
	}
	clear();
	//len = s->len;
	rate = s.rate;
	if(name) free(name);
	if(s.name) name = strdup(s.name);
	else name = 0;
	loop[0] = s.loop[0];
	loop[1] = s.loop[1];
	loop[2] = s.loop[2];
	sample_mode = s.sample_mode;
	addChannel(s.data.size());
}

int	CSample::playback(uint s,uint e)
{
	current_pos = start_pos = MAX(s,0);
	end_pos = MIN(e,len);

	dsp_speed = rate;
	if(isType8())
		dsp_samplesize = 8;
	else
		dsp_samplesize = 16;
	if(isMono())
		dsp_stereo = 0;
	else
		dsp_stereo = 1;

	audio[0] = open (audio_dev[0], O_WRONLY, 0);
	if(audio[0] == -1)
	{
		return -1;
	}
	audio[1] = open (audio_dev[1], O_WRONLY, 0);
	
	if(ioctl(audio[0],SNDCTL_DSP_SPEED,&dsp_speed) == -1)
	{
		close(audio[0]);
		audio[0] = -1;
		if(audio[1] != -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
		return -2;
	}
	if(audio[1] != -1)
	{
		if(ioctl(audio[1],SNDCTL_DSP_SPEED,&dsp_speed) == -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
	}
	if(ioctl(audio[0],SNDCTL_DSP_SAMPLESIZE,&dsp_samplesize) == -1)
	{
		close(audio[0]);
		audio[0] = -1;
		if(audio[1] != -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
		return -3;
	}
	if(audio[1] != -1)
	{
		if(ioctl(audio[1],SNDCTL_DSP_SAMPLESIZE,&dsp_samplesize) == -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
	}
	if(ioctl(audio[0],SNDCTL_DSP_STEREO,&dsp_stereo) == -1)
	{
		close(audio[0]);
		audio[0] = -1;
		if(audio[1] != -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
		return -4;
	}
	if(audio[1] != -1)
	{
		if(ioctl(audio[1],SNDCTL_DSP_STEREO,&dsp_stereo) == -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
	}
	audio_buf_info	ospace;
	if(ioctl(audio[0],SNDCTL_DSP_GETOSPACE,&ospace) >= 0)
	{
		printf("Fragsize for device 0 = %i\n",ospace.fragsize);
		printf("Fragstotal for device 0 = %i\n",ospace.fragstotal);
		printf("Fragments for device 0 = %i\n",ospace.fragments);
		printf("Bytes for device 0 = %i\n",ospace.bytes);
		buffer_size = ospace.bytes;
	}
	else
	{
		int		subdev = 1;
		
		ioctl(audio[0],SNDCTL_DSP_SUBDIVIDE,&subdev);
		buffer_size = 11025 * (isMono() ? 1 : 2) * 2;
		ioctl(audio[0],SNDCTL_DSP_GETBLKSIZE,&buffer_size);
	}
	buffer_size = MIN(buffer_size,m_writesamples * (isType8() ? 1 : 2) * (isMono() ? 1 : 2));
	printf("Buffersize for device 0 = %i\n",buffer_size);
	if(audio[1] != -1)
	{
		int	size = 11025 * (isMono() ? 1 : 2) * 2;
		
		ioctl(audio[1],SNDCTL_DSP_GETBLKSIZE,&size);
		printf("Buffersize for device 1 = %i\n",buffer_size);
		if(buffer_size != size)
		{
			close(audio[1]);
			audio[1] = -1;
		}
	}
	if((buffer[0] = malloc(buffer_size)) == NULL)
	{
		close(audio[0]);
		audio[0] = -1;
		if(audio[1] != -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
		return -5;
	}
	if((lb[0] = malloc(buffer_size)) == NULL)
	{
		free(buffer[0]);
		close(audio[0]);
		audio[0] = -1;
		if(audio[1] != -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
		return -4;
	}
	if((rb[0] = malloc(buffer_size)) == NULL)
	{
		free(buffer[0]);
		free(lb[0]);
		close(audio[0]);
		audio[0] = -1;
		if(audio[1] != -1)
		{
			close(audio[1]);
			audio[1] = -1;
		}
		return -4;
	}
	if(audio[1] != -1)
	{
		if((buffer[1] = malloc(buffer_size)) == NULL)
		{
			close(audio[1]);
			audio[1] = -1;
		}
	}
	if(audio[1] != -1)
	{
		if((lb[1] = malloc(buffer_size)) == NULL)
		{
			free(buffer[1]);
			close(audio[1]);
			audio[1] = -1;
		}
	}
	if(audio[1] != -1)
	{
		if((rb[1] = malloc(buffer_size)) == NULL)
		{
			free(buffer[1]);
			free(lb[1]);
			close(audio[1]);
			audio[1] = -1;
		}
	}

	spb = buffer_size / (dsp_samplesize / 8 * (dsp_stereo + 1));

	return 0;
}

int	CSample::playbackevent()
{
	uint		t,i;

	t = current_pos;
	if(t > end_pos)
	{
		close(audio[0]);
		audio[0] = -1;
		free(buffer[0]);
		free(lb[0]);
		free(rb[0]);
		
		if(audio[1] != -1)
		{
			close(audio[1]);
			audio[1] = -1;
			free(buffer[1]);
			free(lb[1]);
			free(rb[1]);
		}

		return 0;
	}
	else
	{
		if(dsp_samplesize == 8)
		{
			get(0,t,buffer_size,(char*)lb[0]);
			if(dsp_stereo)
				get(1,t,buffer_size,(char*)rb[0]);
			if(audio[1] != -1)
			{
				get(2,t,buffer_size,(char*)lb[1]);
				get(3,t,buffer_size,(char*)rb[1]);
			}
		}
		else
		{
			get(0,t,buffer_size / 2,(short*)lb[0]);
			if(dsp_stereo)
				get(1,t,buffer_size / 2,(short*)rb[0]);
			if(audio[1] != -1)
			{
				get(2,t,buffer_size / 2,(short*)lb[1]);
				get(3,t,buffer_size / 2,(short*)rb[1]);
			}
		}
		for(i = 0;i < spb;i++)
		{
			if(t + i > end_pos)
			{
				for(;i < spb;i++)
				{
					if(dsp_samplesize == 8)
					{
						if(dsp_stereo) 
						{
							((char*)buffer[0])[i * 2] = 127;
							((char*)buffer[0])[i * 2 + 1] = 127;
							if(audio[1] != -1)
							{
								((char*)buffer[1])[i * 2] = 127;
								((char*)buffer[1])[i * 2 + 1] = 127;
							}
						}
						else
						{
							((char*)buffer[0])[i] = 127;
							if(audio[1] != -1)
							{
								((char*)buffer[1])[i] = 127;
							}
						}
					}
					else
					{
						if(dsp_stereo) 
						{
							((short*)buffer[0])[i * 2] = 0;
							((short*)buffer[0])[i * 2 + 1] = 0;
							if(audio[1] != -1)
							{
								((short*)buffer[1])[i * 2] = 0;
								((short*)buffer[1])[i * 2 + 1] = 0;
							}
						}
						else
						{
							((short*)buffer[0])[i] = 0;
							if(audio[1] != -1)
							{
								((short*)buffer[1])[i] = 0;
							}
						}
					}
				}
				write(audio[0],buffer,spb);
				close(audio[0]);
				free(buffer[0]);
				free(lb[0]);
				free(rb[0]);
				audio[0] = -1;
				if(audio[1] != -1)
				{
					write(audio[1],buffer,spb);
					close(audio[1]);
					free(buffer[1]);
					free(lb[1]);
					free(rb[1]);
					audio[1] = -1;
				}

				return 0;
			}
			if(dsp_samplesize == 8)
			{
				if(dsp_stereo) 
				{
					((char*)buffer[0])[i * 2] = 127 - ((char*)lb[0])[i];
					((char*)buffer[0])[i * 2 + 1] = 127 -((char*)rb[0])[i];
				}
				else
				{
				 ((char*)buffer[0])[i] = 127 - ((char*)lb[0])[i];
				}
			}
			else
			{
				if(dsp_stereo) 
				{
					((short*)buffer[0])[i * 2] = (((short*)lb[0])[i]);
					((short*)buffer[0])[i * 2 + 1] = (((short*)rb[0])[i]);
				}
				else
				{
				 ((short*)buffer[0])[i] = (((short*)lb[0])[i]);
				}
			}
		}
		int	written;
		if((written = write(audio[0],buffer[0],buffer_size)) != buffer_size)
		{
			close(audio[0]);
			audio[0] = -1;
			free(buffer[0]);
			free(lb[0]);
			free(rb[0]);
			if(audio[1] != -1)
			{
				close(audio[1]);
				free(buffer[1]);
				free(lb[1]);
				free(rb[1]);
				audio[1] = -1;
			}
			return -5;
		}
		if(audio[1] != -1)
		{
			if(write(audio[1],buffer[1],buffer_size) != buffer_size)
			{
				close(audio[1]);
				audio[1] = -1;
				free(buffer[1]);
				free(lb[1]);
				free(rb[1]);
			}
		}
	}

	current_pos += spb;

	return 0;
}

int	CSample::playback(double s,double e)
{
	return playback(s * rate,e * rate);
}

int	CSample::playback()
{
	return playback(0,len);
}

int	CSample::play(uint s,uint e)
{
	uint		t,i;

	dsp_speed = rate;
	if(isType8())
		dsp_samplesize = 8;
	else
		dsp_samplesize = 16;
	if(isMono())
		dsp_stereo = 0;
	else
		dsp_stereo = 1;

	audio[0] = open (audio_dev[0], O_WRONLY, 0);
	if(audio[0] == -1)
	{
		return -1;
	}
	if(ioctl(audio[0],SNDCTL_DSP_SPEED,&dsp_speed) == -1)
	{
		close(audio[0]);
		audio[0] = -1;
		return -2;
	}
	if(ioctl(audio[0],SNDCTL_DSP_SAMPLESIZE,&dsp_samplesize) == -1)
	{
		close(audio[0]);
		audio[0] = -1;
		return -3;
	}
	if(ioctl(audio[0],SNDCTL_DSP_STEREO,&dsp_stereo) == -1)
	{
		close(audio[0]);
		audio[0] = -1;
		return -4;
	}
	ioctl(audio[0],SNDCTL_DSP_GETBLKSIZE,&buffer_size);
	if((buffer[0] = malloc(buffer_size)) == NULL)
	{
		close(audio[0]);
		audio[0] = -1;
		return -4;
	}
	if((lb[0] = malloc(buffer_size)) == NULL)
	{
		free(buffer[0]);
		close(audio[0]);
		audio[0] = -1;
		return -4;
	}
	if((rb[0] = malloc(buffer_size)) == NULL)
	{
		free(buffer[0]);
		free(lb[0]);
		close(audio[0]);
		audio[0] = -1;
		return -4;
	}

	spb = buffer_size / (dsp_samplesize / 8 * (dsp_stereo + 1));

	for(t = s;t < e;t += spb)
	{
		current_pos = t;
		if(dsp_samplesize == 8)
		{
			get(0,t,buffer_size,(char*)lb[0]);
			if(dsp_stereo) get(1,t,buffer_size,(char*)rb[0]);
		}
		else
		{
			get(0,t,buffer_size / 2,(short*)lb[0]);
			if(dsp_stereo) get(1,t,buffer_size / 2,(short*)rb[0]);
		}
		for(i = 0;i < spb;i++)
		{
			if(t + i > e)
			{
				write(audio[0],buffer,i);
				close(audio[0]);
				free(buffer[0]);
				free(lb[0]);
				free(rb[0]);
				audio[0] = -1;

				return 0;
			}
			if(dsp_samplesize == 8)
			{
				if(dsp_stereo) 
				{
					((char*)buffer[0])[i * 2] = ((char*)lb[0])[i];
					((char*)buffer[0])[i * 2 + 1] = ((char*)rb[0])[i];
				}
				else ((char*)buffer[0])[i] = ((char*)lb[0])[i];
			}
			else
			{
				if(dsp_stereo) 
				{
					((short*)buffer[0])[i * 2] = ((short*)lb[0])[i];
					((short*)buffer[0])[i * 2 + 1] = ((short*)rb[0])[i];
				}
				else ((short*)buffer[0])[i] = ((short*)lb[0])[i];
			}
		}
		if(write(audio[0],buffer,buffer_size) != buffer_size)
		{
			close(audio[0]);
			audio[0] = -1;
			free(buffer[0]);
			free(lb[0]);
			free(rb[0]);
			return -5;
		}
	}

	close(audio[0]);
	audio[0] = -1;
	free(buffer[0]);
	free(lb[0]);
	free(rb[0]);

	return 0;
}

int	CSample::play(double s,double e)
{
	return play(s * rate,e * rate);
}

int	CSample::play()
{
	return play(0,len);
}

void	CSample::stop()
{
	close(audio[0]);
	audio[0] = -1;
	if(audio[1])
	{
		close(audio[1]);
		audio[1] = -1;
	}
}

uint	CSample::currentPlayPos()
{
	return current_pos;
}

double	CSample::currentPlayTime()
{
	return (double)current_pos / (double)rate;
}

int	CSample::playing()
{
	return (audio[0] != -1);
}

QString	CSample::getAudioDevice(int n)
{
	return audio_dev[n];
}

void	CSample::setAudioDevice(QString ad,int n)
{
	audio_dev[n] = ad;
}

/*
void	CSample::flat(QString format,void **data)
{
	QString		str;
	void		*tmp;
	int		t,n,ts,type;

	s = kswp->getSelectionStart();
	e = kswp->getSelectionEnd();
	l = e - s;

	format = "AUDIOSIGNAL:";

	str.sprintf("NC=%i",n = cnpsample->getNChannel());
	
	format += str + ";Type=";

	if(isType8())
	{
		format += "I8";
		type = 0;
		ts = sizeof(char);
	}
	if(isType16())
	{
		format += "I16";
		type = 1;
		ts = sizeof(short);
	}
	if(isType32())
	{
		format += "I32";
		type = 2;
		ts = sizeof(int);
	}
	if(isTypeFL())
	{
		format += "FLT";
		type = 3;
		ts = sizeof(float);
	}
	if(isTypeDB())
	{
		format += "DBL";
		type = 4;
		ts = sizeof(double);
	}

	tmp = malloc(ts * n * l);

	for(t = 0;t < n;t++)
	{
		switch(type)
		{
			case 0:
				get(t,s,l,(char*)(tmp + n * (ts * l)));
			break;
			case 1:
				get(t,s,l,(short*)(tmp + n * (ts * l)));
			break;
			case 2:
				get(t,s,l,(int*)(tmp + n * (ts * l)));
			break;
			case 3:
				get(t,s,l,(float*)(tmp + n * (ts * l)));
			break;
			case 4:
				get(t,s,l,(double*)(tmp + n * (ts * l)));
			break;
		}
	}
	*data = tmp;
}

void	CSample::setFlat(QString format,void **data)
{
	QString		str;
	void		*tmp;
	int		t,n,ts,type;

	s = kswp->getSelectionStart();
	e = kswp->getSelectionEnd();
	l = e - s;

	str = format.left(12);
	if(str != "AUDIOSIGNAL:") return;

	str = format.mid(12,-1);

	str.sprintf("NC=%i",n = cnpsample->getNChannel());
	
	format += str + ";Type=";

	if(isType8())
	{
		format += "I8";
		type = 0;
		ts = sizeof(char);
	}
	if(isType16())
	{
		format += "I16";
		type = 1;
		ts = sizeof(short);
	}
	if(isType32())
	{
		format += "I32";
		type = 2;
		ts = sizeof(int);
	}
	if(isTypeFL())
	{
		format += "FLT";
		type = 3;
		ts = sizeof(float);
	}
	if(isTypeDB())
	{
		format += "DBL";
		type = 4;
		ts = sizeof(double);
	}

	tmp = malloc(ts * n * l);

	for(t = 0;t < n;t++)
	{
		switch(type)
		{
			case 0:
				get(t,s,l,(char*)(tmp + n * (ts * l)));
			break;
			case 1:
				get(t,s,l,(short*)(tmp + n * (ts * l)));
			break;
			case 2:
				get(t,s,l,(int*)(tmp + n * (ts * l)));
			break;
			case 3:
				get(t,s,l,(float*)(tmp + n * (ts * l)));
			break;
			case 4:
				get(t,s,l,(double*)(tmp + n * (ts * l)));
			break;
		}
	}
	*data = tmp;
}
*/

#ifdef TEST_KSAMPLE

#include <stdio.h>

int	main(int argc,char **argv)
{
	CSample		s((uint)0);

	if(argc < 2) return 0;

	s.loadWave(argv[1]);

	printf("Wave:    %s\n",argv[1]);
	printf("Channel: %i\n",s.getNChannel());
	printf("Length:  %i\n",s.getLength());
	printf("Bits:    %i\n",(s.isType16()?16:8));

	return 1;
}

#endif



/**  */
int CSample::bufferSize()
{
	return buffer_size;
}

/** Swaps the both channels n1 and n2 */
void CSample::swapChannel(int n1,int n2)
{
	sampledata	d;
	
	if(n1 < 0 || (unsigned int)n1 >= data.size())
		return;
	if(n2 < 0 || (unsigned int)n2 >= data.size())
		return;
	
	d = data[n1];
	data[n1] = data[n2];
	data[n2] = d;
}

/** sets the number of samples to write per playevent call */
void CSample::setWriteSamples(int s)
{
	if(s < 128)
		s = 128;
	if(s > 128 * 1024)
		s = 128 * 1024;
		
	m_writesamples = s;
}
/** returns the number of samples to write per playevent call */
int CSample::writeSamples()
{
	return m_writesamples;
}
