/*
    KSrnd
    
    A control panel for the miroMEDIA Surround Dolby Pro Logic decoder.

    Copyright (C) 1998-2000 Oliver Gantz (o.gantz@tu-bs.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ----
    3D-Phonic(R) is a registered trademark of VICTOR COMPANY OF JAPAN, LIMITED.
    DOLBY and PRO LOGIC are trademarks of Dolby Laboratories Licensing Corporation.
    miro(R) is a registered trademark of miro Computer Products AG.
*/

#include "ksrnd.h"
#include "ksrnd.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <unistd.h>

#include <qaccel.h>
#include <qpopmenu.h>	// <qpopupmenu.h>
#include <qpixmap.h>
#include <qstring.h>

#include <kiconloader.h>
#include <ktopwidget.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kmsgbox.h>
#include <kapp.h>
#include <kwm.h>

#include "ksrndwidget.h"
#include "ksrndprefs.h"
#include "ksrndsetup.h"
#include "dockwidget.h"


extern KApplication * mykapp;


KSrnd::KSrnd( const char * name ) : KTopLevelWidget( name )
{
	QPixmap pix;
	KIconLoader * KIL = mykapp->getIconLoader();
	QAccel * qAcc = new QAccel( this );

	KsConfig = mykapp->getConfig();

	startVisible = TRUE;

	dockedAndClosed = FALSE;
	
	prefs = new KSrndPrefs( 0 );
	prefs->reset();
	connect( prefs, SIGNAL(apply_clicked()), this, SLOT(newPrefs()) );

	setup = new KSrndSetup( 0, prefs->deviceText() );
	setup->reset();

	view = new KSrndWidget( this, 0 );
	setView(view);
	connect( view, SIGNAL(set_mode(int)), this, SLOT(setMode(int)) );
	connect( view, SIGNAL(set_volume(int)), this, SLOT(setVolume(int)) );
	connect( view, SIGNAL(set_bypass(bool)), this, SLOT(setBypass(bool)) );

	QPopupMenu *file_menu = new QPopupMenu();
	file_menu->insertItem( i18n("&Quit"), this, SLOT(quit()), CTRL+Key_Q );
	qAcc->connectItem( qAcc->insertItem(CTRL+Key_Q), this, SLOT(quit()) );

	QPopupMenu *opt_menu = new QPopupMenu();
	opt_menu->insertItem( i18n("&Mixer"), this, SLOT(invokeMixer()) );
	opt_menu->insertSeparator();
	opt_menu->insertItem( i18n("Hide Menubar"), this, SLOT(toggleMenuBar() ), CTRL+Key_M );
	qAcc->connectItem( qAcc->insertItem(CTRL+Key_M), this, SLOT(toggleMenuBar()) );
	opt_menu->insertItem( i18n("Toggle Toolbar"), this, SLOT(toggleToolBar()), CTRL+Key_T );
	qAcc->connectItem( qAcc->insertItem(CTRL+Key_T), this, SLOT(toggleToolBar()) );
	opt_menu->insertSeparator();
	opt_menu->insertItem( i18n("&Preferences..."), this, SLOT(showPrefs()) );
	opt_menu->insertItem( i18n("Mode &Settings..."), this, SLOT(showSetup()) );

	QPopupMenu *help_menu = new QPopupMenu();
	help_menu->insertItem( i18n("&Contents"), this, SLOT(invokeHelp()), Key_F1 );
	qAcc->connectItem( qAcc->insertItem(Key_F1), this, SLOT(invokeHelp()) );
	help_menu->insertSeparator();
	help_menu->insertItem( i18n("&About ksrnd..."), this, SLOT(showAbout()) );

	menuBar = new KMenuBar( this );
	menuBar->insertItem( i18n("&File"), file_menu );
	menuBar->insertItem( i18n("&Options"), opt_menu );
	menuBar->insertSeparator();
	menuBar->insertItem( i18n("&Help"), help_menu );

	setMenu( menuBar );
	
	if (menuBarOn = prefs->isMenuBar())
		menuBar->show();
	else
		menuBar->hide();

	toolBar = new KToolBar( this );
	pix = KIL->loadIcon( "mixer.xpm" );
	toolBar->insertButton( pix, 0, SIGNAL(clicked()), this, SLOT(invokeMixer()), TRUE, i18n("Mixer") );
	pix = KIL->loadIcon( "tick.xpm" );
	toolBar->insertButton( pix, 1, SIGNAL(clicked()), this, SLOT(showPrefs()), TRUE, i18n("Preferences") );
	pix = KIL->loadIcon( "configure.xpm" );
	toolBar->insertButton( pix, 2, SIGNAL(clicked()), this, SLOT(showSetup()), TRUE, i18n("Mode Settings") );
	pix = KIL->loadIcon( "help.xpm" );
	toolBar->insertButton( pix, 3, SIGNAL(clicked()), this, SLOT(invokeHelp()), TRUE, i18n("Help Contents") );
	pix = KIL->loadIcon( "exit.xpm" );
	toolBar->insertButton( pix, 4, SIGNAL(clicked()), this, SLOT(quit()), TRUE, i18n("Quit") );
	toolBar->alignItemRight( 4, TRUE );

	addToolBar( toolBar );

	if (toolBarOn = prefs->isToolBar())
		toolBar->show();
	else
		toolBar->hide();

	dock_widget = new DockWidget( 0 );
	connect( dock_widget, SIGNAL(dock_mode(int)), this, SLOT(setMode(int)) );
	connect( dock_widget, SIGNAL(dock_bypass(bool)), this, SLOT(setBypass(bool)) );
	connect( dock_widget, SIGNAL(dock_toggle_visible()), this, SLOT(toggleVisible()) );
	connect( dock_widget, SIGNAL(dock_quit()), mykapp, SLOT(quit()) );

	if (prefs->isToolTips()) {
		prefs->addToolTips( TRUE );
		setup->addToolTips( TRUE );
		view->addToolTips( TRUE );
	}

	resetMain();

	if (prefs->isDocking()) {
		dock_widget->dock();
		startVisible = !prefs->startDocked();
	}

	hide();
}


KSrnd::~KSrnd()
{
	delete menuBar;
	delete toolBar;

	if (prefs->isSave())
		saveMain();
}


void KSrnd::resetMain()
{
	KsConfig->setGroup( "Main" );
	setMode( KsConfig->readNumEntry("Mode", 0) );
	setVolume( KsConfig->readNumEntry("Volume", 16) );
	setBypass( KsConfig->readBoolEntry("Bypass", TRUE) );

	dock_widget->markVisible(isVisible());
}


void KSrnd::saveMain()
{
	KsConfig->setGroup( "Main" );
	KsConfig->writeEntry( "Mode", mode );
	KsConfig->writeEntry( "Volume", volume );
	KsConfig->writeEntry( "Bypass", bypass );
	KsConfig->sync();
}


void KSrnd::setMode( int mode )
{
	this->mode = mode;

	view->setMode( mode );
	dock_widget->markMode( mode );
	setup->setMode( mode );
}


void KSrnd::setVolume( int vol )
{
	volume = vol;

	view->setVolume( vol );
	setup->setVolume( vol );
}


void KSrnd::setBypass( bool on )
{
	bypass = on;

	view->setBypass( on );
	dock_widget->markBypass( on );
	setup->setBypass( on );
}


void KSrnd::toggleVisible()
{
	bool visible = isVisible();

	if (visible) {
		dockedAndClosed = TRUE;
		hide();
		recreate( 0,0,QPoint(x(),y()), FALSE );
		mykapp->setTopWidget( this );
	}
	else {
		show();
		dockedAndClosed = FALSE;
		KWM::activate( winId() );
	}
}


void KSrnd::newPrefs()
{
	prefs->addToolTips( prefs->isToolTips() );
	setup->addToolTips( prefs->isToolTips() );
	view->addToolTips( prefs->isToolTips() );

	if (prefs->isDocking() != dock_widget->isDocked()) {
		if (prefs->isDocking()) {
			delete dock_widget;
			createDockWidget();
			dock_widget->markMode( mode );
			dock_widget->markBypass( bypass );
			dock_widget->dock();
		}
		else
			dock_widget->undock();
	}

	if ( prefs->isMenuBar() != menuBarOn )
		toggleMenuBar();
	if ( prefs->isToolBar() != toolBarOn )
		toggleToolBar();
}


void KSrnd::invokeMixer()
{
	if (fork() == 0) {
		execlp(prefs->mixerText(), prefs->mixerText(), 0);
		exit(1);
	}
}


void KSrnd::showPrefs()
{
	prefs->show();
}


void KSrnd::showSetup()
{
	setup->show();
}


void KSrnd::invokeHelp()
{
	mykapp->invokeHTMLHelp( "ksrnd/ksrnd.html", "" );
}


void KSrnd::showAbout()
{
	QString msg;
	char vers[10];

	setup->getSrndVersion( vers );

	msg.sprintf( "KSrnd %s (%s %s)\n", VERSION, i18n("SRND driver"), vers );
	msg += i18n("Copyright  1998-2000 Oliver Gantz (o.gantz@tu-bs.de).\n\n" \
		"Surround Sound control panel for the KDE Desktop Environment.\n\n" \
		"This program is free software and is distributed under the terms\n" \
		"of the GNU General Public License.\n\n" \
		"* 3D-Phonic is a registered trademark of\n" \
		"  VICTOR COMPANY OF JAPAN, LIMITED.\n" \
		"* DOLBY, the double-D symbol and PRO LOGIC are trademarks of\n" \
		"  Dolby Laboratories Licensing Corporation.\n" \
		"* miro is a registered trademark of miro Computer Products AG.");

	KMsgBox::message( 0, i18n("About KSrnd"), msg );
}


void KSrnd::quit()
{
	mykapp->quit();
}


void KSrnd::toggleMenuBar()
{
	menuBarOn = !menuBarOn;
	if (menuBarOn)
		menuBar->show();
	else
		menuBar->hide();
	updateRects();
}


void KSrnd::toggleToolBar()
{
	toolBarOn = !toolBarOn;
	if (toolBarOn)
		toolBar->show();
	else
		toolBar->hide();
	updateRects();
}


void KSrnd::createDockWidget()
{
	dock_widget = new DockWidget( 0 );
	connect( dock_widget, SIGNAL(dock_mode(int)), this, SLOT(setMode(int)) );
	connect( dock_widget, SIGNAL(dock_bypass(bool)), this, SLOT(setBypass(bool)) );
	connect( dock_widget, SIGNAL(dock_toggle_visible()), this, SLOT(toggleVisible()) );
	connect( dock_widget, SIGNAL(dock_quit()), mykapp, SLOT(quit()) );
}


void KSrnd::saveProperties( KConfig * sessionConfig )
{
	sessionConfig->writeEntry( "ksrndvisible", isVisible() );
}


void KSrnd::readProperties( KConfig * sessionConfig )
{
	startVisible = sessionConfig->readBoolEntry( "ksrndvisible", TRUE );
}


void KSrnd::closeEvent( QCloseEvent * e )
{
	dockedAndClosed = TRUE;
	KTopLevelWidget::closeEvent( e );
}


bool KSrnd::event( QEvent * e )
{
	if ((e->type() == Event_Hide) && prefs->isDocking() && !dockedAndClosed) {
		sleep(1);
		if (!KWM::isIconified(winId()))
			return FALSE;
		hide();
		recreate(0, 0, QPoint(x(), y()), FALSE);
		mykapp->setTopWidget( this );
		return TRUE;
	}
	return KTopLevelWidget::event( e );
}
