	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "main.h"
#include "synth.h"
#ifdef HAVE_MINI_STL
#include <ministl/list>
#else
#include <list>
#endif

#include <kapp.h>
#include <ktopwidget.h>
#include <kfiledialog.h>
#include <ksimpleconfig.h>
#include <qlistbox.h>
#include <qtablevw.h>
#include <qpainter.h>
#include <qdrawutl.h>
#include <qwidget.h>
#include <qlayout.h>
#include <qframe.h>
#include <qpixmap.h>
#include <qmsgbox.h>
#include <string.h>
#include <unistd.h>
#include "module.h"
#include "autorouter.h"
#include "portpropdlg.h"
#include "execdlg.h"
#include "utils.h"

#define KSBUILD_VER "ksbuild 0.1.3"

void ModuleWidget::addModule ( Module *module )
{
	newModule = module;
}

ModulePort *ModuleWidget::findPort(Module *m, int xoffset, int direction)
{
	list<ModulePort *> *ports;
	list<ModulePort *>::iterator i;

	int n;

	if(direction == 0) ports = &m->inports; else ports = &m->outports;

	i = ports->begin();
	n = xoffset-1-direction;

	if(n<ports->size() && n>=0)
	{
		while(n>0) { n--; i++; }
		return(*i);
	}
	return(NULL);
}

void ModuleWidget::mousePressEvent( QMouseEvent *e )
{
	if ( e->button() == LeftButton )
	{
		movestart = e->pos();
		int x = findCol(e->x());
		int y = findRow(e->y());

		int cellx = 0, celly=0;

		colXPos(x,&cellx);
		rowYPos(y,&celly);

		cellx = e->x() - cellx;
		celly = e->y() - celly;

		//printf("clicked at %d,%d\n",cellx,celly);

		if(newModule)
		{
			newModule->x = x;
			newModule->y = y;
			if(hasSpace(newModule,x,y))
			{
				unselectAll();
				ModuleList->push_back(newModule);
				newModule->selected = true;
				newModule->modnr = modnr++;
				redrawModule(newModule);
				reRoute();
				
				newModule = NULL;
			}
			else
			{
				printf("beep();\n");
				delete newModule;
				newModule = NULL;
			}
			return;  // done then
		}
	
		Module *m = findModule(x,y);

		ModulePort *port = NULL;

		if(m)
		{
			for(int direction = 0;direction < 2; direction++)
			{
				port = findPort(m,x-(m->x),direction);
				if(port)
				{
					QPoint clickpoint(cellx,celly);
					if(port->clickrect.contains(clickpoint))
					{
						printf("clicked at port %s.%s\n",m->type.data(),port->description.data());
						if(port->selected)
						{
							PortPropDlg *ppd = new PortPropDlg(0,port);
							ppd->exec();
							port->selected = false;
							selectedports--;
						}
						else
						{
							port->selected = true;
							selectedports++;
						}
						if(selectedports == 2)
						{
							if(lastport->direction == port->direction)
							{
								port->selected = false;
								selectedports--;
    							QMessageBox::message( i18n("KSynth Warning"),
i18n("You can only connect an IN-port with an OUT-port, \n\
not two ports with the same direction") );

							}
							else
							{
								if(lastport->direction == ModulePort::in)
									connectPorts(port,lastport);
								else
									connectPorts(lastport,port);
							}
						}
						lastport = port;
						redrawModule(m);
						reRoute();
						return;
					}
				}
			}
		}

		//
		{
			beginUpdate();
			// select operations:
	
			// maintain selected group when
			// - clicking into a module that was already selected
			// - pressing the shift button
	
			bool group = (m && (m->selected || (e->state() & ShiftButton)));
	
			// unselect all before moving, when not grouped
			if(!group) unselectAll();
	
			if(m)
			{
				m->selected = true;
				redrawModule(m);
			}
			endUpdate();
		}
	}
}

void ModuleWidget::mouseMoveEvent( QMouseEvent *e )
{
	int dx = (e->x() - movestart.x())/cellsize;
	int dy = (e->y() - movestart.y())/cellsize;

	if(dx == 0 && dy == 0) return;

	list<Module *>::iterator i;

	for(i = ModuleList->begin();i != ModuleList->end();i++)
	{
		Module *m = *i;
		if(m->selected && !hasSpace(m,m->x+dx,m->y+dy,true)) return;
	}

	beginUpdate();
	for(i = ModuleList->begin();i != ModuleList->end();i++)
	{
		Module *m = *i;
		if(m->selected)
		{
			redrawModule(m);
			m->x += dx;
			m->y += dy;
			redrawModule(m);
		}
	}
	endUpdate();
	
	movestart.setX(movestart.x()+dx*cellsize);
	movestart.setY(movestart.y()+dy*cellsize);
	reRoute();
}

bool ModuleWidget::hasSpace(Module *m,int destx,int desty,bool ignore_selected)
{
	int ddx,ddy;

	for (ddx = 0; ddx < m->width; ddx++)
	{
		for (ddy = 0; ddy < m->height; ddy++)
		{
			if(findModule(destx+ddx,desty+ddy,ignore_selected)) return(false);
		}
	}

	if(destx < 0) return(false);
	if(desty < 0) return(false);
	if(destx+m->width > numCols()) return(false);
	if(desty+m->height > numRows()) return(false);

	return(true);
}

void ModuleWidget::mouseReleaseEvent( QMouseEvent *e )
{
	// something to do here?
}

void ModuleWidget::paintCellBackground(QPainter *p, int y, int x)
{
	QColor bgcolor;

	if((y & 1) == 1)
		bgcolor = QColor(168,168,168);
	else
		bgcolor = QColor(146,168,146);

	p->fillRect(0,0,cellsize,cellsize,QBrush(bgcolor));

	p->setPen(bgcolor.dark(115));
	p->drawLine(0,0,0,cellsize-1);
	p->drawLine(0,0,cellsize-1,0);

	if(x == (numCols()-1))
		p->drawLine(cellsize-1,0,cellsize-1,cellsize-1);
	if(y == (numRows()-1))
		p->drawLine(0,cellsize-1,cellsize-1,cellsize-1);
}

Module *ModuleWidget::findModule(int x,int y,bool ignore_selected)
{
	list<Module *>::iterator i;

	for(i = ModuleList->begin();i != ModuleList->end();i++)
	{
		Module *m = *i;
		if((x >= m->x) && (x < m->x+m->width) && (m->y == y))
		{
			if(!m->selected || !ignore_selected) return(m);
		}
	}
	return(NULL);
}

void ModuleWidget::unselectAll()
{
	list<Module *>::iterator module;
	
	for(module = ModuleList->begin();module != ModuleList->end();module++)
	{
		(*module)->selected = false;
		redrawModule(*module);
	}
}

void ModuleWidget::beginUpdate()
{
	inUpdate = true;
}

void ModuleWidget::endUpdate()
{
	inUpdate = false;

	list<QRect>::iterator i;

	for (i = UpdateList.begin(); i != UpdateList.end(); i++)
	{
		redrawCells(*i);
	}

	UpdateList.erase(UpdateList.begin(),UpdateList.end());
}

void ModuleWidget::redrawModule(Module *m)
{
	QRect r = QRect(m->x,m->y,m->width,m->height);

	if(!inUpdate)
	{
		redrawCells(r);
	}
	else
	{
		UpdateList.push_back(r);
	}
}

void ModuleWidget::redrawCells(QRect &r)
{
	int x,y;

	for(x = r.left(); x<r.width()+r.left(); x++)
	{
		for(y = r.top(); y<r.height()+r.top(); y++)
		{
			updateCell(y,x,false);
		}
	}
}

char *ModuleWidget::cropText(QPainter *p, char *text, int maxlen, int& textwidth)
{
	static char label[100];
	char label2[100];
	bool crop_underscore;
	int n,i,crop_pos;

	strncpy(label, text, 99);
	textwidth = cellsize;
	n = strlen(label);
	while((textwidth > maxlen) && (n>0)){
		crop_underscore=false;
		for(i=0;i<n;i++)
		{
			if(label[i] == '_' && !crop_underscore)
			{
				crop_underscore = true;
				crop_pos=i;
			}
		}

		if(crop_underscore)
		{
			strcpy(label2,&label[crop_pos+1]);
			strcpy(label,label2);
			n = strlen(label);
		}
		else
		{
			label[n--] = 0;
		}
		textwidth = p->fontMetrics().width(label);
	}

	return(label);
}


void ModuleWidget::reRoute()
{
// clear everything
	autorouter->clear();

// add modules (those allocate space) to the router
	list<Module *>::iterator i;

	for(i = ModuleList->begin();i != ModuleList->end();i++)
	{
		int x,y,dx,dy;
		Module *m = *i;

		for(x = m->x; x<(m->x)+(m->width); x++)
		{
			for(y = m->y; y<(m->y)+(m->height); y++)
			{
				for(dx=0; dx<2; dx++)
				{
					for(dy=0; dy<2; dy++)
					{
						autorouter->set(x*2+dx,y*2+dy,AutoRouter::solid);
					}
				}
			}
		}
	}

// add connections to the router

	for(i = ModuleList->begin();i != ModuleList->end();i++)
	{
		Module *m = *i;
		ModulePort *p;
		int xoffset = 1;
		
		while((p = findPort(m,xoffset,0)))
		{
			if(p->conntype == ModulePort::dest)
			{
				ModulePort *src, *dest;

				src = p->connection;
				dest = p;

				printf("autoroute add connection port %s.%s to %s.%s\n",
							src->owner->type.data(),src->description.data(),
							dest->owner->type.data(),dest->description.data());

				int x1 = src->owner->x*2+src->portnr*2+4;
				int y1 = src->owner->y*2+1;

				int x2 = dest->owner->x*2+dest->portnr*2+2;
				int y2 = dest->owner->y*2;

				src->route_owner =
					 autorouter->connect(x1,y1,x2,y2,src->route_owner);
			}
			xoffset++;
		}
	}

// redraw everything

	QRect updaterect(0,0,cols,rows);
	redrawCells(updaterect);
}

void ModuleWidget::paintConnection(QPainter *p, int x, int y, int arx, int ary)
{
	long linetype = autorouter->get(arx,ary);
	long ud_owner = -1, lr_owner = -1, lr_break = 0, ud_break = 0;

	autorouter->getowners(arx,ary,ud_owner,lr_owner);

	p->setPen(QColor(255,255,255));

/*
	if(linetype == AutoRouter::none)
	{
		p->drawPoint(x+cellsize/4,y+cellsize/4);
	}
	if(linetype & AutoRouter::solid)
	{
		QBrush whitefill(QColor(255,255,255));

		p->fillRect(x+cellsize/6,y+cellsize/6,cellsize/6,cellsize/6,whitefill);
	}
*/
	x += cellsize/4;
	y += cellsize/4;

	// both used?
	if(ud_owner != -1 && lr_owner != -1)
	{
		// and not of the same owner?
		if(ud_owner != lr_owner)
		{
			// then we'll have to paint one of them broken
			if(ud_owner > lr_owner)
				lr_break = cellsize/8;
			else
				ud_break = cellsize/8;
		}
	}

	if(linetype & AutoRouter::left)
		p->drawLine(x-cellsize/4,y,x-lr_break,y);
	if(linetype & AutoRouter::right)
		p->drawLine(x+cellsize/4,y,x+lr_break,y);
	if(linetype & AutoRouter::up)
		p->drawLine(x,y-cellsize/4,x,y-ud_break);
	if(linetype & AutoRouter::down)
		p->drawLine(x,y+cellsize/4,x,y+ud_break);
}

void ModuleWidget::paintConnections(QPainter *p, int y, int x)
{
	int dx,dy;
	for(dx = 0; dx < 2; dx++)
	{
		for(dy = 0; dy < 2; dy++)
		{
			paintConnection(p,(cellsize*dx)/2,(cellsize*dy)/2,x*2+dx,y*2+dy);
		}
	}
}

void ModuleWidget::paintCell(QPainter *p, int y, int x)
{
	Module *m = findModule(x,y);

	if(m)
	{
		int xoffset = x-m->x;
		int border = cellsize / 10;  // for the logo
		int ltop = (cellsize-border)/2;
		int lbot = (cellsize+border)/2;

		QColor mcolor(43,43,168);
        static QColorGroup g( white, blue, QColor(164,176,242), mcolor.dark(), mcolor, black, black );
		QBrush fill( mcolor );
		QPen textpen(QColor(255,255,180),1);

		if(xoffset == 0)
		{
			paintCellBackground(p,y,x);
			qDrawShadePanel(p,border,border,cellsize-2*border+1,cellsize-2*border+1,
				g, false, 1, &fill);
			p->fillRect(cellsize-border-1,ltop,cellsize,lbot-ltop+1,fill);
			p->setPen(g.light());
			p->drawLine(cellsize-border,ltop-1,cellsize,ltop-1);
			p->setPen(g.dark());
			p->drawLine(cellsize-border,lbot+1,cellsize,lbot+1);
			if(m->pixmap)
			{
				int destsize = (cellsize-4*border);
				float sx = (float)destsize/(float)m->pixmap->width();
				float sy = (float)destsize/(float)m->pixmap->height();

				QWMatrix matrix;
				matrix.scale(sx,sy);
				QPixmap pmscaled = (m->pixmap)->xForm(matrix);
				p->drawPixmap(border*2,border*2,pmscaled);  
			}

			// there should be no connections over a "module logo", so this
			// is for debugging purposes only
			paintConnections(p,y,x);
			return;
		}

		p->fillRect(0,0,cellsize,cellsize,fill);

		/*
         * take care of the bevel lines around the module
         */

		p->setPen(g.light());
		p->drawLine(0,0,cellsize-1,0);
		if(xoffset < 2)
			p->drawLine(0,0,0,cellsize-1);

		p->setPen(g.dark());
		p->drawLine(cellsize-1,cellsize-1,0,cellsize-1);
		if(xoffset == 0 || xoffset == m->width-1)
			p->drawLine(cellsize-1,cellsize-1,cellsize-1,0);

		/*
		 * now draw the ports
		 */
		int direction;

		for(direction = 0;direction < 2; direction++)
		{
			ModulePort *port = findPort(m, xoffset, direction);

			if(port)
			{
				int border = cellsize/7;
				int textwidth;
				char *label = cropText(p, port->description.data(),
											cellsize/2, textwidth);

				port->clickrect = QRect(border,direction * cellsize/2 + border,
					cellsize/2-2*border, cellsize/2-2*border);
			
				QBrush fillport(fill);
				if(port->isinitarg)
				{
					fillport = QColor(128,128,128);
				}
	
				if(port->selected)
				{
					QBrush fillorange(QColor(255,165,0));
					qDrawShadePanel(p, port->clickrect, g,true,2,&fillorange);
				}
				else
				{
					if(port->conntype == ModulePort::none)
					{
						qDrawShadePanel(p, port->clickrect, g,false,2,&fillport);
					}
					if(port->conntype == ModulePort::dest)
					{
						qDrawShadePanel(p, port->clickrect, g,true,2,&fillport);
					}
					if(port->conntype == ModulePort::source)
					{
						qDrawShadePanel(p, port->clickrect, g,true,2,&fillport);
					}

					if(port->conntype == ModulePort::value)
					{
						QBrush fillp(QColor(100,100,255));
						if(port->isinitarg)
						{
							fillp = QColor(180,180,180);
						}
						qDrawShadePanel(p, port->clickrect, g,true,2,&fillp);
					}
					
				}

				p->setPen(textpen);
				p->drawText((cellsize-border)/2,
								(1+direction) * (cellsize/2)-border,label);
			}
		}

		/*
		 * if it was the rightmost part of the module, it has the module name
		 * and the connection to the logo as well
		 */

		if(xoffset == 1)
		{
			// object type label
			int textwidth;
			char *label = cropText(p, m->type.data(), cellsize-4, textwidth);

			p->setPen(textpen);
			p->fillRect(1,cellsize-16,textwidth+7,15,QBrush(g.dark()));
			p->drawText(4,cellsize-5,label);

			// logo connection
			p->setPen(mcolor);
			p->drawLine(0,ltop,0,lbot);
		}

		/*
		 * when selected, draw a line of white dots around the module
		 */

		if(m->selected)
		{
			QPen pen(white,1,DotLine);

			p->setPen(pen);
			p->drawLine(0,0,cellsize-1,0);
			p->drawLine(0,cellsize-1,cellsize-1,cellsize-1);
			if (x == m->x+1)
				p->drawLine(0,0,0,cellsize-1);
			if (x == m->x+m->width-1)
				p->drawLine(cellsize-1,0,cellsize-1,cellsize-1);
		}
	} else paintCellBackground(p,y,x);
	paintConnections(p,y,x);
}

bool ModuleWidget::connectPorts(ModulePort *src, ModulePort *dest)
{
	printf("Connecting (src)%s.%s -> (dest)%s.%s\n",
		src->owner->type.data(),src->description.data(),
		dest->owner->type.data(),dest->description.data());

// actually connect p2 to p1
	dest->conntype = ModulePort::dest;
	dest->connection = src;

	src->conntype = ModulePort::source;

	selectedports = 0;
	src->selected = false;
	dest->selected = false;

	redrawModule(src->owner);
	redrawModule(dest->owner);
}

// ---------------------------------------------------------------------------
// public part of modulewidget
// ---------------------------------------------------------------------------

void ModuleWidget::setZoom(int zoom)
{
	cellsize = (int)(50.0 * (float)zoom/100);

	setCellHeight(cellsize);
	setCellWidth(cellsize);
	updateTableSize();
}

void ModuleWidget::reInit(int nextmodnr)
{
	selectedports = 0;
	newModule = NULL;
	modnr = nextmodnr;

	reRoute();
}

ModuleWidget::ModuleWidget(list<Module *> *ModuleList, QWidget *parent=0, const char *name=0, WFlags f=0)
		:QTableView(parent,name,f)
{
	this->ModuleList = ModuleList;

	cols = 24;
	rows = 32;

	setNumCols(cols);
	setNumRows(rows);
	setTableFlags(Tbl_autoScrollBars);
	setZoom(100);

	modnr = 0;
	selectedports = 0;
	newModule = NULL;
	autorouter = new AutoRouter(cols*2,rows*2);
};

ModuleView::ModuleView(const char *name, KSynthesizer *Synthesizer) : KTopLevelWidget(name)
{
	this->Synthesizer = Synthesizer;

	menubar = new KMenuBar (this, "menubar");

	QPopupMenu *m_file = new QPopupMenu;
	m_file->insertItem("&Open", this , SLOT(open()));
	m_file->insertItem("&Save", this , SLOT(save()));
    m_file->insertSeparator();
	m_file->insertItem("&Execute Module...", this, SLOT(execute()));
    m_file->insertSeparator();
	m_file->insertItem("&Quit", qApp , SLOT(quit()));
	menubar->insertItem ("&File", m_file);

	QPopupMenu *m_view = new QPopupMenu;
	m_view->insertItem("200%", 200);
	m_view->insertItem("150%", 150);
	m_view->insertItem("100%", 100);
	m_view->insertItem("50%", 50);
	menubar->insertItem ("&View", m_view);

	QPopupMenu *m_modules = new QPopupMenu;
	int i;
	for(i=0;i<(Synthesizer->getModuleCount());i++)
	{
		ModuleInfo *minfo;
		if(Synthesizer->getModuleInfo(i,minfo))
		{
			m_modules->insertItem(minfo->name,i);
		}
	}
	menubar->insertItem ("&Modules", m_modules);

    menubar->insertSeparator();

    QString at = KSBUILD_VER;
    at += i18n("\n\n(C) 1998 Stefan Westerfeld <stefan@twc.de>\n\n"
	       "SoundScape synthesizer designer.\n");
		       
    QPopupMenu *m_help = kapp->getHelpMenu(true, at.data());
    menubar->insertItem( i18n("&Help"), m_help);

	modulewidget = new ModuleWidget(&ModuleList, this, "mwidget");

	setMenu (menubar);      
	setView(modulewidget);

	connect(m_view,SIGNAL(activated(int)),modulewidget,SLOT(setZoom(int)));
	connect(m_modules,SIGNAL(activated(int)),this,SLOT(addModule(int)));
}

void ModuleView::clear()
{
	list<Module *>::iterator mi;

	for(mi = ModuleList.begin(); mi != ModuleList.end(); mi++) delete (*mi);

	ModuleList.erase(ModuleList.begin(), ModuleList.end());
}

void ModuleView::open()
{
	char *filename = KFileDialog::getOpenFileName(0,"*.ks",this).data();
	if(filename)
	{
		printf("open... %s\n",filename);

		// check that the file is ok:

		FILE *infile = fopen(filename,"r");

		if(!infile) return;
		fclose(infile);

		clear();
		KSimpleConfig in(filename);

		int mod;
		for(mod=0;;mod++)
		{
			QString groupname;

			groupname.sprintf("module%d",mod);
			in.setGroup(groupname.data());

			char *moduletype = in.readEntry("type").data();
			if(!moduletype)
			{
				printf("module %s not present -> eof?\n",groupname.data());
				printf("setting up ports (restoring connections)\n");
				list<Module *>::iterator mi;
				for(mi=ModuleList.begin();mi != ModuleList.end(); mi++)
				{
					(*mi)->loadPorts(in,ModuleList);
				}
				modulewidget->reInit(mod);
				return;
			}

			int i;
			for(i=0;i<(Synthesizer->getModuleCount());i++)
			{
				ModuleInfo *minfo;
				if(Synthesizer->getModuleInfo(i,minfo))
				{
					if(strcmp(minfo->name,moduletype) == 0)
					{
						printf("loading module %d which has type %s\n",mod,(char *)minfo->name);

						Module *m = new Module(minfo);

						m->modnr = mod;
						m->load(in);
						ModuleList.push_back(m);
					}
				}
			}
		}	
	}
}

void ModuleView::save()
{
	char *filename = KFileDialog::getSaveFileName(0,"*.ks",this).data();
	if(filename)
	{
		printf("save... %s\n",filename);

		// clear the file first:

		FILE *outfile = fopen(filename,"w");

		if(!outfile) return;
		fclose(outfile);

		KSimpleConfig out(filename);

		list<Module *>::iterator i;
		for(i=ModuleList.begin(); i != ModuleList.end(); i++)
			(*i)->save(out);
	}
}

void ModuleView::execute()
{
	list<Module *>::iterator mi;

	Synthesizer->Execute("clear");

	for(mi=ModuleList.begin();mi != ModuleList.end(); mi++)
	{
		QString command = (*mi)->execute();

		fprintf(stderr,"LINE %s\n",command.data());
		Synthesizer->Execute(command.data());
	}

	char *errors;
	Synthesizer->SyntaxCheck(errors);

	ExecDlg *xd = new ExecDlg(0,Synthesizer);
	xd->exec();
	delete xd;

	Synthesizer->Execute("clear");
	return;
}

void ModuleView::addModule(int module)
{
	ModuleInfo *minfo;

	if(Synthesizer->getModuleInfo(module,minfo))
	{
		Module *m = new Module(minfo);
		modulewidget->addModule(m);
	}
}

int main(int argc, char **argv)
{
	KApplication *Application = new KApplication(argc,argv,"ksbuild");

// ---- connect to synthesizer

	CORBA::ORB_var orb = CORBA::ORB_init( argc, argv, "mico-local-orb" );
	CORBA::BOA_var boa = orb->BOA_init( argc, argv, "mico-local-boa" );

	CORBA::Object_var obj =
		orb->bind ("IDL:KSynthesizer:1.0", "inet:localhost:8888");
	if (CORBA::is_nil (obj)) {
		if(QMessageBox::information(0,Application->getCaption(),
		i18n("Synth server unavailable.\nShould I try to start it (or quit)?"),
		QMessageBox::Yes, QMessageBox::No ) == QMessageBox::Yes)
		{
			system("kvt -e synth_server &");
			sleep(3);

			obj = orb->bind ("IDL:KSynthesizer:1.0", "inet:localhost:8888");
			while (CORBA::is_nil (obj))
			{
				if (QMessageBox::warning(0,Application->getCaption(),
					"Synthesizer still not found...", QMessageBox::Retry,
						QMessageBox::Cancel) == QMessageBox::Cancel)
				{
					exit(1);
				}
				obj = orb->bind ("IDL:KSynthesizer:1.0", "inet:localhost:8888");
			}
		}
		else
		{
			exit(1);
		}
	}

	KSynthesizer_var Synthesizer = KSynthesizer::_narrow (obj);

// ---- now Synthesizer is a valid KSynthesizer ;)

	ModuleView *w = new ModuleView("main", Synthesizer);

	w->setCaption("KSynth Module View");
	w->show();

	Application->setMainWidget(w);

	return Application->exec();
}
