/***************************************************************************
                          ktuner.cpp  -  description
                             -------------------
    begin                : Thu Sep 30 1999
    copyright            : (C) 1999 by Ralf Nolden
    email                : Ralf.Nolden@post.rwth-aachen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// ktuner.cpp
// ktuner v0.5 , Carl van Schaik
// e-mail carl@leg.uct.ac.za
//
// Radiotrack II implementation : Leendert Meyer (leendert@usa.net).
//
// RadioTrack II Protocol : Ben Pfaff

#ifndef KTUNER_CPP
#define KTUNER_CPP

// We use the qt moc precompiler to understand signals / slots
#include "ktuner.h"
//#include "images.h"
#include <unistd.h>
#include <qimage.h>
#include <qkeycode.h>
#include <qmsgbox.h>
#include <kmenubar.h>
#include <kapp.h>

int state=1; // Radio on

radioCard  *cardDev;
char       *channelName[100];
int        *channelFreq, numChannels, curChannel, curFreq;
KMenuBar   *buttons;

QColorGroup blu   (QColor(0,0,0),QColor(0,200,0),QColor(0,200,40),
		   QColor(0,120,20),QColor(0,0,0),QColor(0,150,250),
		   QColor(100,100,100));

QColorGroup blud  (QColor(0,0,0),QColor(0,200,0),QColor(0,200,40),
		   QColor(0,120,20),QColor(0,0,0),QColor(0,80,160),
		   QColor(100,100,100));

extern DockWidget *dock_widget;

Radio::Radio(configFile* cf, radioCard *rc, QWidget *parent=0, const char *name=0)
  : QWidget( parent, name )
{
  resize(400,87);

  config = cf;
  cardDev = rc;
  numChannels = 9 + config->getNumMoreChannels();
  CtrlPress = 0;

  setMinimumSize(130,64);
  setMaximumSize(400,87);
  //setBackgroundPixmap(QPixmap("bg.gif"));

  optionsOn=0;
  optBox = new optionBox(config,cardDev);
  optBox->hide();

  Buts = new Controls(this);
  Disp = new DispBox(cf,this);
  Chans = new Presets(cf,this);
  //Chans = new Presets(cf);
  misc = new Misc(Chans);
  Chans->show();
  Tune = new Tuning(cf,this);
  buttons = new KMenuBar(this);

  Buts->move(2,-3);
  Disp->move(2,2);
  Chans->move(131,2);
  Tune->move(-3,62);
  //misc->move(277,42);
  misc->move(143,40);

  dock_widget->resetMenuChannels();
  for (int tem=0;tem < 9;tem ++)
    {
      buttons->insertItem(cf->getName(tem),cf->getFreq(tem));
      dock_widget->addMenuChannel(cf->getFreq(tem),cf->getName(tem));
    }
  buttons->resize(200,40);
  buttons->move(390,2);
  buttons->setTitle(i18n("Channel Bar"));

  Timer = new sleepTimer(this);
  sleepState=0;

  alarm = new alarmClock(this);
  alarm->setAlarm(config->getAlarmHr(), config->getAlarmMin());
  if (config->setAlarm())
    {
      alarm->toggle();
      Disp->toggleAlarm();
    }

  connect(Tune, SIGNAL(freqChanged(int)), Disp, SLOT(setFreq(int)));
  connect(Chans, SIGNAL(sendFreq(int)), Tune, SLOT(freqSet(int)));
  connect(Chans, SIGNAL(sendVol(int)), Disp, SLOT(setVol(int)));
  connect(Chans, SIGNAL(optionsUpdate()), SLOT(doUpdate2()));
  connect(Buts, SIGNAL(mutePressed()), Disp, SLOT(toggleMute()));
  connect(Buts, SIGNAL(onPressed()), Disp, SLOT(toggleOn()));
  connect(Buts, SIGNAL(offPressed()), Disp, SLOT(toggleOff()));
  connect(Timer, SIGNAL(timeUp()), SLOT(timerUp()));
  connect(misc, SIGNAL(sleepToggle()), SLOT(sleepToggle()));
  connect(misc, SIGNAL(optionsToggled()), optBox, SLOT(show()));
  connect(misc, SIGNAL(alarmToggled()), alarm, SLOT(toggle()));
  connect(alarm, SIGNAL(ring()), Buts, SLOT(toggleOn()));
  connect(alarm, SIGNAL(onOff()), Disp, SLOT(toggleAlarm()));
  connect(optBox, SIGNAL(optionsUpdate()), SLOT(doUpdate()));
  connect(optBox, SIGNAL(listenTo(int)), Tune, SLOT(freqSet(int)));
  connect(dock_widget, SIGNAL(sendOptions()), optBox, SLOT(show()));
  connect(dock_widget, SIGNAL(sendMute()), Disp, SLOT(toggleMute()));
  connect(dock_widget, SIGNAL(sendSleep()), SLOT(sleepToggle()));
  connect(dock_widget, SIGNAL(sendAlarm()), alarm, SLOT(toggle()));
  connect(dock_widget, SIGNAL(sendRadio()), SLOT(toggleState()));
  connect(dock_widget, SIGNAL(sendFreq(int)), SLOT(floatPushed(int)));
  connect(buttons, SIGNAL(activated(int)), SLOT(floatPushed(int)));
}

void Radio::sleepToggle()
{
  if (sleepState == 0) sleepState = 1;
  else sleepState = 0;
  Disp->sleep(sleepState);

  if (sleepState && state)
    {
      Timer->setTimer(config->getSleepTime());
      Timer->Go();
    }
  else
    {
      Timer->resetTimer();
    }
  dock_widget->setRadioState(2+sleepState);
}

void Radio::timerUp()
{
  sleepState=0;
  Disp->sleep(sleepState);
  Disp->toggleOff();
}

void Radio::doUpdate()
{
  alarm->setAlarm(config->getAlarmHr(), config->getAlarmMin());
  alarm->toggle();
  alarm->toggle();         // Twice so we keep it on / off

  Chans->reset();
  Disp->setFreq(-1);
}

void Radio::doUpdate2()
{
  doUpdate();
  optBox->configure();
}

void Radio::keyPressEvent( QKeyEvent* e )
{
  if (CtrlPress)
    {
      CtrlPress = 0;
      if (e->key() == Key_Q)
	  qApp->quit();
    }

  switch(e->key())
    {
    case Key_1:
      {
	Chans->setFreq0();
	return;
      }
    case Key_2:
      {
	Chans->setFreq1();
	return;
      }
    case Key_3:
      {
	Chans->setFreq2();
	return;
      }
    case Key_4:
      {
	Chans->setFreq3();
	return;
      }
    case Key_5:
      {
	Chans->setFreq4();
	return;
      }
    case Key_6:
      {
	Chans->setFreq5();
	return;
      }
    case Key_7:
      {
	Chans->setFreq6();
	return;
      }
    case Key_8:
      {
	Chans->setFreq7();
	return;
      }
    case Key_9:
      {
	Chans->setFreq8();
	return;
      }
     case Key_Plus:
      {
	Chans->volUp();
	return;
      }
     case Key_Minus:
      {
	Chans->volDown();
	return;
      }
     case Key_Left:
      {
	Tune->freqDec();
	return;
      }
     case Key_Right:
      {
	Tune->freqInc();
	return;
      }
     case Key_Space:
      {
	cardDev->cardToggle();
	Disp->toggleMute();
	return;
      }
     case (Key_Control):
      {
	CtrlPress = 1;
	return;
      }
   default:
      e->ignore();
      return;
    }
}

void Radio::toggleState()
{
  state=!state;
  if (state) Disp->toggleOn();
  else Disp->toggleOff();
}

void Radio::floatPushed(int freq)
{
  Tune->freqSet(freq);
}

DispBox::DispBox(configFile* cf, QWidget *parent=0, const char *name=0)
  : QWidget( parent, name )
{
  P = parent;

  resize(120,44);
  setBackgroundColor(QColor(0,0,0));
  QImage temp((const char **)blackMarble);
  QImage temp2((const char **)ktuner_small);
  bgPix = temp;
  cRPix = temp2;
  setBackgroundPixmap(bgPix);

  Mute=0;
  alarm=0;
  timer=0;
  curFreq = cf->getFreq(cf->getDefault());
  curChannel = cf->getDefault();

  QColorGroup redgrn(QColor(0,0,0),QColor(0,200,0),QColor(0,200,40),
		    QColor(0,120,20),QColor(0,0,0),QColor(240,50,0),
		    QColor(100,100,100));

  chNumInd   = new QLCDNumber(1,this);
  freqNumInd = new QLCDNumber(7,this);
  volInd = new QLCDNumber(2,this);

  stNameInd  = new QLabel(i18n("Not a Preset"),this);
  stLockInd  = new QLabel(i18n("Tune"),this);
  stInd      = new QLabel(i18n("oo"),this);
  muteInd    = new QLabel(i18n("MUTE"),this);
  mhzInd     = new QLabel(i18n("MHz"),this);
  Off = new QWidget(this);
  sleepInd   = new QLabel(i18n("Zzz"),this);
  alarmInd   = new QLabel(i18n("Alarm"),this);


  Off->resize(120,44);
  Off->move(0,0);
  dock_widget->setRadioState(5);
  if (!cf->noDefSet())
    {
      dock_widget->setRadioState(4);
      Off->hide();
    }
  Off->setBackgroundColor(QColor(0,0,0));
  Off->setBackgroundPixmap(cRPix);

  chNumInd->move(105,0);
  chNumInd->resize(15,20);
  chNumInd->setMode(QLCDNumber::HEX);
  chNumInd->display(cf->getDefault()+1);
  chNumInd->setBackgroundColor(QColor(0,0,0));
  chNumInd->setPalette(QPalette(redgrn,redgrn,redgrn));
  chNumInd->setFrameStyle(QFrame::NoFrame);
  chNumInd->setBackgroundPixmap(bgPix);
 
  freqNumInd->move(50,26);
  freqNumInd->resize(53,18);
  freqNumInd->setMode(QLCDNumber::DEC);
  //freqNumInd->setSmallDecimalPoint(TRUE);
  freqNumInd->display(double(cf->getFreq(cf->getDefault()))/20.0 + 87.001);
  freqNumInd->setBackgroundColor(QColor(0,0,0));
  freqNumInd->setPalette(QPalette(redgrn,redgrn,redgrn));
  freqNumInd->setFrameStyle(QFrame::NoFrame);
  freqNumInd->setBackgroundPixmap(bgPix);

  stNameInd->move(10,14);
  stNameInd->resize(90,15);
  stNameInd->setText(cf->getName(cf->getDefault()));
  stNameInd->setPalette(QPalette(redgrn,redgrn,redgrn));
  stNameInd->setFont(QFont("helvetica",14,QFont::Bold));
  stNameInd->setBackgroundColor(QColor(0,0,0));
  stNameInd->setAlignment(AlignCenter);
  stNameInd->setBackgroundPixmap(bgPix);

  stLockInd->move(5,0);
  stLockInd->resize(25,15);
  stLockInd->setPalette(QPalette(blud,blud,blud));
  stLockInd->setBackgroundColor(QColor(0,0,0));
  stLockInd->setBackgroundPixmap(bgPix);

  sleepInd->move(35,0);
  sleepInd->resize(20,15);
  sleepInd->setPalette(QPalette(blud,blud,blud));
  sleepInd->setBackgroundColor(QColor(0,0,0));
  sleepInd->setBackgroundPixmap(bgPix);

  alarmInd->move(60,0);
  alarmInd->resize(20,15);
  alarmInd->setPalette(QPalette(blud,blud,blud));
  alarmInd->setBackgroundColor(QColor(0,0,0));
  alarmInd->setBackgroundPixmap(bgPix);

  stInd->move(87,0);
  stInd->resize(15,15);
  stInd->setPalette(QPalette(blud,blud,blud));
  stInd->setBackgroundColor(QColor(0,0,0));
  stInd->setBackgroundPixmap(bgPix);

  muteInd->move(5,28);
  muteInd->resize(30,15);
  muteInd->setPalette(QPalette(blud,blud,blud));
  muteInd->setBackgroundColor(QColor(0,0,0));
  muteInd->setBackgroundPixmap(bgPix);

  mhzInd->move(95,30);
  mhzInd->resize(24,12);
  mhzInd->setPalette(QPalette(redgrn,redgrn,redgrn));
  QFont mhzFnt("helvetica",10,QFont::Normal);
  mhzFnt.setItalic(TRUE);
  mhzInd->setFont(mhzFnt);
  mhzInd->setBackgroundColor(QColor(0,0,0));
  mhzInd->setBackgroundPixmap(bgPix);

  volInd->move(105,19);
  volInd->resize(15,12);
  volInd->display(cf->getVolume());
  volInd->setBackgroundColor(QColor(0,0,0));
  volInd->setPalette(QPalette(redgrn,redgrn,redgrn));
  volInd->setFrameStyle(QFrame::NoFrame);
  volInd->setBackgroundPixmap(bgPix);
}

void DispBox::setFreq(int value)
{
  if (value == -1) value=curFreq;
  else curFreq=value;

  double f;
  f = double(value)/20.0 + 87.001;
  freqNumInd->display(f);

  if (timer) killTimer(timer);
  timer = startTimer( 1000 );
  stLockInd->setPalette(QPalette(blud,blud,blud));

  int j;
  for(j=0;j<numChannels;j++)
    {
      if (channelFreq[j] == value)
	{
	  curChannel = j;
	  stNameInd->setText(channelName[j]);
	  stNameInd->update();
	  chNumInd->display(j+1);
	  char *temp = new char [strlen(channelName[j])+11];
	  temp[0]='\0';
	  strcat(temp,"kTuner - ");
	  strcat(temp,channelName[j]);
	  P->setCaption(temp);
	  delete temp;
	  j = 255;
	}
    }
  if (j < 255)
    {
      curChannel = -1;
      stNameInd->setText("");
      stNameInd->update();
      P->setCaption("kTuner");
      chNumInd->display(15);
    }
  Off->hide();
  if (Mute)
    {
      cardDev->cardToggle();
      toggleMute();
    }
  dock_widget->setRadioState(5);
}

void DispBox::timerEvent( QTimerEvent *e )
{
  killTimer(timer);
  if ((!Mute) && (state))
    {
      if (cardDev->isTuned())
	stLockInd->setPalette(QPalette(blu,blu,blu));
      else
	stLockInd->setPalette(QPalette(blud,blud,blud));
    }
}

void DispBox::setVol(int value)
{
  volInd->display(value);
}

void DispBox::toggleMute()
{
  Mute=!Mute;
  if (Mute)
    muteInd->setPalette(QPalette(blu,blu,blu));
  else
    muteInd->setPalette(QPalette(blud,blud,blud));
  cardDev->cardToggle();
}

void DispBox::toggleOn()
{
  Off->hide();
  state = 1;
  cardDev->cardOn();
  setFreq(-1);
}

void DispBox::toggleOff()
{
  Off->show();
  state = 0;
  cardDev->cardOff();
  dock_widget->setRadioState(4);
}

void DispBox::sleep(int s)
{
  if (s==0)
    sleepInd->setPalette(QPalette(blud,blud,blud));
  else
    sleepInd->setPalette(QPalette(blu,blu,blu));
}

void DispBox::toggleAlarm()
{
  alarm = !alarm;
  if (alarm)
    {
      alarmInd->setPalette(QPalette(blu,blu,blu));
      dock_widget->setRadioState(1);
    }
  else
    {
      alarmInd->setPalette(QPalette(blud,blud,blud));
      dock_widget->setRadioState(0);
    }
}

Presets::Presets(configFile* cf, QWidget *parent=0, const char *name=0)
  : QWidget( parent, name,  WStyle_Customize | WStyle_DialogBorder )
{
  resize(260,55);
  setMinimumSize(260,39);
  setMaximumSize(260,55);

  conf = cf;

  int i;
  QFont f("Helvetica", 8, QFont::Normal);
  for (i=0;i<9;i++)
    {
      channelName[i] = new char [strlen(cf->getName(i))+2];
      strcpy(channelName[i],cf->getName(i));
      ch[i] = new QPushButton(cf->getName(i),this);
      ch[i]->setFont( f );
    }

  Selected = 0;

  for(i=0;i<5;i++)
    {
      ch[i]->move(i*52,0);
      ch[i]->resize(50,17);
    }

  for(i=5;i<9;i++)
    {
      ch[i]->move((i-5)*52,20);
      ch[i]->resize(50,17);
    }

  moreBox = new QComboBox(this);
  moreBox->resize(53,18);
  moreBox->move (207,19);
  editBox = new presetEdit(conf);;
  addBox = new presetAdd(conf);;

  int k = cf->getNumMoreChannels();
  channelFreq = new int[i+9];

  for(i=0;i<9;i++)
      channelFreq[i] = cf->getFreq(i);

  if (k > 0)
    {
      int j;
      for (j=0;j<k;j++)
	{
	  channelFreq[j+9] = cf->getFreq(9+j);
	  moreBox->insertItem(cf->getName(9+j));
	  channelName[j+9] = new char[strlen(cf->getName(9+j))+1];
	  strcpy(channelName[j+9],cf->getName(9+j));
	}
    }
  else
    moreBox->insertItem(i18n("Empty"));
 
  volDownBut = new QPushButton("-",this);
  volUpBut = new QPushButton("+",this);

  for (i=0;i<3;i++)
    Commands[i] = new QPushButton("",this);

  Commands[0]->setText(i18n("Add"));
  Commands[1]->setText(i18n("Delete"));
  Commands[2]->setText(i18n("Edit"));

  Commands[0]->move(0,40);
  Commands[1]->move(35,40);
  Commands[2]->move(70,40);
  volUpBut->move(124,40);
  volDownBut->move(109,40);

  Commands[0]->resize(35,15);
  Commands[1]->resize(35,15);
  Commands[2]->resize(35,15);
  volUpBut->resize(15,15);
  volDownBut->resize(15,15);

  connect(ch[0], SIGNAL(clicked()), SLOT(setFreq0()));
  connect(ch[1], SIGNAL(clicked()), SLOT(setFreq1()));
  connect(ch[2], SIGNAL(clicked()), SLOT(setFreq2()));
  connect(ch[3], SIGNAL(clicked()), SLOT(setFreq3()));
  connect(ch[4], SIGNAL(clicked()), SLOT(setFreq4()));
  connect(ch[5], SIGNAL(clicked()), SLOT(setFreq5()));
  connect(ch[6], SIGNAL(clicked()), SLOT(setFreq6()));
  connect(ch[7], SIGNAL(clicked()), SLOT(setFreq7()));
  connect(ch[8], SIGNAL(clicked()), SLOT(setFreq8()));
  connect(moreBox, SIGNAL(highlighted(int)), SLOT(setFreqMore(int)));

  connect(volUpBut, SIGNAL(clicked()), SLOT(volUp()));
  connect(volDownBut, SIGNAL(clicked()), SLOT(volDown()));
  connect(Commands[2], SIGNAL(clicked()), SLOT(editChannel()));
  connect(editBox, SIGNAL(edited()), SLOT(editDone()));
  connect(Commands[0], SIGNAL(clicked()), SLOT(addChannel()));
  connect(addBox, SIGNAL(edited()), SLOT(editDone()));
  connect(Commands[1], SIGNAL(clicked()), SLOT(deleteChannel()));
}

void Presets::reset()
{
  int i;
  for (i=0;i<9;i++)
    {
      if (channelName[i] != NULL) delete channelName[i];
      channelName[i] = new char [strlen(conf->getName(i))+2];
      strcpy(channelName[i],conf->getName(i));
      ch[i]->setText(conf->getName(i));
      channelFreq[i] = conf->getFreq(i);
    }

  moreBox->clear();
  for (i=9;i<(9+conf->getNumMoreChannels());i++)
    {
      if (channelName[i] != NULL) delete channelName[i];
      channelName[i] = new char [strlen(conf->getName(i))+2];
      strcpy(channelName[i],conf->getName(i));
      moreBox->insertItem(conf->getName(i));
      channelFreq[i] = conf->getFreq(i);
    }
  if (conf->getNumMoreChannels()==0)
    moreBox->insertItem(i18n("Empty"));

  buttons->clear();
  dock_widget->resetMenuChannels();

  for (int tem=0;tem < 9;tem ++)
    {
      buttons->insertItem(conf->getName(tem),conf->getFreq(tem));
      dock_widget->addMenuChannel(conf->getFreq(tem),conf->getName(tem));
    }
  numChannels = 9 + conf->getNumMoreChannels();
}

void Presets::setFreq0() {emit sendFreq(channelFreq[0]);}
void Presets::setFreq1() {emit sendFreq(channelFreq[1]);}
void Presets::setFreq2() {emit sendFreq(channelFreq[2]);}
void Presets::setFreq3() {emit sendFreq(channelFreq[3]);}
void Presets::setFreq4() {emit sendFreq(channelFreq[4]);}
void Presets::setFreq5() {emit sendFreq(channelFreq[5]);}
void Presets::setFreq6() {emit sendFreq(channelFreq[6]);}
void Presets::setFreq7() {emit sendFreq(channelFreq[7]);}
void Presets::setFreq8() {emit sendFreq(channelFreq[8]);}
void Presets::setFreqMore(int Selected) {emit sendFreq((int)channelFreq[Selected+9]);}

void Presets::volUp()
{
  cardDev->volUp();
  emit sendVol(cardDev->getVolume());
}

void Presets::volDown()
{
  cardDev->volDown();
  emit sendVol(cardDev->getVolume());
}

void Presets::editChannel()
{
  editBox->setChanNum(curChannel);
  editBox->show();
}

void Presets::addChannel()
{
  addBox->setInsertPos(9+conf->getNumMoreChannels());
  addBox->setFreq(curFreq);
  addBox->show();
}

void Presets::deleteChannel()
{
  conf->delChannel(curChannel);
  editDone();
}

void Presets::editDone()
{
  emit optionsUpdate();
  conf->save();
}

Tuning::Tuning(configFile* cf, QWidget *parent=0, const char *name=0 )
  : QWidget( parent, name )
{
  resize(400,30);

  Busy=0;
  noTwice=0;

  currentFreq=cf->getFreq(cf->getDefault());
  
  scanDwn = new QPushButton("<<",this);
  fineDwn = new QPushButton("<",this);
  tuneBar = new QSlider(0,440,  //87.00 - 109.00  = 440 increments of 0.05
			1,220,
			QSlider::Horizontal,
			this,"Tuning");
  fineUp  = new QPushButton(">",this);
  scanUp  = new QPushButton(">>",this);

  scanDwn->resize (30,13);
  scanDwn->move (5,5);
  scanUp->resize (30,13);
  scanUp->move (363,5);
  fineDwn->resize (20,13);
  fineDwn->move (35,5);
  fineDwn->setAutoRepeat(TRUE);
  fineUp->resize (20,13);
  fineUp->move (343,5);
  fineUp->setAutoRepeat(TRUE);

  tuneBar->resize(284,18);
  tuneBar->move(57,5);
  tuneBar->setTickmarks(QSlider::Below);
  tuneBar->setTickInterval(20); //20 intervals per MHz
  tuneBar->setValue(currentFreq);

  connect (fineDwn, SIGNAL(clicked()), tuneBar, SLOT(subtractStep()));
  connect (fineUp, SIGNAL(clicked()), tuneBar, SLOT(addStep()));
  connect (tuneBar, SIGNAL(valueChanged(int)), SLOT(freqSetBar(int)));
  connect (scanUp, SIGNAL(clicked()), SLOT(scanUpSlot()));
  connect (scanDwn, SIGNAL(clicked()), SLOT(scanDwnSlot()));
}

void Tuning::freqSet(int iF)
{
  currentFreq=iF;
  noTwice=1;
  tuneBar->setValue(iF);
  freqChg(currentFreq);
}

void Tuning::freqSetBar(int iF)
{
  currentFreq=iF;
  if (noTwice == 0)
    freqChg(currentFreq);
  else
    noTwice = 0;
}

void Tuning::freqChg(int f)
{
  currentFreq=f;
  emit freqChanged(f);

  cardDev->tuneFreq(double(f)/20.0 +87.0);
}
void Tuning::scanDwnSlot()
{
  if (!Busy)
    {
      Busy=1;

      if (timer) killTimer(timer);
      timer = startTimer( 10 );
      scanState = 2;
    }
    else
      Busy = 0;
}

void Tuning::scanUpSlot()
{
  if (!Busy)
    {
      Busy=1;

      if (timer) killTimer(timer);
      timer = startTimer( 10 );
      scanState = 1;
    }
    else
      Busy=0;
}

void Tuning::timerEvent( QTimerEvent *e )
{
  killTimer(timer);
  int f = currentFreq;

  if (scanState == 1)
    {
      if (f < 440) f ++;
        else f = 0;
    }
  else if (scanState == 2)
    {
      if (f > 0) f--;
      else f = 440;
    }

//  cardDev->tuneFreq(f);
  tuneBar->setValue(f);
  usleep(1000);

  if (cardDev->isTuned())
      {
        Busy = 0;
        scanState = 0;
      }
    else
      {
        if (timer) killTimer(timer);
        if (Busy) timer = startTimer( 10 );
      }
  
  currentFreq = f;
}

void Tuning::freqInc()
{
  tuneBar->addStep();
}

void Tuning::freqDec()
{
  tuneBar->subtractStep();
}


Controls::Controls( QWidget *parent=0, const char *name=0)
  : QWidget( parent, name )
{
  resize(400,90);
  P = parent;

  int i;
  for (i=0;i<5;i++)
    but[i] = new QPushButton("",this);
  
  sizeState=0;

  but[0]->setText(i18n("Exit"));
  but[1]->setText(i18n("ON"));
  but[2]->setText(i18n("OFF"));
  but[3]->setText(i18n("Mute"));
  but[4]->setText(i18n(""));

  but[0]->move(0,50);
  but[1]->move(30,50);
  but[2]->move(60,50);
  but[3]->move(90,50);
  but[4]->move(120,5);

  but[0]->resize(30,15);
  but[1]->resize(30,15);
  but[2]->resize(30,15);
  but[3]->resize(30,15);
  but[4]->resize(8,60);

  connect( but[0], SIGNAL(clicked()), qApp, SLOT(quit()));
  connect( but[1], SIGNAL(clicked()), SLOT(toggleOn()));
  connect( but[2], SIGNAL(clicked()), SLOT(toggleOff()));
  connect( but[3], SIGNAL(clicked()), SLOT(toggleMute()));
  connect( but[4], SIGNAL(clicked()), SLOT(toggleResize()));
}

void Controls::toggleMute()
{
  emit mutePressed();
}

void Controls::toggleOn()
{
  emit onPressed();
}

void Controls::toggleOff()
{
  emit offPressed();
}

void Controls::toggleResize()
{
  switch (sizeState)
    {
    case 0:
      {
	sizeState = 1;
	P->resize(130,64);
	break;
      }
    case 1:
      {
	P->resize(400,87);
	sizeState = 0;
	break;
      }
    }
}

Misc::Misc( QWidget *parent=0, const char *name=0)
  : QWidget( parent, name )
{
  resize(120,15);

  int i;
  for (i=0;i<3;i++)
    but[i] = new QPushButton("",this);

  but[0]->setText(i18n("Alarm"));
  but[1]->setText(i18n("Sleep"));
  but[2]->setText(i18n("Options"));

  but[0]->move(0,0);
  but[1]->move(35,0);
  but[2]->move(70,0);

  but[0]->resize(35,15);
  but[1]->resize(35,15);
  but[2]->resize(45,15);

  connect (but[0], SIGNAL(clicked()), SLOT(alarmPushed()));
  connect (but[1], SIGNAL(clicked()), SLOT(sleepPushed()));
  connect (but[2], SIGNAL(clicked()), SLOT(optionsPushed()));
}

void Misc::alarmPushed()
{
  emit alarmToggled();
}

void Misc::sleepPushed()
{
  emit sleepToggle();
}

void Misc::optionsPushed()
{
  emit optionsToggled();
}

#endif
