/*  songlist.h  - class SongList, which holds a list of songs (collection) 
    This file is part of KMid 1.7
    Copyright (C) 1997,98,99,2000  Antonio Larrosa Jimenez
    KMid's homepage : http://www.arrakis.es/~rlarrosa/kmid.html                                         
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Send comments and bug fixes to Antonio Larrosa <larrosa@kde.org>

***************************************************************************/
#ifndef SONGLIST_H
#define SONGLIST_H

/**
 * List of song names (really, file names) . This class is not really part
 * of a Midi library, as it's more like a Playlist class. 
 * 
 * Anyway, it may be useful for some cases, and that's why it's here.
 *
 * @short A List of songs.
 * @author Antonio Larrosa Jimenez <larrosa@kde.org>
 */
class SongList
{
  protected:
    int ntotal;

    struct Song
    {
      int id;
      char *name;  // complete path and file name

      Song *next;
    };

    Song *list;
    Song *last;
    Song *active;

    Song *it; // Iterator, just a helper variable to make easy (and fast) reading
    // all the list

    Song *songID(int id);

    void regenerateID(Song *song,int id);

  public:
    /**
     * Constructor
     */
    SongList(void);

    /**
     * Copy constructor
     */
    SongList(SongList &src);

    /**
     * Destructor
     */ 
    ~SongList();


    /**
     * Adds a song to the list.
     * @return the id number assigned to the song.
     */
    int addSong(const char *song); 

    /**
     * Deletes the song with ID @p id from the list.
     */ 
    void delSong(int id);

    /**
     * Returns the number of songs in the list.
     */ 
    int numberOfSongs(void)     { return ntotal; };

    /**
     * Sets the song with ID @p id as the active song.
     * @see #activeSongID
     * @see #activeSongName
     */
    void setActiveSong(int id);

    /**
     * Returns the ID of the active song.
     * @see #activeSongName
     * @see #setActiveSong
     */
    int activeSongID(void)   {return ((active!=0L)? (active->id  ):(-1)); };

    /**
     * Returns the (file)name of the active song 
     * @see #activeSongID
     * @see #setActiveSong
     */ 
    char *activeSongName(void) 
    {
      return ((active!=0L)? (active->name):((char *)0L)); 
    };

    /**
     * Returns the name of the song with ID @p id
     */

    char *songName(int id);

    /**
     * Sets as active song the previous song.
     */
    void previous(void);

    /**
     * Sets the next song as active song.
     * 
     * @return 1 if everything was OK, and 0 if it was the last song
     * (in this cases, the last song is left active, and it's not set to 0L)
     */
    int next(void); 

    /**
     * Initializes the iterator.
     */
    void iteratorStart(void);

    /**
     * Makes the iterator advance to the next song
     */
    void iteratorNext(void);

    /**
     * Checks if the iterator is at the end of the list
     */
    int iteratorAtEnd (void) {return (it==0L);};

    /**
     * Returns the ID of the song pointed to by the iterator.
     */
    int iteratorID(void);

    /**
     * Returns the name of the song pointed to by the iterator.
     */ 
    char *iteratorName(void);

    /**
     * Clears the list.
     */ 
    void clear(void);

    /**
     * Makes this object a full copy of src (a real copy).
     */
    void copy(SongList &src); 
};

#endif
