#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <util.h>
#include <globals.h>
#include <devicecontrol.h>

DSPDeviceClass::DSPDeviceClass()
{
	opened = false;
	reqopened = false;
	reqclosed = false;
	id = -1;
	pos = 0;
	dspdev = NULL;
}

DSPDeviceClass::DSPDeviceClass( int startid, QString startDeviceName )
{
	opened = false;
	reqopened = false;
	reqclosed = false;
	id = startid;
	pos = 0;
	deviceName = startDeviceName.copy();
	lastError.sprintf( "" );
	dspdev = NULL;
}

DSPDeviceClass::~DSPDeviceClass()
{
	closeDevice();
}

int DSPDeviceClass::openDevice()
{
	bool error;

	if ( dspdev != NULL ) {
		return -2; // Already open!
	}

	dspdev = new dspout( (const char *) deviceName );

	error = true;
	switch ( dspdev->getLastError() ) {
		case dspout::DeviceBusy:
			lastError.sprintf( "Device Busy" );
			error = true;
			break;
		case dspout::DeviceNoGood:
			lastError.sprintf( "Invalid Device" );
			error = true;
			break;
		case dspout::NoError:
			lastError.sprintf( "" );
			error = false;
			break;
		default:
			lastError.sprintf( "Unknown Error" );
			error = true;
			break;
	}

	if ( error ) {
		delete dspdev;
		dspdev = NULL;
		return -1;
	}

	// initialize sound settings
	dspdev->setFormat( dspout::S16_LE );
	dspdev->setChannels( dspout::Stereo );
	dspdev->setSampleRate( 44100 );
	dspdev->setFragmentInfo( 16, 14 );

	opened = true;

	return 0;
}

int DSPDeviceClass::closeDevice()
{
	if ( dspdev == NULL ) {
		return -2; // Already closed!
	}

	delete dspdev;
	dspdev = NULL;

	opened = false;
	return 0;
}

MIDIDeviceClass::MIDIDeviceClass()
{
	id = -1;
	opened = false;
	reqopened = false;
	reqclosed = false;
}

MIDIDeviceClass::MIDIDeviceClass( int startid, QString startDeviceName )
{
	opened = false;
	reqopened = false;
	reqclosed = false;
	id = startid;
	deviceName = startDeviceName.copy();
	lastError.sprintf( "" );
}

MIDIDeviceClass::~MIDIDeviceClass()
{
	closeDevice();
}

int MIDIDeviceClass::openDevice()
{
	return 0;
}

int MIDIDeviceClass::closeDevice()
{
	return 0;
}

DeviceControl::DeviceControl()
{
	int i;

	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		dspDevices[ i ] = NULL;
	}

	for ( i = 0; i < MAX_MIDI_DEVICES; i++ ) {
		midiDevices[ i ] = NULL;
	}

	lastDspDevicenum = 0;
	dspMonoDevice = true; // For thud

	lastMidiDevicenum = 0;

	buildDspDeviceList();
//	buildMidiDeviceList();
}

DeviceControl::~DeviceControl()
{
	int i;

	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		if ( dspDevices[ i ] != NULL ) {
			delete dspDevices[ i ];
		}
	}

	for ( i = 0; i < MAX_MIDI_DEVICES; i++ ) {
		if ( midiDevices[ i ] != NULL ) {
			delete midiDevices[ i ];
		}
	}
}

void DeviceControl::buildDspDeviceList()
{
	QString curfilename;
	int i, cur;

	cur = 0;
	curfilename = "/dev/dsp";
	if ( checkFilename( curfilename ) ) {
		dspDevices[ cur++ ] = new DSPDeviceClass( lastDspDevicenum++, curfilename );
	}

	for ( i = 0; i < MAX_DSP_DEVICES - 1; i++ ) {
		curfilename.sprintf( "/dev/dsp%d", i );
		if ( checkFilename( curfilename ) ) {
			dspDevices[ cur++ ] = new DSPDeviceClass( lastDspDevicenum++, curfilename );
		}
	}
}

DSPDeviceClass *DeviceControl::getDspDevice( int reqid )
{
	unsigned int i;

	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		if ( dspDevices[ i ] != NULL ) {
			if ( dspDevices[ i ]->id == reqid ) {
				return dspDevices[ i ];
			}
		}
	}

	return NULL;
}

void DeviceControl::requestOpenDspDevice( int reqid )
{
	unsigned int i;

	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		if ( dspDevices[ i ] != NULL ) {
			if ( dspDevices[ i ]->id == reqid ) {

				pthread_mutex_lock( &mutex );
				dspDevices[ i ]->reqopened = true;
				pthread_cond_wait( &doneop, &mutex );
				pthread_mutex_unlock( &mutex );

				isadeviceopen = true;
				if ( dspMonoDevice == false ) {
					return;
				}
			} else {
	  			if ( dspMonoDevice == true ) {
					pthread_mutex_lock( &mutex );
	  				dspDevices[ i ]->reqclosed = true;
	  				pthread_cond_wait( &doneop, &mutex );
					pthread_mutex_unlock( &mutex );
	  			}
			}
		}
	}

	if ( dspMonoDevice == true ) {
		for ( i = 0; i < MAX_CHANNELS; i++ ) {
			gsong->setChannelDevice( i, reqid );
		}
	}
}

void DeviceControl::requestCloseDspDevice( int reqid )
{
	unsigned int i;
	bool anotheropen;

	anotheropen = false;
	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		if ( dspDevices[ i ] != NULL ) {
			if ( dspDevices[ i ]->id == reqid ) {

				pthread_mutex_lock( &mutex );
				dspDevices[ i ]->reqclosed = true;
				pthread_cond_wait( &doneop, &mutex );
				pthread_mutex_unlock( &mutex );

			} else {
				if ( dspDevices[ i ]->opened == true ) {
					anotheropen = true;
				}
			}
		}
	}
	isadeviceopen = anotheropen;
}

void DeviceControl::scanDspDevices()
{
	unsigned int i;
	bool anotheropen;

	anotheropen = false;
	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		if ( dspDevices[ i ] != NULL ) {
			if ( dspDevices[ i ]->dspdev != NULL ) {
				anotheropen = true;
			}
		}
	}
	isadeviceopen = anotheropen;
}

void DeviceControl::computeHighestDspFd()
{
	unsigned int i;
	highestdspfd = 0;

	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		int fd;

		if ( dspDevices[ i ] != NULL ) {
			if ( dspDevices[ i ]->dspdev != NULL ) {
				fd = dspDevices[ i ]->dspdev->getFd();
				if ( fd > highestdspfd ) highestdspfd = fd;
			}
		}
	}

	highestdspfd++;
}

void DeviceControl::setAllDspDescriptors( fd_set *fds )
{
	unsigned int i;

	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		if ( dspDevices[ i ] != NULL ) {
			if ( dspDevices[ i ]->opened == true ) {
				int fd = dspDevices[ i ]->dspdev->getFd();
				FD_SET( fd, fds );
			}
		}
	}
}

