#include <sys/time.h>
#include <unistd.h>
#include <globalsong.h>

songPattern::songPattern()
{
	for ( int i = 0; i < MAX_BEATS; i++ ) {
		note[ i ] = -1;
	}
}

songChannel::songChannel()
{
	type = SONG_TYPE_SAMP_THUD;
	state = 0;
	midi_channel = 0;
	devnum = 2;
	last_note = -1;
	for ( int i = 0; i < MAX_PATTERNS; i++ ) {
		data[ i ].index = i;
	}
}

song::song()
{
	int i;

	// Clear the song structures
	clear();

	// Clear device states
	for ( i = 0; i < MAX_DSP_DEVICES; i++ ) {
		dspdevstate[ i ] = false;
	}
}

void song::clear()
{
	int i, j;

	// Clear the arrays
	for ( i = 0; i < MAX_CHANNELS; i++ ) {
		for ( j = 0; j < MAX_PATTERNS; j++ ) {
			data[ i ][ j ] = NULL;
		}
	}

	start_sec = 0;
	start_usec = 0;

	curpat = 0;
	curbeat = 0;

	loop_on = false;
	loop_start = 0;
	loop_end = 1;

	solo_mode = false;

	setTempo( 140 );

	for ( i = 0; i < MAX_CHANNELS; i++ ) {
		for ( j = 0; j < MAX_PATTERNS; j++ ) {
			setSongPattern( i, j, j );
		}
	}
}

void song::startSong()
{
	struct timeval tv;

	gettimeofday( &tv, 0 );

	start_sec = tv.tv_sec;
	start_usec = tv.tv_usec;
	state = SONG_PLAYING;
}

void song::stopSong()
{
	state = SONG_STOPPED;
}

void song::setSongPos( int pattern, int beat )
{
	curpat = pattern;
	curbeat = beat;
}

void song::setTempo( int newtempo )
{
	tempo = newtempo;
	usecpb = 60000000 / newtempo / 4; // should this be 4? 8? 2?
	fpb = (int) ((float) 44100 * 60.0 / ( ( (float) tempo ) * 4.0 ));  // this 4 makes sense
}

void song::setLoopPoints( int startpos, int endpos )
{
	loop_start = startpos;
	loop_end = endpos;
}

void song::incrBeat()
{
	int newp, newb;

	beatAfter( curpat, curbeat, &newp, &newb );
	curpat = newp;
	curbeat = newb;
}

void song::beatAfter( int curp, int curb, int *nextp, int *nextb )
{
	if ( curb == ( MAX_BEATS - 1 ) ) {
		*nextb = 0;
		if ( ( loop_on == true ) && ( loop_end == ( curp + 1 ) ) ) {
			*nextp = loop_start;
		} else {
			*nextp = curp + 1;
		}
	} else {
		*nextp = curp;
		*nextb = curb + 1;
	}
}

void song::getFuturePos( int iter, int *fpat, int *fbeat )
{
	int oldpat, oldbeat, newpat, newbeat, i;

	oldpat = curpat;
	newpat = oldpat;
	oldbeat = curbeat;
	newbeat = oldbeat;
	for ( i = 0; i < iter; i++ ) {
		beatAfter( oldpat, oldbeat, &newpat, &newbeat );
		oldpat = newpat;
		oldbeat = newbeat;
	}

	*fpat = newpat;
	*fbeat = newbeat;
}

int song::getSongNote( int chan, int pat, int beat )
{
	if ( data[ chan ][ pat ] == NULL ) {
		return -1;
	} else {
		return data[ chan ][ pat ]->note[ beat ];
	}
}

int song::getSongPatternIndex( int chan, int pat )
{
	if ( data[ chan ][ pat ] == NULL ) {
		return -1;
	} else {
		return data[ chan ][ pat ]->index;
	}
}

void song::setSongPattern( int chan, int pos, int pat )
{
	if ( pat == -1 ) {
		data[ chan ][ pos ] = NULL;
	} else {
		data[ chan ][ pos ] = &chan_data[ chan ].data[ pat ];
	}
}

int song::getPatternNote( int chan, int pat, int beat )
{
	return chan_data[ chan ].data[ pat ].note[ beat ];
}

void song::setPatternNote( int chan, int pat, int beat, int value )
{
	chan_data[ chan ].data[ pat ].note[ beat ] = value;
}

int *song::getNoteDataPtr( int chan, int pat, int beat )
{
	return &chan_data[ chan ].data[ pat ].note[ beat ];
}

void song::setChannelType( int chan, int type )
{
	chan_data[ chan ].type = type;
}

int song::getChannelType( int chan )
{
	return chan_data[ chan ].type;
}

void song::setMidiChannel( int chan, int mchan )
{
	chan_data[ chan ].midi_channel = mchan;
}

int song::getMidiChannel( int chan )
{
	return chan_data[ chan ].midi_channel;
}

int song::getChannelLastNote( int chan )
{
	return chan_data[ chan ].last_note;
}

void song::setChannelLastNote( int chan, int note )
{
	chan_data[ chan ].last_note = note;
}

int song::getChannelDevice( int chan )
{
	return chan_data[ chan ].devnum;
}

void song::setChannelDevice( int chan, int devnum )
{
	chan_data[ chan ].devnum = devnum;
}

bool song::isChannelMuted( int chan )
{
	return chan_data[ chan ].mute;
}

void song::setChannelMute( int chan, bool mute )
{
	chan_data[ chan ].mute = mute;
}

sampleChannel *song::getChannelSample( int chan )
{
	return &chan_data[ chan ].sampledata;
}

bool song::isChannelSolo( int chan )
{
	return chan_data[ chan ].solo;
}

void song::setChannelSolo( int chan, bool solo )
{
	chan_data[ chan ].solo = solo;

	if ( ( solo_mode == false ) && ( solo == true ) ) {
		solo_mode = true;
	}

	if ( ( solo_mode == true ) && ( solo == false ) ) {
		bool found;
		int i;

		found = false;
		for ( i = 0; i < MAX_CHANNELS; i++ ) {
			if ( chan_data[ chan ].solo == true ) {
				found = true;
				break;
			}
		}
		if ( found == false ) {
			solo_mode = false;
		}
	}
}

void song::saveSong( int fd )
{
	int i, j, k, l;

	// Tempo Information
	write( fd, &tempo, sizeof( int ) );

	// Solo mode Information
	write( fd, &solo_mode, sizeof( bool ) );

	// Loop Information
	write( fd, &loop_on, sizeof( bool ) );
	write( fd, &loop_start, sizeof( int ) );
	write( fd, &loop_end, sizeof( int ) );

	// Channel Information
	int emptypat = -303;
	int nonemptypat = -808;
	for ( i = 0; i < MAX_CHANNELS; i++ ) {
		write( fd, &chan_data[ i ].type, sizeof( int ) );
		write( fd, &chan_data[ i ].state, sizeof( int ) );
		write( fd, &chan_data[ i ].midi_channel, sizeof( int ) );
		write( fd, &chan_data[ i ].devnum, sizeof( int ) );
		write( fd, &chan_data[ i ].mute, sizeof( bool ) );
		write( fd, &chan_data[ i ].solo, sizeof( bool ) );
		for ( j = 0; j < MAX_PATTERNS; j++ ) {
			write( fd, &chan_data[ i ].data[ j ].index, sizeof( int ) );

			bool isempty = true;
			for ( k = 0; k < MAX_BEATS; k++ ) {
				if ( chan_data[ i ].data[ j ].note[ k ] != -1 ) {
					isempty = false;
				}
			}

			if ( isempty == true ) {
				write( fd, &emptypat, sizeof( int ) );
			} else {
				write( fd, &nonemptypat, sizeof( int ) );
				for ( l = 0; l < MAX_BEATS; l++ ) {
					write( fd, &chan_data[ i ].data[ j ].note[ l ], sizeof( int ) );
				}
			}
		}
	}

	// Song Data
	int done = -303;
	for ( i = 0; i < MAX_CHANNELS; i++ ) {
		for ( j = 0; j < MAX_PATTERNS; j++ ) {
			if ( data[ i ][ j ] != NULL ) {
				write( fd, &i, sizeof( int ) );
				write( fd, &j, sizeof( int ) );
				write( fd, &data[ i ][ j ]->index, sizeof( int ) );
			}
		}
	}
	write( fd, &done, sizeof( int ) );
}

void song::loadSong( int fd )
{
	int chanindex, patindex, value, pattype;

	// Tempo Information
	read( fd, &tempo, sizeof( int ) );
	setTempo( tempo );

	// Solo mode Information
	read( fd, &solo_mode, sizeof( bool ) );

	// Loop Information
	read( fd, &loop_on, sizeof( bool ) );
	read( fd, &loop_start, sizeof( int ) );
	read( fd, &loop_end, sizeof( int ) );

	// Channel Information
	for ( int i = 0; i < MAX_CHANNELS; i++ ) {
		read( fd, &chan_data[ i ].type, sizeof( int ) );
		read( fd, &chan_data[ i ].state, sizeof( int ) );
		read( fd, &chan_data[ i ].midi_channel, sizeof( int ) );
		read( fd, &chan_data[ i ].devnum, sizeof( int ) );
		read( fd, &chan_data[ i ].mute, sizeof( bool ) );
		read( fd, &chan_data[ i ].solo, sizeof( bool ) );
		for ( int j = 0; j < MAX_PATTERNS; j++ ) {
			read( fd, &chan_data[ i ].data[ j ].index, sizeof( int ) );
			read( fd, &pattype, sizeof( int ) );
			if ( pattype != -303 ) {
				for ( int k = 0; k < MAX_BEATS; k++ ) {
					read( fd, &chan_data[ i ].data[ j ].note[ k ], sizeof( int ) );
				}
			}
		}
	}

	for (;;) {
		read( fd, &chanindex, sizeof( int ) );
		if ( chanindex == -303 ) break;
		read( fd, &patindex, sizeof( int ) );
		read( fd, &value, sizeof( int ) );
		data[ chanindex ][ patindex ] = &chan_data[ chanindex ].data[ value ];
	}
}

void song::setMaxMidiDevice( int maxdevnum )
{
	maxmididev = maxdevnum;
}

int song::getMaxMidiDevice()
{
	return maxmididev;
}

void song::setMaxDspDevice( int maxdevnum )
{
	maxdspdev = maxdevnum;
}

int song::getMaxDspDevice()
{
	return maxdspdev;
}

void song::getSyncReferenceData( int *lsec, int *lusec, int *lpat, int *lbeat )
{
	*lsec = last_sec;
	*lusec = last_usec;
	*lpat = last_curpat;
	*lbeat = last_curbeat;
}

void song::setSyncReferenceData( int lsec, int lusec, int lpat, int lbeat )
{
	last_sec = lsec;
	last_usec = lusec;
	last_curpat = lpat;
	last_curbeat = lbeat;
}

bool song::dspDeviceActive( int devnum )
{
	return dspdevstate[ devnum ];
}

void song::setDspDeviceActive( int devnum, bool state )
{
	dspdevstate[ devnum ] = state;
}

