#include "k2url.h"

#include <stdio.h>
#include <assert.h>

// This is a hack. Dont want to use Qt here ...
// #include <qdir.h>

bool k2url_parse( K2URL *_url, const char *_txt );

K2URL::K2URL()
{
  m_bIsMalformed = true;
}

K2URL::K2URL( const char *_url )
{
  m_strProtocol = "file";
  m_iPort = -1;
  parse( _url );
}

K2URL::K2URL( string &_url )
{
  m_strProtocol = "file";
  m_iPort = -1;
  parse( _url.c_str() );
}

K2URL::K2URL( const K2URL& _u )
{
  m_strProtocol = _u.m_strProtocol;
  m_strUser = _u.m_strUser;
  m_strPass = _u.m_strPass;
  m_strHost = _u.m_strHost;
  m_strPath = _u.m_strPath;
  m_strQuery_encoded = _u.m_strQuery_encoded;
  m_strRef_encoded = _u.m_strRef_encoded;
  m_bIsMalformed = _u.m_bIsMalformed;
  m_iPort = _u.m_iPort;
}

K2URL::K2URL( const K2URL& _u, const char *_rel_url )
{
  if ( _rel_url[0] == '/' )
  {
    *this = _u;
    setEncodedPathAndQuery( _rel_url );
  }
  else if ( _rel_url[0] == '#' )
  {
    *this = _u;
    setRef( _rel_url + 1 );
  }
  else if ( strstr( _rel_url, ":/" ) != 0 )
  {
    *this = _rel_url;
    /* m_strUser = "";
    m_strPass = "";
    m_strHost = "";
    m_strPath = "";
    m_strQuery_encoded = "";
    m_strRef_encoded = "";
    m_strProtocol = "file";
    m_iPort = -1;     
    m_bIsMalformed = !k2url_parse( this, _rel_url ); */
  }
  else
  {
    *this = _u;
    string tmp;
    decode( tmp );
    setFileName( tmp.c_str() );
  }
}

void K2URL::reset()
{
  m_strProtocol = "file";
  m_strUser = "";
  m_strPass = "";
  m_strHost = "";
  m_strPath = "";
  m_strQuery_encoded = "";
  m_strRef_encoded = "";
  m_bIsMalformed = false;
  m_iPort = -1;
}

void K2URL::parse( const char *_url )
{
  m_bIsMalformed = !k2url_parse( this, _url );
}

K2URL& K2URL::operator=( const char* _url )
{
  reset();  
  parse( _url );

  return *this;
}

K2URL& K2URL::operator=( string& _url )
{
  reset();
  parse( _url.c_str() );

  return *this;
}

K2URL& K2URL::operator=( const K2URL& _u )
{
  m_strProtocol = _u.m_strProtocol;
  m_strUser = _u.m_strUser;
  m_strPass = _u.m_strPass;
  m_strHost = _u.m_strHost;
  m_strPath = _u.m_strPath;
  m_strQuery_encoded = _u.m_strQuery_encoded;
  m_strRef_encoded = _u.m_strRef_encoded;
  m_bIsMalformed = _u.m_bIsMalformed;
  m_iPort = _u.m_iPort;

  return *this;
}

bool K2URL::operator==( const K2URL& _u ) const
{
  if ( isMalformed() || _u.isMalformed() )
    return false;
  
  if ( m_strProtocol == _u.m_strProtocol &&
       m_strUser == _u.m_strUser &&
       m_strPass == _u.m_strPass &&
       m_strHost == _u.m_strHost &&
       m_strPath == _u.m_strPath &&
       m_strQuery_encoded == _u.m_strQuery_encoded &&
       m_strRef_encoded == _u.m_strRef_encoded &&
       m_bIsMalformed == _u.m_bIsMalformed &&
       m_iPort == _u.m_iPort )
    return true;
  
  return false;
}

bool K2URL::operator==( const char* _u ) const
{
  K2URL u( _u );
  return ( *this == u );
}

bool K2URL::cmp( K2URL &_u, bool _ignore_trailing )
{
  if ( _ignore_trailing )
  {
    string path1 = path(1);
    string path2 = _u.path(1);
    if ( path1 != path2 )
      return false;

    if ( m_strProtocol == _u.m_strProtocol &&
	 m_strUser == _u.m_strUser &&
	 m_strPass == _u.m_strPass &&
	 m_strHost == _u.m_strHost &&
	 m_strQuery_encoded == _u.m_strQuery_encoded &&
	 m_strRef_encoded == _u.m_strRef_encoded &&
	 m_bIsMalformed == _u.m_bIsMalformed &&
	 m_iPort == _u.m_iPort )
      return true;

    return false;
  }
  
  return ( *this == _u );
}

void K2URL::setFileName( const char *_txt )
{
  // TODO: clean path at the end
  while( *_txt == '/' ) _txt++;
  
  int len = m_strPath.size();
  if ( len == 0 )
  {
    m_strPath = "/";
    m_strPath += _txt;
    return;
  }
  
  if ( m_strPath[ len - 1 ] == '/' )
  {
    m_strPath += _txt;
    return;
  }
  
  while ( len >= 1 && m_strPath[ len - 1 ] == '/' )
    len--;

  // Does the path only consist of '/' characters ?
  if ( len == 1 && m_strPath[ 1 ] == '/' )
  {
    m_strPath = "/";
    m_strPath += _txt;
    return;
  }
  
  int i = m_strPath.rfind( '/', len - 1 );
  // If ( i == -1 ) => The first character is not a '/' ???
  // This looks strange ...
  if ( i == -1 )
  {
    m_strPath = "/";
    m_strPath += _txt;
    return;
  }
  
  string tmp;
  tmp.assign( m_strPath, i + 1, len - i - 1 );
  tmp += _txt;
}

string K2URL::encodedPathAndQuery( int _trailing, bool _no_empty_path )
{
  string tmp = path( _trailing );
  if ( _no_empty_path && tmp.empty() )
    tmp = "/";
  
  encode( tmp );
  if ( !m_strQuery_encoded.empty() )
  {
    tmp += "?";
    tmp += m_strQuery_encoded;
  }
  
  return tmp;
}

void K2URL::setEncodedPathAndQuery( const char *_txt )
{
  string tmp = _txt;
  int pos = tmp.find( '?' );
  if ( pos == -1 )
  {
    m_strPath = tmp;
    m_strQuery_encoded = "";
  }
  else
  { 
    m_strPath.assign( tmp, 0, pos );
    m_strQuery_encoded = _txt + pos + 1;
  }

  decode( m_strPath );
}

string K2URL::path( int _trailing ) const
{
  string result = path();

  if ( _trailing == 0 )
    return result;
  else if ( _trailing == 1 )
  {
    int len = result.size();
    if ( len == 0 )
      result = "";
    else if ( result[ len - 1 ] != '/' )
      result += "/";
    return result;
  }
  else if ( _trailing == -1 )
  {
    if ( result == "/" )
      return result;
    int len = result.size();
    if ( len != 0 && result[ len - 1 ] == '/' )
      result.erase( len - 1, 1 );
    return result;
  }
  else
    assert( 0 );
}

bool K2URL::isLocalFile()
{
  if ( m_strProtocol != "file" )
    return false;
  
  if ( m_strRef_encoded.empty() )
    return true;
  
  K2URL u( m_strRef_encoded.c_str() );
  if ( u.isMalformed() )
    return true;
  
  return false;
}

bool K2URL::hasSubURL()
{
  if ( m_strRef_encoded.empty() )
    return false;
  
  K2URL u( m_strRef_encoded.c_str() );
  if ( u.isMalformed() )
    return false;

  return true;
}

string K2URL::url()
{
  string result = url( 0 );
  return result;
}

string K2URL::url( int _trailing )
{
  // HACK encode parts here!

  string u = m_strProtocol;
  if ( hasHost() )
  {
    u += "://";
    if ( hasUser() )
    {
      u += m_strUser;
      if ( hasPass() )
      {
	u += ":";
	u += m_strPass;
      }
      u += "@";
    }
    u += m_strHost;
    if ( m_iPort != -1 )
    {
      char buffer[ 100 ];
      sprintf( buffer, ":%i", m_iPort );
      u += buffer;
    }
  }
  else
    u += ":";
  string tmp;
  if ( _trailing == 0 )
    tmp = m_strPath;
  else
    tmp = path( _trailing );
  encode( tmp );
  u += tmp;
    
  if ( !m_strQuery_encoded.empty() )
  {
    u += "?";
    u += m_strQuery_encoded;
  }
  
  if ( hasRef() )
  {
    u += "#";
    u += m_strRef_encoded;
  }
  
  return u;
}

bool K2URL::split( const char *_url, list<K2URL>& lst )
{
  string tmp;
  
  do
  {
    K2URL u( _url );
    if ( u.isMalformed() )
      return false;
    
    if ( u.hasSubURL() )
    {
      tmp = u.ref();
      _url = tmp.c_str();
      u.setRef( "" );
      lst.push_back( u );
    }
    else
    {
      lst.push_back( u );
      return true;
    }
  } while( 1 );
}

void K2URL::join( list<K2URL>& lst, string& _dest )
{
  _dest = "";
  list<K2URL>::iterator it = lst.begin();
  for( ; it != lst.end(); ++it )
  {
    string tmp = it->url();
    _dest += tmp;
    list<K2URL>::iterator it2 = it;
    it2++;
    if ( it2 != lst.end() )
      _dest += "#";
  }
}

string K2URL::filename( bool _strip_trailing_slash )
{
  string fname;

  int len = m_strPath.size();
  if ( len == 0 )
    return fname;
  
  if ( _strip_trailing_slash )
  {    
    while ( len >= 1 && m_strPath[ len - 1 ] == '/' )
      len--;
  }
  else if ( m_strPath[ len - 1 ] == '/' )
    return fname;
  
  // Does the path only consist of '/' characters ?
  if ( len == 1 && m_strPath[ 1 ] == '/' )
    return fname;
  
  int i = m_strPath.rfind( '/', len - 1 );
  // If ( i == -1 ) => The first character is not a '/' ???
  // This looks like an error to me.
  if ( i == -1 )
    return fname;
  
  fname.assign( m_strPath, i + 1, len - i - 1 );
  return fname;
}

void K2URL::addPath( const char *_txt )
{
  if ( *_txt == 0 )
    return;
  
  int len = m_strPath.size();
  // Add the trailing '/' if it is missing
  if ( _txt[0] != '/' && ( len == 0 || m_strPath[ len - 1 ] != '/' ) )
    m_strPath += "/";
    
  // No double '/' characters
  if ( len != 0 && m_strPath[ len - 1 ] == '/' )
    while( *_txt == '/' )
      _txt++;
  
  m_strPath += _txt;
}

string K2URL::directory( bool _strip_trailing_slash_from_result, bool _ignore_trailing_slash_in_path )
{
  string result;
  if ( _ignore_trailing_slash_in_path )
    result = path( -1 );
  else
    result = m_strPath;
 
  if ( result.empty() || result == "/" )
    return result;
    
  int i = result.rfind( "/" );
  if ( i == -1 )
    return result;
  
  if ( i == 0 )
  {
    result = "/";
    return result;
  }
  
  if ( _strip_trailing_slash_from_result )
    result.assign( m_strPath, 0, i );
  else
    result.assign( m_strPath, 0, i + 1 );

  return result;
}

void K2URL::encode( string& _url )
{
  int old_length = _url.size();

  if ( !old_length )
    return;
   
  // a worst case approximation
  char *new_url = new char[ old_length * 3 + 1 ];
  int new_length = 0;
     
  for ( int i = 0; i < old_length; i++ )
  {
    // 'unsave' and 'reserved' characters
    // according to RFC 1738,
    // 2.2. URL Character Encoding Issues (pp. 3-4)
    // Torben: Added the space characters
    if ( strchr("<>#@\"&%$:,;?={}|^~[]\'`\\ \n\t\r", _url[i]) )
    {
      new_url[ new_length++ ] = '%';

      char c = _url[ i ] / 16;
      c += (c > 9) ? ('A' - 10) : '0';
      new_url[ new_length++ ] = c;

      c = _url[ i ] % 16;
      c += (c > 9) ? ('A' - 10) : '0';
      new_url[ new_length++ ] = c;
	    
    }
    else
      new_url[ new_length++ ] = _url[i];
  }

  new_url[new_length] = 0;
  _url = new_url;
  delete [] new_url;
}

char K2URL::hex2int( char _char )
{
  if ( _char >= 'A' && _char <='F')
    return _char - 'A' + 10;
  if ( _char >= 'a' && _char <='f')
    return _char - 'a' + 10;
  if ( _char >= '0' && _char <='9')
    return _char - '0';
  return 0;
}

void K2URL::decode( string& _url )
{
  int old_length = _url.size();
  if ( !old_length )
    return;
    
  int new_length = 0;

  // make a copy of the old one
  char *new_url = new char[ old_length + 1];

  int i = 0;
  while( i < old_length )
  {
    char character = _url[ i++ ];
    if ( character == '%' )
    {
      character = hex2int( _url[i] ) * 16 + hex2int( _url[i+1] );
      i += 2;
    }
    new_url [ new_length++ ] = character;
  }
  new_url [ new_length ] = 0;
  _url = new_url;
  delete [] new_url;
}

bool urlcmp( K2URLList& _url1, K2URLList& _url2 )
{
  unsigned int size = _url1.size();
  if ( _url2.size() != size )
    return false;
  
  K2URLList::iterator it1 = _url1.begin();
  K2URLList::iterator it2 = _url2.begin();
  for( ; it1 != _url1.end() && it2 != _url2.end(); ++it1, ++it2 )
    if ( *it1 != *it2 )
      return false;
  
  return true;
}

bool urlcmp( const char *_url1, const char *_url2 )
{
  K2URLList list1;
  K2URLList list2;

  bool res1 = K2URL::split( _url1, list1 );
  bool res2 = K2URL::split( _url2, list2 );

  if ( !res1 || !res2 )
    return false;

  return urlcmp( list1, list2 );
}

bool urlcmp( const char *_url1, const char *_url2, bool _ignore_trailing, bool _ignore_ref )
{
  K2URLList list1;
  K2URLList list2;

  bool res1 = K2URL::split( _url1, list1 );
  bool res2 = K2URL::split( _url2, list2 );

  if ( !res1 || !res2 )
    return false;

  unsigned int size = list1.size();
  if ( list2.size() != size )
    return false;

  if ( _ignore_ref )
  {    
    list1.back().setRef("");
    list2.back().setRef("");
  }
  
  K2URLList::iterator it1 = list1.begin();
  K2URLList::iterator it2 = list2.begin();
  for( ; it1 != list1.end() && it2 != list2.end(); ++it1, ++it2 )
    if ( !it1->cmp( *it2, _ignore_ref ) )
      return false;

  return true;
}
