#include <time.h>
#include <iostream.h>
#include <qdatetm.h> 
#include <qfileinf.h> 
#include <qfile.h>
#include <qdir.h>
#include "StringToken.h"
#include "ircClient.h"
#include "webServer.h"
#include "webServer.moc"
#include "WebParser.h"
#include "ircDefine.h"
#include "ircClient.h"
#include "ircApp.h"

QList<WebClient> WebClient::clientList;

WebClient::WebClient(QObject* parent, const char* name)
    : IrcCore(parent, name)
{
  webServer = (WebServer*)parent;
  webParser = new WebParser(this);
  clientList.setAutoDelete(false);
  clientInfo.setAutoDelete(true);
  host   = 0;
  socket = 0;
  sn     = 0;
  clientList.append(this);
}

WebClient::~WebClient()
{
  webServer->current--;
  delete host;
  delete socket;
  delete sn;
  delete webParser;
  clientList.removeRef(this);
}

void WebClient::setSocket(ClientSocket* cs)
{
  socket = cs;
  InetAddress inAddr=socket->getInetAddress();
  host = new IrcServer(inAddr.getHostAddress(), socket->getPeerPort());
  sn = new QSocketNotifier(socket->getSocket(), QSocketNotifier::Read, this);
  QObject::connect(sn, SIGNAL(activated(int)), this, SLOT(slotMsgParse(int)));
  connect(socket,
	  SIGNAL(signError()),
	  SLOT  (slotDisconnected()));
  emit signConnected(host);
  int to=ircapp->readNumEntry("WebServerClientTimeout", 30);
  setCusInterval(to);
  connect(this,
	  SIGNAL(signCusTimeout(IrcCore*)),
	  SLOT  (slotCusTimeout(IrcCore*)));
  start();
}

void WebClient::slotCusTimeout(IrcCore*)
{
  slotDisconnected();
}

void WebClient::slotDisconnected()
{
#ifdef EDEBUG
  cout << "WebClient::slotDisconnect:"<<endl;
#endif
  emit signDisconnected(host);
  delete this;
}

void WebClient::slotMsgParse(int sd)
{
#ifdef EDEBUG
  cout << "WebClient::slotMsgParse:"<<endl;
#endif
  stop();
  if (sd<1)
     return;
  QString d(4096);
  if (socket->readLine(d)<1){
     sn->setEnabled(false);
     return;
  } 
#ifdef EDEBUG
  cout << ":"<<d<<":" << endl;
#endif
  if (d.isEmpty())
     doReply();
  else
     webParser->parseLine(d);  
}

void WebClient::doReply()
{
#ifdef EDEBUG
  cout << "doReply"<<endl;
#endif
  QString root = ircapp->readEntry("WebServerRootDir", ircapp->keircHome);
  QString fullPath = root+"/"+path;
  QFileInfo fi(fullPath);
  sendHeader();
  if (!fi.exists()){
     slotDisconnected();
     return;
  }

  if (fi.isDir())
     sendDir(fullPath);
  else
     sendFile(fullPath);
  start();
}

void WebClient::sendFile(const char* File)
{
  QString s;
  QString mt;
  
  QFile f(File);
  QFileInfo fi(f);
  if (!f.open(IO_Raw|IO_ReadOnly)){
    s="Cannot open "+QString(f.name());
    emit signWriteMsg(TYPE_ERROR|TYPE_IMG, s);
    return;
  }
  QString fs = File;
  int idx = fs.findRev('.');
  QString suffix;
  if (idx<0)
     suffix = "txt";
  else
     suffix = fs.right(fs.length()-idx-1);
  mt = webServer->mimeTypes->find(suffix);
  if (!mt)
     mt = "text/plain";

  QString size;
  size.setNum(fi.size());
  socket->writeLine("Content-type: "+mt);
  socket->writeLine("Content-length: "+size);
  QDateTime dt=fi.lastModified();
  socket->writeLine("Last-modified: "+ dt.toString());
  socket->writeLine("Connection: Keep-Alive");
  QString to = ircapp->readEntry("WebServerClientTimeout", "30");
  QString max= ircapp->readEntry("WebServerClientMax", "5");
  socket->writeLine("Keep-Alive: timeout="+to+", max="+max);
  socket->writeLine("");
  QString d(1024);
  while (!f.atEnd()){
    int bytesread = f.readBlock(d.data(), 1024);
    ::write(socket->getSocket(), d.data(), bytesread);
  }
  f.close();
}

void WebClient::sendDir(const char* File)
{
  QDir d(File);
  d.setFilter(QDir::Files | QDir::Hidden | QDir::NoSymLinks | QDir::Dirs);

  QString  index = ircapp->readEntry("WebServerIndex", "index.html");
  QString rnd;
  rnd.setNum((ulong)time(NULL));
  QString fs=QString(File)+"/"+rnd+".html";
  QFile f(fs);
  if (!f.open(IO_WriteOnly)){
     cerr << "ERROR:Cannot open "<<fs<<endl;
     return;
  }
  QTextStream ts(&f);
  ts<<"<HEAD><TITLE>Index of "+path+QString(" </TITLE></HEAD><BODY>")<<endl;
  ts<<"<H1>Index of "+path+QString(" </H1>")<<endl;
  ts<<"<PRE>"<<endl;
  QString s(4096);
  s.sprintf("%-5s %-20s %-25s %-10s %-10s", 
	    " ", "Name", "Last modified", "Size", "Description");
  ts<<s<<endl;
  ts<<"<HR>"<<endl;

  const QFileInfoList *list = d.entryInfoList();
  QFileInfoListIterator it(*list );
  QFileInfo *fi;
  for (fi=it.current();(fi=it.current())!=0L;++it){
    QString s(4096);
    QDateTime dt=fi->lastModified();
    if (fi->isDir()){
       QString dir=fi->fileName()+"/";
       if (!stricmp(dir, "./"))
	  continue;
       QString ref=dir+"</A>";
       s.sprintf("%-5s <A HREF=\"%s\">%-24s %-25s %-10s %-10s", 
		 "DIR", dir.data(), ref.data(), dt.toString().data(), "-", "-");
       ts<<s<<endl;
    }
    else{
       QString name=fi->fileName();
       if (!stricmp(name, rnd+".html"))
	  continue;
       if (!stricmp(name, index)){
	  f.close();
	  ::unlink(fs);
	  sendFile(fi->absFilePath());
	  return;
       }
       QString ref=name+"</A>";
       s.sprintf("%-5s <A HREF=\"%s\">%-24s %-25s %-10d %-10s", 
		 " ", name.data(), ref.data(), dt.toString().data(), fi->size(), "-");
       ts<<s<<endl;
    }
  }
  s = "</PRE></BODY>";
  ts<<s<<endl;
  f.flush();
  f.close();
  sendFile(fs);
  ::unlink(fs);
}

void WebClient::sendHeader()
{
  QStrList list;
  list.setAutoDelete(true);
  QString s;
  s = protocol+" 200 OK";
  socket->writeLine(s);
  QDateTime dateTime=QDateTime::currentDateTime();
  s = "Date: "+dateTime.toString();
  socket->writeLine(s);
  s = "Server: "+webServer->version;
  socket->writeLine(s);
}

void WebServer::Init()
{
  state  = 0;
  port   = 0;
  current= 0;
  mimeTypes = new QDict<char>(30, false, true);
  mimeTypes->setAutoDelete(true);
}

WebServer::WebServer(IrcClient* client, QObject* Parent, const char* Name):
  ClientCore(Parent, Name)
{
  WebServer::Init();
  setName("Web Server");
  version = "lesehan/0.0.1";
  this->client = client;
}


WebServer::~WebServer()
{
#ifdef EDEBUG
  cout << "Destruktor WebServer" << endl;
#endif
  
  WebClient* wc;
  for (wc=WebClient::clientList.first();(wc=WebClient::clientList.current())!=0L;){
      delete wc;
  }
  server.Close();
}

int WebServer::slotOpen()
{
  port = ircapp->readNumEntry("WebServerPort", 6789);
  return slotOpen(port);
}

int WebServer::slotOpen(int port)
{
  if (state){
     slotClosed();
     return 0;
  }

  client->addOutputPage("Web Server", MDI_DETACHED|MDI_CLOSED, false, "keirc-webserver.xpm");
  if (port<0)
     this->port = 0;
  else
     this->port = port;

  QString ps;
  ps.setNum(this->port);
  QString s = "Try to serve connection on "+ps+" , please wait...";
  client->slotWritePage("Web Server", TYPE_INFO|TYPE_IMG, s);

  client->Sleep(3);
  slotServe(this);
  return state;
}

void WebServer::slotServe(IrcCore*)
{
#ifdef EDEBUG
  cout << "WebServer::slotServe:"<< endl;
#endif

  QString host=ircapp->localHost;

  if (server.Open(host, port)<1){
     slotClosed();
     return;
  }
  state = 1;
  QString as, ps;
  ulong up = htonl(server.getLocalInetAddress().getAddress());
  as.setNum(up);
  ps.setNum(server.getLocalPort());
  QString s="Serve connection on "+host +" port "+ps;
  client->slotWritePage("Web Server", TYPE_INFO|TYPE_IMG, s);
  connect(&server,
	  SIGNAL(signAccept(ClientSocket*)),
	  SLOT  (slotAccept(ClientSocket*)));
}

void WebServer::slotAccept(ClientSocket* cs)
{
#ifdef EDEBUG
  cout << "WebServer::slotAccept:"<<endl;
#endif
  int max = ircapp->readNumEntry("WebServerClientMax", 5);
  if (current<max)
     current++;
  else{
     delete cs;
     return;
  }

  WebClient* wc = new WebClient(this);
  connect(wc,
	  SIGNAL(signConnected(IrcServer*)),
	  SIGNAL(signConnected(IrcServer*)));
  connect(wc,
	  SIGNAL(signDisconnected(IrcServer*)),
	  SIGNAL(signDisconnected(IrcServer*)));
  connect(wc,
	  SIGNAL(signWriteMsg(int, const char*, bool)),
	  SIGNAL(signWriteMsg(int, const char*, bool)));
  wc->setSocket(cs);
}

void WebServer::slotClosed()
{
#ifdef EDEBUG
  cout << "WebServer::slotClosed:"<<endl;
#endif
  state = 0;
  server.Close();
  QString s="Server closed...";
  client->slotWritePage("Web Server", TYPE_INFO|TYPE_IMG, s);
  client->Sleep(2);
  client->delOutputPage("Web Server");
}




