import os, string, time, ConfigParser, re
from stat import *
from ftp import *
from localtoremotetransfer import LocalToRemoteTransfer
from remotetolocaltransfer import RemoteToLocalTransfer
import dbutils

class Update:
    """
    This class mirrors a local directory to an ftp server.

    It accepts parameters:
    NAME=name of the project
    CONFIG=filename of configuration file
    (If absent the filename is taken to be ~/update.config)
    The configuration file must have permissions 0600.
    
    It has format:
    
    [name]
    put=1 or 0 (upload files to the remote directory
                or download files from the remote directory)
    ldir=the local dir
    rdir=the remote dir  (this may be left blank)
    db=filemodification time db
    host=the remote host
    login=the remote username
    passwd=remote password
    log=file which logs output from sessions
    kill=1 or 0 (do or don't remove remote files and directories that
                 don't occur locally)
    ignore=some python regular exp (files matching the regular exp will not
    be transferred). This field is optional.

    A filename must match the regular expression completely to be ignored.
    The part of the filename considered for matching is the full path but with
    the source dir removed. Thus for source dir /foo and file /foo/bar/baz only
    bar/baz is matched. /bar/ is called the prefix. The special escape sequence
    \p matches the prefix.
    Example: ignore=(\p\..*)|(.*~)|(.*\.pyc$) will mean any file or directory
    beginning with . or ending with ~ or .pyc will be ignored.)
    
    Likewise there are the
    only=
    ignoredirs=
    onlydirs=

    settings which ignore files which don't match a reg exp, ignore dirs
    which match a reg exp, and ignore dirs that don't macth a reg exp, resp.

    ignorekill=1 or 0 This deletes on the target files or directories
    that are ignored on the source directory.
    
    There can be several different [name] sections.
    
    if log= entry is missing the default is ~/update.name.log
    The default for the db= entry is ~/update.name.mod
    The default for kill is 0 (i.e. not remove unnecessary remote files.)

    readconfig() reads the configuration file
    initftp() logs in to the ftp server
    init_transfer() starts a transfer context
    readdb() reads the database of file time stamps.
    init() does the updating process.
    
    init() uploads files in the local directory that don't exist remotely
    or have a newer modification time than last time as judged by the
    database in the db=entry of the config file (or it's default replacement).

    If any errors are encountered the update session continues and files
    that should have been updated but were not will be updated next time.

    isStopped() checks to see if the session should be stopped.
    stop() sets a stop status.

    isDry() checks to see if the session is a dry run. setDry() sets
    the dry run status.

    isConfirmDel() checks to see whether delete confirmations should be made.
    setConfirmDel() sets the delete confirmation status.

    LIMITATIONS: assumes unix locally in a few places, needs ftp server
    to support MDTM SIZE

    Normal usage is:

    update=Update(name,config) #NAME is a project name and CONFIG an optional
                               #alternate configuration file
                               
    update.readconfig()        #read configuration file for NAME
    update.initftp()           #login to remote ftp server
    update.init_transfer()     #start up transfer context
    update.readdb()            #read time stamps database
    update.init()              #transfer files
    """
    def __init__(self,name,config=None):
        self.name=name
        self.config=config
        if self.config==None:
            self.config=os.path.expanduser("~")+'/update.config'
        self.setDry(0) #by default not a dry run
        self.stop(0)   #by default not stopped
        self.setConfirmDel(0) #by default no delete confirmations

    def readconfig(self):
        "Read configuration file."
        if not os.path.isfile(self.config):
            print "No such config file",self.config
            return 0
        #first check that config  file has 600 perms
        m=S_IMODE(os.stat(self.config)[ST_MODE])
        if m & (S_IXUSR| S_IRGRP| S_IWGRP|S_IXGRP| S_IROTH |S_IWOTH |S_IXOTH):
            print "Configuration file",self.config,"should have permissions 0600."
            return 0
        p=ConfigParser.ConfigParser()
        p.read(self.config)
        if self.name not in p.sections():
            print "No configuration file for project",self.name
            return 0
        if 'ignore' not in p.options(self.name):
            self.pat=None
        else:
            self.pat=p.get(self.name,'ignore')
        if 'only' not in p.options(self.name):
            self.onlypat=None
        else:
            self.onlypat=p.get(self.name,'only')

        if 'ignoredirs' not in p.options(self.name):
            self.patdirs=None
        else:
            self.patdirs=p.get(self.name,'ignoredirs')
        if 'onlydirs' not in p.options(self.name):
            self.onlypatdirs=None
        else:
            self.onlypatdirs=p.get(self.name,'onlydirs')

        if 'ignorekill' not in p.options(self.name):
            self.ignorekill=0
        else:
            try:
                self.ignorekill=p.getint(self.name,'ignorekill')
                if self.ignorekill not in (0,1): raise ValueError
            except ValueError:
                print "The ignorekill option in the configuration file",self.config,"must be 0 or 1."
                return 0

        if 'kill' not in p.options(self.name):
            self.kill=0
        else:
            try:
                self.kill=p.getint(self.name,'kill')
                if self.kill not in (0,1): raise ValueError
            except ValueError:
                print "The kill option in the configuration file",self.config,"must be 0 or 1."
                return 0
        if 'log' not in p.options(self.name):
            self.logfile=''
        else:
            self.logfile=p.get(self.name,'log')
        if self.logfile=='': self.logfile=os.path.expanduser('~')+'/update.'+self.name+'.log'
        if 'db' not in p.options(self.name):
            self.dbfile=''
        else:
            self.dbfile=p.get(self.name,'db')
        if self.dbfile=='': self.dbfile=os.path.expanduser('~')+'/update.'+self.name+'.mod'
        if 'rdir' not in p.options(self.name):
            self.rdir=''
        else:
            self.rdir=p.get(self.name,'rdir')
        if 'login1' not in p.options(self.name):
            self.login1=''
        else:
            self.login1=p.get(self.name,'login1')
        if 'passwd1' not in p.options(self.name):
            self.passwd1=''
        else:
            self.passwd1=p.get(self.name,'passwd1')
        try:
            self.put=p.getint(self.name,'put')
            if self.put not in (0,1): raise ValueError
            self.passwd=p.get(self.name,'passwd')
            if len(self.passwd)==0:
                raise ValueError,"password field empty"
            self.login=p.get(self.name,'login')
            if len(self.login)==0:
                raise ValueError,"username field empty"
            self.ldir=p.get(self.name,'ldir')
            if len(self.ldir)==0:
                raise ValueError,"local directory field empty"
            self.host=p.get(self.name,'host')
            if len(self.host)==0:
                raise ValueError,"remote host field empty"
        except (ConfigParser.InterpolationError,ConfigParser.NoOptionError,ValueError),detail:
            print "Failed to get option from the configuration file",self.config
            print detail
            return 0
        return 1
        
    def init(self):
        "Begin update."
        if self.openlog():
            if self.isDry(): self.logpr("Dry run, no files will actually be transferred or deleted.")
            self.logpr("Session on "+time.asctime(time.localtime(time.time())))
            self.logpr()
            if not os.path.isdir(self.ldir):
                try:
                    os.mkdir(self.ldir)
                except os.error,detail:
                    self.logpr("Can't create directory "+d,detail)
                    return self.exit(1)
                else:
                    self.logpr("Created directory "+self.ldir)
            msg="Transferring files from "
            if self.put:
                msg = msg+self.ldir+" to "+self.rdir
            else:
                msg = msg+self.rdir+" to "+self.ldir
            self.logpr(msg)
            if self.put:
                self.logpr("Processing: "+self.ldir)
            else:
                self.logpr("Processing: "+self.rdir)
            self.logpr("")
            self.process2()
            self.logpr()
            self.logpr("Done")
            self.logpr()
            self.logpr()
            self.logfd.close()
            self.ftpQuit()
            return self.exit(0)

    def ftpQuit(self):
        if self.xfer.ftp!=None:
            try:
                #attempt to quit gracefully
                self.xfer.ftp.quit()
            except all_errors,detail:
                #this probably doesn't matter
                print "(Non serious) error on quitting the ftp server"
                print detail
                #try deleting ftp
                if 'ftp' in dir(self.xfer):
                    del self.xfer.ftp

    def openlog(self):
        "Open log file."
        try:
            self.logfd=open(self.logfile,'a')
        except IOError,detail:
            print "Can't open log file for appending."
            print detail
            return 0
        return 1
        
    def initftp(self):
        "Connect to ftp server."
        self.ftp=Ftp()
        try:
            self.ftp.connect(self.host)
            self.ftp.login(self.login,self.passwd)
        except all_errors,detail:
            self.error=("Can't login to the remote host.",detail)
            return 0
        else:
            self.pr("Logged into "+self.host+" as user "+self.login)
        if self.login1!='':
            try:
                self.ftp.sendcmd('USER '+self.login1)
                self.ftp.sendcmd('PASS '+self.passwd1)
            except all_errors,detail:
                self.error=("Can't login as: "+self.login1,detail)
                return 0
            else:
                self.pr("Logged in as user: "+self.login1)
        self.ftp.set_pasv(1)
        if self.rdir!='':
            try:
                self.ftp.cwd(self.rdir)
            except all_errors,detail:
                self.pr("Couldn't cd into: "+self.rdir)
                self.pr(detail)
                self.pr("Trying to create it.")
                try:
                    self.ftp.mkd(self.rdir)
                except all_errors,detail:
                    self.error=("Couldn't create directory: "+self.rdir,detail)
                    return 0
                else:
                    self.pr("Created "+self.rdir)
                    try:
                        self.ftp.cwd(self.rdir)
                    except all_errors,detail:
                        self.error=("Couldn't cd into: "+self.rdir,detail)
                        return 0
        try:
            self.rdir=self.ftp.pwd()
        except all_errors,detail:
            self.error=("Couldn't issue pwd in: "+self.rdir,detail)
            return 0
        self.pr("Local directory: "+self.ldir)
        self.pr("Remote directory: "+self.rdir)
        self.pr()
        return 1

    def applyre(self,isfile,alist,d):
        if isfile:
            onlypat=self.onlypat
            pat=self.pat
            filetxt="file"
        else:
            onlypat=self.onlypatdirs
            pat=self.patdirs
            filetxt="directory"

        if pat==None and onlypat==None: return alist[:],[]
        if pat==None: pat='' #we ignore files matching the pattern so ''
                             #is the default--ignore none
        if onlypat==None: onlypat='.*' #we ignore files not matching the pattern
                                       #so .* is the default--ignore none
#        print "*",isfile,pat,onlypat
        prefix=self.xfer.sremove_top(d)
        if prefix!='': prefix=prefix+'/'
        try:
            rexp=re.compile(string.replace(pat,"\p",prefix))
            onlyrexp=re.compile(string.replace(onlypat,"\p",prefix))
        except re.error,detail:
            self.pr("Error in regular expression")
            self.pr((pat, "or", onlypat))
            self.pr(detail)
            return alist[:],[]
        returnlist=alist[:]   #this will contain the non-ignored files
        complist=[]           #this will contain the ignored files
        for item in alist:
            file=self.xfer.sremove_top(os.path.join(d,item))
#            print file
            ans=onlyrexp.match(file)
            #if there is a not a complete match
            if ans==None or ans.group()!=file:
#                if ans!=None: print "**",ans.group()
                returnlist.remove(item)
                complist.append(item)
                self.logpr("Ignoring "+filetxt+": "+os.path.join(d,item))
                continue
            ans=rexp.match(file)
            #if there is a complete match
            if ans!=None and ans.group()==file:
                returnlist.remove(item)
                complist.append(item)
                self.logpr("Ignoring "+filetxt+": "+os.path.join(d,item))
        return returnlist,complist

    def isStopped(self):
        if self._stopped:
            self.logpr("User termination")
        return self._stopped

    def stop(self,n=1):
        self._stopped=n

    def isDry(self):
        return self._dry

    def setConfirmDel(self,n):
        self._confirmDel=n

    def isConfirmDel(self):
        return self._confirmDel

    def yesno(self,f):
        ans=raw_input("Delete "+f+"? [y]n: ")
        if ans=='' or ans[0]=='y' or ans[0]=='Y': return 1
        return 2

    def setYesno(self,f):
        self.yesno=f

    def setDry(self,n=1):
        self._dry=n
        
    def process2(self,d=None):
        "Process the source directory D."

        #I skip out if the user chooses to abort
        if self.isStopped(): return 0
        if d==None:
            d=self.xfer.sdir
        target=self.xfer.path_sTot(d)
        origfiles,origdirs=self.xfer.sfilesdirs(d)
        #apply reg expressions to obtain ignored files and directories
        files,ignoredfiles=self.applyre(1,origfiles,d) 
        dirs,ignoreddirs=self.applyre(0,origdirs,d)   
                                                       
        #we clean ignored files and files in subdirectories
        #of ignored directories from the database
        #
        #This may not be strictly necessary
#        self.xfer.clean_file_db(d,ignoredfiles)
#        self.xfer.clean_dir_db(d,ignoreddirs)

        #get target files and directories
        tfiles,tdirs=self.xfer.tfilesdirs(target)

        if self.isStopped(): return 0

        #delete ignored files and directories on target if requested
        if self.ignorekill:
            for f in tfiles[:]:
                if self.isStopped(): return 0
                if f in ignoredfiles:
                    if self.isConfirmDel() and self.yesno(os.path.join(target,f))==2:continue
                    self.logpr("Deleting file: "+os.path.join(target,f))
                    if not self.isDry():
                        if not self.xfer.tdelete(os.path.join(target,f)):
                            self.logpr(self.xfer.error)
                        else:
                            del tfiles[tfiles.index(f)]
                            self.logpr("done")
            for f in tdirs[:]:
                if self.isStopped(): return 0
                if f in ignoreddirs:
                    if self.isConfirmDel() and self.yesno(os.path.join(target,f))==2:continue
                    self.logpr("Deleting directory: "+os.path.join(target,f))
                    if not self.isDry():
                        if not self.xfer.temptydir(os.path.join(target,f)):
                            self.logpr(self.xfer.error)
                        else:
                            del tdirs[tdirs.index(f)]
                            self.logpr("done")

        if self.isStopped(): return 0


        #create any target directories
        for f in dirs:
            if self.isStopped(): return 0
            if f not in tdirs:
                #must create target
                targetfile=os.path.join(target,f)
                if f in tfiles:
                    #file with that name so delete it
                    if not self.isDry():
                        if not self.xfer.tdelete(targetfile):
                            self.logpr(self.xfer.error)
                        else:
                            del tfiles[tfiles.index(f)]
                if not self.isDry():
                    if not self.xfer.tmkd(targetfile):
                        self.logpr(self.xfer.error)
                    else:
                        self.logpr("Created directory: "+targetfile)
                        tdirs.append(f)
                else:
                    self.logpr("Created directory: "+targetfile)
                    tdirs.append(f)

        if self.isStopped(): return 0


        #now update files from source to target
        for f in files:
            if self.isStopped(): return 0
            srcfile=os.path.join(d,f)
            modf=self.xfer.smdtm(srcfile)
            if srcfile not in self.xfer.db.keys() or modf > self.xfer.db[srcfile] or f not in tfiles:
                self.logpr("Transferring file: "+srcfile+" ...")
                targetfile=os.path.join(target,f)
                if f in tdirs:
                    #uh oh we want to upload a file there so
                    #we must kill the directory first
                    if not self.isDry():
                        if not self.xfer.temptydir(targetfile):
                            self.logpr("Can't transfer "+srcfile+" because there is a remote directory with the same name"+self.xfer.error)
                        else:
                            del tdirs[tdirs.index(f)]
                    else:
                        del tdirs[tdirs.index(f)]
                if not self.isDry():
                    if not self.xfer.put(srcfile):
                        self.logpr(self.xfer.error)
                        if self.isStopped(): return 0
                    else:
                        self.xfer.db[srcfile]=modf
                        self.logpr("done")
                        if f not in tfiles:
                            tfiles.append(f)
                else:
                    if f not in tfiles:
                        tfiles.append(f)

        if self.isStopped(): return 0

        #kill unnecessary target files and directories if requested
        if self.kill:
            for f in tfiles:
                if self.isStopped(): return 0
                if f not in origfiles:
                    targetfile=os.path.join(target,f)
                    if self.isConfirmDel() and self.yesno(targetfile)==2:continue
                    self.logpr("Deleting file: "+targetfile+" ...")
                    if not self.isDry():
                        if not self.xfer.tdelete(targetfile):
                            self.logpr(self.xfer.error)
                        else:
                            self.logpr("done")

            for f in tdirs:
                if self.isStopped(): return 0
                if f not in origdirs:
                    targetfile=os.path.join(target,f)
                    if self.isConfirmDel() and self.yesno(targetfile)==2:continue
                    self.logpr("Deleting directory: "+targetfile+" ...")
                    if not self.isDry():
                        if not self.xfer.temptydir(targetfile):
                            self.logpr(self.xfer.error)
                        else:
                            self.logpr("done")

        if self.isStopped(): return 0

        #now recurse directories
        for e in dirs:
            if self.process2(os.path.join(d,e))==0: return 0

        return 1
        
    def pr(self,s=''):
        "This can be conveniently overridden in other classes"
        if type(s)!=type(()):s=[s]
        for item in s:
            print str(item)

    def logpr(self,s=''):
        "Write to stdout and the log file."
        if type(s)!=type(()):
            s=[s]
        for msg in s:
            self.pr(msg)
            self.logfd.write(str(msg)+'\n')

    def exit(self,n):
        "Clean up on exit."

        if not self.isDry():
            d=dbutils.Dbutils(self.xfer.db,self.dbfile)
            if not d.write():
                self.pr(d.error)
                n=1
        if n:
            self.logpr("Abandoning session")
        self.logfd.close()
        return n

    def init_transfer(self,noftplogin=0):
        if noftplogin==1: self.ftp=None
        if self.put:
            self.xfer=LocalToRemoteTransfer(self.ldir,self.rdir,self.ftp,self.dbfile)
        else:
            self.xfer=RemoteToLocalTransfer(self.rdir,self.ldir,self.ftp,self.dbfile)

    def empty_db(self):
        self.xfer.db={}

    def init_db(self):
        if not self.xfer.init_sdb():
            self.error=self.xfer.error
            return 0
        return 1

    def cleardb(self):
        if not self.xfer.cleardb():
            self.error=self.xfer.error
            return 0
        return 1

    def readdb(self):
        if not self.xfer.readdb():
            self.error=self.xfer.error
            return 0
        return 1

    def setEvent(self,f):
        self.xfer.ftp.setEvent(f)

    def setProgress(self,init_p,do_p,finish_p):
        self.xfer.setProgress(init_p,do_p,finish_p)
        