/* $Id: kget.cpp,v 1.8 1999/03/02 21:05:29 koss Exp $
   $Log: kget.cpp,v $
   Revision 1.8  1999/03/02 21:05:29  koss
   bugfix : readTransfer now sets processedSize ( hack for statusbar )

   Revision 1.7  1999/02/23 21:50:38  koss
   Replaced about dialog and help menu with getHelpMenu() from
   KApplication
   getSize() is now called only for queued transfers ( not for delayed )
   statusbar update for speed and remaining time

   Revision 1.6  1999/02/21 11:39:25  koss
   Bugfix - displaying + flickering, now timer for animations is always
   started and only slotAnimTimeout calls repaint()

   Revision 1.5  1999/02/19 15:43:55  koss
   Fixed bug in showing speed, it remained from the times of kget 0.5.1
   when only ppp speed was showed.
   Fixed transfer mode switching + updateToolBar() + checkQueue()

   Revision 1.4  1999/02/07 17:27:53  matt
   updated for new setting dialogs
   new item statuses
   get sizes function
   slotError() update
   updated for new icons

   Revision 1.3  1998/12/19 19:40:34  matt
   new philosophy - mode and status
   new toolbar items
   new popup
   renaming items

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   */

extern "C" {
#include <mediatool.h>
}

#include "docking.h"

#include <qdir.h>
#include <qpainter.h>
#include <qclipboard.h>

#include <kapp.h>
#include <kiconloader.h>
#include <kstdaccel.h>
#include <kmsgbox.h>
#include <kwm.h>
#include <kaudio.h>
#include <kfiledialog.h>
#include <ksimpleconfig.h>
#include <kstring.h>
#include <kdebug.h>

#include <kseparator.h> // !!! remove this

#include <k2url.h>
#include <kio_job.h>
#include <kio_linedit_dlg.h>
#include <kio_error.h>

#include "kget-defaults.h"
#include "kget.h"
#include "version.h"
#include "kgetoptdlg.h"
#include "configdlg.h"


KMainWidget*	p_kmain;
DockWidget*     dock_widget;
KStdAccel*      keys;

QString currentDirectory;

// StatusBar field IDs
#define ID_TOTAL_TRANSFERS 1
#define ID_TOTAL_FILES     2
#define ID_TOTAL_SIZE      3
#define ID_TOTAL_TIME      4
#define ID_TOTAL_SPEED     5


// utility methods
QString getStringFromBool( bool x );
QString getStringFromInt( int x );
bool getBoolFromString( QString s );
int getIntFromString( QString s );


int main( int argc, char **argv ) {
  KConfig* cfg;

  KApplication a( argc, argv, "kget" );

  cfg = KApplication::getKApplication()->getConfig();
  keys = new KStdAccel(cfg);

  KIOJob::initStatic();
  KMyTabListBox::InitStatic();

  KMainWidget kmain;
  p_kmain = &kmain;
  
  kdebug( KDEBUG_INFO, 5001,"Init window" );

  a.setMainWidget(&kmain);

  return a.exec();
}



KMainWidget::KMainWidget( const char *name )
  : KTopLevelWidget( name )
{
  audio = 0L;
  confdlg = 0L;

  GlobalItemList.setAutoDelete( TRUE );
  b_deleteMode = FALSE;
  
  readSettings();

  if ( properties != "" )
    setGeometry(KWM::setProperties(winId(), properties));
  else
    resize( 450, 200 );

  setCaption("KGet "KGETVERSION);
  setMinimumSize( 350, 150 );

  setupMenuBar();
  setupToolBar();
  setupStatusBar();
  setupTabListBox();

  // Setup animation timer
  animTimer = new QTimer( this );
  animCounter = 0;
  connect( animTimer, SIGNAL( timeout() ),
	   this, SLOT( slotAnimTimeout() ) );

  if ( b_useAnimation )
    animTimer->start( 400 );
  else
    animTimer->start( 1000 );

  // Setup sound
  if ( b_useSound ) {
    if ( ! audio )
      audio = new KAudio;
    if ( audio->serverStatus() )
      kdebug( KDEBUG_ERROR, 5001,"Failed contacting audio server !");
  }
  else if ( audio ){
    delete audio;
    audio = 0L;
  }

  // Setup transfer timer
  transferTimer = new QTimer( this );
  connect( transferTimer, SIGNAL( timeout() ),
	   this, SLOT( slotTransferTimeout() ) );
  transferTimer->start( 60000 );

  // Setup autosave timer
  autosaveTimer = new QTimer( this );
  connect( autosaveTimer, SIGNAL( timeout() ),
	   this, SLOT( slotAutosaveTimeout() ) );

  if ( b_autoSave )
    autosaveTimer->start( autoSaveInterval*60000 );

  // Setup docking
  dock_widget = new DockWidget("dockw");

  // we need to add a separator, because we may start in expert mode
  currentDirectory = QDir::currentDirPath() + QDir::separator();
  readTransfers();

  if ( b_dockWindow ){
    b_dockWindow = !b_dockWindow;
    toggleDocking();
  }
  else
    this->show();
}


KMainWidget::~KMainWidget () {
  if ( animTimer ) {
    animTimer->stop();
    delete animTimer;
  }
  
  if ( audio )
    delete audio;
}


void KMainWidget::setupMenuBar(){
  file = new QPopupMenu ();
  options = new QPopupMenu ();

  options->setCheckable(true);
  op_statusbar = options->insertItem( i18n("Show &Statusbar"),
		       this, SLOT(slotShowStatusbar()) );
  options->setItemChecked( op_statusbar, showStatusbar );

  options->insertSeparator();
  op_useanimation = options->insertItem( i18n("Use &Animation"),
		       this, SLOT(slotAnimateStatus()) );
  options->setItemChecked( op_useanimation, b_useAnimation );

  op_usesound = options->insertItem( i18n("Use &Sound"),
		       this, SLOT(slotSoundStatus()) );
  options->setItemChecked( op_usesound, b_useSound );

  options->insertSeparator();
  options->insertItem( i18n("P&references..."),
		       this, SLOT(slotPreferences()));

  file->insertItem(i18n("&Open Transfer"),
		   this, 	SLOT(openTransfer()), keys->open());
  file->insertItem(i18n("&Paste Transfer"),
		   this, 	SLOT(pasteTransfer()), keys->paste());
  file->insertSeparator();
  file->insertItem(i18n("&Export Transfer List"),
		   this, 	SLOT(exportTransfers()) );
  file->insertItem(i18n("&Import Transfer List"),
		   this, 	SLOT(importTransfers()) );
  file->insertSeparator();
  file->insertItem (i18n("&Quit"),
		    this,	SLOT(quit()), keys->quit());


  QString about_text;

  about_text.sprintf("KGet Version %s\n\n%s", KGETVERSION,
		     i18n("Copyright (c) 1997, 1999\nMatej Koss\nkoss@napri.sk") );

  help = KApplication::getKApplication()->getHelpMenu( false, about_text.data());
  
  myMenuBar = new KMenuBar (this, "myMenuBar");
  myMenuBar->insertItem (i18n("&File"), file);
  myMenuBar->insertItem (i18n("&Options"), options);
  myMenuBar->insertSeparator();
  myMenuBar->insertItem (i18n("&Help"), help);

  setMenu(myMenuBar);
  myMenuBar->setMenuBarPos( menubarPos );
  
  // Setup popup menu for listbox
  popup = new QPopupMenu;
  popup->setCheckable( true );

  popup->insertItem(i18n("Copy URL to clipboard"), this,
		    SLOT(slotCopyToClipboard()));
  popup->insertSeparator();
  popup->insertItem(i18n("Configure transfer"), this,
		    SLOT(configCurrent()));
  popup->insertItem(i18n("Open download window"), this,
		    SLOT(openDownloadWindow()));
  popup->insertSeparator();

  pop_resume = popup->insertItem(i18n("Resume"), this,
				 SLOT(resumeCurrent()));
  pop_pause = popup->insertItem(i18n("Pause"), this,
				SLOT(pauseCurrent()));
  popup->insertItem(i18n("Delete"), this,
		    SLOT(deleteCurrent()));
  popup->insertSeparator();

  pop_queued = popup->insertItem(i18n("Queued"), this,
				 SLOT(queueCurrent()));
  pop_scheduled = popup->insertItem(i18n("Scheduled"), this,
				    SLOT(timerCurrent()));
  pop_delayed = popup->insertItem(i18n("Delayed"), this,
				  SLOT(delayCurrent()));
  popup->insertSeparator();

  pop_begin = popup->insertItem(i18n("Move to beginning"), this,
				SLOT(slotMoveToBegin()));
  pop_end = popup->insertItem(i18n("Move to end"), this,
			      SLOT(slotMoveToEnd()));
}



void KMainWidget::setupToolBar(){
  myToolBar = new KToolBar( this );

  myToolBar->insertButton(Icon("tool_resume.xpm"), TOOL_RESUME,
			  SIGNAL(clicked()), this,
			  SLOT(resumeCurrent()), FALSE, i18n("Resume"));

  myToolBar->insertButton(Icon("tool_pause.xpm"), TOOL_PAUSE,
			  SIGNAL(clicked()), this,
			  SLOT(pauseCurrent()), FALSE, i18n("Pause"));

  myToolBar->insertButton(Icon("tool_delete.xpm"), TOOL_DELETE,
			  SIGNAL(clicked()), this,
			  SLOT(deleteCurrent()), FALSE, i18n("Delete"));

  myToolBar->insertButton(Icon("tool_pause_all.xpm"), TOOL_PAUSE_ALL,
			  SIGNAL(clicked()), this,
			  SLOT(pauseAll()), FALSE, i18n("Pause all"));

//  myToolBar->insertLineSeparator();
   myToolBar->insertWidget( 54, 5, new KSeparator( QFrame::VLine, myToolBar ), -1 );
			   

  myToolBar->insertButton(Icon("tool_queue.xpm"), TOOL_QUEUE,
			  SIGNAL(clicked()), this,
			  SLOT(queueCurrent()), FALSE, i18n("Queued"));
  myToolBar->setToggle(TOOL_QUEUE, TRUE);

  myToolBar->insertButton(Icon("tool_timer.xpm"), TOOL_TIMER,
			  SIGNAL(clicked()), this,
			  SLOT(timerCurrent()), FALSE, i18n("Scheduled"));
  myToolBar->setToggle(TOOL_TIMER, TRUE);

  myToolBar->insertButton(Icon("tool_delay.xpm"), TOOL_DELAY,
			  SIGNAL(clicked()), this,
			  SLOT(delayCurrent()), FALSE, i18n("Delayed"));
  myToolBar->setToggle(TOOL_DELAY, TRUE);

//  myToolBar->insertLineSeparator();
   myToolBar->insertWidget( 55, 5, new KSeparator( QFrame::VLine, myToolBar ), -1 );

  myToolBar->insertButton(Icon("tool_preferences.xpm"), TOOL_PREFERENCES,
			  SIGNAL(clicked()), this,
			  SLOT(slotPreferences()), TRUE, i18n("Preferences"));

  myToolBar->insertButton(Icon("editpaste.xpm"), TOOL_PASTE,
			  SIGNAL(clicked()), this,
			  SLOT(pasteTransfer()), TRUE, i18n("Paste transfer"));

//  myToolBar->insertLineSeparator();
   myToolBar->insertWidget( 56, 5, new KSeparator( QFrame::VLine, myToolBar ), -1 );

  myToolBar->insertButton(Icon("tool_expert.xpm"), TOOL_EXPERT,
			  SIGNAL(clicked()), this,
			  SLOT(toggleExpertMode()), TRUE, i18n("Expert mode"));
  myToolBar->setToggle(TOOL_EXPERT, true);
  ((KToolBarButton*)myToolBar->getButton(TOOL_EXPERT))->on(b_expertMode);

  myToolBar->insertButton(Icon("tool_disconnect.xpm"), TOOL_DISCONNECT,
			  SIGNAL(clicked()), this,
			  SLOT(toggleAutoDisconnect()), TRUE, i18n("Auto Disconnect"));
  myToolBar->setToggle(TOOL_DISCONNECT, true);
  ((KToolBarButton*)myToolBar->getButton(TOOL_DISCONNECT))->on(b_autoDisconnect);

  myToolBar->insertButton(Icon("tool_shutdown.xpm"), TOOL_SHUTDOWN,
			  SIGNAL(clicked()), this,
			  SLOT(toggleAutoShutdown()), TRUE, i18n("Auto Shutdown"));
  myToolBar->setToggle(TOOL_SHUTDOWN, true);
  ((KToolBarButton*)myToolBar->getButton(TOOL_SHUTDOWN))->on(b_autoShutdown);

//  myToolBar->insertLineSeparator();
   myToolBar->insertWidget( 57, 5, new KSeparator( QFrame::VLine, myToolBar ), -1 );

  myToolBar->insertSeparator();

  myToolBar->insertButton(Icon("tool_dock.xpm"), TOOL_DOCK,
			  SIGNAL(clicked()), this,
			  SLOT(toggleDocking()), TRUE, i18n("Dock window"));
  myToolBar->setToggle(TOOL_DOCK, true);
  ((KToolBarButton*)myToolBar->getButton(TOOL_DOCK))->on(b_dockWindow);

  addToolBar(myToolBar);

  myToolBar->setBarPos( toolbarPos );
}



void KMainWidget::setupStatusBar(){
  myStatusBar = new KStatusBar( this );

  myStatusBar->setInsertOrder(KStatusBar::LeftToRight);

  QString tmpstr;

  ksprintf( &tmpstr, " %s : %d ", i18n("Transfers"), 99 );
  myStatusBar->insertItem( tmpstr.data(), ID_TOTAL_TRANSFERS);

  ksprintf( &tmpstr, " %s : %d ", i18n("Files"), 555 );
  myStatusBar->insertItem( tmpstr.data(), ID_TOTAL_FILES);

  ksprintf( &tmpstr, " %s : %s ", i18n("Size"), "134.56 kB" );
  myStatusBar->insertItem( tmpstr.data(), ID_TOTAL_SIZE);

  ksprintf( &tmpstr, " %s : %s ", i18n("Time"), "00:00:00" );
  myStatusBar->insertItem( tmpstr.data(), ID_TOTAL_TIME);

  myStatusBar->insertItem( "123.34 kB/sec", ID_TOTAL_SPEED);

  setStatusBar( myStatusBar );

  myStatusBar->message(i18n("Welcome to KGet"), 1000);
 
  if ( !showStatusbar )
    myStatusBar->enable( KStatusBar::Hide );
  else
    myStatusBar->enable( KStatusBar::Show );

  updateStatusBar();
}


void KMainWidget::setupTabListBox() {
  myTabListBox = new KMyTabListBox(this, "transferList", 9);

  myTabListBox->setTableFont( tablistboxfont );

  setView( myTabListBox, true);

  KDNDDropZone *myDropZone = new KDNDDropZone(this, DndURL);
  connect( myDropZone, SIGNAL( dropAction( KDNDDropZone *) ), 
	   this, SLOT( slotDropEvent( KDNDDropZone *) ) );

  connect( myTabListBox, SIGNAL( highlighted( int, int )), 
	   this, SLOT( slotHighlightedEvent( int )));
  connect( myTabListBox, SIGNAL( midClick( int, int )), 
	   this, SLOT( slotMidClickEvent( int )));
  connect( myTabListBox, SIGNAL( selected( int, int )), 
	   this, SLOT( openDownloadWindow()));
  connect( myTabListBox, SIGNAL( popupMenu( int, int )), 
	   this, SLOT( slotPopupMenu( int )));
}


void KMainWidget::slotPreferences() {
  KGETOptDlg dlg;
  QStrList strList( true );

  // transfer data to DlgConnection
  strList.clear();
  strList.append( getStringFromBool( b_reconnectOnError ) );
  strList.append( getStringFromInt( reconnectTime ) );
  strList.append( getStringFromInt( reconnectRetries ) );
  strList.append( getStringFromBool( b_reconnectOnBroken ) );
  strList.append( getStringFromInt( timeoutData ) );
  strList.append( getStringFromInt( timeoutDataNoResume ) );
//   strList.append( getStringFromInt( timeoutSearch ) );
  
  dlg.setConnectionData(&strList);

  // set data in DlgProxy
  dlg.setProxyData();

  // transfer data to DlgAutomation
  strList.clear();
  strList.append( getStringFromBool( b_autoSave ) );
  strList.append( getStringFromInt( autoSaveInterval ) );
  strList.append( getStringFromBool( b_autoDisconnect ) );
  strList.append( disconnectCommand );
  strList.append( getStringFromBool( b_autoShutdown ) );
//  strList.append( getStringFromBool( b_netscapeIntegrate ) );

  dlg.setAutomationData(&strList);

  // transfer data to DlgLimits
  strList.clear();
  strList.append( getStringFromInt( maxSimultaneousConnections ) );
  strList.append( getStringFromInt( minimumBandwidth ) );
  strList.append( getStringFromInt( maximumBandwidth ) );

  dlg.setLimitsData(&strList);

  // transfer data to DlgAdvanced
  strList.clear();
  strList.append( getStringFromBool( b_addQueued ) );
  strList.append( getStringFromBool( b_showDownloadWindows ) );
  strList.append( getStringFromBool( b_startIconified ) );
  strList.append( getStringFromBool( b_removeOnSuccess ) );
  strList.append( getStringFromBool( b_getSizes ) );
  //  strList.append( getStringFromBool( b_useCookies ) );
  strList.append( getStringFromBool( b_expertMode ) );

  dlg.setAdvancedData(&strList);

  // transfer data to DlgSystem
  strList.clear();
  strList.append( getStringFromBool( b_useSound ) );
  strList.append( audioFinished.data() );
  strList.append( audioFinishedAll.data() );
  strList.append( audioStartScheduled.data() );

  strList.append( getStringFromBool( b_useAnimation ) );
  strList.append( getStringFromBool( b_dockWindow ) );

  dlg.setSystemData(&strList);
  dlg.setFontData( tablistboxfont );

  // show the dialog
  int ret = dlg.exec();
  if( ret == QDialog::Accepted )
    {

      // write entries to the config file
      KConfig* config = KApplication::getKApplication()->getConfig();

      // get back the entries from DlgConnection
      strList = dlg.dataConnection();

      b_reconnectOnError = getBoolFromString( strList.first() );
      reconnectTime = getIntFromString( strList.next() );
      reconnectRetries = getIntFromString( strList.next() );
      b_reconnectOnBroken = getBoolFromString( strList.next() );
      timeoutData = getIntFromString( strList.next() );
      timeoutDataNoResume = getIntFromString( strList.next() );

//       timeoutSearch = getIntFromString( strList.next() );

      config->setGroup( "Connection" );
      config->writeEntry("ReconnectOnError", b_reconnectOnError);
      config->writeEntry("ReconnectTime", reconnectTime);
      config->writeEntry("ReconnectRetries", reconnectRetries);
      config->writeEntry("ReconnectOnBroken", b_reconnectOnBroken);
      config->writeEntry("TimeoutData", timeoutData);
      config->writeEntry("TimeoutDataNoResume", timeoutDataNoResume);
//       config->writeEntry("TimeoutSearch", timeoutSearch);
      
      // get back the entries from DlgProxy
      dlg.dataProxy();

      // get back the entries from DlgAutomation
      strList = dlg.dataAutomation();

      b_autoSave = getBoolFromString( strList.first() );
      autoSaveInterval = getIntFromString( strList.next() );

      if ( b_autoSave ) {
	autosaveTimer->stop();
	autosaveTimer->start( autoSaveInterval*60000 );
      }

      bool b_tmp = getBoolFromString( strList.next() );

      if ( b_tmp != b_autoDisconnect )
	toggleAutoDisconnect();

      disconnectCommand = strList.next();

      b_tmp = getBoolFromString( strList.next() );

      if ( b_tmp != b_autoShutdown )
	toggleAutoShutdown();

//      b_netscapeIntegrate = getBoolFromString( strList.next() );

      config->setGroup( "Automation" );
      config->writeEntry("AutoSave", b_autoSave);
      config->writeEntry("AutoSaveInterval", autoSaveInterval);
      config->writeEntry("DisconnectCommand", disconnectCommand );
//       config->writeEntry("NetscapeIntegrate", b_netscapeIntegrate );

      // get back the entries from DlgLimits
      strList = dlg.dataLimits();

      maxSimultaneousConnections = getIntFromString( strList.first() );
      minimumBandwidth = getIntFromString( strList.next() );
      maximumBandwidth = getIntFromString( strList.next() );
      checkQueue();

      config->setGroup( "Limits" );
      config->writeEntry("MaxSimConnections", maxSimultaneousConnections);
      config->writeEntry("MinimumBandwidth", minimumBandwidth);
      config->writeEntry("MaximumBandwidth", maximumBandwidth);

      // get back the entries from DlgAdvanced
      strList = dlg.dataAdvanced();

      b_addQueued = getBoolFromString( strList.first() );
      b_showDownloadWindows = getBoolFromString( strList.next() );
      b_startIconified = getBoolFromString( strList.next() );
      b_removeOnSuccess = getBoolFromString( strList.next() );
      b_getSizes = getBoolFromString( strList.next() );
      //      b_useCookies = getBoolFromString( strList.next() );

      config->setGroup( "Advanced" );
      config->writeEntry("AddQueued", b_addQueued);
      config->writeEntry("StartIconified", b_startIconified);
      config->writeEntry("RemoveOnSuccess", b_removeOnSuccess );
      config->writeEntry("GetSizes", b_getSizes );
      //      config->writeEntry("UseCookies", b_useCookies );

      b_tmp = getBoolFromString( strList.next() );
      if ( b_tmp != b_expertMode )
	toggleExpertMode();

      // get back the entries from DlgSystem

      tablistboxfont = dlg.dataFont();
      myTabListBox->setTableFont( tablistboxfont );

      strList = dlg.dataSystem();

      b_tmp = getBoolFromString( strList.first() );

      if ( b_tmp != b_useSound )
	slotSoundStatus();

      audioFinished = strList.next();
      audioFinishedAll = strList.next();;
      audioStartScheduled = strList.next();;

      b_tmp = getBoolFromString( strList.next() );

      if ( b_tmp != b_useAnimation )
	slotAnimateStatus();

      b_tmp = getBoolFromString( strList.next() );

      if ( b_tmp != b_dockWindow )
	toggleDocking();

      config->setGroup( "System" );
      config->writeEntry("Finished", audioFinished);
      config->writeEntry("FinishedAll", audioFinishedAll);
      config->writeEntry("StartScheduled", audioStartScheduled);

    }
}


void KMainWidget::readSettings(){

  KConfig *config = KApplication::getKApplication()->getConfig();

  // read system options
  config->setGroup("System");
  b_useSound = config->readBoolEntry("UseSound", DEF_UseSound);

  QString path = KApplication::getKApplication()->kde_datadir() + DEF_SoundFinished;
  audioFinished = config->readEntry("Finished", path.data());

  path = KApplication::getKApplication()->kde_datadir() + DEF_SoundFinishedAll;
  audioFinishedAll = config->readEntry("FinishedAll", path.data());

  path = KApplication::getKApplication()->kde_datadir() + DEF_SoundStartScheduled;
  audioStartScheduled = config->readEntry("StartScheduled", path.data());

  b_useAnimation = config->readBoolEntry("UseAnimation", DEF_UseAnimation);
  b_dockWindow = config->readBoolEntry("DockWindow", DEF_DockWindow);

  // read font options
  config->setGroup( "Font" );
  QFont font( DEF_Font, DEF_FontSize );
  tablistboxfont = config->readFontEntry("Font", &font );

  // read connection options
  config->setGroup("Connection");

  b_reconnectOnError = config->readBoolEntry("ReconnectOnError",DEF_ReconnectOnError);
  reconnectTime = config->readNumEntry("ReconnectTime", DEF_ReconnectTime);
  reconnectRetries = config->readNumEntry("ReconnectRetries", DEF_ReconnectRetries);
  b_reconnectOnBroken = config->readBoolEntry("ReconnectOnBroken", DEF_ReconnectOnBroken);

  timeoutData = config->readNumEntry("TimeoutData", DEF_TimeoutData);
  timeoutDataNoResume = config->readNumEntry("TimeoutDataNoResume", DEF_TimeoutDataNoResume);
//   timeoutSearch = config->readNumEntry("TimeoutSearch", DEF_TimeoutSearch);

  // read automation options
  config->setGroup("Automation");

  b_autoSave = config->readBoolEntry("AutoSave", DEF_AutoSave);
  autoSaveInterval = config->readNumEntry("AutoSaveInterval", DEF_AutoSaveInterval);

  b_autoDisconnect = config->readBoolEntry("AutoDisconnect", DEF_AutoDisconnect );
  disconnectCommand = config->readEntry("DisconnectCommand", DEF_DisconnectCommand);

  b_autoShutdown = config->readBoolEntry("AutoShutdown", DEF_AutoShutdown);
//   b_netscapeIntegrate = config->readBoolEntry("NetscapeIntegrate", DEF_NetscapeIntegrate);

  // read limits options
  config->setGroup("Limits");

  maxSimultaneousConnections = config->readNumEntry("MaxSimConnections", DEF_MaxSimConnections);
  minimumBandwidth = config->readNumEntry("MinimumBandwidth", DEF_MinimumBandwidth);
  maximumBandwidth = config->readNumEntry("MaximumBandwidth", DEF_MaximumBandwidth);

  // read advanced options
  config->setGroup("Advanced");

  b_addQueued = config->readBoolEntry("AddQueued", DEF_AddQueued);
  b_showDownloadWindows = config->readBoolEntry("ShowDownloadWindows", DEF_ShowDownloadWindows);
  b_startIconified = config->readBoolEntry("StartIconified", DEF_StartIconified);

  b_removeOnSuccess = config->readBoolEntry("RemoveOnSuccess", DEF_RemoveOnSuccess );
  b_getSizes = config->readBoolEntry("GetSizes", DEF_GetSizes );
  //  b_useCookies = config->readBoolEntry("UseCookies", DEF_UseCookies );
  b_expertMode = config->readBoolEntry("ExpertMode", DEF_ExpertMode );

  // read misc settings
  config->setGroup( "Misc" );

  QString entry = config->readEntry("Menubar", DEF_Menubar);
  if ( entry == "top" )
    menubarPos = KMenuBar::Top;
  if ( entry == "bottom" )
    menubarPos = KMenuBar::Bottom;
  if ( entry == "flat" )
    menubarPos = KMenuBar::Flat;

  entry = config->readEntry("Toolbar", DEF_Toolbar);
  if ( entry == "top" )
    toolbarPos = KToolBar::Top;
  else if ( entry == "left" )
    toolbarPos = KToolBar::Left;
  else if ( entry == "right" )
    toolbarPos = KToolBar::Right;    
  else if ( entry == "bottom" )
    toolbarPos = KToolBar::Bottom;    
  else if ( entry == "floating" )
    toolbarPos = KToolBar::Floating;    
  else if ( entry == "flat" )
    toolbarPos = KToolBar::Flat;    

  showStatusbar = config->readBoolEntry("Statusbar", DEF_Statusbar);

  // read geometry settings
  config->setGroup("Geometry");
  properties = config->readEntry("Properties", "" );
}



void KMainWidget::writeSettings(){
  KConfig *config = KApplication::getKApplication()->getConfig();
  QString entry;

  // connection and limits options are written from slotPreferences

  // write automation options
  config->setGroup( "Automation" );
  config->writeEntry( "AutoDisconnect", b_autoDisconnect );// only need to write these
  config->writeEntry( "AutoShutdown", b_autoShutdown );

  // write advanced options
  config->setGroup( "Advanced" );
  config->writeEntry("ShowDownloadWindows", b_showDownloadWindows);
  config->writeEntry( "ExpertMode", b_expertMode ); // only need to write this

  // write system options
  config->setGroup( "System" );
  config->writeEntry("UseSound", b_useSound);
  config->writeEntry( "UseAnimation", b_useAnimation );
  config->writeEntry( "DockWindow", b_dockWindow );

  config->setGroup( "Font" );
  config->writeEntry("Font", tablistboxfont );

  // write misc settings
  config->setGroup( "Misc" );

  if ( myMenuBar->menuBarPos() == KMenuBar::Top )
    config->writeEntry( "Menubar", "top" );
  else if ( myMenuBar->menuBarPos() == KMenuBar::Bottom )
    config->writeEntry( "Menubar", "bottom" );
  else if ( myMenuBar->menuBarPos() == KMenuBar::Flat )
    config->writeEntry( "Menubar", "flat" );

  if ( myToolBar->barPos() == KToolBar::Top )
    config->writeEntry( "Toolbar", "top" );
  else if ( myToolBar->barPos() == KToolBar::Bottom )
    config->writeEntry( "Toolbar", "bottom" );
  else if ( myToolBar->barPos() == KToolBar::Left )
    config->writeEntry( "Toolbar", "left" );
  else if ( myToolBar->barPos() == KToolBar::Right )
    config->writeEntry( "Toolbar", "right" );
  else if ( myToolBar->barPos() == KToolBar::Floating )
    config->writeEntry( "Toolbar", "floating" );
  else if ( myToolBar->barPos() == KToolBar::Flat )
    config->writeEntry( "Toolbar", "flat" );

  config->writeEntry( "Statusbar", showStatusbar );

  config->setGroup( "Geometry" );
  properties = KWM::getProperties(winId());
  config->writeEntry("Properties", properties );

  config->sync();
}


void KMainWidget::importTransfers() {
  readTransfers( true );
}


void KMainWidget::readTransfers( bool ask_for_name ) {
  kdebug( KDEBUG_INFO, 5001,"readTransfers()");

  QString txt;
  if ( ask_for_name )
    txt = KFileDialog::getOpenFileName(currentDirectory.data());
  else
    txt = KApplication::localkdedir() + "/share/apps/kget/transfers";

  KSimpleConfig config( txt );
  
  config.setGroup( "Common" );

  int num = config.readNumEntry( "Count", 0 );

  QString str, f, g;
  QString modeIcon;
  int mode;

  KItem *item;

  QTime time;
  QDate date;
  QDateTime qdt;

  while ( num-- ) {
    ksprintf( &str, "Item%d", num );
    config.setGroup( str.data() );
    
    f = config.readEntry( "Source", "" );
    g = config.readEntry( "Dest", "" );

    if ( f.isEmpty() || g.isEmpty() )
      continue;

    K2URL url( f.data() );
    if ( url.isMalformed() ){
      QString msg;
      ksprintf( &msg, "%s\n%s", i18n("Malformed URL :"), f.data() );
      KMsgBox::message( this, i18n( "KGet Error" ), msg.data() );
      return;
    }

    mode = config.readNumEntry( "Mode", KItem::MD_QUEUED );

    if ( mode == KItem::MD_SCHEDULED ) {

      time.setHMS( config.readNumEntry("Hour"), config.readNumEntry("Minute"), 0 );

      date.setYMD( config.readNumEntry("Year"),
 		   config.readNumEntry("Month"),
 		   config.readNumEntry("Day") );

      qdt.setTime( time );
      qdt.setDate( date );
      if ( !qdt.isValid() ){
	QString msg;
	ksprintf( &msg, "%s\n%s", i18n("Invalid timer setting : "),
		  qdt.toString().data() );
	KMsgBox::message( this, i18n( "KGet Error" ), msg.data() );

	mode = KItem::MD_QUEUED;
	qdt = QDateTime::currentDateTime();
      }
    }

    switch ( mode ) {

    case KItem::MD_QUEUED:
      modeIcon = "queued";
      break;
    case KItem::MD_SCHEDULED:
      modeIcon = "scheduled";
      break;
    case KItem::MD_DELAYED:
      modeIcon = "delayed";
      break;
    }

    int status = config.readNumEntry( "Status", KItem::ST_RUNNING );

    KIOJob *job = new KIOJob();

    if ( status != KItem::ST_FINISHED ) {
      item = new KItem( f, g, job->id(), job, mode );
      item->retryCount = reconnectRetries;
      Connect( job );
    }
    else {
      item = new KItem( f, g, job->id(), 0L, mode );
      item->status = KItem::ST_FINISHED;
      modeIcon = "finished";
    }

    item->startTime = qdt;
    item->canResume = config.readBoolEntry( "CanResume", true );
    item->totalSize = config.readNumEntry( "TotalSize", 0 );
    item->processedSize = config.readNumEntry( "ProcessedSize", 0 );
    item->totalFiles = config.readNumEntry( "TotalFiles", 1 );
    item->processedFiles = config.readNumEntry( "ProcessedFiles", 0 );
    item->speed = config.readNumEntry( "Speed", 0 );

    if ( item->totalSize == 0 )
      item->percent = 0;
    else
      item->percent = (item->processedSize / item->totalSize) * 100;

    QString statusString, tmps;

    // Create one line in listbox

    // icon 
    statusString = modeIcon + "\n";

    // filename
    statusString += url.filename().c_str();
    statusString += "\n";

    // resume
    if ( item->canResume )
      statusString += i18n("Yes");
    else
      statusString += i18n("No");
    statusString += "\n";

    // processed / total files
    tmps.setNum( item->processedFiles );
    statusString += tmps;
    tmps.setNum( item->totalFiles );
    statusString += (" / " + tmps + "\n");

    // percents
    if ( item->percent == 100 )
	statusString += "OK";
    else {
      tmps.setNum( item->percent );
      statusString += tmps.data();
    }
    statusString += "\n";

    // total size
    statusString += convertSize( item->totalSize );
    statusString += "\n";

    // speed
    statusString += convertSize( item->speed );
    statusString += "/s\n";

    // remaining time
    statusString += " \n";

    // url
    statusString += f.data();

    myTabListBox->insertItem( statusString.data() );

    // append new KItem to global list
    GlobalItemList.append ( item );

  }

  checkQueue();

  myTabListBox->unmarkAll();

}


void KMainWidget::exportTransfers() {
  writeTransfers( true );
}


void KMainWidget::writeTransfers( bool ask_for_name ) {
  kdebug( KDEBUG_INFO, 5001,"writeTransfers()");

  KItem *item;
  QString str;

  QString txt;
  if ( ask_for_name )
    txt = KFileDialog::getSaveFileName(currentDirectory.data());
  else
    txt = KApplication::localkdedir() + "/share/apps/kget/transfers";

  KSimpleConfig config( txt );

  int num = GlobalItemList.count();
  config.setGroup( "Common" );
  config.writeEntry( "Count", num );

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()){
    num--;
    ksprintf( &str, "Item%d", num );
    config.setGroup( str.data() );
    config.writeEntry( "Source", item->src );
    config.writeEntry( "Dest", item->dest );
    config.writeEntry( "Mode", item->mode );
    config.writeEntry( "Status", item->status );
    config.writeEntry( "CanResume", item->canResume );
    config.writeEntry( "TotalSize", item->totalSize );
    config.writeEntry( "ProcessedSize", item->processedSize );
    config.writeEntry( "TotalFiles", item->totalFiles );
    config.writeEntry( "ProcessedFiles", item->processedFiles );
    config.writeEntry( "Speed", item->speed );

    QTime time = item->startTime.time();
    QDate date = item->startTime.date();
    config.writeEntry( "Hour", time.hour() );
    config.writeEntry( "Minute", time.minute() );
    config.writeEntry( "Day", date.day() );
    config.writeEntry( "Month", date.month() );
    config.writeEntry( "Year", date.year() );
  }

  config.sync();
}


void KMainWidget::quit(){
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->status == KItem::ST_RUNNING && !b_expertMode ) {
      int res = KMsgBox::yesNo(this, i18n("KGet Message"), i18n("Some transfers are still running. Are you sure you want to close KGet ?"));
      if ( res == 2 )
    	return;
    }

  writeTransfers();

  b_deleteMode = TRUE;
  writeSettings();
  KApplication::getKApplication()->quit();
}


void KMainWidget::resumeCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Resuming job %d", id);

  KItem *it = GlobalItemList.at(id);

  ASSERT( it->status == KItem::ST_STOPPED );

  KIOJob *job = new KIOJob();
  Connect( job );

  it->id = job->id();
  it->job = job;

  // copy selected KIOJob
  job->copy( it->src.data(), it->dest.data() );

  it->status = KItem::ST_TRYING;
  it->mode = KItem::MD_QUEUED;
  
  checkQueue();
}


void KMainWidget::pauseCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Pausing job %d", id);

  KItem *it = GlobalItemList.at(id);

  ASSERT( it->status != KItem::ST_STOPPED && it->status != KItem::ST_FINISHED &&
	  it->status != KItem::ST_SIZE_CHECK );

  if ( it->status != KItem::ST_RETRYING )
    it->job->kill( true ); // stop selected KIOJob

  it->status = KItem::ST_STOPPED;
  it->mode = KItem::MD_DELAYED;

  checkQueue();
}


void KMainWidget::deleteCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  if ( !b_expertMode ) {
    int res = KMsgBox::yesNo(this, i18n("KGet Message"), i18n("Are you sure you want to delete this transfer ?"));
    if ( res == 2 )
      return;
  }

  KItem *it = GlobalItemList.at(id);

  kdebug( KDEBUG_INFO, 5001, "Deleting job %d", id);

  if ( it->status == KItem::ST_RUNNING || it->status == KItem::ST_TRYING )
    it->job->kill(); // kill selected KIOJob only when running
  else
    slotCanceled( it->id );
}


void KMainWidget::pauseAll() {
  if ( myTabListBox->count() == 0 )
    return;

  kdebug( KDEBUG_INFO, 5001, "Pausing all jobs" );

  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()) {
    if ( item->status == KItem::ST_RUNNING || item->status == KItem::ST_TRYING ) {
      item->job->kill( true ); // stop selected KIOJob
    }
    item->mode = KItem::MD_DELAYED;
    item->status = KItem::ST_STOPPED;
  }

  checkQueue();
}


void KMainWidget::queueCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Queuing job %d" , id);

  KItem *it = GlobalItemList.at(id);

  if ( it->mode == KItem::MD_QUEUED )
    return;

  it->mode = KItem::MD_QUEUED;

  checkQueue();
}


void KMainWidget::timerCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Timing job %d" , id);

  KItem *it = GlobalItemList.at(id);

  if ( it->mode == KItem::MD_SCHEDULED )
    return;

  it->mode = KItem::MD_SCHEDULED;
  it->startTime = QDateTime::currentDateTime();
  configCurrent();

  checkQueue();
}


void KMainWidget::delayCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Delaying job %d" , id);

  KItem *it = GlobalItemList.at(id);

  if ( it->mode == KItem::MD_DELAYED )
    return;

  if ( it->status == KItem::ST_RUNNING || it->status == KItem::ST_TRYING )
    it->job->kill( true ); // stop selected KIOJob

  it->mode = KItem::MD_DELAYED;
  it->status = KItem::ST_STOPPED;

  checkQueue();
}


void KMainWidget::openTransfer() {
  KLineEditDlg *box = new KLineEditDlg(i18n("Open transfer :"), "", this);
  box->show();

  if (!box->result())   /* cancelled */
    return;

  QString newtransfer = box->text();
  delete box;

  if ( newtransfer.isEmpty() ) /* answer is "" */
    return;

  addTransfer( newtransfer );
}



void KMainWidget::pasteTransfer() {
  QString newtransfer;
  QClipboard *cb = QApplication::clipboard();

  newtransfer = cb->text();
  newtransfer = newtransfer.stripWhiteSpace();

  if ( ! b_expertMode ) {
    KLineEditDlg *box = new KLineEditDlg(i18n("Open transfer :"),
					       newtransfer.data(), this );
    box->show();
    
    if (!box->result())   // cancelled
      return;
    
    newtransfer = box->text();
    delete box;
  }

  if ( newtransfer.isEmpty() )
    return;

  addTransfer( newtransfer );
}


void KMainWidget::addTransfer( QString s ) {
  K2URL url( s.data() );

  // don't download file URL's
  if ( strcmp( url.protocol(), "file" ) == 0L )
    return;

  if ( url.isMalformed() ){
    QString msg;
    ksprintf( &msg, "%s\n%s", i18n("Malformed URL :"), s.data() );
    KMsgBox::message( this, i18n( "KGet Error" ), msg.data() );
    return;
  }

  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->src == s ){ // if we find this URL in the list
      QString e;
      e << i18n( "Already saving URL \n" ) << s.data();
      KMsgBox::message( this, i18n( "KGet Error" ), e.data() );
      return;
    }

  QString dest;

  if ( !b_expertMode ) {
    KFileDialog *dlg = new KFileDialog( currentDirectory.data(), "*",
					0L , "filedialog", true, false );
    dlg->setSelection( currentDirectory + "/" + url.filename().c_str() );

    int res = dlg->exec();

    if ( !res ) {
      delete dlg;
      return;
    }
    else {
      dest = "file:";
      dest += dlg->selectedFile();
      currentDirectory = dlg->dirPath();
      delete dlg;
    }
  }
  else
    dest = "file:" + currentDirectory + url.filename().c_str();

  QString file = url.filename().c_str();
  QString modeIcon;
  int mode;

  if ( b_addQueued ){
    mode = KItem::MD_QUEUED;
    modeIcon = "queued\n";
  }
  else{
    mode = KItem::MD_DELAYED;
    modeIcon = "delayed\n";
  }

  myTabListBox->insertItem( modeIcon + file + "\n \n \n \n \n \n \n" + s.data() );

  KIOJob *job = new KIOJob();

  item = new KItem( s, dest, job->id(), job, mode );
  item->retryCount = reconnectRetries;

  // append new KItem to global list
  GlobalItemList.append ( item );

  Connect( job );
  
  // get size if flag is set, but only if mode is queued !!! ( later checkOnline() )
  if ( b_getSizes && item->mode == KItem::MD_QUEUED ){
    item->status = KItem::ST_SIZE_CHECK;
    job->getSize( s.data() );
  }

  checkQueue();

  myTabListBox->unmarkAll();
}


void KMainWidget::checkQueue() {
  uint numRun = 0;
  int status;
  KItem *item;

  kdebug( KDEBUG_INFO, 5001,"Checking queue");

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    {
      status = item->status;
      if ( status == KItem::ST_RUNNING || status == KItem::ST_TRYING )
	numRun++;
    }

  item=GlobalItemList.first();
  while ( numRun < maxSimultaneousConnections && item != 0 ){
    if ( ( item->status == KItem::ST_STOPPED || item->status == KItem::ST_RETRYING )
	 && item->mode == KItem::MD_QUEUED )
      {
	kdebug( KDEBUG_INFO, 5001,"Resuming another queued job");
	myStatusBar->message( i18n("Resuming another queued job."), 1000);
	
	KIOJob *job = new KIOJob();
	
	Connect( job );
	
	item->id = job->id();
	item->job = job;

    	job->copy( item->src.data(), item->dest.data());

//   	job->copy( item->src.data(), "ftp://matt:rockys@localhost/home/matt/kde/kget/WWW" );

//   	list<string> lst;
//   	lst.push_back( item->src.data() );
//   	job->copy( lst, "file:/home/matt/kde/tmp" );

	item->status = KItem::ST_TRYING;
	numRun++;
      }

    item=GlobalItemList.next();
  }

  if ( numRun > 0 )
    myToolBar->setItemEnabled( TOOL_PAUSE_ALL, TRUE);
  else
    myToolBar->setItemEnabled( TOOL_PAUSE_ALL, FALSE);

  updateToolBar();
  updateStatusBar();
}


void KMainWidget::slotAnimTimeout(){
  KItem *item;
  QString s;
  int index = 0;

  if ( GlobalItemList.isEmpty() ){
    dock_widget->setAnim( false );
    return;
  }

  bool isTransfer = false;

  animCounter++;
  if ( animCounter == myTabListBox->getPhasesNum() )
    animCounter = 0;

  myTabListBox->setAutoUpdate( false );

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()) {

    if ( item->status == KItem::ST_RUNNING ){
      if ( b_useAnimation )
	s.sprintf("connect%i", animCounter);
      else
	s = "connect5";
      isTransfer = true;
    } else if ( item->status == KItem::ST_TRYING ){
      if ( b_useAnimation )
	s.sprintf("try%i", animCounter);
      else
	s = "try0";
      isTransfer = true;
    } else if ( item->status == KItem::ST_RETRYING ){
      s = "retrying";
      isTransfer = true;
    } else if ( item->status == KItem::ST_STOPPED ||
		item->status == KItem::ST_SIZE_CHECK ){
      if ( item->mode == KItem::MD_QUEUED )
	s = "queued";
      else if ( item->mode == KItem::MD_SCHEDULED )
	s = "scheduled";
      else
	s = "delayed";
    } else if ( item->status == KItem::ST_FINISHED )
      s = "finished";

    myTabListBox->changeItemPart( s, index, TB_PIXMAP );

    index++;
  }

  myTabListBox->setAutoUpdate( true );

  if ( this->isVisible() )
    myTabListBox->repaint();

  updateStatusBar();

  if ( dock_widget->isDocked() )
    dock_widget->setAnim( isTransfer );
}


void KMainWidget::slotTransferTimeout() {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()){
    QTime curtime = QTime::currentTime();
    QTime time = item->startTime.time();
    
    if ( item->mode == KItem::MD_SCHEDULED &&
	 item->startTime.date() <= QDate::currentDate() &&
	 time.hour() <= curtime.hour() &&
	 time.minute() <= curtime.minute() )
      {
	item->mode = KItem::MD_QUEUED;
	checkQueue();
      }
  }
}


void KMainWidget::slotAutosaveTimeout() {
  writeTransfers();
}


void KMainWidget::slotError( int id, int errid, const char* errortext ) {
  popup->hide();

  kdebug( KDEBUG_INFO, 5001,"Error %d %s", errid, errortext);
  // get formated error message
  QString msg = kioErrorString( errid, errortext );

  // pause all transfers if Internet connection has been broken
  if ( ! checkOnline() )
    pauseAll();

  KItem *item;
  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      
      // if this error comes from user cancelation, then only cancel and quit
      if ( errid == ERR_USER_CANCELED ) {
	slotCanceled( item->id );
	return;
      }

      // show message only if we are in normal mode
      if ( !b_expertMode ){
	KMsgBox *m = new KMsgBox( 0L, i18n("Error"), 
				  msg.data(), KMsgBox::EXCLAMATION, 
				  i18n("Keep"), 
				  i18n("Cancel" ) );
	m->show();
	if (m->result() == 2)
	  slotCanceled( item->id );
      }

      switch ( errid ) {

      case ERR_CONNECTION_BROKEN:
	{
	  // when this option is set, automatically set for restart
	  if ( b_reconnectOnBroken ) {
	    item->status = KItem::ST_RETRYING;
	    item->mode = KItem::MD_QUEUED;
	  } else {
	    item->status = KItem::ST_STOPPED;
	    item->mode = KItem::MD_DELAYED;
	    myStatusBar->message( i18n("Reconnecting broken transfer"), 1000);
	  }
	  break;
	}
      case ERR_CANNOT_OPEN_FOR_READING:
      case ERR_DOES_NOT_EXIST:
      case ERR_ACCESS_DENIED:
      case ERR_CANNOT_ENTER_DIRECTORY:
      case ERR_COULD_NOT_CREATE_SOCKET:
      case ERR_COULD_NOT_CONNECT:
      case ERR_UNKNOWN_HOST:
      case ERR_UNKNOWN_PROXY_HOST:
      case ERR_COULD_NOT_READ:
      case ERR_COULD_NOT_LOGIN:
	{
	  // when this option is set, start timeout for restart
	  if ( b_reconnectOnError ) {
	    item->retryCount--;
	    if ( item->retryCount == 0 ) {  // no more retries
	      item->status = KItem::ST_STOPPED;
	      item->mode = KItem::MD_DELAYED;
	    } else {
	      item->status = KItem::ST_RETRYING;
	      item->mode = KItem::MD_SCHEDULED;
	      item->startTime=QDateTime::currentDateTime().addSecs(reconnectTime*60);
	      QString tmps;
	      ksprintf( &tmps, "%s %d", i18n("Attempt number"), item->retryCount );
	      myStatusBar->message( tmps, 1000);
	    }
	  } else { // if reconnecting is not enabled - simply set to delayed
	    item->status = KItem::ST_STOPPED;
	    item->mode = KItem::MD_DELAYED;
	  }
	  break;
	}
      }
      checkQueue();
      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotError for KIOJob : %d", id);
}


void KMainWidget::slotFinished( int id ) {
  popup->hide();

  if ( b_deleteMode )     // so that we don't start new jobs 
    return;

  kdebug( KDEBUG_INFO, 5001,"slotFinished");
  KItem *item;
  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {

      // if we were only checking the size, then don't remove from the list
      if ( item->status == KItem::ST_SIZE_CHECK ) {
	item->status = KItem::ST_STOPPED;
	checkQueue();
	return;
      }

      int index = GlobalItemList.find( item );

      if ( b_removeOnSuccess ) {
	GlobalItemList.remove( item );

	myTabListBox->removeItem( index );

	if ( myTabListBox->count() == 0)
	  myToolBar->setItemEnabled( TOOL_PAUSE_ALL, FALSE);

      } else
	item->status = KItem::ST_FINISHED;

      if ( b_useSound && audio )
	if ( downloadStatus() == STATUS_EMPTY )
	  audio->play( audioFinishedAll );
	else
	  audio->play( audioFinished );

      checkQueue();

      int stat = downloadStatus();

      if ( stat != STATUS_LOADING) { // disconnect only when we are not downloading
	if ( b_autoDisconnect )
	  disconnect();
      }
      else if ( stat != STATUS_SCHEDULED ) { // shutdown only when there are no
	if ( b_autoShutdown )                // scheduled downloads
	  quit();
      }

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotFinished for KIOJob : %d", id);
}


void KMainWidget::slotCanceled( int id ) {
  popup->hide();

  kdebug( KDEBUG_INFO, 5001,"slotCanceled");
  KItem *item;
  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find( item );

      GlobalItemList.remove( item );

      myTabListBox->removeItem( index );

      if ( myTabListBox->count() == 0)
	myToolBar->setItemEnabled( TOOL_PAUSE_ALL, FALSE);

      checkQueue();

      int stat = downloadStatus();

      if ( stat != STATUS_LOADING) { // disconnect only when we are not downloading
	if ( b_autoDisconnect )
	  disconnect();
      }
      else if ( stat != STATUS_SCHEDULED ) { // shutdown only when there are no
	if ( b_autoShutdown )                // scheduled downloads
	  quit();
      }

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotCanceled for KIOJob : %d", id);
}


void KMainWidget::slotCopying( int id, const char *_from, const char* _to) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      K2URL url( _to );
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( _from, index, TB_URL);
      myTabListBox->changeItemPart(url.filename().c_str(), index, TB_LOCAL_FILENAME );
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotCopying for KIOJob : %d", id );
}


void KMainWidget::slotRenamed( int id, const char *_new ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      K2URL url( _new );
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( url.filename().c_str(), index, TB_LOCAL_FILENAME);
      myTabListBox->changeItemPart( _new, index, TB_URL);
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotRenamed for KIOJob : %d", id);
}


void KMainWidget::slotCanResume( int id, bool _resume ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      QString tmp;
      if ( _resume )
	tmp = i18n("Yes");
      else
	tmp = i18n("No");
	
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( tmp.data(), index, TB_RESUME);
      myTabListBox->setAutoUpdate( true );

      item->canResume = _resume;

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotResumed for KIOJob : %d", id);
}


void KMainWidget::slotSpeed( int id, unsigned long _bytes_per_second ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );
      item->speed = _bytes_per_second;

      unsigned long secs = ( item->totalSize - item->processedSize ) / _bytes_per_second;
      int hr = secs / ( 60 * 60 );
      int mn = ( secs - hr * 60 * 60 ) / 60;
      int sc = ( secs - hr * 60 * 60 - mn * 60 );

      item->remainingTime.setHMS( hr, mn, sc );

      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( item->remainingTime.toString().data(),
				    index, TB_REMAINING_TIME );

      QString tmps;
      ksprintf( &tmps, "%s/s" , convertSize(_bytes_per_second).data() );
      myTabListBox->changeItemPart( tmps.data(), index, TB_SPEED );
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotSpeed for KIOJob : %d", id);
}


void KMainWidget::slotTotalSize( int id, unsigned long _bytes ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      item->totalSize = _bytes;
      int index = GlobalItemList.find ( item );

      if ( item->status == KItem::ST_TRYING )
	item->status = KItem::ST_RUNNING;

      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( convertSize( _bytes ).data(), index, TB_TOTAL);
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotTotalSize for KIOJob : %d", id);
}


void KMainWidget::slotProcessedSize( int id, unsigned long _bytes ) {
  KItem *item;

  kdebug( KDEBUG_INFO, 5001, "slotProcessedSize");
  for ( item =GlobalItemList.first(); item != 0; item=GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      int old = item->percent;

      item->processedSize = _bytes;
      item->percent = (int)(( (float)_bytes / (float)item->totalSize ) * 100.0);
      
      QString s;

      if ( item->percent == 100 )
	s = "OK";
      else
	s.sprintf("%d", item->percent);

      if ( item->percent != old ) {
	myTabListBox->setAutoUpdate( false );
	myTabListBox->changeItemPart( s, index, TB_PROGRESS);
	myTabListBox->setAutoUpdate( true );
      }

      return;
    } 
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotProcessedSize for KIOJob : %d", id);
}


void KMainWidget::slotTotalFiles( int id, unsigned long _files ) {
  KItem *item;

  for ( item =GlobalItemList.first(); item != 0; item=GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );
      item->totalFiles = _files;

      QString s;
      s.sprintf("0 / %d", _files);
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( s, index, TB_COUNT );
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotTotalFiles for KIOJob : %d", id);
}


void KMainWidget::slotProcessedFiles( int id, unsigned long _files ) {
  KItem *item;

  for ( item =GlobalItemList.first(); item != 0; item=GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );
      item->processedFiles = _files;

      QString s;
      s.sprintf("%d / %d", _files, item->totalFiles );
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( s, index, TB_COUNT );
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotProcessedFiles for KIOJob : %d", id);
}


void KMainWidget::slotDropEvent( KDNDDropZone * _dropZone ) {
  QString s;

  QStrList & list = _dropZone->getURLList();

  for ( s = list.first(); s != 0L; s = list.next() ) {
    addTransfer( s );
  }
  myTabListBox->unmarkAll();
}


void KMainWidget::slotHighlightedEvent( int Index ) {
  updateToolBar();
}


void KMainWidget::slotMidClickEvent( int Index ) {
  myTabListBox->unmarkItem( Index );
  updateToolBar();
}


void KMainWidget::slotPopupMenu( int Index ) {

  myTabListBox->setCurrentItem( Index );
  myTabListBox->unmarkAll();
  myTabListBox->markItem( Index );

  KItem *it = GlobalItemList.at(Index);

  switch ( it->status ){
  case KItem::ST_TRYING :
  case KItem::ST_RUNNING :
    popup->setItemEnabled( pop_pause, TRUE );
    popup->setItemEnabled( pop_resume, FALSE );
    break;
  case KItem::ST_STOPPED :
    popup->setItemEnabled( pop_pause, FALSE );
    popup->setItemEnabled( pop_resume, TRUE );
    break;
  }

  popup->setItemChecked( pop_queued, FALSE );
  popup->setItemChecked( pop_scheduled, FALSE );
  popup->setItemChecked( pop_delayed, FALSE );
  switch ( it->mode ){
  case KItem::MD_QUEUED :
    popup->setItemChecked( pop_queued, TRUE );
    break;
  case KItem::MD_SCHEDULED :
    popup->setItemChecked( pop_scheduled, TRUE );
    break;
  case KItem::MD_DELAYED :
    popup->setItemChecked( pop_delayed, TRUE );
    break;
  }

  if ( it == GlobalItemList.last() )
    popup->setItemEnabled( pop_end, FALSE );
  else
    popup->setItemEnabled( pop_end, TRUE );

  if ( it == GlobalItemList.first() )
    popup->setItemEnabled( pop_begin, FALSE );
  else
    popup->setItemEnabled( pop_begin, TRUE );
    
  popup->popup(QCursor::pos());
}


void KMainWidget::slotCopyToClipboard()
{
    int index = myTabListBox->currentItem();

    if ( index < 0 )
	return;

    QClipboard *cb = QApplication::clipboard();
    cb->setText( GlobalItemList.at(index)->src.data() );
    updateToolBar();
}


void KMainWidget::slotMoveToBegin() {
  int index = myTabListBox->currentItem();

  if ( index <= 0 )
    return;

  KItem *item = GlobalItemList.take( index );
  GlobalItemList.insert( 0, item );

  QString str = myTabListBox->text( index );
  myTabListBox->removeItem( index );

  myTabListBox->insertItem( str, 0 );
  myTabListBox->unmarkAll();
  updateToolBar();
}


void KMainWidget::slotMoveToEnd() {
  int index = myTabListBox->currentItem();

  if ( index < 0 )
    return;

  KItem *item = GlobalItemList.take( index );
  GlobalItemList.append( item );

  QString str = myTabListBox->text( index );
  myTabListBox->removeItem( index );

  myTabListBox->insertItem( str );
  myTabListBox->unmarkAll();
  updateToolBar();
}


void KMainWidget::configCurrent() {
  int index = myTabListBox->currentItem();

  if ( index < 0 )
    return;

  confdlg = new ConfigDlg( this, GlobalItemList.at( index ) );

  confdlg->show();
}


void KMainWidget::openDownloadWindow() {
  int index = myTabListBox->currentItem();

  if ( index < 0 )
    return;

  KItem *item = GlobalItemList.at( index );
  item->job->showGUI();
}


void KMainWidget::closeEvent( QCloseEvent * ){
  quit();
}


void KMainWidget::slotShowStatusbar() {

  showStatusbar = !showStatusbar;
  options->setItemChecked( op_statusbar, showStatusbar );
  if ( !showStatusbar )
    myStatusBar->enable( KStatusBar::Hide );
  else
    myStatusBar->enable( KStatusBar::Show );
  resizeEvent( 0L );
}


void KMainWidget::slotAnimateStatus() {
  b_useAnimation = !b_useAnimation;
  options->setItemChecked( op_useanimation, b_useAnimation );

  if ( !b_useAnimation && animTimer->isActive() ) {
    animTimer->stop();
    animTimer->start( 1000 );
  }
  else {
    animTimer->stop();
    animTimer->start( 400 );
  }

  dock_widget->setAnim( b_useAnimation );
}


void KMainWidget::slotSoundStatus() {
  b_useSound = !b_useSound;
  options->setItemChecked( op_usesound, b_useSound );

  if ( b_useSound ) {
      if ( ! audio )
	  audio = new KAudio;
      if ( audio->serverStatus() )
	  kdebug( KDEBUG_ERROR, 5001,"Failed contacting audio server !");
  }
  else if ( audio ) {
    delete audio;
    audio = 0L;
  }
}


void KMainWidget::toggleExpertMode() {
  b_expertMode = !b_expertMode;

  if ( b_expertMode ) {
    ((KToolBarButton*)myToolBar->getButton(TOOL_EXPERT))->on(true);
    myStatusBar->message( i18n("Expert mode on."), 1000);
  }
  else {
    ((KToolBarButton*)myToolBar->getButton(TOOL_EXPERT))->on(false);
    myStatusBar->message( i18n("Expert mode off."), 1000);
  }
}


void KMainWidget::toggleAutoDisconnect() {
  b_autoDisconnect = !b_autoDisconnect;

  if ( b_autoDisconnect ) {
    ((KToolBarButton*)myToolBar->getButton(TOOL_DISCONNECT))->on(true);
    myStatusBar->message( i18n("Auto disconnect on."), 1000);
  }
  else {
    ((KToolBarButton*)myToolBar->getButton(TOOL_DISCONNECT))->on(false);
    myStatusBar->message( i18n("Auto disconnect off."), 1000);
  }
}


void KMainWidget::toggleAutoShutdown() {
  b_autoShutdown = !b_autoShutdown;

  if ( b_autoShutdown ) {
    ((KToolBarButton*)myToolBar->getButton(TOOL_SHUTDOWN))->on(true);
    myStatusBar->message( i18n("Auto shutdown on."), 1000);
  }
  else {
    ((KToolBarButton*)myToolBar->getButton(TOOL_SHUTDOWN))->on(false);
    myStatusBar->message( i18n("Auto shutdown off."), 1000);
  }
}


void KMainWidget::toggleDocking(){
  b_dockWindow = !b_dockWindow;

  if ( b_dockWindow ) {
    ((KToolBarButton*)myToolBar->getButton(TOOL_DOCK))->on(true);
    dock_widget->dock();
    this->hide();
  }
  else {
    ((KToolBarButton*)myToolBar->getButton(TOOL_DOCK))->on(false);
    dock_widget->undock();
    this->show();
  }
}


void KMainWidget::updateToolBar() {
  int id;

  ((KToolBarButton*)myToolBar->getButton(TOOL_QUEUE))->on(FALSE);
  ((KToolBarButton*)myToolBar->getButton(TOOL_TIMER))->on(FALSE);
  ((KToolBarButton*)myToolBar->getButton(TOOL_DELAY))->on(FALSE);

  int count = myTabListBox->count();
  if ( count == 0)
    goto updcln;

  id = myTabListBox->currentItem();
  if ( id >= count || id < 0)
    goto updcln;

  if ( myTabListBox->isMarked(id) ){
    myToolBar->setItemEnabled( TOOL_DELETE, TRUE);

    KItem *item = GlobalItemList.at(id);
    K2URL url( item->src.data() );
    QString f = url.protocol();

    switch ( item->status ){
    case KItem::ST_RUNNING :
    case KItem::ST_TRYING :
      myToolBar->setItemEnabled( TOOL_PAUSE, TRUE);
      myToolBar->setItemEnabled( TOOL_RESUME, FALSE);
      break;
    case KItem::ST_STOPPED :
      myToolBar->setItemEnabled( TOOL_PAUSE, FALSE);
      myToolBar->setItemEnabled( TOOL_RESUME, TRUE);
      break;
    }

    if ( item->status != KItem::ST_FINISHED ) {
      myToolBar->setItemEnabled( TOOL_QUEUE, TRUE);
      myToolBar->setItemEnabled( TOOL_TIMER, TRUE);
      myToolBar->setItemEnabled( TOOL_DELAY, TRUE);
    }
    else
      goto updcln2;

    switch ( item->mode ){
    case KItem::MD_QUEUED :
      ((KToolBarButton*)myToolBar->getButton(TOOL_QUEUE))->on(TRUE);
      break;
    case KItem::MD_SCHEDULED :
      ((KToolBarButton*)myToolBar->getButton(TOOL_TIMER))->on(TRUE);
      break;
    case KItem::MD_DELAYED :
      ((KToolBarButton*)myToolBar->getButton(TOOL_DELAY))->on(TRUE);
      break;
    }
    return;
  }
  
 updcln:
  myToolBar->setItemEnabled( TOOL_DELETE, FALSE); 

 updcln2:
  myToolBar->setItemEnabled( TOOL_RESUME, FALSE);
  myToolBar->setItemEnabled( TOOL_PAUSE, FALSE);
  myToolBar->setItemEnabled( TOOL_QUEUE, FALSE);
  myToolBar->setItemEnabled( TOOL_TIMER, FALSE);
  myToolBar->setItemEnabled( TOOL_DELAY, FALSE); 
  return;
}


void KMainWidget::updateStatusBar() {
  KItem *item;
  QString tmpstr;

  int totalFiles = 0;
  int totalSize = 0;
  int totalSpeed = 0;
  QTime remTime;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    {
      totalSize += ( item->totalSize - item->processedSize );
      totalFiles += ( item->totalFiles - item->processedFiles );
      totalSpeed += item->speed;

      if ( item->remainingTime > remTime )
	remTime = item->remainingTime;
    }

  ksprintf( &tmpstr, " %s : %d ", i18n( "Transfers"), GlobalItemList.count() );
  myStatusBar->changeItem( tmpstr.data(), ID_TOTAL_TRANSFERS);

  ksprintf( &tmpstr, " %s : %d ", i18n( "Files"), totalFiles );
  myStatusBar->changeItem( tmpstr.data(), ID_TOTAL_FILES);

  ksprintf( &tmpstr, " %s : %s ", i18n( "Size"), convertSize( totalSize ).data() );
  myStatusBar->changeItem( tmpstr.data(), ID_TOTAL_SIZE);

  ksprintf( &tmpstr, " %s : %s ", i18n( "Time"), remTime.toString().data() );
  myStatusBar->changeItem( tmpstr.data(), ID_TOTAL_TIME);

  ksprintf( &tmpstr, " %s/s ", convertSize( totalSpeed ).data() );
  myStatusBar->changeItem( tmpstr.data(), ID_TOTAL_SPEED);
}


int KMainWidget::downloadStatus() {
  KItem *item;

  int status;

  if ( GlobalItemList.count() <= 0 )
    return STATUS_EMPTY;
  else {
    for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
      {
	status = item->status;
	if ( status == KItem::ST_RUNNING || status == KItem::ST_TRYING ||
	     status == KItem::ST_RETRYING )
	  return STATUS_LOADING;
	else if ( item->mode == KItem::MD_SCHEDULED )
	  return STATUS_SCHEDULED;
      }
  }
  return STATUS_DELAYED;
}



void KMainWidget::Connect( KIOJob *job ) {
  job->enableGUI( b_showDownloadWindows );
  job->startIconified( b_startIconified );

//   job->cacheToPool( false );

  connect( job, SIGNAL( sigSpeed( int, unsigned long ) ),
	   this, SLOT( slotSpeed( int, unsigned long ) ) );
  connect( job, SIGNAL( sigProcessedSize( int, unsigned long ) ),
	   this, SLOT( slotProcessedSize( int, unsigned long ) ) );
  connect( job, SIGNAL( sigTotalSize( int, unsigned long ) ),
	   this, SLOT( slotTotalSize( int, unsigned long ) ) );
  connect( job, SIGNAL( sigTotalFiles( int, unsigned long ) ),
	   this, SLOT( slotTotalFiles( int, unsigned long ) ) );
  connect( job, SIGNAL( sigProcessedFiles( int, unsigned long ) ),
	   this, SLOT( slotProcessedFiles( int, unsigned long ) ) );
  connect( job, SIGNAL( sigFinished( int ) ),
	   this, SLOT( slotFinished( int ) ) );
  connect( job, SIGNAL( sigCanceled( int ) ),
	   this, SLOT( slotCanceled( int ) ) );
  connect( job, SIGNAL( sigError( int, int, const char* ) ),
	   this, SLOT( slotError( int, int, const char* ) ) );
  connect( job, SIGNAL( sigRenamed( int, const char* ) ),
	   this, SLOT( slotRenamed( int, const char* ) ) );
  connect( job, SIGNAL( sigCopying( int, const char*, const char* ) ),
	   this, SLOT( slotCopying( int, const char*, const char* ) ) );
  connect( job, SIGNAL( sigCanResume( int, bool ) ),
	   this, SLOT( slotCanResume( int, bool ) ) );
}


void KMainWidget::disconnect() {
  system( disconnectCommand.data() );
}


bool KMainWidget::checkOnline() {
  return true;
}


QString KMainWidget::convertSize( int size ) {
  float fsize;
  QString s;
  if ( size > 1048576 ){
    fsize = (float) size / (float) 1048576;
    s.sprintf ( "%.1f MB", fsize);
  } else if ( size > 1024 ){
    fsize = (float) size / (float) 1024;
    s.sprintf ( "%.1f kB", fsize);
  } else {
    s.sprintf ( "%d B", size);
  }
  return s;
}


//
//  Helper functions
//

QString getStringFromBool( bool x ) {
  if ( x )
    return "true";
  else
    return "false";
}


QString getStringFromInt( int x ) {
  QString s;
  s.setNum( x );
  return s;
}


bool getBoolFromString( QString s ) {
  if ( s == "true" )
    return true;
  else
    return false;
}


int getIntFromString( QString s ) {
  return s.toUInt();
}


////////////
void openFileManagerWindow( const char * )
{
  assert( 0 );
}
