/***************************************************************************
                          kash.cpp  -  description
                             -------------------
    begin                : Son Nov 21 15:23:42 CET 1999
    copyright            : (C) 1999 by Stephan Kahnt
    email                : stephan.kahnt@ipk.fhg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// standard include
#include <iostream>

// include files for QT
#include <qdir.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qfileinfo.h>

// include files for KDE
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kconfig.h>
#include <ksimpleconfig.h>
#include <kstddirs.h>

// application specific includes
#include "kash.h"
#include "kashview.h"
#include "kashdoc.h"
#include "resource.h"


KashApp::KashApp() {
  config=kapp->config();

  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initMenuBar();
  initToolBar();
  initStatusBar();
  initKeyAccel();
  initDocument();
  initView();
	
  readOptions();

  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar items at startup
  disableCommand(ID_FILE_SAVE);
  disableCommand(ID_FILE_SAVE_AS);
  disableCommand(ID_FILE_PRINT);
 	
  disableCommand(ID_EDIT_CUT);
  disableCommand(ID_EDIT_COPY);
  disableCommand(ID_EDIT_PASTE);
}

KashApp::~KashApp()
{

}

void KashApp::initKeyAccel()
{
  keyAccel = new KAccel(this);
	
  // fileMenu accelerators
  keyAccel->connectItem(KAccel::New, this, SLOT(slotFileNew()));
  keyAccel->connectItem(KAccel::Open, this, SLOT(slotFileOpen()));
  keyAccel->connectItem(KAccel::Save, this, SLOT(slotFileSave()));
  keyAccel->connectItem(KAccel::Close, this, SLOT(slotFileClose()));
  keyAccel->connectItem(KAccel::Print, this, SLOT(slotFilePrint()));
  keyAccel->connectItem(KAccel::Quit, this, SLOT(slotFileQuit()));
  // editMenu accelerators
  keyAccel->connectItem(KAccel::Cut, this, SLOT(slotEditCut()));
  keyAccel->connectItem(KAccel::Copy, this, SLOT(slotEditCopy()));
  keyAccel->connectItem(KAccel::Paste, this, SLOT(slotEditPaste()));

  keyAccel->connectItem(KAccel::Help, kapp, SLOT(appHelpActivated()));
			
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_NEW, KAccel::New);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_OPEN, KAccel::Open);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_SAVE, KAccel::Save);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_CLOSE, KAccel::Close);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_PRINT, KAccel::Print);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_QUIT, KAccel::Quit);

  keyAccel->changeMenuAccel(editMenu, ID_EDIT_CUT, KAccel::Cut);
  keyAccel->changeMenuAccel(editMenu, ID_EDIT_COPY, KAccel::Copy);
  keyAccel->changeMenuAccel(editMenu, ID_EDIT_PASTE, KAccel::Paste);

  keyAccel->readSettings();	
}

void KashApp::initMenuBar()
{
  ///////////////////////////////////////////////////////////////////
  // MENUBAR
  recentFilesMenu = new QPopupMenu();
  connect( recentFilesMenu, SIGNAL( activated( int ) ), 
	   SLOT( slotFileOpenRecent( int ) ) );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry fileMenu
  fileMenu = new QPopupMenu();
  fileMenu->insertItem(kapp->miniIcon(), i18n("New &Window"), ID_FILE_NEW_WINDOW);
  fileMenu->insertSeparator();
  fileMenu->insertItem(BarIcon("filenew"), i18n("&New"), ID_FILE_NEW);
  fileMenu->insertItem(BarIcon("fileopen"), i18n("&Open..."), ID_FILE_OPEN);
  fileMenu->insertItem(i18n("Open &recent"), recentFilesMenu, ID_FILE_OPEN_RECENT);

  fileMenu->insertItem(i18n("&Close"), ID_FILE_CLOSE);
  fileMenu->insertSeparator();
  fileMenu->insertItem(BarIcon("filefloppy") ,i18n("&Save"), ID_FILE_SAVE);
  fileMenu->insertItem(i18n("Save &As..."), ID_FILE_SAVE_AS);
  fileMenu->insertSeparator();
  fileMenu->insertItem(BarIcon("fileprint"), i18n("&Print..."), ID_FILE_PRINT);
  fileMenu->insertSeparator();
  fileMenu->insertItem(i18n("E&xit"), ID_FILE_QUIT);
	
  ///////////////////////////////////////////////////////////////////
  // menuBar entry editMenu
  editMenu = new QPopupMenu();
  editMenu->insertItem(BarIcon("editcut"), i18n("Cu&t"), ID_EDIT_CUT);
  editMenu->insertItem(BarIcon("editcopy"), i18n("&Copy"), ID_EDIT_COPY);
  editMenu->insertItem(BarIcon("editpaste"), i18n("&Paste"), ID_EDIT_PASTE);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry viewMenu
  viewMenu = new QPopupMenu();
  viewMenu->setCheckable(true);
  viewMenu->insertItem(i18n("&Toolbar"), ID_VIEW_TOOLBAR);
  viewMenu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry helpMenu
  QString aboutstring = "kash: the K Homebanking software\n\n"
    "Version 0.1\n"
    "Author: Stephan Kahnt (stephan.kahnt@ipk.fhg.de)\n";
    //kapp->name();

  helpMenu_ = helpMenu( aboutstring );

  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // insert your popup menus with the according menubar entries in the order
  // they will appear later from left to right
  menuBar()->insertItem(i18n("&File"), fileMenu);
  menuBar()->insertItem(i18n("&Edit"), editMenu);
  menuBar()->insertItem(i18n("&View"), viewMenu);

  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), helpMenu_);

  ///////////////////////////////////////////////////////////////////
  // CONNECT THE MENU SLOTS WITH SIGNALS
  // for execution slots and statusbar messages

  connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(fileMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(editMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(editMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(viewMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(viewMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

}

void KashApp::initToolBar() {

  ///////////////////////////////////////////////////////////////////
  // TOOLBAR
  toolBar()->insertButton( BarIcon( "filenew" ), ID_FILE_NEW, true, i18n( "New File" ) );
  toolBar()->insertButton( BarIcon( "fileopen" ), ID_FILE_OPEN, true, i18n( "Open File" ) );
  toolBar()->insertButton( BarIcon( "filefloppy" ), ID_FILE_SAVE, true, i18n( "Save File" ) );
  toolBar()->insertButton( BarIcon( "fileprint" ), ID_FILE_PRINT, true, i18n( "Print" ) );
  toolBar()->insertSeparator();
  toolBar()->insertButton( BarIcon( "editcut" ), ID_EDIT_CUT, true, i18n( "Cut" ) );
  toolBar()->insertButton( BarIcon( "editcopy" ), ID_EDIT_COPY, true, i18n( "Copy" ) );
  toolBar()->insertButton( BarIcon( "editpaste" ), ID_EDIT_PASTE, true, i18n( "Paste" ) );
  toolBar()->insertSeparator();
  toolBar()->insertButton( BarIcon( "help" ), ID_HELP_CONTENTS, SIGNAL( clicked() ),
			   kapp, SLOT( appHelpActivated() ), true,i18n( "Help" ) );

  ///////////////////////////////////////////////////////////////////
  // INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE WITH toolBar(n)


  ///////////////////////////////////////////////////////////////////
  // CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars by their according number
  // connect for invoking the slot actions
  connect( toolBar(), SIGNAL( clicked( int ) ), SLOT( commandCallback( int ) ) );
  // connect for the status help on holing icons pressed with the mouse button
  connect( toolBar(), SIGNAL( pressed( int ) ), SLOT( statusCallback( int ) ) );
}

void KashApp::initStatusBar() {
  ///////////////////////////////////////////////////////////////////
  // STATUSBAR
  // TODO: add your own items you need for displaying current application status.
  statusBar()->insertItem( i18n( IDS_STATUS_DEFAULT ), ID_STATUS_MSG );
}

void KashApp::initDocument() {
  doc = new KashDoc( this );
  doc->newDocument();
}

void KashApp::initView() { 
  ////////////////////////////////////////////////////////////////////
  // create the main widget here that is managed by KTMainWindow's view-region and
  // connect the widget to your document to display document contents.

  view = new KashView( this );
  doc->addView( view );
  setView( view );	
  setCaption( doc->getTitle() );
}

void KashApp::enableCommand( int id_ ) {
  ///////////////////////////////////////////////////////////////////
  // enable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled( id_, true );
  toolBar()->setItemEnabled( id_, true );
}

void KashApp::disableCommand( int id_ ) {
  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled( id_, false );
  toolBar()->setItemEnabled( id_, false );
}

void KashApp::addRecentFile( const QString& file ) {
  if( recentFiles.find( file ) == -1 ) {
    if( recentFiles.count() < 5 ) {
      recentFiles.insert( 0, file );
    }
    else {
      recentFiles.remove( 4 );
      recentFiles.insert( 0, file );
    }
    recentFilesMenu->clear();
    for ( int i = 0; i < ( int )recentFiles.count(); i++ ) {
      recentFilesMenu->insertItem( recentFiles.at( i ), i );
    }
  }
}

void KashApp::openDocumentFile( const char* _cmdl ) {
  slotStatusMsg( i18n( "Opening file..." ) );

  doc->openDocument( _cmdl );
  slotStatusMsg( i18n( IDS_STATUS_DEFAULT ) );
}


KashDoc *KashApp::getDocument() const {
  return doc;
}

void KashApp::saveOptions() {	
  config->setGroup( "General Options" );
  config->writeEntry( "Geometry", size() );
  config->writeEntry( "Show Toolbar", toolBar()->isVisible() );
  config->writeEntry( "Show Statusbar",statusBar()->isVisible() );
  config->writeEntry( "MenuBarPos", ( int )menuBar()->menuBarPos() );
  config->writeEntry( "ToolBarPos", ( int )toolBar()->barPos() );
  config->writeEntry( "Recent Files", recentFiles );
}


void KashApp::readOptions() {
  config->setGroup( "General Options" );

  // bar status settings
  bool bViewToolbar = config->readBoolEntry( "Show Toolbar", true );
  viewMenu->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
  if(!bViewToolbar)
  {
     enableToolBar(KToolBar::Hide);
  }
	
  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  viewMenu->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
  if(!bViewStatusbar)
  {
    enableStatusBar(KStatusBar::Hide);
  }

  // bar position settings
  KMenuBar::menuPosition menuBarPos;
  menuBarPos=(KMenuBar::menuPosition) config->readNumEntry("MenuBarPos", KMenuBar::Top);

  KToolBar::BarPosition toolBarPos;
  toolBarPos=(KToolBar::BarPosition) config->readNumEntry("ToolBarPos", KToolBar::Top);

  menuBar()->setMenuBarPos(menuBarPos);
  toolBar()->setBarPos(toolBarPos);
	
  // initialize the recent file list
  recentFiles.setAutoDelete(TRUE);
  config->readListEntry("Recent Files", recentFiles);
	
  for ( int i = 0; i < ( int )recentFiles.count(); i++ ) {
    recentFilesMenu->insertItem( recentFiles.at( i ), i );
  }

  QSize size = config->readSizeEntry( "Geometry" );
  if( !size.isEmpty() ) {
    resize( size );
  }

}

void KashApp::saveProperties(KConfig *_cfg) {
  if( doc->getTitle() != i18n( UNTITLED_STRING ) && !doc->isModified() ) {
    // saving to tempfile not necessary
  }
  else {
    QString filename=doc->getAbsFilePath();	
    _cfg->writeEntry( "filename", filename );
    _cfg->writeEntry( "modified", doc->isModified() );
		
    QString tempname = kapp->tempSaveName( filename );
    doc->saveDocument( tempname );
  }
}

void KashApp::readProperties( KConfig* _cfg ) {
  QString filename = _cfg->readEntry("filename", "");
  bool modified = _cfg->readBoolEntry("modified", false);
  if(modified)
  {
    bool canRecover;
    QString tempname = kapp->checkRecoverFile(filename, canRecover);
  	
    if(canRecover) {
      doc->openDocument(tempname);
      doc->setModified();
      QFileInfo info(filename);
      doc->setAbsFilePath(info.absFilePath());
      doc->setTitle(info.fileName());
      QFile::remove(tempname);
    }
  }
  else {
    if(!filename.isEmpty()) {
      doc->openDocument(filename);
    }
  }

  QString caption=kapp->caption();	
  setCaption( caption + ": " + doc->getTitle() );
}		

bool KashApp::queryClose() {
  return doc->saveModified();
}

bool KashApp::queryExit()
{
  saveOptions();
  return true;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void KashApp::slotFileNewWindow()
{
  slotStatusMsg(i18n("Opening a new application window..."));
	
  KashApp *new_window= new KashApp();
  new_window->show();

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotFileNew() {
  slotStatusMsg( i18n( "Creating new document...") );
  
  if( !doc->saveModified() ) {
    // here saving wasn't successful

  }
  else {	
    doc->newDocument();		
    
    QString caption=kapp->caption();	
    setCaption( caption + ": " + doc->getTitle() );
  }
  
  slotStatusMsg( i18n( IDS_STATUS_DEFAULT ) );
}

void KashApp::slotFileOpen() {
  slotStatusMsg( i18n( "Opening file..." ) );
	
  if( !doc->saveModified() ) {
    // here saving wasn't successful
  }
  else {
    QString fileToOpen;
    QFileInfo openInfo;
    int wantResel = KMessageBox::No;
    do {
      fileToOpen = KFileDialog::getOpenFileName( QDir::homeDirPath(),
						 i18n( "*|All files" ), 
						 this, i18n( "Open File..." ) );
      if ( fileToOpen.isEmpty() ) {
	break;
      }
      openInfo.setFile( fileToOpen );
      if ( !openInfo.exists() || !openInfo.isFile() || !openInfo.isReadable() ) {
	wantResel = 
	  KMessageBox::warningYesNo( this, 
				     i18n( "The selected file doesn't exist,\n"
					   "it isn't a file or it isn't readable.\n"
					   "Do you want to reselect?"),
				     i18n( "Warning" ) );
	fileToOpen = "";
      }
    } while( wantResel == KMessageBox::Yes );
    if( !fileToOpen.isEmpty() ) {
      doc->openDocument( fileToOpen );
      QString caption=kapp->caption();
      setCaption( caption + ": " + doc->getTitle() );
      addRecentFile( fileToOpen );
    }
  }
  
  slotStatusMsg( i18n( IDS_STATUS_DEFAULT ) );
}

void KashApp::slotFileOpenRecent( int id_ ) {
  slotStatusMsg( i18n( "Opening file..." ) );
  
  if( !doc->saveModified() ) {
    // here saving wasn't successful
  }
  else {
    doc->openDocument( recentFiles.at( id_ ) );
    QString caption=kapp->caption();	
    setCaption( caption + ": " + doc->getTitle() );
  }
  
  slotStatusMsg( i18n( IDS_STATUS_DEFAULT ) );
}

void KashApp::slotFileSave() {
  slotStatusMsg( i18n( "Saving file..." ) );

  if ( doc->getTitle() == i18n( UNTITLED_STRING ) ) {
    slotFileSaveAs();
  }
  else {
    doc->saveDocument( doc->getAbsFilePath() );
  }

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotFileSaveAs() {
  slotStatusMsg( i18n( "Saving file with a new filename..." ) );

  bool exists;
  int wantOver = KMessageBox::Yes;
  QFileInfo saveAsInfo;
  QString newName;
  do {
    newName = KFileDialog::getSaveFileName( QDir::currentDirPath(),
					    i18n( "*|All files " ), this, 
					    i18n( "Save as..." ) );
    if( newName.isEmpty() ) {
      wantOver = KMessageBox::Cancel;
      break;
    }
    saveAsInfo.setFile( newName );
    if ( ( exists = saveAsInfo.exists() ) ) {
      wantOver = 
	KMessageBox::warningYesNoCancel( this, 
					 i18n( "A file with the selected "
					       "name already exists.\n"
					       "Do you want to overwrite it?"),
					 i18n( "Warning" ) );
    }
  } while ( wantOver == KMessageBox::No );
  
  if ( wantOver == KMessageBox::Yes ) {
    doc->setTitle( saveAsInfo.fileName() );
    doc->setAbsFilePath( saveAsInfo.absFilePath() );
    doc->saveDocument( newName );
    addRecentFile( newName );
  
    QString caption=kapp->caption();	
    setCaption( caption + ": " + doc->getTitle() );
  }
  
  slotStatusMsg( i18n( IDS_STATUS_DEFAULT ) );
}

void KashApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing file..."));
	
  close();
  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotFilePrint()
{
  slotStatusMsg(i18n("Printing..."));

  QPrinter printer;
  if (printer.setup(this))
  {
    view->print(&printer);
  }

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  saveOptions();
  // close the first window, the list makes the next one the first again.
  // This ensures that queryClose() is called on each window to ask for closing
  KTMainWindow* w;
  if(memberList)
  {
    for(w=memberList->first(); w!=0; w=memberList->first())
    {
      // only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
      // the window and the application stay open.
      if(!w->close())
	break;
    }
  }	
  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotEditCut()
{
  slotStatusMsg(i18n("Cutting selection..."));

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotEditCopy()
{
  slotStatusMsg(i18n("Copying selection to clipboard..."));

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotEditPaste()
{
  slotStatusMsg(i18n("Inserting clipboard contents..."));

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotViewToolBar()
{
  slotStatusMsg(i18n("Toggle the toolbar..."));
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
  if( viewMenu->isItemChecked(ID_VIEW_TOOLBAR))
  {
    viewMenu->setItemChecked(ID_VIEW_TOOLBAR, false);
    enableToolBar(KToolBar::Hide);
  }
  else
  {
    viewMenu->setItemChecked(ID_VIEW_TOOLBAR, true);
    enableToolBar(KToolBar::Show);
  }		

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KashApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  if( viewMenu->isItemChecked(ID_VIEW_STATUSBAR))
  {
    viewMenu->setItemChecked(ID_VIEW_STATUSBAR, false);
    enableStatusBar(KStatusBar::Hide);
  }
  else
  {
    viewMenu->setItemChecked(ID_VIEW_STATUSBAR, true);
    enableStatusBar(KStatusBar::Show);
  }

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}


void KashApp::slotStatusMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KashApp::slotStatusHelpMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message of whole statusbar temporary (text, msec)
  statusBar()->message(text, 2000);
}



void KashApp::commandCallback(int id_)
{
  switch (id_)
  {
    case ID_FILE_NEW_WINDOW:
         slotFileNewWindow();
    	 break;

    case ID_FILE_NEW:
    	 slotFileNew();
         break;

    case ID_FILE_OPEN:
         slotFileOpen();
         break;

    case ID_FILE_SAVE:
         slotFileSave();
         break;

    case ID_FILE_SAVE_AS:
         slotFileSaveAs();
         break;

    case ID_FILE_CLOSE:
         slotFileClose();
         break;

    case ID_FILE_PRINT:
         slotFilePrint();
         break;

    case ID_FILE_QUIT:
         slotFileQuit();
         break;

    case ID_EDIT_CUT:
         slotEditCut();
         break;

    case ID_EDIT_COPY:
         slotEditCopy();
         break;

    case ID_EDIT_PASTE:
         slotEditPaste();
         break;
  
    case ID_VIEW_TOOLBAR:
         slotViewToolBar();
         break;

    case ID_VIEW_STATUSBAR:
         slotViewStatusBar();
         break;

    default:
         break;
  }
}

void KashApp::statusCallback(int id_)
{
  switch (id_)
  {
    case ID_FILE_NEW_WINDOW:
         slotStatusHelpMsg(i18n("Opens a new application window"));
         break;

    case ID_FILE_NEW:
         slotStatusHelpMsg(i18n("Creates a new document"));
         break;

    case ID_FILE_OPEN:
         slotStatusHelpMsg(i18n("Opens an existing document"));
         break;

    case ID_FILE_OPEN_RECENT:
         slotStatusHelpMsg(i18n("Opens a recently used file"));
         break;

    case ID_FILE_SAVE:
         slotStatusHelpMsg(i18n("Saves the actual document"));
         break;

    case ID_FILE_SAVE_AS:
         slotStatusHelpMsg(i18n("Saves the actual document as..."));
         break;

    case ID_FILE_CLOSE:
         slotStatusHelpMsg(i18n("Closes the actual document"));
         break;

    case ID_FILE_PRINT:
         slotStatusHelpMsg(i18n("Prints out the actual document"));
         break;

    case ID_FILE_QUIT:
         slotStatusHelpMsg(i18n("Quits the application"));
         break;

    case ID_EDIT_CUT:
         slotStatusHelpMsg(i18n("Cuts the selected section and puts it to the clipboard"));
         break;

    case ID_EDIT_COPY:
         slotStatusHelpMsg(i18n("Copies the selected section to the clipboard"));
         break;

    case ID_EDIT_PASTE:
         slotStatusHelpMsg(i18n("Pastes the clipboard contents to actual position"));
         break;

    case ID_VIEW_TOOLBAR:
         slotStatusHelpMsg(i18n("Enables/disables the toolbar"));
         break;

    case ID_VIEW_STATUSBAR:
         slotStatusHelpMsg(i18n("Enables/disables the statusbar"));
         break;

    default:
         break;
  }
}
