/**************************************************************************
 * $Id: SamplinPlotData.cpp 1.2 Thu, 18 Feb 1999 16:48:37 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include <limits.h>
#include <unistd.h>
#include <stdio.h>

#include <qwt_math.h>

#include "SamplinPlotData.h"

#define GRAPH_DATA_ID "SampLin_d_1.2"
#define GRAPH_DATA_ID3 "SampLin_d_1.3"

static char *axisNames[]={"Left","Right","Bottom","Top",NULL
};
static char *legendPositionNames[]={"None","Bottom","Right",NULL
};



SamplinPlotData::SamplinPlotData()
{
   init();
//   setSaved();
}

SamplinPlotData::SamplinPlotData( const char *lbl)
{
   d_label = lbl;
   init();
}


SamplinPlotData::~SamplinPlotData()
{
   d_curveList.clear();
   d_curveDict.clear();
   
   for (int i=0;i<axisCnt;i++)
     delete d_scale[i];
}


void SamplinPlotData::init(void)
{
   d_fileid=GRAPH_DATA_ID;
   d_curveList.setAutoDelete( TRUE );
   d_curveDict.setAutoDelete( FALSE );   

   d_grid.enableXMin(false);
   d_grid.enableYMin(false);
   d_grid.setPen(QPen(black,0,DotLine));
   d_grid.setXAxis(xBottom);
   d_grid.setYAxis(yLeft);
   
   d_scale[yLeft] = new SamplinScaleBox(SamplinScaleBox::Left);
   d_scale[yRight] = new SamplinScaleBox(SamplinScaleBox::Right);
   d_scale[xTop] = new SamplinScaleBox(SamplinScaleBox::Top);
   d_scale[xBottom] = new SamplinScaleBox(SamplinScaleBox::Bottom);

   d_lpos=LegendPosition(None);
   d_cpos=LegendPosition(None);
   d_lColor=QColor(black);
   d_cColor=QColor(black);
   d_lFont=QFont("Helvetica",8);
   d_cFont=QFont("Helvetica",8);
   
}

int SamplinPlotData::axis(const char *str)
{
   char **s=axisNames;
   int i=0;
   
   while(*s!=0){
      if(!strcasecmp(*s,str)&&strlen(*s)==strlen(str))return i;
      ++s;++i;
   }
    
   return -1;
}

/*
int SamplinPlotData::verifyXAxis(int axis)
{
   if ((axis == xBottom) || (axis == xTop))
             return axis;
       else
            return xBottom;
}

int SamplinPlotData::verifyYAxis(int axis)
{
   if ((axis == yLeft) || (axis == yRight))
             return axis;
       else
            return yLeft;
}

bool SamplinPlotData::axisValid(int axis) const
{
       return ((axis >= yLeft) && (axis < axisCnt));
}
*/

SamplinCurve::CurveStyle SamplinPlotData::curveStyle(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->style();
    }
    else
       return (SamplinCurve::CurveStyle) 0;
}

const QwtSymbol & SamplinPlotData::curveSymbol(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->symbol();
    }
    else
       return dummySymbol;
}

const QPen& SamplinPlotData::curvePen(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->pen();
    }
    else
       return dummyPen;
    
}

int SamplinPlotData::curveSplineType(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->splineType();
    }
    else
       return 0;

}

int SamplinPlotData::curveSplineSize(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->splineSize();
    }
    else
       return 0;
}

QString SamplinPlotData::curveName(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->name();
    }
    else
       return QString("");
}

uint SamplinPlotData::curveKey(const char *name)
{
   const char *cname;
   SamplinCurve *crv;
   
   crv=d_curveList.first();
   while(crv!=0)
     {
	cname=crv->name();
	if((strlen(name)==strlen(cname)) && !strcmp(name,cname))
	  return(crv->key());
	   
	crv=d_curveList.next();
     }
   return(0);
}

QArray<uint> SamplinPlotData::curveKeys()
{
   uint i;
   QArray<uint> rv(d_curveList.count());
   SamplinCurve *crv;
   
   crv=d_curveList.first();
   i = 0;
   while (crv && (i < rv.size()))
     {
	rv[i] = crv->key();
	crv=d_curveList.next();
	++i;
     }	
   return rv;
}

int SamplinPlotData::curveXAxis(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->xAxis();
    }
    else
       return -1;    
}

int SamplinPlotData::curveYAxis(uint key)
{
    SamplinCurve *c;

    if ((c = findCurve(key)))
    {
	return c->yAxis();
    }
    else
       return -1;    
}

uint SamplinPlotData::newCurveKey()
{
   uint newkey = d_curveList.count() + 1;

   if (newkey > 1)			// size > 0
     {
	if (findCurve(newkey))	// key size+1 exists => there must be a
	  // free key <= size
	  {
	     // find the first available key <= size
	    newkey = 1;
	     while (newkey <= d_curveList.count())
	       {
		  if (findCurve(newkey))
		    newkey++;
		  else
		    break;
	       }
	     
	     if (newkey > d_curveList.count())
	       {
		  while (!findCurve(newkey))
		    {
		       newkey++;
		       if (newkey > 10000)
			 {
			    newkey = 0;
			    break;
			 }
		    }
	       }
	  }
     }
   return newkey;
}

bool SamplinPlotData::insertCurve(uint key, const char *name, int xAxis, int yAxis)
{
   int rv = false;
   if (findCurve(key) == 0)
     {
	SamplinCurve *nc = new SamplinCurve();
	if (nc) 
	  {
	     nc->setxAxis(verifyXAxis(xAxis));
	     nc->setyAxis(verifyYAxis(yAxis));
	     nc->setName(name);
	     nc->setKey(key);
	     nc->setBackgroundColor(d_plotColor);
	     d_curveList.insert(d_curveList.count(),nc);
	     d_curveDict.insert(key,nc);
	     rv = true;
	  }
     }
   autoRefresh();
   return rv;
}

uint SamplinPlotData::insertCurve(const char *name, int xAxis, int yAxis)
{
    uint newkey = newCurveKey();
    if (newkey > 0)
	if (!insertCurve(newkey, name, xAxis, yAxis))
	   newkey = 0;
    return newkey;
}

uint SamplinPlotData::insertCurve(SamplinCurve *curve)
{
   uint newkey = newCurveKey();
   if (newkey > 0 && curve){
      curve->setKey(newkey);
      curve->setBackgroundColor(d_plotColor);
      d_curveList.insert(d_curveList.count(),curve);
      d_curveDict.insert(newkey,curve);
      autoRefresh();
   }
   else newkey=0;
   
   return newkey;
}

bool SamplinPlotData::removeCurve(uint key)
{
   SamplinCurve *crv;
   
   crv=findCurve(key);
   
   if(crv!=0)
     if (d_curveList.remove(crv))
     {
	d_curveDict.remove(key);
	autoRefresh();
	return true;
     }
   
   return false;
}

void SamplinPlotData::removeCurves()
{
   d_curveList.clear();
   d_curveDict.clear();
   autoRefresh();
}

bool SamplinPlotData::setCurvePen(uint key, const QPen &pen)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setPen(pen);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}

bool SamplinPlotData::setCurveSymbol(uint key, const QwtSymbol &s)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setSymbol(s);
	autoRefresh();
       rv = true;
    }
    return rv;
    
    
}

bool SamplinPlotData::setCurveRawData(uint key, double *xdat, double *ydat, int size)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setRawData(xdat, ydat, size);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}

bool SamplinPlotData::setCurveRawData(uint key, double begin, double step, double *ydat, int size)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setRawData(begin, step, ydat, size);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}

bool SamplinPlotData::setCurveRawData(uint key, double *xdat, double begin, double step, int size)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setRawData(xdat, begin, step, size);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}

bool SamplinPlotData::setCurveData(uint key, double *xdat, double *ydat, int size)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setData(xdat, ydat, size);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}
bool SamplinPlotData::setCurveData(uint key, double begin, double step, double *ydat, int size)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setData(begin, step, ydat, size);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}

bool SamplinPlotData::setCurveData(uint key, double *xdat, double begin, double step, int size)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setData(xdat, begin, step, size);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}

void SamplinPlotData::mapCurve(uint key, int axis)
{
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	switch(axis)
	{
	case xTop:
	case xBottom:
	    c->setxAxis(axis);
	    break;
	case yLeft:
	case yRight:
	    c->setyAxis(axis);
	    break;
	}
       autoRefresh();
    }
    
}

bool SamplinPlotData::setCurveStyle(uint key, SamplinCurve::CurveStyle s)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setStyle(s);
	autoRefresh();
       rv = true;
    }
  
    return rv;
  
}

bool SamplinPlotData::setCurveSplineType(uint key, int t)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setSplineType(t);
	autoRefresh();
       rv = true;
    }
    return rv;
}

bool SamplinPlotData::setCurveSplineSize(uint key, int s)
{
    bool rv = false;
    
    SamplinCurve *c;
    if ((c = findCurve(key)))
    {
	c->setSplineType(s);
	autoRefresh();
       rv = true;
    }
    return rv;
    
}




QDataStream &operator<<( QDataStream &s, SamplinPlotData &series )
{
   series.sendData(s);
   return(s);
}


QDataStream &operator>>( QDataStream &s, SamplinPlotData &series )
{
   series.receiveData(s);
   return(s);
}

void SamplinPlotData::receiveData(QDataStream &s){
   SamplinCurve *tmp;
   int i;
   uint count;
   QString str;
   char b;
   
   removeCurves();
   s >> str;
   setFileName(str);

   s >> d_title;
   s >> d_titleFont;
   s >> d_titleColor;
   
   s >> d_plotColor;
   s >> d_panelColor;
   
   s >> d_grid;

   for(i=0;i<axisCnt;++i){
      s >> d_axisEnabled[i];
      s >> *d_scale[i];   
   }
  
   
   s >> i;
   if(i==11){
      s >> d_printfit;
      s >> d_printwidth;
      s >> d_printheight;
   }
   s >> i;
   if(i==12){
      s >> d_lpos;
      s >> d_lColor;
      s >> d_lFont;
      s >> d_cpos;
      s >> d_cColor;
      s >> d_cFont;
      s >> d_comment;
   }
   else{
      d_lpos=0;
      d_cpos=0;
      d_comment="";
   }
   s >> i;
   s >> i;
   s >> i;
   
   s >> count;
   while(count>0){
      tmp = new SamplinCurve();
      s >> *tmp;
      insertCurve(tmp);
      --count;
   }
}
  
void SamplinPlotData::sendData(QDataStream &s){
   int i;
   
   s << fileName();

   s << d_title;
   s << d_titleFont;
   s << d_titleColor;
   
   s << d_plotColor;
   s << d_panelColor;
   
   s << d_grid;
   
/*
 for(i=0;i<axisCnt;++i){
      s << d_axisEnabled[i];
      s << d_axisTitle[i];
      s << d_axisTitleFont[i];
      s << d_axisTitleColor[i];   
      s << d_axisFont[i];
      s << d_axisPen[i];   
      s << d_axisAutoScale[i];
      s << d_axisScaleMin[i];   
      s << d_axisScaleMax[i];      
      s << d_axisScaleStep[i];         
      s << d_axisReference[i];         
      s << d_axisMaxMinor[i];         
      s << d_axisMaxMajor[i];            
      s << d_axisOptions[i];            
   }
 */
   for(i=0;i<axisCnt;++i){
      s << d_axisEnabled[i];
      s << *d_scale[i];
   }
   
   s << (int)11;
   s << d_printfit;
   s << d_printwidth;
   s << d_printheight;
   s << (int)12;
   s << d_lpos;
   s << d_lColor;
   s << d_lFont;
   s << d_cpos;
   s << d_cColor;
   s << d_cFont;
   s << d_comment;
   
   s << (int)0;
   s << (int)0;
   s << (int)0;
   
   s << curveCount();
   
   SamplinCurve *crv=firstCurve();
   while(crv!=NULL){
      s << *crv;
      crv=nextCurve();
   }
}

bool SamplinPlotData::loadTempl(const char *fn)
{
   return TRUE;   
}

bool SamplinPlotData::saveTempl(const char *fn)
{
   return TRUE;
}

bool SamplinPlotData::load(const char *fn)
{
   QString tmp;
   bool ret;
   char ver_id[16];
   Q_UINT16 chksum,nchksum;
   char *tmp_buf;
   uint len;
   
   d_fileerror="";
   tmp=filename;
   if(fn!=NULL)tmp=fn;
   
   QFile f(tmp);
   if((ret=f.open( IO_ReadOnly))==TRUE){
      QDataStream s(&f);

      f.readBlock(ver_id,strlen(GRAPH_DATA_ID));
            
      if(!strncmp(GRAPH_DATA_ID,ver_id,strlen(GRAPH_DATA_ID))){
	 // load version 1.2
	 oldload2(s);
      }
      else if(!strncmp(GRAPH_DATA_ID3,ver_id,strlen(GRAPH_DATA_ID3))){
	 // first compute & check checksum then
	 f.at(f.size()-sizeof(Q_UINT16));
	 s >> chksum;
	 //	 printf("read chksum:%i\n",chksum);
	 len=f.size()-sizeof(Q_UINT16)-strlen(GRAPH_DATA_ID3);
	 if((tmp_buf=new char[len+1])){
	    f.at(strlen(GRAPH_DATA_ID3));
	    f.readBlock(tmp_buf,len);	 
	    nchksum=qChecksum(tmp_buf,len);
	    //	    printf("file len=%i,computed chksum:%i\n",len,nchksum);
	    delete tmp_buf;

	    if(nchksum==chksum){
	       f.at(strlen(GRAPH_DATA_ID3));
	       s >> *this;
	    }
	    else {
	       d_fileerror="File "+tmp+" is corrupted.";
	       ret=FALSE;
	    }
	 }
	 else {
	    d_fileerror="Can't allocate enough memory.";
	    ret=FALSE;
	 }
      }
      else {
	 // the oldest version
	 f.at(0);
	 oldload(s);
      }
      
      f.close();
      autoRefresh();
      setSaved();
   }
   else d_fileerror="Can't open file "+tmp+" for reading";
   return(ret);
   
}

void SamplinPlotData::oldload(QDataStream &s){
   QString str;
   SamplinCurve *tmp;
   int i;
   uint cnt;
   QColor color;
   float f, begin,step;
   uint count,len;
   QArray<double> data;
   
   removeCurves();
   
   s >> str;
   setFileName(str);
   s >> str;
   setLabel(str);
   s >> str;
   setTitle(str);
   s >> str;
   setAxisTitle(2,str);
   s >> str;
   setAxisTitle(0,str);
   s >> i;
   enableAxis(yLeft);
   enableAxis(xBottom);
   enableAxis(yRight,FALSE);
   enableAxis(xTop,FALSE);
   for(i=yLeft;i<axisCnt;++i){
      d_scale[i]->setAutoScale(TRUE);
      d_scale[i]->setOptions(0);
   }
   
   //   setGrid(i);
   s >> cnt;
   while(cnt>0){
      tmp = new SamplinCurve();
      //------
      s >> str;
      tmp->setName(str);
      s >> str;
      tmp->setLegend(str);
      s >> i;
      //   series.setLayer(i);
      s >> i;
      //   series.setType(i);
      s >> i;
      s >> color;
      tmp->setPen(QPen(QColor(color),i,SolidLine));
      s >> begin;
      s >> step;
      
      s >> count;
      len=count;
      data.resize(count);
      while(count>0){
	 --count;
	 s >> f;
	 data[count]=f;
      }
      tmp->setData( begin, step, data.data(), len );
      data.resize(0);
      //------      
      insertCurve(tmp);
      --cnt;
   }
   
}

void SamplinPlotData::oldload2(QDataStream &s){

   SamplinCurve *tmp;
   int i,a;
   uint count;
   QString str;
   char b;
   double d;
   QFont f;
   QColor c;
   QPen p;
   
   removeCurves();
   s >> str;
   setFileName(str);

   s >> d_title;
   s >> d_titleFont;
   s >> d_titleColor;
   
   s >> d_plotColor;
   s >> d_panelColor;
   
   s >> d_grid;

   d_axisEnabled[yLeft]=TRUE;
   d_axisEnabled[xBottom]=TRUE;
   d_axisEnabled[yRight]=FALSE;
   d_axisEnabled[xTop]=FALSE;
   
   for(i=0;i<axisCnt;++i){
      s >> b;//d_axisEnabled[i];
      s >> str;
      d_scale[i]->setTitle(str);
      s >> f;
      s >> c;
      s >> f;
      s >> p;
      s >> b;
      s >> d;
      s >> d;
      s >> d;
      s >> d;
      s >> a;
      s >> a;
      s >> a;
      d_scale[i]->setAutoScale(TRUE);
      d_scale[i]->setOptions(0);
   }
   
   s >> i;
   if(i==11){
      s >> d_printfit;
      s >> d_printwidth;
      s >> d_printheight;
   }
   s >> i;
   s >> i;
   s >> i;
   s >> i;
   
   s >> count;
   while(count>0){
      tmp = new SamplinCurve();
      s >> *tmp;
      insertCurve(tmp);
      --count;
   }
}

bool SamplinPlotData::save(const char *fn)
{
   QString tmp;
   bool ret;
   Q_UINT16 chksum;
   
   d_fileerror="";
   tmp=filename;
   if(fn!=NULL)tmp=fn;
   
   QFile f(tmp);
   if(f.exists()==TRUE)rename(tmp,tmp+"~");
   
   if((ret=f.open( IO_WriteOnly))==TRUE){
      QDataStream s(&f);
      f.writeBlock(GRAPH_DATA_ID3,strlen(GRAPH_DATA_ID3));
      filename=tmp;

      //checksum
      QBuffer sbuf;
      sbuf.open(IO_WriteOnly);
      QDataStream str(&sbuf);
      sendData(str);
      sbuf.close();
   
      chksum=qChecksum(sbuf.buffer().data(),sbuf.buffer().size());
//      printf("len=%i,writing checksum:%i\n",sbuf.buffer().size(),chksum);
      sbuf.buffer().resize(0);   
      //end of checksum
      
      sendData(s);
      s << chksum;
      f.close();
      setSaved();
   }
   else d_fileerror="Can't open file "+tmp+" for writing.";
   return(ret);
   
}

bool SamplinPlotData::exportData(const char *fn)
{
   SamplinCurve *crv;
   QString tmp;
   bool ret;
   unsigned int i, max_len,n_ser;
   
   tmp=fn;
   d_fileerror="";
   QFile f(tmp);
   if(f.exists()==TRUE)rename(tmp,tmp+"~");
   
   if((ret=f.open( IO_WriteOnly))==TRUE){

      QTextStream s(&f);

      s << "*** SampLin exported data file***\n\n";
      s << QString("Filename: ")+fileName()+"\n";
      s << QString("Label: ")+label()+"\n";
      s << QString("Title: ")+title()+"\n\n";

      for(i=0;i<axisCnt;++i){
	 s << QString("Axis ") << i << QString(" title: ")+d_scale[i]->title()+"\n";
      }
      
      s << "\n";
      
      i=0;max_len=0;
      crv=d_curveList.first();
      while(crv){

	 s << "Series " << i <<": " << crv->name() << "; ";
	 if(crv->size()>max_len)max_len=crv->size();

	 switch(crv->dataStyle()){
	  case SamplinCurve::XandY: s << "Axis " << crv->xAxis() << ": X data; Axis " << crv->yAxis() <<  ": Y data"; break;
	  case SamplinCurve::Xonly: 
	    s << "Axis "<< crv->xAxis() << ": X data; Axis "<< crv->yAxis() << ": Y range [begin = " << crv->begin() << ", step = " << crv->step() << "]"; 
	    break;
	  case SamplinCurve::Yonly:
	    s << "Axis "<< crv->xAxis() << ": X range [begin = " << crv->begin() << ", step = " << crv->step() << "]; Axis " << crv->yAxis() << ": Y data"; 
	    break;	    
   
	 }
	 
	 s << "\n";
	 
	 ++i;
	 crv=d_curveList.next();
      }
      n_ser=i;
      
      s << "\n";
      s.width(7);
      s << "Index:";

      i=0;
      crv=d_curveList.first();
      while(crv){
	 
	 switch(crv->dataStyle()){
	  case SamplinCurve::XandY:
	    s.width(12);
	    tmp.sprintf("X(%u)",i);
	    s << tmp;
	    s.width(12);
	    tmp.sprintf("Y(%u)",i);
	    s << tmp;
	    break;
	  case SamplinCurve::Xonly: 
	    s.width(12);
	    tmp.sprintf("X(%u)",i);
	    s << tmp;
	    break;
	  case SamplinCurve::Yonly:
	    s.width(12);
	    tmp.sprintf("Y(%u)",i);
	    s << tmp;
	    break;	    
	    
	 }
	 ++i;
	 crv=d_curveList.next();
      }
      
      s << "\n";
      
      for(i=0;i<max_len;++i){

	 s.setf(QTextStream::fixed);
	 s.width(7);
	 s.precision(0);
	 s << i << " ";

	 crv=d_curveList.first();
	 while(crv){
	    
	    switch(crv->dataStyle()){
	     case SamplinCurve::XandY:
	       if(crv->size()>i){
		  tmp.sprintf("%11g ",crv->x(i));
		  s << tmp;
		  tmp.sprintf("%11g ",crv->y(i));
		  s << tmp;
	       }
	       else{
	       s.width(12);
	       s << "- ";	       
	       s.width(12);
	       s << "- ";	       		  
	       }
	       break;
	     case SamplinCurve::Xonly: 
	       if(crv->size()>i){
		  tmp.sprintf("%11g ",crv->x(i));
		  s << tmp;
	       }
	       else{
		  s.width(12);
		  s << "- ";	       
	       }
	       break;
	     case SamplinCurve::Yonly:
	       if(crv->size()>i){
		  tmp.sprintf("%11g ",crv->y(i));
		  s << tmp;
	       }
	       else{
		  s.width(12);
		  s << "- ";	       
	       }
	       break;	    
	    }
	    crv=d_curveList.next();
	 }
	 s << "\n";
      }
      
      f.close();
   }
   else d_fileerror="Can't open file "+tmp+" for writing.";
   return(ret);
}

void SamplinPlotData::setTitle(const char *t)
{
   d_title=t;
   autoRefresh();
}

void SamplinPlotData::setTitleColor(const QColor &c)
{
   d_titleColor=c;
   autoRefresh();
}

void SamplinPlotData::setTitleFont(const QFont &f)
{
   d_titleFont=f;
   autoRefresh();
}

void SamplinPlotData::setLegendColor(const QColor &c)
{
   d_lColor=c;
   autoRefresh();
}

void SamplinPlotData::setLegendFont(const QFont &f)
{
   d_lFont=f;
   autoRefresh();
}
void SamplinPlotData::setLegendPosition(char pos)
{
   d_lpos=pos;
   autoRefresh();
}
void SamplinPlotData::setCommentColor(const QColor &c)
{
   d_cColor=c;
   autoRefresh();
}
void SamplinPlotData::setCommentFont(const QFont &f)
{
   d_cFont=f;
   autoRefresh();
}
void SamplinPlotData::setCommentPosition(char pos)
{
   d_cpos=pos;
   autoRefresh();
}

int SamplinPlotData::curveIndex(uint key)
{
   SamplinCurve *crv;
   
   if((crv=findCurve(key)))
      return(d_curveList.findRef(crv));
   else return -1;
   
/*   crv=d_curveList.first();
   while(crv!=0){
      if(crv->key()==key)return index;
      crv=d_curveList.next();
      ++index;
   }
   return -1;*/
}

bool SamplinPlotData::isSaved(void)
{
   char *ptr1,*ptr2;
   int size1,size2,i;
   FILE *f1,*f2;
   
   QBuffer sbuf;
   sbuf.open(IO_WriteOnly);
   QDataStream str(&sbuf);
   sendData(str);
   sbuf.close();
   
   if(savedData==sbuf.buffer())return TRUE;
     else {
/*	ptr1=sbuf.buffer().data();
	ptr2=savedData.data();
	size1=sbuf.buffer().size();
	size2=savedData.size();
	f1=fopen("data1.cmp","wb");
	for(i=0;i<size1;++i)fputc(*(ptr1+i),f1);
	fclose(f1);
	f2=fopen("data2.cmp","wb");
	for(i=0;i<size2;++i)fputc(*(ptr2+i),f2);	
	fclose(f2);
*/	
	return FALSE;
     }
   
}

void SamplinPlotData::setSaved(void)
{
   savedData.resize(0);
   QBuffer sbuf(savedData);
   sbuf.open(IO_WriteOnly);
   QDataStream str(&sbuf);
   sendData(str);
   sbuf.close();
}

void SamplinPlotData::setGridPen(const QPen &p)
{
         d_grid.setPen(p);
         autoRefresh();
}
void SamplinPlotData::setGridMinPen(const QPen &p)
{
         d_grid.setMinPen(p);
         autoRefresh();
}

void SamplinPlotData::setGridMajPen(const QPen &p)
{
         d_grid.setMajPen(p);
         autoRefresh();
}

void SamplinPlotData::setGridAxis(int axis)
{
          if ((axis==xBottom)||(axis==xTop))
     {
	                d_grid.setXAxis(axis);
	                d_grid.setXDiv(d_scale[axis]->scaleDiv());
     }
          else if ((axis==yLeft) || (axis == yRight))
     {
	                d_grid.setYAxis(axis);
	                d_grid.setYDiv(d_scale[axis]->scaleDiv());
     }
          autoRefresh();
   
}
void SamplinPlotData::enableGridX(bool tf)
{
          d_grid.enableX(tf);
          autoRefresh();
}

void SamplinPlotData::enableGridY(bool tf)
{
          d_grid.enableY(tf);
          autoRefresh();
}

void SamplinPlotData::enableGridXMin(bool tf)
{
          d_grid.enableXMin(tf);
          autoRefresh();
}

void SamplinPlotData::enableGridYMin(bool tf)
{
          d_grid.enableYMin(tf);
          autoRefresh();
}

void SamplinPlotData::enableAxis(int axis, bool tf)
{
       if (axisValid(axis))
     {
	        d_axisEnabled[axis] = tf;
	        autoRefresh();
     }
}
bool SamplinPlotData::axisEnabled(int axis)
{
      if (axisValid(axis))
     {
	        return d_axisEnabled[axis];
     }
      return FALSE;
}

SamplinScaleBox * SamplinPlotData::axis(int axis)
{
   if (axisValid(axis))
     {
	return d_scale[axis];
     }
   return d_scale[0];
}


void SamplinPlotData::lowerCurve(uint key)
{
   SamplinCurve *c;
   int index;
   
   index=curveIndex(key);
   if (index!=-1 && (c = findCurve(key))){
      if(d_curveList.insert(d_curveList.count(),c)){
	 d_curveList.take(index);
	 autoRefresh();
      }
   }
}

void SamplinPlotData::raiseCurve(uint key)
{
   SamplinCurve *c;
   int index;
   
   index=curveIndex(key);
   if (index!=-1 && (c = findCurve(key))){
      if(d_curveList.insert(0,c)){
	 d_curveList.take(index+1);
	 autoRefresh();
      }
   }
}

int SamplinPlotData::lposition(const char *str)
{
   char **s=legendPositionNames;
   int i=0;
   
   while(*s!=0){
      if(!strcasecmp(*s,str)&&strlen(*s)==strlen(str))return i;
      ++s;++i;
   }
    
   return -1;
}
