/**************************************************************************
 * $Id: scriptlcd.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "scriptlcd.h"
#include <qbitarry.h>
#include <qpainter.h>
#include <stdio.h>

#include "scriptlcd.moc"

static QString int2string( int num, int base, int ndigits, bool *oflow )
{
    QString s;
    bool negative;
    if ( num < 0 ) {
	negative = TRUE;
	num	 = -num;
    } else {
	negative = FALSE;
    }
   switch( base ) {
	case LCDisplay::HEX:
	    s.sprintf( "%*lx", ndigits, num );
	    break;
	case LCDisplay::DEC:
	case LCDisplay::SCI:       
	    s.sprintf( "%*li", ndigits, num );
	    break;
	case LCDisplay::OCT:
	    s.sprintf( "%*lo", ndigits, num );
	    break;
	case LCDisplay::BIN: {
	    char buf[42];
	    char *p = &buf[41];
	    uint n = num;
	    int len = 0;
	    *p = '\0';
	    do {
		*--p = (char)((n&1)+'0');
		n >>= 1;
		len++;
	    } while ( n != 0 );
	    len = ndigits - len;
	    if ( len > 0 )
	    s.fill( ' ', len );
	    s += p;
	    }
	    break;
    }
    if ( negative ) {
	for ( int i=0; i<(int)s.length(); i++ ) {
	    if ( s[i] != ' ' ) {
		if ( i != 0 ) {
		    s[i-1] = '-';
		} else {
		    s.insert( 0, '-' );
		}
		break;
	    }
	}
    }
   if ( oflow ){
      *oflow = (int)s.length() > ndigits;
      if(*oflow)s="OL";
   }
   return s;
}


static QString double2string( double num, int base, int ndigits, int nddigits, bool *oflow )
{
    QString s;
   if ( base != LCDisplay::DEC && base != LCDisplay::SCI) {
	bool of = num >= 2147483648.0 || num < -2147483648.0;
	if ( of ) {				// oops, integer overflow
	    if ( oflow )
		*oflow = TRUE;
	    return s;
	}
	s = int2string( (int)num, base, ndigits, 0 );
    } else {					// decimal base
       if(base==LCDisplay::DEC){
	 s.sprintf( "%*.*f", ndigits, nddigits, num );
       }
       else{
	 s.sprintf( "%*.*E", ndigits, nddigits, num );
       }
       
	int i = s.find('e');
	if ( i > 0 && s[i+1]=='+' ) {
	    s[i] = ' ';
	    s[i+1] = 'e';
	}
    }
   if ( oflow ){
      *oflow = (int)s.length() > ndigits;
      if(*oflow)s="OL";
   }
   return s;
}


static char *getSegments( char ch )		// gets list of segments for ch
{
    static char segments[30][8] =
       { { 0, 1, 2, 4, 5, 6,99, 0},		// 0	0 / O
	 { 2, 5,99, 0, 0, 0, 0, 0},		// 1	1
	 { 0, 2, 3, 4, 6,99, 0, 0},		// 2	2
	 { 0, 2, 3, 5, 6,99, 0, 0},		// 3	3
	 { 1, 2, 3, 5,99, 0, 0, 0},		// 4	4
	 { 0, 1, 3, 5, 6,99, 0, 0},		// 5	5 / S
	 { 0, 1, 3, 4, 5, 6,99, 0},		// 6	6
	 { 0, 2, 5,99, 0, 0, 0, 0},		// 7	7
	 { 0, 1, 2, 3, 4, 5, 6,99},		// 8	8
	 { 0, 1, 2, 3, 5, 6,99, 0},		// 9	9 / g
	 { 3,99, 0, 0, 0, 0, 0, 0},		// 10	-
	 { 7,99, 0, 0, 0, 0, 0, 0},		// 11	.
	 { 0, 1, 2, 3, 4, 5,99, 0},		// 12	A
	 { 1, 3, 4, 5, 6,99, 0, 0},		// 13	B
	 { 0, 1, 4, 6,99, 0, 0, 0},		// 14	C
	 { 2, 3, 4, 5, 6,99, 0, 0},		// 15	D
	 { 0, 1, 3, 4, 6,99, 0, 0},		// 16	E
	 { 0, 1, 3, 4,99, 0, 0, 0},		// 17	F
	 { 1, 3, 4, 5,99, 0, 0, 0},		// 18	h
	 { 1, 2, 3, 4, 5,99, 0, 0},		// 19	H
	 { 1, 4, 6,99, 0, 0, 0, 0},		// 20	L
	 { 3, 4, 5, 6,99, 0, 0, 0},		// 21	o
	 { 0, 1, 2, 3, 4,99, 0, 0},		// 22	P
	 { 3, 4,99, 0, 0, 0, 0, 0},		// 23	r
	 { 4, 5, 6,99, 0, 0, 0, 0},		// 24	u
	 { 1, 2, 4, 5, 6,99, 0, 0},		// 25	U
	 { 1, 2, 3, 5, 6,99, 0, 0},		// 26	Y
	 { 8, 9,99, 0, 0, 0, 0, 0},		// 27	:
	 { 0, 1, 2, 3,99, 0, 0, 0},		// 28	'
	 {99, 0, 0, 0, 0, 0, 0, 0} };		// 29	empty

    if (ch >= '0' && ch <= '9')
	return segments[ch - '0'];
    if (ch >= 'A' && ch <= 'F')
	return segments[ch - 'A' + 12];
    if (ch >= 'a' && ch <= 'f')
	return segments[ch - 'a' + 12];

    int n;
    switch ( ch ) {
	case '-':
	    n = 10;  break;
	case 'O':
	    n = 0;   break;
	case 'g':
	    n = 9;   break;
	case '.':
	    n = 11;  break;
	case 'h':
	    n = 18;  break;
	case 'H':
	    n = 19;  break;
	case 'l':
	case 'L':
	    n = 20;  break;
	case 'o':
	    n = 21;  break;
	case 'p':
	case 'P':
	    n = 22;  break;
	case 'r':
	case 'R':
	    n = 23;  break;
	case 's':
	case 'S':
	    n = 5;   break;
	case 'u':
	    n = 24;  break;
	case 'U':
	    n = 25;  break;
	case 'y':
	case 'Y':
	    n = 26;  break;
	case ':':
	    n = 27;  break;
	case '\'':
	    n = 28;  break;
	default:
	    n = 29;  break;
    }
    return segments[n];
}


LCDisplay::LCDisplay( QWidget *parent, const char *name )
	: QFrame( parent, name )
{
    ndigits = 7;
    init();
}


LCDisplay::LCDisplay( uint numDigits, QWidget *parent, const char *name )
	: QFrame( parent, name )
{
    ndigits = numDigits;
    init();
}


void LCDisplay::init()
{
   initMetaObject();
   setFrameStyle( QFrame::Box | QFrame::Raised );
   val	       = 0;
   base       = DEC;
   smallPoint = FALSE;
   setNumDecDigits(2);
   setNumDigits( ndigits );
   setSegmentStyle( Outline );
}

LCDisplay::~LCDisplay()
{
}


void LCDisplay::setNumDigits( int numDigits )
{
    if ( numDigits > 99 ) {
#if defined(CHECK_RANGE)
	warning( "LCDisplay::setNumDigits: Max 99 digits allowed" );
#endif
	numDigits = 99;
    }
    if ( digitStr.isNull() ) {			// from constructor
	ndigits = numDigits;
	digitStr.fill( ' ', ndigits );
	points.fill( 0, ndigits );
	digitStr[ndigits - 1] = '0';		// "0" is the default number
    }
    else {
	if ( numDigits == ndigits )		// no change
	    return;
	register int i;
	int dif;
	if ( numDigits > ndigits ) {		// expand
	    dif = numDigits - ndigits;
	    QString buf;
	    buf.fill( ' ', dif );
	    digitStr.insert( 0, buf );
	    points.resize( numDigits );
	    for ( i=numDigits-1; i>=dif; i-- )
		points.setBit( i, points.testBit(i-dif) );
	    for ( i=0; i<dif; i++ )
		points.clearBit( i );
	}
	else {					// shrink
	    dif = ndigits - numDigits;
	    digitStr = digitStr.right( numDigits );
	    QBitArray tmpPoints = points.copy();
	    points.resize( numDigits );
	    for ( i=0; i<(int)numDigits; i++ )
		points.setBit( i, tmpPoints.testBit(i+dif) );
	}
	ndigits = numDigits;
	update();
    }
}

void LCDisplay::setNumDecDigits( int numDigits )
{
   if ( numDigits > 9 ) {
      numDigits = 9;
   }
   if ( digitStr.isNull() ) {			// from constructor
      ndecdigits = numDigits;
   }
   else    
   if(numDigits!=ndecdigits){
      ndecdigits=numDigits;
      display(val/*0.0*/);
   }

   return;   
}

bool LCDisplay::checkOverflow( int num ) const
{
    bool of;
    int2string( num, base, ndigits, &of );
    return of;
}

bool LCDisplay::checkOverflow( double num ) const
{
    bool of;
    double2string( num, base, ndigits, ndecdigits, &of );
    return of;
}

LCDisplay::Mode LCDisplay::mode() const
{
    return (LCDisplay::Mode) base;
}

double LCDisplay::value() const
{
    return val;
}

int LCDisplay::intValue() const
{
    return (int)(val < 0 ? val - 0.5 : val + 0.5);
}

void LCDisplay::display( int num )
{
    val = (double)num;
    bool of;
    QString s = int2string( num, base, ndigits, &of );
    if ( of )
	emit overflow();
//    else
	internalDisplay( s );
}

void LCDisplay::display( double num )
{
    val = num;
    bool of;
    QString s = double2string( num, base, ndigits, ndecdigits, &of );
    if ( of )
	emit overflow();
//    else
	internalDisplay( s );
}

void LCDisplay::display( const char *s )
{
   val = 0;
   internalDisplay( s );
}

void LCDisplay::setHexMode()
{
   setMode( HEX );
}

void LCDisplay::setDecMode()
{
   setMode( DEC );
}

void LCDisplay::setOctMode()
{
   setMode( OCT );
}

void LCDisplay::setBinMode()
{
   setMode( BIN );
}

void LCDisplay::setSciMode()
{
   setMode( SCI );
}

void LCDisplay::setMode( Mode m )
{
   base = m;
   display( 0.0 );

}

void LCDisplay::setSmallDecimalPoint( bool b )
{
   smallPoint = b;
   display( 0.0 );
}

void LCDisplay::resizeEvent( QResizeEvent * )
{
}


void LCDisplay::drawContents( QPainter *p )
{
    if ( smallPoint )
	drawString( digitStr, *p, &points, FALSE );
    else
	drawString( digitStr, *p, 0, FALSE );
}


void LCDisplay::internalDisplay( const char *s )
{
    QPainter p;
    QString buffer(ndigits+1);
    int i;
    int len = strlen(s);

    p.begin( this );
    if ( !smallPoint ) {
	if ( len >= ndigits ) {
	    for( i=0; i<(int)ndigits; i++ )	// trunc too long string
		buffer[i] = s[len - ndigits + i];
	} else {
	    for( i=0; i<ndigits-len; i++ )	// pad with spaces
		buffer[i] = ' ';
	    for( i=0; i<len; i++ )
		buffer[ndigits - len + i] = s[i];
	}
	drawString( buffer, p );
   } else {
	int  index = -1;
	bool lastWasPoint = TRUE;
	QBitArray newPoints(ndigits);
	newPoints.clearBit(0);
	for ( i=0; i<len; i++ ) {
	    if ( s[i] == '.' ) {
		if ( lastWasPoint ) {		// point already set for digit?
		    if ( index == ndigits - 1 ) // no more digits
			break;
		    index++;
		    buffer[index] = ' ';	// 2 points in a row, add space
		}
		newPoints.setBit(index);	// set decimal point
		lastWasPoint = TRUE;
	    } else {
		if ( index == ndigits - 1 )
		    break;
		index++;
		buffer[index] = s[i];
		newPoints.clearBit(index);	// decimal point default off
		lastWasPoint = FALSE;
	    }
	}
	if ( index < ((int) ndigits) - 1 ) {
	    for( i=index; i>=0; i-- ) {
		buffer[ndigits - 1 - index + i] = buffer[i];
		newPoints.setBit( ndigits - 1 - index + i,
				   newPoints.testBit(i) );
	    }
	    for( i=0; i<ndigits-index-1; i++ ) {
		buffer[i] = ' ';
		newPoints.clearBit(i);
	    }
	}
	drawString(buffer,p,&newPoints);
    }
    p.end();
}


void LCDisplay::drawString( const char *s, QPainter &p,
			     QBitArray *newPoints, bool newString )
{
    QPoint  pos;

    int digitSpace = smallPoint ? 2 : 1;
    int xSegLen	   = width()*5/(ndigits*(5 + digitSpace) + digitSpace);
    int ySegLen	   = height()*5/12;
    int segLen	   = ySegLen > xSegLen ? xSegLen : ySegLen;
    int xAdvance   = segLen*( 5 + digitSpace )/5;
    int xOffset	   = ( width() - ndigits*xAdvance + segLen/5 )/2;
    int yOffset	   = ( height() - segLen*2 )/2;

    for ( int i=0;  i<ndigits; i++ ) {
	pos = QPoint( xOffset + xAdvance*i, yOffset );
	if ( newString )
	    drawDigit( pos, p, segLen, s[i], digitStr[i] );
	else
	    drawDigit( pos, p, segLen, s[i]);
	if ( newPoints ) {
	    char newPoint = newPoints->testBit(i) ? '.' : ' ';
	    if ( newString ) {
		char oldPoint = points.testBit(i) ? '.' : ' ';
		drawDigit( pos, p, segLen, newPoint, oldPoint );
	    } else {
		drawDigit( pos, p, segLen, newPoint );
	    }
	}
    }
    if ( newString ) {
	digitStr = s;
	if ( (int)digitStr.length() > ndigits )
	    digitStr.resize( ndigits );
	if ( newPoints )
	    points = *newPoints;
    }
}


void LCDisplay::drawDigit( const QPoint &pos, QPainter &p, int segLen,
			    char newCh, char oldCh )
{
// Draws and/or erases segments to change display of a single digit
// from oldCh to newCh

    char updates[18][2];	// can hold 2 times number of segments, only
				// first 9 used if segment table is correct
    int	 nErases;
    int	 nUpdates;
    char *segs;
    int	 i,j;

    const char erase	  = 0;
    const char draw	  = 1;
    const char leaveAlone = 2;

    segs = getSegments(oldCh);
    for ( nErases=0; segs[nErases] != 99; nErases++ ) {
	updates[nErases][0] = erase;		// get segments to erase to
	updates[nErases][1] = segs[nErases];	// remove old char
    }
    nUpdates = nErases;
    segs = getSegments(newCh);
    for(i = 0 ; segs[i] != 99 ; i++) {
	for ( j=0;  j<nErases; j++ )
	    if ( segs[i] == updates[j][1] ) {	// same segment ?
		updates[j][0] = leaveAlone;	// yes, already on screen
		break;
	    }
	if ( j == nErases ) {			// if not already on screen
	    updates[nUpdates][0] = draw;
	    updates[nUpdates][1] = segs[i];
	    nUpdates++;
	}
    }
    for ( i=0; i<nUpdates; i++ ) {
	if ( updates[i][0] == draw )
	    drawSegment( pos, updates[i][1], p, segLen );
	if (updates[i][0] == erase)
	    drawSegment( pos, updates[i][1], p, segLen, TRUE );
    }
}


static void addPoint( QPointArray &a, const QPoint &p )
{
    uint n = a.size();
    a.resize( n + 1 );
    a.setPoint( n, p );
}


void LCDisplay::drawSegment( const QPoint &pos, char segmentNo, QPainter &p,
			      int segLen, bool erase )
{
    QPoint pt = pos;
    QColorGroup g = colorGroup();
    QColor lightColor,darkColor,fgColor;
    if ( erase ){
	lightColor = backgroundColor();
	darkColor  = lightColor;
	fgColor    = lightColor;
    } else {
	lightColor = g.text().light();
	darkColor  = g.text().dark();
	fgColor    = g.text(); // !!! foreground
    }
    int width = segLen/5;

#define LINETO(X,Y) addPoint( a, QPoint(pt.x() + (X),pt.y() + (Y)))
#define LIGHT 
#define DARK 

    if ( fill ) {
	QPointArray a(0);

	//The following is an exact copy of the switch below.
	//don't make any changes here
	switch ( segmentNo ) {
	case 0 :
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(segLen - 1,0);
	    DARK;
	    LINETO(segLen - width - 1,width);
	    LINETO(width,width);
	    LINETO(0,0);
	    break;
	case 1 :
	    pt.ry()++;
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,width);
	    DARK;
	    LINETO(width,segLen - width/2 - 2);
	    LINETO(0,segLen - 2);
	    LIGHT;
	    LINETO(0,0);
	    break;
	case 2 :
	    pt.rx() += (QCOORD)(segLen - 1);
	    pt.ry()++;
	    p.moveTo(pt);
	    DARK;
	    LINETO(0,segLen - 2);
	    LINETO(-width,segLen - width/2 - 2);
	    LIGHT;
	    LINETO(-width,width);
	    LINETO(0,0);
	    break;
	case 3 :
	    pt.ry() += (QCOORD)segLen;
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,-width/2);
	    LINETO(segLen - width - 1,-width/2);
	    LINETO(segLen - 1,0);
	    DARK;
	    if (width & 1) {		// adjust for integer division error
		LINETO(segLen - width - 3,width/2 + 1);
		LINETO(width + 2,width/2 + 1);
	    } else {
		LINETO(segLen - width - 1,width/2);
		LINETO(width,width/2);
	    }
	    LINETO(0,0);
	    break;
	case 4 :
	    pt.ry() += (QCOORD)(segLen + 1);
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,width/2);
	    DARK;
	    LINETO(width,segLen - width - 2);
	    LINETO(0,segLen - 2);
	    LIGHT;
	    LINETO(0,0);
	    break;
	case 5 :
	    pt.rx() += (QCOORD)(segLen - 1);
	    pt.ry() += (QCOORD)(segLen + 1);
	    p.moveTo(pt);
	    DARK;
	    LINETO(0,segLen - 2);
	    LINETO(-width,segLen - width - 2);
	    LIGHT;
	    LINETO(-width,width/2);
	    LINETO(0,0);
	    break;
	case 6 :
	    pt.ry() += (QCOORD)(segLen*2);
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,-width);
	    LINETO(segLen - width - 1,-width);
	    LINETO(segLen - 1,0);
	    DARK;
	    LINETO(0,0);
	    break;
	case 7 :
	    if ( smallPoint )	// if smallpoint place'.' between other digits
		pt.rx() += (QCOORD)(segLen + width/2);
	    else
		pt.rx() += (QCOORD)(segLen/2);
	    pt.ry() += (QCOORD)(segLen*2);
	    p.moveTo(pt);
	    DARK;
	    LINETO(width,0);
	    LINETO(width,-width);
	    LIGHT;
	    LINETO(0,-width);
	    LINETO(0,0);
	    break;
	case 8 :
	    pt.ry() += (QCOORD)(segLen/2 + width);
	    pt.rx() += (QCOORD)(segLen/2 - width/2 + 1);
	    p.moveTo(pt);
	    DARK;
	    LINETO(width,0);
	    LINETO(width,-width);
	    LIGHT;
	    LINETO(0,-width);
	    LINETO(0,0);
	    break;
	case 9 :
	    pt.ry() += (QCOORD)(3*segLen/2 + width);
	    pt.rx() += (QCOORD)(segLen/2 - width/2 + 1);
	    p.moveTo(pt);
	    DARK;
	    LINETO(width,0);
	    LINETO(width,-width);
	    LIGHT;
	    LINETO(0,-width);
	    LINETO(0,0);
	    break;
#if defined(CHECK_RANGE)
	default :
	    warning( "LCDisplay::drawSegment: Internal error."
		     "  Illegal segment id: %d\n", segmentNo );
#endif
	}
	// End exact copy
	p.setPen( fgColor );
	p.setBrush( fgColor );
	p.drawPolygon( a );
	p.setBrush( NoBrush );

	pt = pos;
    } 
#undef LINETO
#undef LIGHT
#undef DARK

#define LINETO(X,Y) p.lineTo(QPoint(pt.x() + (X),pt.y() + (Y)))
#define LIGHT p.setPen(lightColor)
#define DARK  p.setPen(darkColor)
    if ( shadow )
	switch ( segmentNo ) {
	case 0 :
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(segLen - 1,0);
	    DARK;
	    LINETO(segLen - width - 1,width);
	    LINETO(width,width);
	    LINETO(0,0);
	    break;
	case 1 :
	    pt.ry()++;
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,width);
	    DARK;
	    LINETO(width,segLen - width/2 - 2);
	    LINETO(0,segLen - 2);
	    LIGHT;
	    LINETO(0,0);
	    break;
	case 2 :
	    pt.rx() += (QCOORD)(segLen - 1);
	    pt.ry()++;
	    p.moveTo(pt);
	    DARK;
	    LINETO(0,segLen - 2);
	    LINETO(-width,segLen - width/2 - 2);
	    LIGHT;
	    LINETO(-width,width);
	    LINETO(0,0);
	    break;
	case 3 :
	    pt.ry() += (QCOORD)segLen;
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,-width/2);
	    LINETO(segLen - width - 1,-width/2);
	    LINETO(segLen - 1,0);
	    DARK;
	    if (width & 1) {		// adjust for integer division error
		LINETO(segLen - width - 3,width/2 + 1);
		LINETO(width + 2,width/2 + 1);
	    } else {
		LINETO(segLen - width - 1,width/2);
		LINETO(width,width/2);
	    }
	    LINETO(0,0);
	    break;
	case 4 :
	    pt.ry() += (QCOORD)(segLen + 1);
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,width/2);
	    DARK;
	    LINETO(width,segLen - width - 2);
	    LINETO(0,segLen - 2);
	    LIGHT;
	    LINETO(0,0);
	    break;
	case 5 :
	    pt.rx() += (QCOORD)(segLen - 1);
	    pt.ry() += (QCOORD)(segLen + 1);
	    p.moveTo(pt);
	    DARK;
	    LINETO(0,segLen - 2);
	    LINETO(-width,segLen - width - 2);
	    LIGHT;
	    LINETO(-width,width/2);
	    LINETO(0,0);
	    break;
	case 6 :
	    pt.ry() += (QCOORD)(segLen*2);
	    p.moveTo(pt);
	    LIGHT;
	    LINETO(width,-width);
	    LINETO(segLen - width - 1,-width);
	    LINETO(segLen - 1,0);
	    DARK;
	    LINETO(0,0);
	    break;
	case 7 :
	    if ( smallPoint )	// if smallpoint place'.' between other digits
		pt.rx() += (QCOORD)(segLen + width/2);
	    else
		pt.rx() += (QCOORD)(segLen/2);
	    pt.ry() += (QCOORD)(segLen*2);
	    p.moveTo(pt);
	    DARK;
	    LINETO(width,0);
	    LINETO(width,-width);
	    LIGHT;
	    LINETO(0,-width);
	    LINETO(0,0);
	    break;
	case 8 :
	    pt.ry() += (QCOORD)(segLen/2 + width);
	    pt.rx() += (QCOORD)(segLen/2 - width/2 + 1);
	    p.moveTo(pt);
	    DARK;
	    LINETO(width,0);
	    LINETO(width,-width);
	    LIGHT;
	    LINETO(0,-width);
	    LINETO(0,0);
	    break;
	case 9 :
	    pt.ry() += (QCOORD)(3*segLen/2 + width);
	    pt.rx() += (QCOORD)(segLen/2 - width/2 + 1);
	    p.moveTo(pt);
	    DARK;
	    LINETO(width,0);
	    LINETO(width,-width);
	    LIGHT;
	    LINETO(0,-width);
	    LINETO(0,0);
	    break;
#if defined(CHECK_RANGE)
	default :
	    warning( "LCDisplay::drawSegment: Internal error."
		     "  Illegal segment id: %d\n", segmentNo );
#endif
	}

#undef LINETO
#undef LIGHT
#undef DARK
}

void LCDisplay::setSegmentStyle( SegmentStyle s )
{
    fill = ( s == Flat || s == Filled );
    shadow = ( s == Outline || s == Filled );
}

LCDisplay::SegmentStyle LCDisplay::segmentStyle() const
{
    ASSERT( fill || shadow );
    if ( !fill && shadow )
	return Outline;
    if ( fill && shadow )
	return Filled;
    return Flat;
}
